//LabPlot : Dialog.cc

#include <iostream>
#include <qlabel.h>
#include <qfiledialog.h>

#include <klocale.h>
#include <kdebug.h>
#include <kfilterdev.h>
#include <kfilterbase.h>
#include <kconfig.h>
#include <kmessagebox.h>

#include "Dialog.h"
#include "Plot2DSurface.h"
#include "FileInfoDialog.h"
#include "importitems.h"
#include "filteritems.h"
#include "defs.h"

using namespace std;

Dialog::Dialog(MainWin *mw, const char *name)
	: KDialog((QWidget *)mw, name), mw(mw), p(mw->activeWorksheet()), s(mw->activeSpreadsheet())
{
	vbox = new QGroupBox(1,QGroupBox::Horizontal,QString(""),this);
	gbox = new QGroupBox(1, QGroupBox::Vertical, QString(""), this);

	ok = new KPushButton(i18n("OK"),gbox);
	apply = new KPushButton(i18n("Apply"),gbox);
	cancel = new KPushButton(i18n("Cancel"),gbox);
	QObject::connect(cancel,SIGNAL(clicked()),SLOT(reject()));
}

void Dialog::resizeEvent(QResizeEvent *e) {
//	kdDebug()<<"Dialog::resizeEvent()\n";

	QSize s = e->size();
	gbox->setGeometry(0,s.height()-50,s.width(),50);
	vbox->setGeometry(0,0,s.width(),s.height()-50);
}

//! used from dialogs for surface plots
QVBox* Dialog::surfaceStyle(QTabWidget *tw, bool fresh) {
	kdDebug()<<"Dialog::surfaceStyle()"<<endl;
	QVBox *styletab = new QVBox(tw);
	KConfig *config = mw->Config();
	config->setGroup( "Plot Surface Style" );

	//TODO : bug?
	Plot *tmpplot=0;
	if(p) {
		tmpplot = p->getPlot(p->API());
		kdDebug()<<"	Getting plot "<<p->API()<<" from worksheet"<<endl;
	}
	Plot2DSurface *plot=0;
	if (tmpplot && tmpplot->Type() == PSURFACE)
		plot = (Plot2DSurface *)tmpplot;
	else {
		kdDebug()<<"Creating new Surface Plot"<<endl;
		if(p) plot = new Plot2DSurface(p);
	}
	if(plot) kdDebug()<<"	gl->Number()="<<plot->getGraphList()->Number()<<endl;

	QHBox *hb = new QHBox(styletab);
	dcb = new QCheckBox(i18n("density"),hb);
	dcb->setChecked(fresh?config->readBoolEntry("Density Enabled",true):plot->densityEnabled());
	ccb = new QCheckBox(i18n("contour"),hb);
	ccb->setChecked(fresh?config->readBoolEntry("Contour Enabled",true):plot->contourEnabled());

	hb = new QHBox(styletab);
	new QLabel(i18n("number of level : "),hb);
	numberle = new KLineEdit(QString::number(fresh?config->readNumEntry("Contour Level",10):
		plot->Number()),hb);
	numberle->setValidator(new QIntValidator(numberle));
	numberle->setMaximumWidth(100);
	new QLabel(i18n("contour line color : "),hb);
	contourcolor = new KColorButton(fresh?config->readColorEntry("Contour Color",&Qt::black):
		plot->ContourColor(),hb);

	hb = new QHBox(styletab);
	new QLabel(i18n("Brush : "),hb);
	dbrushcb = new KComboBox(hb);
	fillBrushBox(dbrushcb,SRECT,Qt::blue,FFULL,Qt::red);
	dbrushcb->setCurrentItem(fresh?config->readNumEntry("Density Brush",1):plot->Brush());

	hb = new QHBox(styletab);
	coloredcb = new QCheckBox(i18n("colored contour lines"),hb);
	coloredcb->setChecked(fresh?config->readBoolEntry("Colored Contour",false):plot->ColoredContour());
	meshcb = new QCheckBox(i18n("show mesh"),hb);
	meshcb->setChecked(fresh?config->readBoolEntry("Show Mesh",false):plot->Mesh());
	relativecb = new QCheckBox(i18n("relative color scale"),hb);
	relativecb->setChecked(fresh?config->readBoolEntry("Relative Colorscale",true):plot->Relative());

	hb = new QHBox(styletab);
	new QLabel(i18n(" Threshold : "),hb);
	thresholdle = new KLineEdit(QString::number(fresh?config->readDoubleNumEntry("Threshold",-INF):
		plot->Threshold()),hb);
	thresholdle->setValidator(new QDoubleValidator(thresholdle));
	thresholdle->setMaximumWidth(100);

	hb = new QHBox(styletab);
	new QLabel(i18n(" Color Palette : "),hb);
	//palabel->setAlignment(Qt::AlignHCenter);
	pcb = new KComboBox(hb);
	for (int j=0;j<21;j++) {		// all palettes
		QPainter pa;
		QPixmap pm(255,25);
		pa.begin(&pm);
		pm.fill(Qt::white);

		for (int i=0;i<255;i++) {
			pa.setPen(plot->Color(i,j));
			pa.drawLine(i,0,i,25);
		}

		pa.end();
		pcb->insertItem(pm);
	}
	pcb->setCurrentItem(fresh?config->readNumEntry("Palette",0):plot->Palette());

	return styletab;
}

void Dialog::setStyle(Style *s) {
	cb2->setCurrentItem(s->Type());
	boxwidthle->setText(QString::number(s->BoxWidth()));
	autobox->setChecked(s->AutoBoxWidth());
	color->setColor(s->Color());
	widthle->setText(QString::number(s->Width()));
	pencb->setCurrentItem(s->PenStyle());
	filled->setChecked(s->isFilled());
	fcolor->setColor(s->FillColor());
	brushcb->setCurrentItem(s->Brush());
}

void Dialog::setSymbol(Symbol *s) {
	symbolcb->setCurrentItem(s->Type());
	scolor->setColor(s->Color());
	ssize->setText(QString::number(s->Size()));
	symbolfillcb->setCurrentItem(s->Fill());
	sfcolor->setColor(s->FillColor());
	sbrushcb->setCurrentItem(s->Brush());
}

//! used from dialogs for simple plots
QVBox* Dialog::simpleStyle(QTabWidget *tw, Style *style, Symbol *symbol) {
	QVBox *styletab = new QVBox(tw);
	KConfig *config = mw->Config();
	config->setGroup( "Plot Simple Style" );

	QHBox *hb = new QHBox(styletab);
	new QLabel(i18n("Graph Style : "),hb);
	cb2 = new KComboBox(hb);
	QStringList stylelist;
	stylelist<<i18n("Lines")<<i18n("NoCurve")<<i18n("Steps")<<i18n("Boxes")<<i18n("Impulses")<<i18n("Y Boxes");
	cb2->insertStringList(stylelist);
	cb2->setCurrentItem(style==0?config->readNumEntry("Graph Style",0):style->Type());

	hb = new QHBox(styletab);
	new QLabel(i18n(" Box width : "),hb);
	boxwidthle = new KLineEdit(QString::number(style==0?config->readNumEntry("Box Width",10):style->BoxWidth()),hb);
	boxwidthle->setValidator(new QIntValidator(1,1000,boxwidthle));
	autobox = new QCheckBox(i18n("auto box width "),hb);
	autobox->setChecked(style==0?config->readBoolEntry("Auto Box Width",false):style->AutoBoxWidth());
	
	hb = new QHBox(styletab);
	new QLabel(i18n("   Color : "),hb);
	color = new KColorButton(style==0?config->readColorEntry("Style Color",&Qt::blue):style->Color(),hb);
	QObject::connect(color,SIGNAL(changed(const QColor &)),this,SLOT(styleChanged()));
	hb = new QHBox(styletab);
	new QLabel(i18n(" Line Width : "),hb);
	widthle = new KLineEdit(QString::number(style==0?config->readNumEntry("Style Width",1):style->Width()),hb);
	widthle->setValidator(new QIntValidator(widthle));
	new QLabel(i18n(" Style : "),hb);
	pencb = new KComboBox(hb);
	pencb->clear();
	for (int i=0;i<6;i++) {
		QPainter pa;
		QPixmap pm( 50, 30 );
		pm.fill(Qt::white);
        	pa.begin( &pm );
		
		pa.setPen((PenStyle)i);
		pa.drawLine(5,15,45,15);
		pa.end();
		
		pencb->insertItem(pm);
	}
	pencb->setCurrentItem(style==0?config->readNumEntry("Pen Style",1):style->PenStyle());

	hb = new QHBox(styletab);
	filled = new QCheckBox(i18n("Filled "),hb);
	filled->setChecked(style==0?config->readBoolEntry("Filled",false):style->isFilled());
	fcolor = new KColorButton(style==0?config->readColorEntry("Fill Color",&Qt::green):style->FillColor(),hb);
	QObject::connect(fcolor,SIGNAL(changed(const QColor &)),this,SLOT(styleChanged()));

	hb = new QHBox(styletab);
	new QLabel(i18n("    Brush : "),hb);
	brushcb = new KComboBox(hb);
	fillBrushBox(brushcb,SRECT,Qt::blue,FFULL,fcolor->color());
	brushcb->setCurrentItem(style==0?config->readNumEntry("Brush",0):style->Brush());

	// read Symbol values from config or from symbol if defined
	SType stype = (SType) (symbol==0?config->readNumEntry("Symbol Type",SNONE):symbol->Type());
	QColor sycolor = (symbol==0?config->readColorEntry("Symbol Color",&Qt::blue):symbol->Color());
	FType sfill = (FType) (symbol==0?config->readNumEntry("Symbol Fill",FNONE):symbol->Fill());
	QColor syfillcolor = (symbol==0?config->readColorEntry("Symbol Fill Color",&Qt::red):symbol->FillColor());
	int sysize = (symbol==0?config->readNumEntry("Symbol Size",5):symbol->Size());
	int sbrush = (symbol==0?config->readNumEntry("Symbol Brush",1):symbol->Brush());
	
	hb = new QHBox(styletab);
	new QLabel(i18n("Graph Symbol : "),hb);
	symbolcb = new KComboBox(hb);
	fillSymbolBox(sycolor,sfill,syfillcolor,sbrush);
	symbolcb->setCurrentItem(stype);
	QObject::connect(symbolcb,SIGNAL(activated(int)),this,SLOT(symbolChanged()));
	hb = new QHBox(styletab);
	new QLabel(i18n("    Color : "),hb);
	scolor = new KColorButton(sycolor,hb);
	QObject::connect(scolor,SIGNAL(changed(const QColor &)),this,SLOT(symbolChanged()));
	
	hb = new QHBox(styletab);
	new QLabel(i18n("    Size : "),hb);
	ssize = new KLineEdit(QString::number(sysize),hb);
	ssize->setValidator(new QIntValidator(1,20,ssize));
	
	hb = new QHBox(styletab);
	new QLabel(i18n("    Fill : "),hb);
	symbolfillcb = new KComboBox(hb);
	fillSymbolFillBox(stype,sycolor,syfillcolor,sbrush);
	symbolfillcb->setCurrentItem(sfill);
	QObject::connect(symbolfillcb,SIGNAL(activated(int)),this,SLOT(symbolChanged()));
		
	ssize->setText(QString::number(sysize));
	sfcolor = new KColorButton(syfillcolor,hb);
	QObject::connect(sfcolor,SIGNAL(changed(const QColor &)),this,SLOT(symbolChanged()));

	hb = new QHBox(styletab);
	new QLabel(i18n("    Brush : "),hb);
	sbrushcb = new KComboBox(hb);
	fillBrushBox(sbrushcb,stype,sycolor,sfill,syfillcolor);
	sbrushcb->setCurrentItem(sbrush);
	QObject::connect(sbrushcb,SIGNAL(activated(int)),this,SLOT(symbolChanged()));

	return styletab;
}

// called from configure dialog
void Dialog::saveSimpleStyle() {
	KConfig *config = mw->Config();
	config->setGroup( "Plot Simple Style" );
	config->writeEntry("Graph Style",cb2->currentItem());
	config->writeEntry("Box Width",boxwidthle->text().toInt());
	config->writeEntry("Auto Box Width",autobox->isChecked());
	config->writeEntry("Style Color",color->color());
	config->writeEntry("Style Width",widthle->text().toInt());
	config->writeEntry("Pen Style",pencb->currentItem());
	config->writeEntry("Filled",filled->isChecked());
	config->writeEntry("Fill Color",fcolor->color());
	config->writeEntry("Brush",brushcb->currentItem());
	
	config->writeEntry("Symbol Type",symbolcb->currentItem());
	config->writeEntry("Symbol Color",scolor->color());
	config->writeEntry("Symbol Fill",symbolfillcb->currentItem());
	config->writeEntry("Symbol Fill Color",sfcolor->color());
	config->writeEntry("Symbol Brush",sbrushcb->currentItem());
}

// called from configure dialog
void Dialog::saveSurfaceStyle() {
	KConfig *config = mw->Config();
	config->setGroup( "Plot Surface Style" );
	
	config->writeEntry("Density Enabled",dcb->isChecked());
	config->writeEntry("Contour Enabled",ccb->isChecked());
	config->writeEntry("Contour Level",numberle->text().toInt());
	config->writeEntry("Contour Color",contourcolor->color());
	config->writeEntry("Density Brush",dbrushcb->currentItem());
	config->writeEntry("Colored Contour",coloredcb->isChecked());
	config->writeEntry("Show Mesh",meshcb->isChecked());
	config->writeEntry("Relative Colorscale",relativecb->isChecked());
	config->writeEntry("Threshold",thresholdle->text().toDouble());
	config->writeEntry("Palette",pcb->currentItem());
}

//! any style properties changed
void Dialog::styleChanged() {
	fillBrushBox(brushcb,SRECT,color->color(),FFULL,fcolor->color());
}
//! any symbol properties changed
void Dialog::symbolChanged() {
	fillSymbolBox(scolor->color(),(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	fillSymbolFillBox((SType)symbolcb->currentItem(),scolor->color(),sfcolor->color(),sbrushcb->currentItem());
	fillBrushBox(sbrushcb,(SType)symbolcb->currentItem(),scolor->color(),
		(FType)symbolfillcb->currentItem(),sfcolor->color());
}

QVBox* Dialog::annotateValuesTab(QTabWidget *tw, Graph *graph) {
	QVBox *annotatetab = new QVBox(tw);
	QStringList list;
	
	QHBox *hb = new QHBox(annotatetab);
	new QLabel(i18n("Type : "),hb);
	typecb = new KComboBox(hb);
	list<<i18n("None")<<i18n("x")<<i18n("y")<<i18n("z")<<i18n("x,y")<<i18n("x,y,z");
	typecb->insertStringList(list);
	if (graph)
		typecb->setCurrentItem(graph->getAnnotateValues().Type());

	hb = new QHBox(annotatetab);
	new QLabel(i18n("Position : "),hb);
	positioncb = new KComboBox(hb);
	list.clear();
	list<<i18n("top")<<i18n("bottom")<<i18n("left")<<i18n("right");
	positioncb->insertStringList(list);
	if (graph)
		positioncb->setCurrentItem(graph->getAnnotateValues().Position());

	hb = new QHBox(annotatetab);
	new QLabel(i18n("Distance : "),hb);
	QString distance("10");
	if (graph)
		distance=QString::number(graph->getAnnotateValues().Distance());
	distancele = new KLineEdit(distance,hb);
	distancele->setValidator(new QIntValidator(0,100,ssize));
	
	return annotatetab;
}

void Dialog::fillBrushBox(KComboBox *cb, SType t, QColor c,FType f, QColor s) {
	int item = cb->currentItem();
	cb->clear();
	// Qt : number of different brush styles
	int BRUSHNR=15;
	for (int i=0;i<BRUSHNR;i++) {
		QPainter pa;
		QPixmap pm( 30, 30 );
		pm.fill(Qt::white);
        	pa.begin( &pm );

		Symbol symbol((SType)t,c,10,f,s,i);
		symbol.draw(&pa,15,15);
		pa.end();

		cb->insertItem(pm);
	}
	cb->setCurrentItem(item);
}

//! fill the symbol combo box with all symbols
void Dialog::fillSymbolBox(QColor c, FType f, QColor s, int b) {
	int item = symbolcb->currentItem();
	symbolcb->clear();
	for (int i=0;i<SYMBOLNR;i++) {
		QPainter pa;
		QPixmap pm( 30, 30 );
		pm.fill(Qt::white);
        	pa.begin( &pm );
		
		Symbol symbol((SType)i,c,10,f,s,b);
		symbol.draw(&pa,15,15);
		pa.end();
		
		symbolcb->insertItem(pm);
	}
	symbolcb->setCurrentItem(item);
}

//! fill the symbol combo box with all symbols
void Dialog::fillSymbolFillBox(SType t, QColor c, QColor s, int b) {
	int item = symbolfillcb->currentItem();
	symbolfillcb->clear();
	for (int i=0;i<SYMBOLFILLNR;i++) {
		QPainter pa;
		QPixmap pm( 30, 30 );
		pm.fill(Qt::white);
        	pa.begin( &pm );
		
		Symbol symbol(t,c,10,(FType)i,s,b);
		symbol.draw(&pa,15,15);
		pa.end();
		
		symbolfillcb->insertItem(pm);
	}
	symbolfillcb->setCurrentItem(item);
}
 
void Dialog::fileInfo() {
	QStringList fns = QStringList::split(";",filele->text());
	 for ( QStringList::Iterator it = fns.begin(); it != fns.end(); ++it ) {
		QString filename = *it;
	
		QIODevice *file = KFilterDev::deviceForFile(filename,QString::null,true);
		if(file==0) file = new QFile(filename);
		
		if ( file->open( IO_ReadOnly ))
			(new FileInfoDialog(mw,0,filename))->show();
		else
			KMessageBox::error(this, i18n("Sorry. Could not open file for reading!"));
	}
}

void Dialog::selectFile() {
	QStringList f = QFileDialog::getOpenFileNames(QString(""),QString(""), this );
	if (! f.isEmpty() )
		filele->setText(f.join(";"));
}

// widget for import and data dialog
void Dialog::importWidget(QVBox *vb, QString filename, InputFilter filter) {
	QGroupBox *gb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Data Import"),vb);
	
	QHBox *hb = new QHBox(gb);
	
	filele = new KLineEdit(filename,hb);
	KPushButton *newFile = new KPushButton(i18n("Browse"),hb);
	QObject::connect(newFile,SIGNAL(clicked()),SLOT(selectFile()));
	
	hb = new QHBox(gb);
	KPushButton *infopb = new KPushButton(i18n("File Info"),hb);
	infopb->setMaximumWidth(100);
	QObject::connect(infopb,SIGNAL(clicked()),SLOT(fileInfo()));
	
	hb = new QHBox(gb);
	new QLabel(i18n("Filter : "),hb);
	filtercb = new KComboBox(hb);
	int i=0;
	while(filteritems[i] != 0) filtercb->insertItem(i18n(filteritems[i++]));
#ifndef HAVE_KEXIDB
	filtercb->removeItem(FKEXIDB);
#endif
	filtercb->setMaximumWidth(350);
	QObject::connect(filtercb,SIGNAL(activated (int)),SLOT(updateFilter(int)));
	filtercb->setCurrentItem(filter);
	
	hb = new QHBox(gb);
	simplifycb = new QCheckBox(i18n("simplify whitespaces"),hb);
	simplifycb->setChecked(true);
	emptycb = new QCheckBox(i18n("allow empty entries"),hb);
	emptycb->setChecked(false);
	hb = new QHBox(gb);
	headercb = new QCheckBox(i18n("import header"),hb);
	headercb->setChecked(false);

	hb = new QHBox(gb);
	seplabel = new QLabel(i18n("separating character : "),hb);
	sccb = new KComboBox(hb);
	sccb->setEditable(true);
	sccb->insertStrList(separatoritems);
	sccb->setMaximumWidth(100);

	hb = new QHBox(gb);
	commlabel = new QLabel(i18n("comment character : "),hb);
	commcb = new KComboBox(hb);
	commcb->setEditable(true);
	commcb->insertStrList(commentitems);
	commcb->setMaximumWidth(100);
	hb = new QHBox(gb);
	startlabel = new QLabel(i18n("Start Row : "),hb);
	startle = new KLineEdit("1",hb);
	startle->setValidator(new QIntValidator(startle));
	endlabel = new QLabel(i18n(" End Row : "),hb);
	endle = new KLineEdit(i18n("END"),hb);

	// only for binary data
	hb = new QHBox(gb);
	varlabel = new QLabel(i18n("variables : "),hb);
	varle =  new KLineEdit("2",hb);
	varle->setValidator(new QIntValidator(1,INF,varle));
	varlabel->hide();
	varle->hide();
	hb = new QHBox(gb);
	binarytypelabel = new QLabel(i18n("Format : "),hb);
        binarytypecb = new KComboBox(hb);
	binarytypelabel->hide();
	binarytypecb->hide();
	QStringList blist;
	blist<<QString("double")<<QString("float")<<QString("long");
	blist<<QString("int (8 Bit)")<<QString("int (16 Bit)")<<QString("int (32 Bit)")<<QString("int (64 Bit)");
	binarytypecb->insertStringList(blist);
	binarytypecb->setCurrentItem(0);
}

// update import settings when changing filter
void Dialog::updateFilter(int item) {
	// standard widgets (show it when previously hidden)
	filele->setReadOnly(false);
	seplabel->show();
	sccb->show();
	commcb->show();
	commlabel->show();
	startle->show();
	startlabel->show();
	endle->show();
	endlabel->show();
	simplifycb->show();
	varlabel->hide();
	varle->hide();
	binarytypelabel->hide();
	binarytypecb->hide();

	simplifycb->setChecked(true);
	emptycb->setChecked(false);
	commcb->setCurrentItem(CSHARP);
	endle->setText(i18n("END"));
	
	switch(item) {
	case FUSER:	// dont change settings
		break;
	case FTDI:
		simplifycb->setChecked(false);
		sccb->setCurrentItem(STAB);
		startle->setText("2");
		emptycb->setChecked(true);
		break;
	case FLECROY:	// from Floppy
		sccb->setCurrentItem(SCOMMA);
		startle->setText("6");
		break;
	case FLECROYXY:	// XYPairs
		sccb->setCurrentItem(SCOMMA);
		startle->setText("4");
		break;
	case FGOLDSTAR:
		sccb->setCurrentItem(SAUTO);
		startle->setText("20");
		break;
	case FBINARY:
		seplabel->hide();
		commlabel->hide();
		simplifycb->hide();
		sccb->hide();
		commcb->hide();
		
		varlabel->show();
		varle->show();
		binarytypelabel->show();
		binarytypecb->show();
		break;
	case FKEXIDB:
		//TODO : disable file linedit
		filele->setReadOnly(true);
		seplabel->hide();
		sccb->hide();
		commcb->hide();
		commlabel->hide();
		startle->hide();
		startlabel->hide();
		endle->hide();
		endlabel->hide();
		simplifycb->hide();
		emptycb->hide();
		break;
	}
}

// split a line when reading data
QStringList Dialog::splitLine(QString line,QString sep, bool empty) {
	QStringList oneline;
	
	if (sep == QString("auto"))
		oneline = QStringList::split(' ', line, empty );
	else if(sep == QString("TAB"))
		oneline = QStringList::split(QRegExp(QString("\t")), line, empty );
	else
		oneline = QStringList::split(sep.at(0), line, empty );

	return oneline;
}

// read binary value from d of type type
double Dialog::getBinaryValue(QDataStream *d, int type) {
	double value=0;
	
	switch(type) {
	case 0: {
		double var;
		*d>>var;
		value=var;
		} break;
	case 1: {
		float var;
		*d>>var;
		value=var;
		} break;
	case 2: {
		Q_LONG var;
		*d>>var;
		value=var;
		} break;
	case 3: {
		Q_INT8 var;
		*d>>var;
		value=var;
		} break;
	case 4: {
		Q_INT16 var;
		*d>>var;
		value=var;
		} break;
	case 5: {
		Q_INT32 var;
		*d>>var;
		value=var;
		} break;
	case 6: {
		Q_INT64 var;
		*d>>var;
		value=var;
		} break;
	}
	
	return value;
}
