/*=========================================================================

  Program:   VtkQt
  Module:    $RCSfile: vtkQtRenderWindowInteractor.cxx,v $
  Language:  C++
  Date:      $Date: 2002/11/12 10:52:38 $
  Version:   $Revision: 1.1.1.1 $

  Copyright (c) 2002 Denis Shamonin
  Section Computational Science
  University of Amsterdam
  Kruislaan 403, 1098 SJ Amsterdam
  the Netherlands

  E-mail        : dshamoni@science.uva.nl
  URL           : http://www.science.uva.nl/~dshamoni/
  
  All rights reserved.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkQtRenderWindowInteractor.h"
#include "vtkQtRenderWindow.h"
#include <vtkCommand.h>

#include <X11/Shell.h>
#include <X11/keysym.h>
#include <X11/X.h>
#include <X11/Xutil.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// Construct an instance so that the light follows the camera motion.
vtkQtRenderWindowInteractor::vtkQtRenderWindowInteractor(QWidget *parent, const char *name ):QWidget(parent, name)
{
  this->Top = NULL;
  this->OwnTop = 0;
  this->TopLevelShell = NULL;
  this->BreakLoopFlag = 0;
  this->TimerId = 0;
  this->InitializedWidgetSize = 0;
}

vtkQtRenderWindowInteractor::~vtkQtRenderWindowInteractor()
{
  this->Disable();
  if(this->OwnTop)
    {
     delete Top;
    }
}

void BreakXtLoop(void *iren)
{
  ((vtkQtRenderWindowInteractor*)iren)->SetBreakLoopFlag(1);
}

// This will start up the X event loop and never return. If you
// call this method it will loop processing X events until the
// application is exited.
void vtkQtRenderWindowInteractor::Start()
{
}

// Initializes the event handlers without an XtAppContext.  This is
// good for when you don't have a user interface, but you still
// want to have mouse interaction.
void vtkQtRenderWindowInteractor::Initialize()
{
  if (this->HasObserver(vtkCommand::StartEvent))
    {
    this->InvokeEvent(vtkCommand::StartEvent,NULL);
    return;
    }
    
  vtkQtRenderWindow *ren;

  // make sure we have a RenderWindow and camera
  if ( ! this->RenderWindow)
    {
    vtkErrorMacro(<<"No renderer defined!");
    return;
    }

  this->Initialized = 1;
  ren = (vtkQtRenderWindow *)(this->RenderWindow);
  
  ren->SetDisplayId(qt_xdisplay());
  ren->SetWindowId(this->winId());
  int *WidgetPos = ren->GetPosition();
  int *WidgetSize = ren->GetSize();
  this->move(WidgetPos[0],WidgetPos[1]);
    
  this->resize(WidgetSize[0],WidgetSize[1]);
  char *WindowName = ren->GetWindowName();
  this->setCaption(QString(WindowName));
  ren->Start();
  this->Enable();
}

void vtkQtRenderWindowInteractor::Enable()
{
  // avoid cycles of calling Initialize() and Enable()
  if (this->Enabled)
    {
    return;
    }
    
  this->setEnabled (TRUE);  
  this->Enabled = 1;
  this->Modified();
}

void vtkQtRenderWindowInteractor::Disable()
{
  if (!this->Enabled)
    {
    return;
    }
  this->setEnabled (FALSE);     
  this->Enabled = 0;
  this->Modified();
}


void vtkQtRenderWindowInteractor::PrintSelf(ostream& os, vtkIndent indent)
{ 
  this->Superclass::PrintSelf(os,indent);
}

void vtkQtRenderWindowInteractorTimer(XtPointer client_data,
                                     XtIntervalId *vtkNotUsed(id))
{
  vtkQtRenderWindowInteractor *me;
  me = (vtkQtRenderWindowInteractor *)client_data;

  if (me->Enabled) 
    {
    me->InvokeEvent(vtkCommand::TimerEvent,NULL);
    }
}

int vtkQtRenderWindowInteractor::CreateTimer(int vtkNotUsed(timertype)) 
{
  this->AddTimeOut(10);
  return 1;
}

int vtkQtRenderWindowInteractor::DestroyTimer(void) 
{
  this->killTimers();
  return 1;
}

int vtkQtRenderWindowInteractor::AddTimeOut(unsigned long interval)
{
  this->TimerId = startTimer(interval);
  return this->TimerId;
}

void vtkQtRenderWindowInteractor::GetMousePosition(int *x, int *y)
{
  Window root,child;
  int root_x,root_y;
  unsigned int keys;

  XQueryPointer(qt_xdisplay(),this->winId(),
                &root,&child,&root_x,&root_y,x,y,&keys);

  *y = this->Size[1] - *y - 1;
}

void vtkQtRenderWindowInteractor::Render()
{
  if (this->RenderWindow) 
    {
    this->RenderWindow->Render(); 
    }
}

void vtkQtRenderWindowInteractor::timerEvent( QTimerEvent *e )
{
 if ( e->timerId() == this->TimerId ) 
   {
    if (this->Enabled) 
      {
      this->InvokeEvent(vtkCommand::TimerEvent,NULL);
      }
   }
}

void vtkQtRenderWindowInteractor::SetRenderWindow(vtkQtRenderWindow *aren)
{
  if (this->RenderWindow != aren)
    {
    // to avoid destructor recursion
    vtkRenderWindow *temp = this->RenderWindow;
    this->RenderWindow = aren;
    if (temp != NULL)
      {
      temp->UnRegister(this);
      }
    if (this->RenderWindow != NULL)
      {
      this->RenderWindow->Register(this);
      if (this->RenderWindow->GetInteractor() != this)
        {
        this->RenderWindow->SetInteractor(this);
        }
      this->Initialize();
      }
    }
}

bool vtkQtRenderWindowInteractor::x11Event ( XEvent * event)
{
  if (this->Initialized)
     vtkQtRenderWindowInteractorCallback(NULL,this,event,NULL);
  return FALSE; 
}

void vtkQtRenderWindowInteractorCallback(Widget vtkNotUsed(w),
                                        XtPointer client_data,
                                        XEvent *event,
                                        Boolean *vtkNotUsed(ctd))
{
  vtkQtRenderWindowInteractor *me;
  int xp, yp;

  me = (vtkQtRenderWindowInteractor *)client_data;
  
  switch (event->type) 
    {
    case Expose:
      {
      XEvent result;
      while (XCheckTypedWindowEvent(qt_xdisplay(), 
                                    me->winId(),
                                    Expose, 
                                    &result))
        {
        // just getting the expose configure event
        event = &result;
        }
      int width = (reinterpret_cast<XConfigureEvent *>(event))->width;
      int height = (reinterpret_cast<XConfigureEvent *>(event))->height;
      me->SetEventSize(width, height);
      xp = (reinterpret_cast<XButtonEvent*>(event))->x;
      yp = (reinterpret_cast<XButtonEvent*>(event))->y;
      yp = me->Size[1] - xp - 1;
      me->SetEventPosition(xp, yp);
      // only render if we are currently accepting events
      if (me->GetEnabled())
        {
        me->InvokeEvent(vtkCommand::ExposeEvent,NULL);
        me->GetRenderWindow()->Render();
        }
      }
      break;

    case MapNotify:
      {
      // only render if we are currently accepting events
      if (me->GetEnabled() && me->GetRenderWindow()->GetNeverRendered())
        {
        me->GetRenderWindow()->Render();
        }
      }
      break;

    case ConfigureNotify: 
      {
      XEvent result;
      while (XCheckTypedWindowEvent(qt_xdisplay(), me->winId(),
                                    ConfigureNotify, &result))
        {
        // just getting the last configure event
        event = &result;
        }
      int width = (reinterpret_cast<XConfigureEvent *>(event))->width;
      int height = (reinterpret_cast<XConfigureEvent *>(event))->height;
      if (width != me->Size[0] || height != me->Size[1])
        {
        me->UpdateSize(width, height);
        xp = (reinterpret_cast<XButtonEvent*>(event))->x;
        yp = (reinterpret_cast<XButtonEvent*>(event))->y;
        me->SetEventPosition(xp, me->Size[1] - yp - 1);
        // only render if we are currently accepting events
        if (me->GetEnabled())
          {
          me->InvokeEvent(vtkCommand::ConfigureEvent,NULL);
          me->GetRenderWindow()->Render();
          }
        }
      }
      break;
            
    case ButtonPress: 
      {
      if (!me->Enabled) 
        {
        return;
        }
      int ctrl = 
        (reinterpret_cast<XButtonEvent *>(event))->state & ControlMask ? 1 : 0;
      int shift =
        (reinterpret_cast<XButtonEvent *>(event))->state & ShiftMask ? 1 : 0;
      xp = (reinterpret_cast<XButtonEvent*>(event))->x;
      yp = (reinterpret_cast<XButtonEvent*>(event))->y;
      me->SetEventInformationFlipY(xp, 
                                   yp,
                                   ctrl, 
                                   shift);
      switch ((reinterpret_cast<XButtonEvent *>(event))->button)
        {
        case Button1:  
          me->InvokeEvent(vtkCommand::LeftButtonPressEvent,NULL);
          break;
        case Button2: 
          me->InvokeEvent(vtkCommand::MiddleButtonPressEvent,NULL);
          break;
        case Button3: 
          me->InvokeEvent(vtkCommand::RightButtonPressEvent,NULL);
          break;
        }
      }
      break;
      
    case ButtonRelease: 
      {
      if (!me->Enabled) 
        {
        return;
        }
      int ctrl = 
        (reinterpret_cast<XButtonEvent *>(event))->state & ControlMask ? 1 : 0;
      int shift =
        (reinterpret_cast<XButtonEvent *>(event))->state & ShiftMask ? 1 : 0;
      xp = (reinterpret_cast<XButtonEvent*>(event))->x;
      yp = (reinterpret_cast<XButtonEvent*>(event))->y; 
      me->SetEventInformationFlipY(xp, 
                                   yp,
                                   ctrl, 
                                   shift);
      switch ((reinterpret_cast<XButtonEvent *>(event))->button)
        {
        case Button1: 
          me->InvokeEvent(vtkCommand::LeftButtonReleaseEvent,NULL);
          break;
        case Button2: 
          me->InvokeEvent(vtkCommand::MiddleButtonReleaseEvent,NULL);
          break;
        case Button3: 
          me->InvokeEvent(vtkCommand::RightButtonReleaseEvent,NULL);
          break;
        }
      }
      break;

    case EnterNotify:
      {
      // Force the keyboard focus to be this render window
      if (me->TopLevelShell != NULL)
        {
        //XtSetKeyboardFocus(me->TopLevelShell, me->Top);
        }
      if (me->Enabled)
        {
        XEnterWindowEvent *e = reinterpret_cast<XEnterWindowEvent *>(event);
        me->SetEventInformationFlipY(e->x, 
                                     e->y,
                                     (e->state & ControlMask) != 0, 
                                     (e->state & ShiftMask) != 0);
        me->InvokeEvent(vtkCommand::EnterEvent, NULL);
        }
      }
      break;

    case LeaveNotify:
      {
      if (me->Enabled)
        {
        XLeaveWindowEvent *e = reinterpret_cast<XLeaveWindowEvent *>(event);
        me->SetEventInformationFlipY(e->x, 
                                     e->y,
                                     (e->state & ControlMask) != 0, 
                                     (e->state & ShiftMask) != 0); 
        me->InvokeEvent(vtkCommand::LeaveEvent, NULL);
        }
      }
      break;

    case KeyPress:
      {
      if (!me->Enabled) 
        {
        return;
        }
      int ctrl = 
        (reinterpret_cast<XButtonEvent *>(event))->state & ControlMask ? 1 : 0;
      int shift =
        (reinterpret_cast<XButtonEvent *>(event))->state & ShiftMask ? 1 : 0;
      KeySym ks;
      static char buffer[20];
      buffer[0] = '\0';
      XLookupString(reinterpret_cast<XKeyEvent *>(event),buffer, 20, &ks,NULL);
      xp = (reinterpret_cast<XKeyEvent*>(event))->x;
      yp = (reinterpret_cast<XKeyEvent*>(event))->y;
      me->SetEventInformationFlipY(xp, 
                                   yp,
                                   ctrl, 
                                   shift, 
                                   buffer[0], 
                                   1, 
                                   buffer);
      me->InvokeEvent(vtkCommand::KeyPressEvent, NULL);
      me->InvokeEvent(vtkCommand::CharEvent, NULL);
      }
      break;      
      
    case KeyRelease:
      {
      if (!me->Enabled) 
        {
        return;
        }
      int ctrl = 
        (reinterpret_cast<XButtonEvent *>(event))->state & ControlMask ? 1 : 0;
      int shift =
        (reinterpret_cast<XButtonEvent *>(event))->state & ShiftMask ? 1 : 0;
      KeySym ks;
      static char buffer[20];
      buffer[0] = '\0';
      XLookupString(reinterpret_cast<XKeyEvent *>(event),buffer, 20, &ks,NULL);
      xp = (reinterpret_cast<XKeyEvent *>(event))->x;
      yp = (reinterpret_cast<XKeyEvent *>(event))->y;
      me->SetEventInformationFlipY(xp, 
                                   yp,
                                   ctrl, 
                                   shift, 
                                   buffer[0], 
                                   1, 
                                   XKeysymToString(ks));
      me->InvokeEvent(vtkCommand::KeyReleaseEvent, NULL);
      }
      break;      
      
    case MotionNotify: 
      {
      // Let's Qt to care about this with mouseMoveEvent(QMouseEvent* e)
      }
      break;
    }
}

void vtkQtRenderWindowInteractor::mouseMoveEvent(QMouseEvent* e)
{
  if (!this->Enabled)
  return;
      
  int state = e->state();
  int ctrl = 0;
  if (state & ControlButton)
    ctrl = 1;
  int shift = 0;
  if (state & ShiftButton)
    shift = 1;
			
  int xp = e->x();
  int yp = e->y();
  yp = Size[1] - yp - 1;
  this->SetEventInformation(xp, yp, ctrl, shift);
  this->InvokeEvent(vtkCommand::MouseMoveEvent, NULL);
}

