/***************************************************************************
 *
 *  $Id: kzenartist.cpp,v 1.14 2005/06/02 07:09:01 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <qpainter.h>
#include <qpixmap.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>

#include "kzenconfig.h"
#include "kzentrack.h"
#include "kzenalbum.h"
#include "kzenartist.h"

QStringList KZenArtist::fArtists;

KZenArtist::KZenArtist(const char * name)
{
  setName(name);

  fTracks = new QPtrList<KZenTrack>(); 
  fTracks->setAutoDelete(false);
  
  fAlbums = new QPtrList<KZenAlbum>();
  fAlbums->setAutoDelete(true);
}

KZenArtist::~KZenArtist()
{
  delete fTracks;
  delete fAlbums;
}

void KZenArtist::setName(const char * name)
{
  fName = name;

  if (fName==i18n("Various Artists")) return;

  if (!KZenArtist::getArtists().contains(name)) {
    KZenArtist::getArtists().append(name);
    KZenArtist::getArtists().sort();
  }
}

void KZenArtist::addTrack(KZenTrack * track)
{
  fTracks->append(track);

  KZenAlbum * temp = 0;
  QString album = track->getAlbum();
  if (album.isNull() || album.isEmpty()) {
    album = i18n("Unknown Album");
    track->setAlbum(album);
  }
  
  for (uint a=0;a<fAlbums->count();a++) {
    temp = fAlbums->at(a);
    if (album==temp->getName()) break;
    temp = 0;
  }
    
  if (!temp) {
    temp = new KZenAlbum(album);
    temp->setArtist(this);
    fAlbums->append(temp);
  }
  
  temp->addTrack(track);
  track->setZenAlbum(temp);
}

QStringList & KZenArtist::getArtists()
{
  return fArtists;
}

uint KZenArtist::getTrackCount()
{
  return fTracks->count();
}

uint KZenArtist::getAlbumCount()
{
  return fAlbums->count();
}

uint KZenArtist::getLength()
{
  int length = 0;
  KZenTrack * track;
  for (uint tr=0;tr<fTracks->count();tr++) {
    track = fTracks->at(tr);
    length += track->getLength();
  }
  
  return length;
}

KZenArtistItem::KZenArtistItem(QListView * parent, KZenArtist * Artist)
  :KZenItem(parent,KZenItem::ItemTypeArtist)
{
  fArtist = Artist;

  setPixmap(0, UserIcon("artist", 16));
  setText(0,fArtist->getName());

  fSortColumn = KZen::FieldAlbum;
  fSortOrder = Qt::Ascending;
}

KZenArtistItem::~KZenArtistItem()
{

}

void KZenArtistItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
  bool sd = KZenConfig::get()->showDetailsArtist();

  QFontMetrics fml(listView()->font());
  int margin = listView()->itemMargin()*2;
  int h = fml.lineSpacing();
  if (sd) h += fml.lineSpacing()*2;
  if (pixmap(0)->height()>h) h = pixmap(0)->height();
  if (h % 2 > 0) h++;
  
  setHeight(h + margin + 2);

  KListView *lv = (KListView *)listView();

  static QPixmap buffer;
  buffer.resize(width, height());

  if(buffer.isNull()) {
    KListViewItem::paintCell(p, cg, column, width, align);
    return;
  }
  
  QPainter pBuf(&buffer, true);
  if (isSelected()) {
    pBuf.fillRect(buffer.rect(),cg.highlight());
  } else {
    pBuf.fillRect(buffer.rect(),KZen::backgroundColor(this));
  }

  QFont font(p->font());
  QFontMetrics fm(p->fontMetrics());

  QRect rect(((lv->treeStepSize()-9) / 2) + 1, (height()-9) / 2, 9, 9);
  pBuf.setPen(cg.mid());
  pBuf.drawRect(rect);
  //fill the rect with base color if the item has alternate color and viceversa
  QColor color = KZen::backgroundColor(this) == lv->alternateBackground() ? cg.base() : lv->alternateBackground();
  pBuf.fillRect(rect.x()+1, rect.y()+1, rect.width()-2, rect.height()-2, color);
  // +/- drawing
  pBuf.setPen(cg.text());
  pBuf.drawLine(rect.x()+2, rect.y()+4, rect.x()+6, rect.y()+4);
  if(!isOpen())
    pBuf.drawLine(rect.x()+4, rect.y()+2, rect.x()+4, rect.y()+6);

  int text_x = pixmap(0)->width() + 26;
  int text_y = 1;
  int textHeight = fm.lineSpacing() + lv->itemMargin() + 1;

  pBuf.drawPixmap(20, (height() - pixmap(0)->height())/2,
		  *pixmap(0));
  
  if (isSelected()) {
    pBuf.setPen(cg.highlightedText());
  } else {
    pBuf.setPen(cg.text());
  }

  if (sd) font.setBold(true);
  pBuf.setFont(font);
  QFontMetrics fmName(font);

  QString name = text(0);

  if(fmName.width(name) + text_x + lv->itemMargin()*2 > width ) {
    int ellWidth = fmName.width(i18n("..."));
    QString text = QString::fromLatin1("");
    int i = 0;
    int len = name.length();
    while ( i < len && fmName.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
      text += name[i];
      i++;
    }
    name = text + i18n("...");
  }

  pBuf.drawText(text_x, text_y, width, textHeight, AlignVCenter, name);

  if (sd) {
    font.setBold(false);
    pBuf.setFont(font);
    QFontMetrics fmInfo(font);

    QString info;
    info += i18n("1 Album", "%n Albums", fArtist->getAlbumCount());
        
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }
    pBuf.drawText(text_x, text_y + textHeight, width, fm.lineSpacing(), AlignVCenter, info);

    font.setItalic(true);
    pBuf.setFont(font);
    fmInfo = QFontMetrics(font);

    info = "";
    info += i18n("1 Track", "%n Tracks", fArtist->getTrackCount());
    info += QString(i18n(" - [%2]")).arg(KZen::formatTime(fArtist->getLength()));
    
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }
    pBuf.drawText(text_x, text_y + textHeight*2, width, fm.lineSpacing(), AlignVCenter, info);
  }
  
  pBuf.end();
  p->drawPixmap(0, 0, buffer);
}
