/***************************************************************************
 *
 *  $Id: dirselector.cpp,v 1.1 2005/06/28 18:54:30 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <kdebug.h>

#include "kzenconfig.h"

#include "dirselector.h"

DirSelector::DirSelector(QWidget * parent, const char * name)
  :QListView(parent, name)
{
  addColumn(QString::null);
  setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
  setResizeMode(QListView::LastColumn);
  reinterpret_cast<QWidget*>(header())->hide();
}

DirSelector::~DirSelector()
{

}

void DirSelector::setDirectories(const QStringList & l)
{
  fDirectories = l;
}

void DirSelector::setTopDir(const KURL & url) 
{ 
  if (fTopDir!=url) {
    fDirectories.clear();
  }
  fTopDir = url; 
}

void DirSelector::refresh()
{
  kdDebug() << "DirSelector::refresh" << endl;
  
  clear();
  new DirSelectorItem(this, fTopDir);
}

DirSelectorItem::DirSelectorItem(QListView * parent, const KURL & url)
  :QCheckListItem(parent, url.path(), QCheckListItem::CheckBox)
  ,fDirLister(true)
  ,fURL(url)
  ,fListed(false)
{
  fDirLister.setDirOnlyMode(true);

  connect(&fDirLister, SIGNAL(newItems(const KFileItemList &)), this, SLOT(newItems(const KFileItemList &)));
  setOpen(true);
  setVisible(true);
}

DirSelectorItem::DirSelectorItem(QListViewItem * parent, const KURL & url)
  :QCheckListItem(parent, url.fileName(), QCheckListItem::CheckBox)
  ,fDirLister(true)
  ,fURL(url)
  ,fListed(false)
{
  fDirLister.setDirOnlyMode(true);

  setExpandable(true);
  connect(&fDirLister, SIGNAL(newItems(const KFileItemList &)), this, SLOT(newItems(const KFileItemList &)));
  connect(&fDirLister, SIGNAL(completed()), this, SLOT(completed()));
  connect(&fDirLister, SIGNAL(canceled()), this, SLOT(completed()));
}

QCheckListItem * DirSelectorItem::parent() const
{
  return (QCheckListItem*) QListViewItem::parent();
}

bool DirSelectorItem::isDisabled() const
{
  return parent() && parent()->isOn();
}

void DirSelectorItem::setOpen(bool b)
{
  if (!fListed) {
    fDirLister.openURL(fURL, true);
    fListed = true;
  }
  
  QListViewItem::setOpen(b);
}

void DirSelectorItem::stateChange(bool b)
{
  for(QListViewItem *item=firstChild();item;item=item->nextSibling()) {
    static_cast<QCheckListItem*>(item)->QCheckListItem::setOn(b);
  }

  DirSelector * dirs = (DirSelector*) listView();
  QStringList::Iterator it = dirs->directories().find(fURL.path());
  if (isOn()) {
    if (it==dirs->directories().end()) {
      dirs->directories() << fURL.path();
    }
  } else {   
    dirs->directories().erase(it);
  }
  
  listView()->triggerUpdate();
}

void DirSelectorItem::activate()
{
  if(!isDisabled())
    QCheckListItem::activate();
}

void DirSelectorItem::newItems(const KFileItemList & list)
{
  DirSelector * dirs = (DirSelector*) listView();

  for (KFileItemListIterator it(list);*it;++it) {
    KURL url = (*it)->url();
    DirSelectorItem *item = new DirSelectorItem(this, url);

    item->setOn(isOn() || 
		dirs->directories().contains(item->url().path()));
  }
}

void DirSelectorItem::paintCell(QPainter * p, const QColorGroup & cg, int column, int width, int align)
{
  DirSelector * dirs = (DirSelector*) listView();

  bool dirty = false;
  
  for (uint i=0;i<dirs->directories().count();i++) {
    if (dirs->directories()[i].startsWith(fURL.path())) dirty = true;
  }
  
  QColorGroup _cg = cg;
  if (dirty) _cg.setColor(QColorGroup::Text, listView()->colorGroup().link());

  QCheckListItem::paintCell(p, isDisabled() ? listView()->palette().disabled() : _cg, column, width, align);
}

void DirSelectorItem::completed() 
{
  if (childCount()==0) { 
    setExpandable(false); 
    repaint(); 
  } 
}

#include "dirselector.moc"
