/***************************************************************************
 *   Copyright (C) 2005 by Roland Weigert   *
 *   roweigert@t-online.de   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ripper.h"

extern databasewrapper *database;
extern rcdatei settingsfile;

ripperdialog::ripperdialog(QWidget *pointer)
{
    //define sizes fo widgets
    coverdialog=NULL;
    QSize combosize(300,30);
    QSize spinboxsize(80,30);
    //make new Groupboxes for the layout
    ripperlayout=new QGroupBox(5,Qt::Vertical,pointer);
    multitagbox=new QGroupBox(4,Qt::Horizontal,ripperlayout);
    multitagbox->setTitle(i18n("Album information"));
    //artist-widgets
    artistlabel=new QLabel(i18n("Artist"),multitagbox);
    artistbox=new QComboBox(TRUE,multitagbox);
    artistbox->setAutoCompletion(settingsfile.autocomplete);
    artistbox->setDuplicatesEnabled(FALSE);
    artistbox->setMinimumSize(combosize);
    //albumwidgets
    albumlabel=new QLabel(i18n("Album"),multitagbox);
    albumbox=new QComboBox(TRUE,multitagbox);
    albumbox->setAutoCompletion(settingsfile.autocomplete);
    albumbox->setDuplicatesEnabled(FALSE);
    albumbox->setMinimumSize(combosize);
    //genrewidgets
    genrelabel=new QLabel(i18n("Genre"),multitagbox);
    genrebox=new QComboBox(TRUE,multitagbox);
    genrebox->setAutoCompletion(settingsfile.autocomplete);
    genrebox->setDuplicatesEnabled(FALSE);
    genrebox->setMinimumSize(combosize);
    if(settingsfile.databasetype)
    {
        //fill all comboboxes,if we are using mysqldatabase
        database->fillCombo(genrebox,"genres","genrename");
        database->fillCombo(albumbox,"albums","albumname");
        database->fillCombo(artistbox,"artists","artistname");
    }
    //yearwidgets
    yearlabel=new QLabel(i18n("Releaseyear"),multitagbox);
    yearbox=new QSpinBox(multitagbox);
    yearbox->setMinimumSize(spinboxsize);
    yearbox->setMaximumSize(spinboxsize);
    yearbox->setRange(0,9999);
    //fill songbox with cd-contents
    cdbox=new QGroupBox(2,Qt::Horizontal,ripperlayout);
    songbox=new QGroupBox(1,Qt::Horizontal,cdbox);
    songbox->setTitle(i18n("Songinformation"));
    //initialize every pointer with NULL
    songs=NULL;
    scrollarea=NULL;
    selectallbutton=NULL;
    cd=NULL;
    scrollarea=new QScrollView(songbox);
    songs=new albumdialog(scrollarea,cd);//generate an empty songwidget conaining only text
    //Cd-Cover-Buttons
    coverbox=new QGroupBox(1,Qt::Horizontal,cdbox);
    coverbox->setTitle(i18n("CD-Cover"));
    coverlabel=new QLabel(coverbox);
    coverlabel->setText(i18n("no cover set"));
    loadcoverbutton=new QPushButton(i18n("Select Cover"),coverbox);
    removecoverbutton=new QPushButton(i18n("Remove Cover"),coverbox);
    findcoverbutton=new QPushButton(i18n("Search for Cover"),coverbox);
    actionbox=new QGroupBox(2,Qt::Horizontal,ripperlayout);
    startbutton=new QPushButton(i18n("Start ripping"),actionbox);
    refreshbutton=new QPushButton(i18n("Refresh CD-info"),actionbox);
    startbutton->setEnabled(FALSE);
    connect(artistbox,SIGNAL(textChanged(const QString &)),SLOT(artistChanged()));
    connect(albumbox,SIGNAL(textChanged(const QString &)),SLOT(albumChanged()));
    connect(genrebox,SIGNAL(textChanged(const QString &)),SLOT(genreChanged()));
    connect(refreshbutton,SIGNAL(clicked()),SLOT(refreshcdinfoSlot()));
    connect(startbutton,SIGNAL(clicked()),SLOT(readinDiscSlot()));
    connect(loadcoverbutton,SIGNAL(clicked()),this,SLOT(loadpicture()));
    connect(removecoverbutton,SIGNAL(clicked()),this,SLOT(removecoverslot()));
    connect(findcoverbutton,SIGNAL(clicked()),this,SLOT(findcoverslot()));
}

ripperdialog::~ripperdialog()
{
    cerr<<"ripper destructor"<<endl;
    if (songs)
        delete songs;
    if (cd)
        delete cd;
    if (scrollarea)
        delete scrollarea;
    delete artistlabel;
    delete albumlabel;
    delete yearlabel;
    delete genrelabel;
    delete loadcoverbutton;
    delete coverlabel;
    delete removecoverbutton;
    delete findcoverbutton;
    delete yearbox;
    delete startbutton;
    delete refreshbutton;
    if (selectallbutton)
        delete selectallbutton;
    delete artistbox;
    delete albumbox;
    delete genrebox;
    delete actionbox;
    delete multitagbox;
    delete coverbox;
    delete songbox;
    delete cdbox;
    delete ripperlayout;
}

void ripperdialog::databasechangeslot()
{
    //deactivate all connections to the combos, so they don't realise the change
    disconnect(artistbox,SIGNAL(textChanged(const QString &)),this,SLOT(artistChanged()));
    disconnect(albumbox,SIGNAL(textChanged(const QString &)),this,SLOT(albumChanged()));
    disconnect(genrebox,SIGNAL(textChanged(const QString &)),this,SLOT(genreChanged()));
    if (settingsfile.databasetype)
    {
        database->fillCombo(genrebox,"genres","genrename");
        database->fillCombo(albumbox,"albums","albumname");
        database->fillCombo(artistbox,"artists","artistname");
    }
    else
    {
        //only clear boxes, and put back old values
        QString oldartist=artistbox->currentText();
        QString oldalbum=albumbox->currentText();
        QString oldgenre=genrebox->currentText();
        artistbox->clear();
        albumbox->clear();
        genrebox->clear();
        artistbox->setCurrentText(oldartist);
        albumbox->setCurrentText(oldalbum);
        genrebox->setCurrentText(oldgenre);
    }
    //reactivate all slots
    connect(artistbox,SIGNAL(textChanged(const QString &)),SLOT(artistChanged()));
    connect(albumbox,SIGNAL(textChanged(const QString &)),SLOT(albumChanged()));
    connect(genrebox,SIGNAL(textChanged(const QString &)),SLOT(genreChanged()));
}

void ripperdialog::loadpicture()
{
    QString coverfile=QFileDialog::getOpenFileName(QDir::homeDirPath(),i18n("pictures (*.jpg)"),this,i18n("load file"),i18n("please choose cover-picture"));
    frontcover.load(coverfile);
    QImage scaledcover;
    scaledcover=frontcover;
    scaledcover=scaledcover.smoothScale(150,150,QImage::ScaleMin);
    coverlabel->setPixmap(scaledcover);
    if (!frontcover.isNull())
        removecoverbutton->setEnabled(TRUE);
}

void ripperdialog::removecoverslot()
{
    frontcover.resize(0,0);
    coverlabel->setText(i18n("no cover set"));
    removecoverbutton->setEnabled(FALSE);
}

void ripperdialog::imageDropped(QImage droppedimage)
{
    frontcover=droppedimage;
    QImage scaledcover;
    scaledcover=frontcover;
    scaledcover=scaledcover.smoothScale(150,150,QImage::ScaleMin);
    coverlabel->setPixmap(scaledcover);
}

void ripperdialog::coverabort()
{
    delete coverdialog;
    coverdialog=NULL;
}

void ripperdialog::findcoverslot()
{
    //Create a findstring, out of the already found tags
   if (!(artistbox->currentText().length()==0)||!(albumbox->currentText().length()==0))
    {
        QString findstring=QString("%1+%2").arg(artistbox->currentText(),albumbox->currentText());
        findstring=findstring.replace(' ','+');
        coverdialog=new coverfinder2(findstring);
        coverdialog->show();
        connect (coverdialog,SIGNAL(pictureselected(QImage)),SLOT(coverupdate(QImage)));
        connect (coverdialog,SIGNAL(abort()),SLOT(coverabort()));
    }
    else
    QMessageBox::information(0,i18n("Missing information"),i18n("No values to search for!"),QMessageBox::Ok);
}

void ripperdialog::coverupdate(QImage newcover)
{
    if (coverdialog)
    {
        delete coverdialog;
        coverdialog=NULL;
    }
    frontcover=newcover;
    QImage scaledcover;
    scaledcover=frontcover;
    scaledcover=scaledcover.smoothScale(150,150,QImage::ScaleMin);
    coverlabel->setPixmap(scaledcover);
    removecoverbutton->setEnabled(TRUE);
}

void ripperdialog::refreshcdinfoSlot()
{
    selectall=FALSE;
    progressmainwindow *progresswindow=new progressmainwindow(FALSE);
    progresswindow->show();
    frontcover.resize(0,0);//delete old cover
    coverlabel->setText(i18n("no cover set"));
    progressinfoonly *progress=progresswindow->startInfoOnly(i18n("Reading Disc-Info"));
    progress->setText(i18n("Checking for CD"));
    if (songs)
        delete songs;
    if (scrollarea)
        delete scrollarea;
    if (selectallbutton)
        delete selectallbutton;
    if (cd)
        delete cd;
    cd=new compactdisc();
    scrollarea=new QScrollView(songbox);
    songs=new albumdialog(scrollarea,cd);
    if (cd->cdcontent->trackcount)//if we have a cd in cdrom
    {
        bool found=FALSE;
        if (settingsfile.databasetype)
        {
            progress->setText(i18n("Checking local database for CD"));
            //first try to find cd in own database
            found=database->checkdiscid(cd->discid,artistbox,albumbox);
            if (found)
            {
                QStringList songlist;
                long albumid=database->translatealbum(albumbox->currentText());
                long artistid=database->translateartist(artistbox->currentText());
                if (database->albumexists(artistid,albumid,&songlist))
                {
                    //enter the values of the stringlist into the songledits
                    songs->setalltracks(&songlist);
                    //fetch the year of album
                    long year=database->getyear(albumid);
                    yearbox->setValue(year);
                }
            }
        }
        if (!found&&settingsfile.freedbsupport)
        {
            progress->setText(i18n("Fetching songs from CDDB"));
            freedbconnect *freedb=new freedbconnect();
            freedb->getcdinfo(cd->cddbquery,cd->discid);
            if (freedb->queryOk)
            {
                if (settingsfile.databasetype)
                {
                    //when mysql is used, save the album into database
                    database->writealbum(cd->discid,freedb->artist,freedb->album,freedb->genre,freedb->releaseyear,freedb->tracklist);
                }
                //deactivate the connections to the boxes, while we are changing them
                disconnect(artistbox,SIGNAL(textChanged(const QString &)),this,SLOT(artistChanged()));
                disconnect(albumbox,SIGNAL(textChanged(const QString &)),this,SLOT(albumChanged()));
                disconnect(genrebox,SIGNAL(textChanged(const QString &)),this,SLOT(genreChanged()));
                //set all combos to the values got by freedb
                artistbox->setCurrentText(freedb->artist);
                albumbox->setCurrentText(freedb->album);
                genrebox->setCurrentText(freedb->genre);
                yearbox->setValue(freedb->releaseyear);
                //put all songnames into the songs-widget
                songs->setalltracks(freedb->tracklist);
                connect(artistbox,SIGNAL(textChanged(const QString &)),SLOT(artistChanged()));
                connect(albumbox,SIGNAL(textChanged(const QString &)),SLOT(albumChanged()));
                connect(genrebox,SIGNAL(textChanged(const QString &)),SLOT(genreChanged()));
            }
            else
            {
                if (!freedb->userabort)
                    QMessageBox::information(this,i18n("Freedb-Search"),i18n("Got no result from freedb."));
            }
            delete freedb;
        }
    }
    else
    {
        QMessageBox::information(this,i18n("Disc-error"),i18n("There is no audio-cd in your cdrom!"));
    }
    scrollarea->repaintContents(TRUE);
    scrollarea->show();
    if (songs->counter)
    {
        selectallbutton=new QPushButton(i18n("Select all"),songbox);
        selectallbutton->show();
        connect(selectallbutton,SIGNAL(clicked()),SLOT(selectallslot()));
        startbutton->setEnabled(TRUE);
    }
    else
    {
        selectallbutton=NULL;
        startbutton->setEnabled(FALSE);
    }
    delete progress;
    delete progresswindow;
}

void ripperdialog::readinDiscSlot()
{
    //First of all check if the directories given in settings are still valid
    QFileInfo filecheck(settingsfile.destination_dir);
    if (!filecheck.exists())
    {
        QMessageBox::warning(this,i18n("Access error"),i18n("The destination directory doesn't exist"),"&Ok",QString::null,QString::null,1);
        return;
    }
    filecheck.setFile(settingsfile.mp3_dir);
    if (!filecheck.exists())
    {
        QMessageBox::warning(this,i18n("Access error"),i18n("The temp directory doesn't exist"),"&Ok",QString::null,QString::null,1);
        return;
    }
    //Check if the selected encoding-engine-executable exists
    switch (settingsfile.encodingengine)
    {
    case 0:
        {
            //lame
            filecheck.setFile(settingsfile.lameexe);
            break;
        }
    case 1:
        {
            //gogo
            filecheck.setFile(settingsfile.gogoexe);
            break;
        }
    case 2:
        {
            //bladeenc
            filecheck.setFile(settingsfile.bladeencexe);
            break;
        }
    case 3:
        {
            //oggenc
            filecheck.setFile(settingsfile.oggencexe);
            break;
        }
    }
    if (!filecheck.exists())
    {
        QMessageBox::warning(this,i18n("Encoder error"),i18n("Can't find the executable for selected Encoding-engine! Is it installed?"),"&Ok",QString::null,QString::null,1);
        return;
    }

    QStringList *readymp3Songs=new QStringList();
    long itemcounter=0;
    progressmainwindow *progresswindow=new progressmainwindow(TRUE);
    int tracknumber=1;
    //First store the values in the comboboxes to database
    if (settingsfile.databasetype)
    {
        //First catch all tracknames, and enter them in a stringlist
        QStringList *tracklist=new QStringList();
        while(tracknumber<=(cd->cdcontent->trackcount))
        {
            tracklist->append(songs->songfield[tracknumber-1]->songledit->text());
            tracknumber++;
        }
        database->writealbum(cd->discid,artistbox->currentText(),albumbox->currentText(),genrebox->currentText(),yearbox->value(),tracklist);
        delete tracklist;
    }
    tracknumber=1;
    while(tracknumber<=((cd->cdcontent->trackcount))&&!progresswindow->userabort)
    {
        if (songs->songfield[tracknumber-1]->songcheck->isChecked())
        {
            //Construct wavname
            mediatag Song;
            Song.album=albumbox->currentText();
            Song.artist=artistbox->currentText();
            Song.genre=genrebox->currentText();
            Song.number=tracknumber;
            Song.year=yearbox->value();
            Song.cdcover=frontcover;
            //Fill up all empty fields, so cleanup can work propper afterwards
            if (Song.album.isEmpty())
                Song.album="No Album";
            if (Song.artist.isEmpty())
                Song.artist="No Artist";
            if (Song.genre.isEmpty())
                Song.genre="No Genre";
            QString destinationname;
            QString wavname;
            QString format;
            if (settingsfile.encodingengine==0||settingsfile.encodingengine==1||settingsfile.encodingengine==2)
                format=QString("mp3");
            else if(settingsfile.encodingengine==3)
                format=QString("ogg");
            if (!(songs->songfield[tracknumber-1]->songledit->text().isEmpty()))
            {
                Song.title=songs->songfield[tracknumber-1]->songledit->text();
                wavname=QString("%1%2.wav").arg(settingsfile.mp3_dir,songs->songfield[tracknumber-1]->songledit->text());
                destinationname=QString("%1%2.%3").arg(settingsfile.mp3_dir,songs->songfield[tracknumber-1]->songledit->text(),format);
            }
            else
            {
                Song.title=QString("Track%1").arg(QString::number(tracknumber));
                wavname=QString("%1Track%2.wav").arg(settingsfile.mp3_dir,QString::number(tracknumber));
                destinationname=QString("%1Track%2.%3").arg(settingsfile.mp3_dir,QString::number(tracknumber),format);
            }
            if(!progresswindow->iscancelled)
                cd->getTrack(tracknumber,wavname,progresswindow);
            if(!progresswindow->iscancelled)
            {
                //check if file exists, before encoding
                QFileInfo filecheck(wavname);
                if (filecheck.exists())
                {
                    new encoder(wavname,destinationname,Song,progresswindow);
                    readymp3Songs->append(destinationname);
                    itemcounter++;
                }
            }
        }
        tracknumber++;
    }
    while (progresswindow->runningprogresses)
        kapp->processEvents();
    //delete all working files, if user aborted
    if (progresswindow->userabort)
    {
        progressinfoonly *wipeprocess=progresswindow->startInfoOnly(i18n("Deleting all temporary files"));
        for (int i=1;i<=tracknumber;i++)
        {
            mediatag Song;
            Song.album=albumbox->currentText();
            Song.artist=artistbox->currentText();
            Song.genre=genrebox->currentText();
            Song.number=tracknumber;
            Song.year=yearbox->value();
            QString mp3name;
            QString wavname;
            if (!(songs->songfield[i-1]->songledit->text().isEmpty()))
            {
                Song.title=songs->songfield[i-1]->songledit->text();
                wavname=QString("%1%2.wav").arg(settingsfile.mp3_dir,songs->songfield[i-1]->songledit->text());
                mp3name=QString("%1%2.mp3").arg(settingsfile.mp3_dir,songs->songfield[i-1]->songledit->text());
            }
            else
            {
                Song.title=QString("Track%1").arg(QString::number(i));
                wavname=QString("%1Track%2.wav").arg(settingsfile.mp3_dir,QString::number(i));
                mp3name=QString("%1Track%2.mp3").arg(settingsfile.mp3_dir,QString::number(i));
            }
            QFileInfo wavfile(wavname);
            if (wavfile.exists())
            {
                QFile wav(wavname);
                wav.remove();
            }
            QFileInfo mp3file(mp3name);
            if(mp3file.exists())
            {
                QFile mp3(mp3name);
                mp3.remove();
            }
        }
        delete wipeprocess;
        delete progresswindow;
    }
    else
    {
        if (itemcounter)
        {
            delete progresswindow;
            cleanup *filemove=new cleanup(this);
            if (filemove->filterok&&filemove->directoriesok)
            {
                filemove->cleanupcd(readymp3Songs,itemcounter);
                delete filemove;
                if (settingsfile.verbose)
                    QMessageBox::information(this,i18n("Process finished"),i18n("Selected Tracks where ripped."),"&Ok");
            }
            else
                delete filemove;
        }
        else
        {
            delete progresswindow;
            QMessageBox::information(this,i18n("Process finished"),i18n("You have to select some tracks for ripping."),"&Ok");
        }
    }
}

void ripperdialog::selectallslot()
{
    if (selectall)
    {
        selectall=FALSE;
        selectallbutton->setText(i18n("Select all"));
    }
    else
    {
        selectall=TRUE;
        selectallbutton->setText(i18n("Deselect all"));
    }
    {
        for (int tracknumber=1;tracknumber<=((cd->cdcontent->trackcount));tracknumber++)
        {
            if ((cd->cdcontent->cdtoc[tracknumber-1].mode==0)||(cd->cdcontent->cdtoc[tracknumber-1].mode==2))
                songs->songfield[tracknumber-1]->songcheck->setChecked(selectall);
        }
    }
}

void ripperdialog::artistChanged()
{
    if(settingsfile.databasetype)
    {
        //deactivate all other comboslots,to stop recursion
        disconnect(artistbox,SIGNAL(textChanged(const QString &)),this,SLOT(artistChanged()));
        disconnect(albumbox,SIGNAL(textChanged(const QString &)),this,SLOT(albumChanged()));
        disconnect(genrebox,SIGNAL(textChanged(const QString &)),this,SLOT(genreChanged()));
        //Read in all Combos, and build a sql-query with them
        QString artist=artistbox->currentText();
        QString album=albumbox->currentText();
        QString genre=genrebox->currentText();
        long artistid=database->translateartist(artist);
        if (!artistid)
        {
            //if no artist with that name in database,reread all boxes
            database->fillCombo(artistbox,"artists","artistname");
            database->fillCombo(albumbox,"albums","albumname");
            database->fillCombo(genrebox,"genres","genrename");
        }
        else
        {
            long albumid=database->translatealbum(album);
            QString querystring=QString("artist_id=\"%1\"").arg(artistid);
            database->fillSortedCombo(albumbox,"albums",querystring,"albumname");
            database->fillSortedCombo(genrebox,"tracks",querystring,"genre_id","genres");
            //refetch albumcombo
            album=albumbox->currentText();
            albumid=database->translatealbum(album);
            //Check if a right combination is in the comboboxes, and than modify all songledits
            if(artistid&&albumid)
            {
                QStringList songlist;
                if (database->albumexists(artistid,albumid,&songlist))
                {
                    //enter the values of the stringlist into the songledits
                    songs->setalltracks(&songlist);
                    //fetch the year of album
                    long year=database->getyear(albumid);
                    yearbox->setValue(year);
                }
            }
        }
        //reactivate all slots
        connect(artistbox,SIGNAL(textChanged(const QString &)),SLOT(artistChanged()));
        connect(albumbox,SIGNAL(textChanged(const QString &)),SLOT(albumChanged()));
        connect(genrebox,SIGNAL(textChanged(const QString &)),SLOT(genreChanged()));
    }
}

void ripperdialog::albumChanged()
{
    if (settingsfile.databasetype)
    {
        //deactivate all other comboslots,to stop recursion
        disconnect(artistbox,SIGNAL(textChanged(const QString &)),this,SLOT(artistChanged()));
        disconnect(albumbox,SIGNAL(textChanged(const QString &)),this,SLOT(albumChanged()));
        disconnect(genrebox,SIGNAL(textChanged(const QString &)),this,SLOT(genreChanged()));
        //Read in all Combos, and build a sql-query with them
        QString artist=artistbox->currentText();
        QString album=albumbox->currentText();
        QString genre=genrebox->currentText();
        long albumid=database->translatealbum(album);
        if (!albumid)
        {
            //if no album with that name in database,reread all boxes
            database->fillCombo(artistbox,"artists","artistname");
            database->fillCombo(albumbox,"albums","albumname");
            database->fillCombo(genrebox,"genres","genrename");
        }
        else
        {
            long artistid=database->translateartist(artist);
            QString querystring=QString("album_id=\"%1\"").arg(albumid);
            database->fillSortedCombo(artistbox,"tracks",querystring,"artist_id","artists");
            database->fillSortedCombo(genrebox,"tracks",querystring,"genre_id","genres");
            //refetch albumcombo
            artist=artistbox->currentText();
            artistid=database->translateartist(artist);
            //Check if a right combination is in the comboboxes, and than modify all songledits
            if(artistid&&albumid)
            {
                QStringList songlist;
                if (database->albumexists(artistid,albumid,&songlist))
                {
                    //enter the values of the stringlist into the songledits
                    songs->setalltracks(&songlist);
                    //fetch the year of album
                    long year=database->getyear(albumid);
                    yearbox->setValue(year);
                }
            }
        }
        //reactivate all slots
        connect(artistbox,SIGNAL(textChanged(const QString &)),SLOT(artistChanged()));
        connect(albumbox,SIGNAL(textChanged(const QString &)),SLOT(albumChanged()));
        connect(genrebox,SIGNAL(textChanged(const QString &)),SLOT(genreChanged()));
    }
}

void ripperdialog::genreChanged()
{
    if (settingsfile.databasetype)
    {
        //deactivate all other comboslots,to stop recursion
        disconnect(artistbox,SIGNAL(textChanged(const QString &)),this,SLOT(artistChanged()));
        disconnect(albumbox,SIGNAL(textChanged(const QString &)),this,SLOT(albumChanged()));
        disconnect(genrebox,SIGNAL(textChanged(const QString &)),this,SLOT(genreChanged()));
        //Read in all Combos, and build a sql-query with them
        QString artist=artistbox->currentText();
        QString album=albumbox->currentText();
        QString genre=genrebox->currentText();
        long genreid=database->translategenre(genre);
        if (!genreid)
        {
            //if no album with that name in database,reread all boxes
            database->fillCombo(artistbox,"artists","artistname");
            database->fillCombo(albumbox,"albums","albumname");
            database->fillCombo(genrebox,"genres","genrename");
        }
        else
        {
            long artistid=database->translateartist(artist);
            long albumid=database->translatealbum(album);
            QString querystring=QString("genre_id=\"%1\"").arg(genreid);
            database->fillSortedCombo(artistbox,"tracks",querystring,"artist_id","artists");
            database->fillSortedCombo(albumbox,"tracks",querystring,"album_id","albums");
            //refetch albumcombo and artistcombo
            album=albumbox->currentText();
            albumid=database->translatealbum(album);
            artist=artistbox->currentText();
            artistid=database->translateartist(artist);
            //Check if a right combination is in the comboboxes, and than modify all songledits
            if(artistid&&albumid)
            {
                QStringList songlist;
                if (database->albumexists(artistid,albumid,&songlist))
                {
                    //enter the values of the stringlist into the songledits
                    songs->setalltracks(&songlist);
                    //fetch the year of album
                    long year=database->getyear(albumid);
                    yearbox->setValue(year);
                }
            }
        }
        //reactivate all slots
        connect(artistbox,SIGNAL(textChanged(const QString &)),SLOT(artistChanged()));
        connect(albumbox,SIGNAL(textChanged(const QString &)),SLOT(albumChanged()));
        connect(genrebox,SIGNAL(textChanged(const QString &)),SLOT(genreChanged()));
    }
}

void ripperdialog::setAutocomplete(bool autocomplete)
{
    artistbox->setAutoCompletion(autocomplete);
    albumbox->setAutoCompletion(autocomplete);
    genrebox->setAutoCompletion(autocomplete);
}
