/***************************************************************************
 *   Copyright (C) 2005 by Roland Weigert   *
 *   roweigert@t-online.de   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "playerinterface.h"

extern rcdatei settingsfile;

playerinterface::playerinterface()
{
    mplayerlink=NULL;
    isplaying=FALSE;
    ispaused=FALSE;
    songtime=0;
    this->setCaption("K-Yamo Player");
    mainlayout=new QGroupBox(1,Qt::Horizontal,this);
    nowplayinglabel=new QLabel(mainlayout);
    songtimelabel=new QLabel(mainlayout);
    songtimelabel->setText(i18n("0:00 of 0:00"));
    songtimeslider=new QSlider(Qt::Horizontal,mainlayout,i18n("Playing time"));
    songtimeslider->setMinValue(0);
    songtimeslider->setMaxValue(0);
    buttonlayout=new QGroupBox(3,Qt::Horizontal,mainlayout);
    pausebutton=new QPushButton(i18n("Pause"),buttonlayout);
    stopbutton=new QPushButton(i18n("Stop"),buttonlayout);
    nextbutton=new QPushButton(i18n("Next"),buttonlayout);
    //load the buttonpicutures
    KIconLoader *loader=new KIconLoader();
    pausepic=loader->loadIcon("player_pause.png",KIcon::Desktop);
    playpic=loader->loadIcon("player_play.png",KIcon::Desktop);
    stoppic=loader->loadIcon("player_stop.png",KIcon::Desktop);
    nextpic=loader->loadIcon("player_end.png",KIcon::Desktop);
    QIconSet buttonset(pausepic);
    pausebutton->setIconSet(buttonset);
    buttonset=QIconSet(stoppic);
    stopbutton->setIconSet(buttonset);
    buttonset=QIconSet(nextpic);
    nextbutton->setIconSet(buttonset);
    delete loader;
    switch (settingsfile.playertype)
    {
    case 0:
        {
            //mplayer
            mplayerlink=new mplayerinterface;
            connect (mplayerlink,SIGNAL(songlength(int)),this,SLOT(getsonglength(int)));
            connect (mplayerlink,SIGNAL(songtime(int)),this,SLOT(getplayertime(int)));
            connect (mplayerlink,SIGNAL(playerstopped()),this,SLOT(songend()));
            break;
        }
    }
    connect(pausebutton,SIGNAL(clicked()),SLOT(pauseplaying()));
    connect(stopbutton,SIGNAL(clicked()),SLOT(completestop()));
    connect(nextbutton,SIGNAL(clicked()),SLOT(nextsong()));
    connect(songtimeslider,SIGNAL(sliderPressed()),SLOT(sliderchstart()));
    connect(songtimeslider,SIGNAL(sliderReleased()),SLOT(sliderchend()));
}

playerinterface::~playerinterface()
{
    if (mplayerlink)
        delete mplayerlink;
    delete nowplayinglabel;
    delete songtimelabel;
    delete stopbutton;
    delete nextbutton;
    delete pausebutton;
    delete songtimeslider;
    delete buttonlayout;
    delete mainlayout;
}

void playerinterface::closeEvent(QCloseEvent* ce)
{
    completestop();
    ce->ignore();
}

int playerinterface::addTrack(QString song)
{
    int success;
    songlist.append(song);
    if (!isplaying)
    {
        songtimeslider->setTickmarks(QSlider::Below);
        songtimeslider->setTickInterval(10);

        playingatm=song;
        playdtrack=songlist.begin();
        switch (settingsfile.playertype)
        {
        case 0:
            {
                //mplayer
                int success=mplayerlink->play(song);
                if (success==1)
                {
                    songtimelabel->setText(i18n("<center>00:00 of 00:00"));
                    isplaying=TRUE;
                    this->show();
                }
                break;
            }
        }
    }
    return success;
}

int playerinterface::createList(QString song)
{
    int success=0;
    switch (settingsfile.playertype)
    {
    case 0:
        {
            if (isplaying)
            {
                mplayerlink->stop();
                songlist.clear();
            }
            playdtrack=songlist.begin();
            success=mplayerlink->play(song);
            if (success==1)
            {
                songtimelabel->setText(i18n("<center>00:00 of 00:00"));
                playingatm=song;
                songtimeslider->setMaxValue(0);
                isplaying=TRUE;
                QString shownname=playingatm.right(playingatm.length()-(playingatm.findRev("/")+1));
                nowplayinglabel->setText(QString("<center>%1").arg(shownname));
                mainlayout->adjustSize();
                this->adjustSize();
                this->show();
            }
            break;
        }
    }
    return success;
}

bool playerinterface::nextsong()
{
    ispaused=FALSE;
    QIconSet buttonset(pausepic);
    pausebutton->setIconSet(buttonset);
    pausebutton->setText(i18n("Pause"));
    switch (settingsfile.playertype)
    {
    case 0:
        {
            if (mplayerlink->isplaying)
            {
                mplayerlink->stop();
            }
            QStringList::Iterator songtoremove=playdtrack;
            playdtrack++;
            if (playdtrack!=songlist.end())
            {    //songtime=songtimeslider->value();

                playingatm=(*playdtrack);
                int success=mplayerlink->play(playingatm);
                if (success==1)
                {
                    QString shownname=playingatm.right(playingatm.length()-(playingatm.findRev("/")+1));
                    nowplayinglabel->setText(QString("<center>%1").arg(shownname));
                    mainlayout->adjustSize();
                    this->adjustSize();
                    this->show();
                }
                else if (success==-1)
                {
                    nextsong();
                }
                else if(success==0)
                {
                    isplaying=FALSE;
                    playingatm=QString::null;
                    playdtrack=NULL;
                    nowplayinglabel->setText(QString::null);
                    this->hide();
                }
            }
            else
            {
                isplaying=FALSE;
                playingatm=QString::null;
                playdtrack=NULL;
                nowplayinglabel->setText(QString::null);
                this->hide();
            }
        }
    }
    return TRUE;
}

void playerinterface::completestop()
{
    ispaused=FALSE;
    QIconSet buttonset(pausepic);
    pausebutton->setIconSet(buttonset);
    pausebutton->setText(i18n("Pause"));
    switch (settingsfile.playertype)
    {
    case 0:
        {
            //mplayer
            mplayerlink->stop();
            break;
        }
    }
    songlist.clear();
    isplaying=FALSE;
    this->hide();
    playdtrack=NULL;
}

bool playerinterface::pauseplaying()
{
    switch (settingsfile.playertype)
    {
    case 0:
        {
            //mplayer
            mplayerlink->pause();
            break;
        }
    }
    if (ispaused)
    {
        ispaused=FALSE;
        QIconSet buttonset(pausepic);
        pausebutton->setIconSet(buttonset);
        pausebutton->setText(i18n("Pause"));
    }
    else
    {
        ispaused=TRUE;
        QIconSet buttonset(playpic);
        pausebutton->setIconSet(buttonset);
        pausebutton->setText(i18n("Play"));
    }
    return ispaused;
}

void playerinterface::sliderchstart()
{
    //disconnect the signals from playerProcess, so we don´t get confused by sliderchanges, that are not from user
    switch (settingsfile.playertype)
    {
    case 0:
        {
            //mplayer
            disconnect (mplayerlink,SIGNAL(songtime(int)),this,SLOT(getplayertime(int)));
            break;
        }
    }
    //check if we are paused, and set pause, if we are not paused
    if (!ispaused)
    {
        pauseplaying();
    }
}

void playerinterface::sliderchend()
{
    //get the current sliderposition, and check the differnce between this position and before
    int newsliderpos=songtimeslider->value();
    if (newsliderpos<songtimeslider->maxValue())
    {
        int offset=newsliderpos-songtime;
        if (offset<0)
        {
            //scrollback
            switch (settingsfile.playertype)
            {
            case 0:
                {
                    //mplayer
                    mplayerlink->seekbackward(offset);
                    break;
                }
            }
        }
        else if(offset>0)
        {
            switch (settingsfile.playertype)
            {
            case 0:
                {
                    //mplayer
                    mplayerlink->seekforward(offset);
                    break;
                }
            }
        }
        ispaused=FALSE;
        //just for sure, that it is saying pause again.
        QIconSet buttonset(pausepic);
        pausebutton->setIconSet(buttonset);
        pausebutton->setText(i18n("Pause"));
        switch (settingsfile.playertype)
        {
        case 0:
            {
                //mplayer
                connect (mplayerlink,SIGNAL(songtime(int)),this,SLOT(getplayertime(int)));
                break;
            }
        }
    }
    else
        nextsong();
}

void playerinterface::getsonglength(int songlength)
{
    songtimeslider->setMaxValue(songlength);
    songtimeslider->setTickmarks(QSlider::Below);
    songtimeslider->setTickInterval(10);
    QString minutes=QString::number(songlength/60);
    if (minutes.length()==1)
        minutes=QString("0%1").arg(minutes);
    QString seconds=QString::number(songlength%60);
    if (seconds.length()==1)
        seconds=QString("0%1").arg(seconds);
    songlengthstring=QString("%1:%2").arg(minutes,seconds);
    QString songtimestring=QString(i18n("<center>00:00 of %1")).arg(songlengthstring);
    songtimelabel->setText(songtimestring);
}

void playerinterface::getplayertime(int newvalue)
{
    songtime=newvalue;
    songtimeslider->setValue(songtime);
    QString minutes=QString::number(songtime/60);
    if (minutes.length()==1)
        minutes=QString("0%1").arg(minutes);
    QString seconds=QString::number(songtime%60);
    if (seconds.length()==1)
        seconds=QString("0%1").arg(seconds);
    QString songtimestring=QString(i18n("<center>%1:%2 of %3")).arg(minutes,seconds,songlengthstring);
    songtimelabel->setText(songtimestring);
}

void playerinterface::songend()
{
    ispaused=FALSE;
    QIconSet buttonset(pausepic);
    pausebutton->setIconSet(buttonset);
    pausebutton->setText(i18n("Pause"));
    //Look on the list, if we have other stuff to play, and start next song
    QStringList::Iterator songtoremove=playdtrack;
    playdtrack++;
    if (playdtrack!=songlist.end())
    {
        playingatm=(*playdtrack);
        switch (settingsfile.playertype)
        {
        case 0:
            {
                //mplayer
                mplayerlink->play(playingatm);
                break;
            }
        }
        QString shownname=playingatm.right(playingatm.length()-(playingatm.findRev("/")+1));
        nowplayinglabel->setText(shownname);
        mainlayout->adjustSize();
        this->adjustSize();
        this->show();
    }
    else
    {
        isplaying=FALSE;
        playingatm=QString::null;
        nowplayinglabel->setText(QString::null);
        this->hide();
    }
}
