//=============================================================================
//
//   File : kvi_command.cpp
//   Creation date : Sun Jul 02 2000 15:39:55 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_command.h"
#include "kvi_uparser.h"
#include "kvi_window.h"
#include "kvi_error.h"
#include "kvi_datacontainer.h"
#include "kvi_string.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_malloc.h"
#include "kvi_locale.h"


#include <ctype.h>

// FIXME: Merge the constructors!

KviCommand::KviCommand(const char * cmdBuffer,KviWindow *wnd,KviCommand * pParent,KviDataContainer * pExtendedScopeDataContainer)
{
	m_pParent       = pParent;
	m_szCmdBuffer   = cmdBuffer;
	m_pWnd          = wnd;
	m_pParamList    = 0;
	m_pContextStack = new QStrList(false); // SHALLOW COPIES OF THE STACK STRINGS!!!!
	m_pContextStack->setAutoDelete(false);
	m_pSwitchList   = new KviPtrList<KviStr>;
	m_pSwitchList->setAutoDelete(true);
	m_ptr           = m_szCmdBuffer.ptr();
	m_iError        = KviError_success;
	m_pDataContainer = new KviDataContainer(false);
	m_pExtendedScopeDataContainer = pExtendedScopeDataContainer;
	m_bOwnParams    = true;
	m_pScopeObject  = 0;
	m_pThisPointer  = 0;
	m_state         = Success;
	m_pReturnParameterList = 0;
	m_pAliasSwitchList = 0;
}

KviCommand::KviCommand(const QString & cmdBuffer,KviWindow *wnd,KviCommand * pParent,KviDataContainer * pExtendedScopeDataContainer)
{
	m_pParent       = pParent;
	m_szCmdBuffer   = cmdBuffer.utf8().data();
	m_pWnd          = wnd;
	m_pParamList    = 0;
	m_pContextStack = new QStrList(false); // SHALLOW COPIES OF THE STACK STRINGS!!!!
	m_pContextStack->setAutoDelete(false);
	m_pSwitchList     = new KviPtrList<KviStr>;
	m_pSwitchList->setAutoDelete(true);
	m_ptr           = m_szCmdBuffer.ptr();
	m_iError        = KviError_success;
	m_pDataContainer = new KviDataContainer(false);
	m_pExtendedScopeDataContainer = pExtendedScopeDataContainer;
	m_bOwnParams    = true;
	m_pScopeObject  = 0;
	m_pThisPointer  = 0;
	m_state         = Success;
	m_pReturnParameterList = 0;
	m_pAliasSwitchList = 0;
}



KviCommand::KviCommand(const KviStr & cmdBuffer,KviWindow *wnd,KviCommand * pParent,KviDataContainer * pExtendedScopeDataContainer)
{
	m_pParent       = pParent;
	m_szCmdBuffer   = cmdBuffer;
	m_pWnd          = wnd;
	m_pParamList    = 0;
	m_pContextStack = new QStrList(false); // SHALLOW COPIES OF THE STACK STRINGS!!!!
	m_pContextStack->setAutoDelete(false);
	m_pSwitchList     = new KviPtrList<KviStr>;
	m_pSwitchList->setAutoDelete(true);
	m_ptr           = m_szCmdBuffer.ptr();
	m_iError        = KviError_success;
	m_pDataContainer = new KviDataContainer(false);
	m_pExtendedScopeDataContainer = pExtendedScopeDataContainer;
	m_bOwnParams    = true;
	m_pScopeObject  = 0;
	m_pThisPointer  = 0;
	m_state         = Success;
	m_pReturnParameterList = 0;
	m_pAliasSwitchList = 0;
}

KviCommand::KviCommand(const char * begin,const char * end,KviWindow *wnd,KviCommand * pParent)
{
	m_pParent       = pParent;
	m_szCmdBuffer.extractFromString(begin,end);
	m_pWnd          = wnd;
	m_pParamList    = 0;
	m_pContextStack = new QStrList();
	m_pContextStack->setAutoDelete(false);
	m_pSwitchList     = new KviPtrList<KviStr>;
	m_pSwitchList->setAutoDelete(true);
	m_ptr           = m_szCmdBuffer.ptr();
	m_iError        = KviError_success;
	m_pDataContainer = new KviDataContainer(false);
	m_pExtendedScopeDataContainer = 0;
	m_bOwnParams    = false;
	m_pScopeObject  = 0;
	m_pThisPointer  = 0;
	m_pReturnParameterList = 0;
	m_state         = Success;
	m_pAliasSwitchList = 0;
}

KviCommand::~KviCommand()
{
	delete m_pContextStack;
	if(m_pDataContainer)delete m_pDataContainer;
	if(m_pExtendedScopeDataContainer)delete m_pExtendedScopeDataContainer;
	clearSwitchList();
	delete m_pSwitchList;
	if(m_pParamList && m_bOwnParams)delete m_pParamList;
	if(m_pReturnParameterList)delete m_pReturnParameterList;
}

void KviCommand::setDataContainer(KviDataContainer * c)
{
	if(m_pDataContainer)delete m_pDataContainer;
	m_pDataContainer = c;
}

#define KVI_INVALID_DICTIONARY_NAME ";"
#define KVI_INVALID_ARRAY_NAME ";"

void KviCommand::setParsingDictionaryAssignment(bool bYes)
{
	if(bYes)
	{
		m_state = ParsingDictionaryAssignment;
	} else {
		if((m_state == ParsingDictionaryAssignment) || (m_state == HaveDictionaryReturn))
		{
			m_pDataContainer->removeDictionary(KVI_INVALID_DICTIONARY_NAME);
			m_state = Success;
		}
	}
}

void KviCommand::setParsingArrayAssignment(bool bYes)
{
	if(bYes)
	{
		m_state = ParsingArrayAssignment;
	} else {
		if((m_state == ParsingArrayAssignment) || (m_state == HaveArrayReturn))
		{
			m_pDataContainer->removeArray(KVI_INVALID_ARRAY_NAME);
			m_state = Success;
		}
	}
}

void KviCommand::beginListArrayOrDictionaryReturnIdentifier()
{
	if(m_pReturnParameterList)return; //OK! We will append the result to the return parameter list!
	if((m_state == ParsingArrayAssignment) || (m_state == HaveArrayReturn))
	{
		KviStringArray * a = m_pDataContainer->lookupArray(KVI_INVALID_ARRAY_NAME,false);
		a->clear();
		m_state = HaveArrayReturn;
		return;
	}
	if((m_state == ParsingDictionaryAssignment) || (m_state == HaveDictionaryReturn))
	{
		KviDictionary * d = m_pDataContainer->lookupDictionary(KVI_INVALID_DICTIONARY_NAME,false);
		d->clear();
		m_state = HaveDictionaryReturn;
	}
}

void KviCommand::addListArrayOrDictionaryReturnValue(int id,const char * val,KviStr &buffer)
{
	if(m_pReturnParameterList)
	{
		if(id == 0)
		{
			buffer.append(val);
		} else {
			m_pReturnParameterList->append(new KviStr(buffer));
			buffer = val;
		}
		return;
	}

	if(m_state == HaveArrayReturn)
	{
		KviStringArray * d = m_pDataContainer->lookupArray(KVI_INVALID_ARRAY_NAME,false);
		d->insert(id,new KviStr(val));
		return;
	}

	if(m_state == HaveDictionaryReturn)
	{
		KviDictionary * d = m_pDataContainer->lookupDictionary(KVI_INVALID_DICTIONARY_NAME,false);
		KviStr tmp(KviStr::Format,"%d",id);
		d->replace(tmp.ptr(),new KviStr(val));
		return;
	}

	if(id > 0)buffer.append(',');
	buffer.append(val);
}

void KviCommand::addListArrayOrDictionaryReturnValue(int id,const KviStr &val,KviStr &buffer)
{
	if(m_pReturnParameterList)
	{
		if(id == 0)
		{
			buffer.append(val);
		} else {
			m_pReturnParameterList->append(new KviStr(buffer));
			buffer = val;
		}
		return;
	}

	if(m_state == HaveArrayReturn)
	{
		KviStringArray * d = m_pDataContainer->lookupArray(KVI_INVALID_ARRAY_NAME,false);
		d->insert(id,new KviStr(val));
		return;
	}

	if(m_state == HaveDictionaryReturn)
	{
		KviDictionary * d = m_pDataContainer->lookupDictionary(KVI_INVALID_DICTIONARY_NAME,false);
		KviStr tmp(KviStr::Format,"%d",id);
		d->replace(tmp.ptr(),new KviStr(val));
		return;
	}
	if(id > 0)buffer.append(',');
	buffer.append(val);
}

KviDictionary * KviCommand::dictionaryReturn()
{
	if(m_state != HaveDictionaryReturn)return 0;
	KviDictionary * d = m_pDataContainer->lookupDictionary(KVI_INVALID_DICTIONARY_NAME,false);
	m_pDataContainer->removeDictionaryNoDelete(KVI_INVALID_DICTIONARY_NAME);
	return d;
}

KviStringArray * KviCommand::arrayReturn()
{
	if(m_state != HaveArrayReturn)return 0;
	KviStringArray * a = m_pDataContainer->lookupArray(KVI_INVALID_ARRAY_NAME,false);
	m_pDataContainer->removeArrayNoDelete(KVI_INVALID_ARRAY_NAME);
	return a;
}

void KviCommand::setBreakEncountered()
{
//	m_iError = KviError_breakCalled;
	m_state = Break;
}

bool KviCommand::breakEncountered()
{
	return (m_state == Break);
}

void KviCommand::setBreakHandled(int iContextStackSize)
{
	if(iContextStackSize < 1)debug("internal error in setBreakHandled(): context stack size is %d",iContextStackSize);
	while(m_pContextStack->count() > (unsigned int)iContextStackSize)m_pContextStack->removeLast();
	m_state = Success;
}

void KviCommand::setCmdBuffer(const char * cmdBuffer)
{
	m_szCmdBuffer = cmdBuffer;
	m_ptr = m_szCmdBuffer.ptr();
	m_iError = KviError_success;
	m_state = Success;
}

void KviCommand::setParams(KviParameterList * params,bool bTransferOwnership)
{
	if(m_pParamList && m_bOwnParams)delete m_pParamList;
	m_pParamList = params;
	m_bOwnParams = bTransferOwnership;
}

void KviCommand::skipWhiteSpace()
{
	while(*m_ptr && isspace(*m_ptr))m_ptr++;
	if(*m_ptr == '\\')
	{
		//escape char
		m_ptr++;
		if(*m_ptr && isspace(*m_ptr)){
			//escaped whitespace
			m_ptr++;
			skipWhiteSpace();
		} else m_ptr--;
	}
}

void KviCommand::skipSpace()
{
	while((*m_ptr == ' ')||(*m_ptr == '\t'))m_ptr++;
	if(*m_ptr == '\\')
	{
		//escape char
		m_ptr++;
		if(*m_ptr == '\n')
		{
			//escaped newline
			m_ptr++;
			skipSpace();
		} else m_ptr--;
	}
}

bool KviCommand::hasSwitch(char letter)
{
	for(KviStr * s=m_pSwitchList->first();s;s=m_pSwitchList->next())
	{
		if(tolower(*(s->ptr())) == tolower(letter))return true;
	}
	return false;
}

bool KviCommand::getSwitchValue(char letter,KviStr &buffer)
{
	for(KviStr * s=m_pSwitchList->first();s;s=m_pSwitchList->next())
	{
		if(tolower(*(s->ptr())) == tolower(letter))
		{
			if(s->len() < 2)return false;
			buffer = s->ptr();
			buffer.cutLeft(2);
			return true;
		}
	}
	return false;
}

void KviCommand::getParamCount(KviStr &buffer)
{
	if(!hasParams())buffer.append("0");
	else {
		KviStr num(KviStr::Format,"%d",m_pParamList->count());
		buffer.append(num);
	}
}

// $1 , $1- $0- $0-4 $21-24 $12-

void KviCommand::getSingleParam(int param,KviStr &buffer)
{
	if(!hasParams())return;
	if(m_pParamList->count() > ((unsigned int)param))
	{
		KviStr * par = m_pParamList->at(param);
		if(par)buffer.append(*par);
	}
}

void KviCommand::getParam(int fromParam,int toParam,KviStr &buffer)
{
	if(!hasParams())return;
	if(((unsigned int)fromParam) >= m_pParamList->count())return;

	KviStr * par = m_pParamList->at(fromParam);
	if(par)
	{
		buffer.append(*par);
		if(toParam < fromParam)
		{
			// up to the end
			for(par = m_pParamList->next();par;par = m_pParamList->next())
			{
				buffer.append(' ');
				buffer.append(*par);
			}
		} else {
			if(((unsigned int)toParam) >= m_pParamList->count())toParam = m_pParamList->count();
			fromParam++;
			par = m_pParamList->next();
			while(par && (fromParam <= toParam))
			{
				buffer.append(' ');
				buffer.append(*par);
				par = m_pParamList->next();
				fromParam++;
			}
		}
	}
}


bool KviCommand::noIrcContext()
{
	warnNoIrcContext();
	return leaveStackFrame();
}

bool KviCommand::notConnectedToServer()
{
	warnNotConnectedToServer();
	return leaveStackFrame();
}

void KviCommand::warnNoIrcContext()
{
	warning(__tr2qs("This window has no associated IRC context"));
}

void KviCommand::warnNotConnectedToServer()
{
	warning(__tr2qs("You're not connected to a server"));
}

bool KviCommand::warning(const QString &format,...)
{
	if(KVI_OPTION_BOOL(KviOption_boolAvoidParserWarnings))return true;
	kvi_va_list list;
	kvi_va_start_by_reference(list,format);
	QString s;
	KviQString::vsprintf(s,format,list);
	kvi_va_end(list);
	QString c = m_pContextStack->last();
	if(c.isEmpty())c = __tr2qs("Unknown");
	m_pWnd->output(KVI_OUT_PARSERWARNING,__tr2qs("[Parser warning in '%Q']: %Q"),&c,&s);
	return true; // false if treat warnings as errors...
}



bool KviCommand::warning(const char *format,...)
{
	if(KVI_OPTION_BOOL(KviOption_boolAvoidParserWarnings))return true;
	kvi_va_list list;
	kvi_va_start(list,format);
	QString s;
	QString szFormat(format);
	KviQString::vsprintf(s,szFormat,list);
	kvi_va_end(list);
	QString c = m_pContextStack->last();
	if(c.isEmpty())c = __tr2qs("Unknown");
	m_pWnd->output(KVI_OUT_PARSERWARNING,__tr2qs("[Parser warning in '%Q']: %Q"),&c,&s);
	return true; // false if treat warnings as errors... // false if treat warnings as errors...
}

bool KviCommand::error(int errCode,const char * format,...)
{
	m_iError = errCode;
	m_state = Error;
	if(!format)m_szErrDetail = "";
	kvi_va_list list;
	kvi_va_start(list,format);
	QString s;
	QString szFormat(format);
	KviQString::vsprintf(s,szFormat,list);
	kvi_va_end(list);
	m_szErrDetail=s;
	return false;
}
