/***************************************************************************
 *   Copyright (C) 2004-2008 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <netinet/in.h>

#include <qstringlist.h>

#include <klocale.h>

#include "icmp-protocol.h"
#include "ip-protocol.h"
#include "../packet.h"
#include "../packetstructures.h"

IcmpProtocol::IcmpProtocol()
{
  m_packet = NULL;
  m_icmp = (struct IcmpHdr *) NULL;
}


IcmpProtocol::IcmpProtocol(Packet *packet)
{
  m_packet = packet;
  m_icmp = (struct IcmpHdr *) m_packet;
}


void IcmpProtocol::setPacket(Packet *packet)
{
  m_packet = packet;
  m_icmp = (struct IcmpHdr *) m_packet;
}


u_int8_t IcmpProtocol::type() const
{
  return m_icmp->type;
}


QString IcmpProtocol::strType() const
{
  /**
   * icmp-codes [RFC 792]
   *
   * Summary of Message Types
   *   0  Echo Reply
   *   3  Destination Unreachable
   *   4  Source Quench
   *   5  Redirect
   *   8  Echo
   *  11  Time Exceeded
   *  12  Parameter Problem
   *  13  Timestamp
   *  14  Timestamp Reply
   *  15  Information Request
   *  16  Information Reply
   */

  QString stringType;
  switch (type())
  {
    case 0:
      stringType = i18n("ICMP type", "Echo reply");
      break;
    case 3:
      stringType = i18n("ICMP type", "Destination unreachable");
      break;
    case 4:
      stringType = i18n("ICMP type", "Source quench");
      break;
    case 5:
      stringType = i18n("ICMP type", "Redirect");
      break;
    case 8:
      stringType = i18n("ICMP type", "Echo");
      break;
    case 11:
      stringType = i18n("ICMP type", "Time exceeded");
      break;
    case 12:
      stringType = i18n("ICMP type", "Parameter problem");
      break;
    case 13:
      stringType = i18n("ICMP type", "Timestamp");
      break;
    case 14:
      stringType = i18n("ICMP type", "Timestamp reply");
      break;
    case 15:
      stringType = i18n("ICMP type", "Information request");
      break;
    case 16:
      stringType = i18n("ICMP type", "Information reply");
      break;
  }

  return stringType;
}

QString IcmpProtocol::strCode() const
{
  QString stringCode;
  switch (code())
  {
    case 1:
      stringCode = i18n("ICMP code type", "Host unreachable");
      break;
  }

  return stringCode;
}


QString IcmpProtocol::strInfo() const
{
  QString ret = strType();

  if (!strCode().isEmpty())
    ret += " (" + strCode() + ")";

  return ret;
}


u_int8_t IcmpProtocol::code() const
{
  return m_icmp->code;
}


u_int16_t IcmpProtocol::checksum() const
{
  return ntohs(m_icmp->checksum);
}


u_int16_t IcmpProtocol::id() const
{
  return m_icmp->un.echo.id;
}


u_int16_t IcmpProtocol::sequence() const
{
  return m_icmp->un.echo.sequence;
}


QStringList IcmpProtocol::headerLines() const
{
  QStringList list, ipList;
  IpProtocol ip(m_packet, sizeof(struct IcmpHdr));

  list << i18n("ICMP protocol field", "Type: %1 (%2)").arg( type() ).arg( strType() );
  if (strCode().isEmpty())
    list << i18n("ICMP protocol field", "Code: %1").arg( code() );
  else
    list << i18n("ICMP protocol field", "Code: %1 (%2)").arg( code() ).arg( strCode() );
  list << i18n("ICMP protocol field", "Checksum: 0x%1").arg( checksum(), 0, 16 );
  if (type() == 3 || type() == 4 || type() == 11 || type() == 12)
  {
    list << "*open";
    list << i18n("IP information:");
    ipList = ip.headerLines();
    for ( QStringList::Iterator it = ipList.begin(); it != ipList.end(); ++it )
    {
      list << *it;
    }
    list << "*close";
  }
  //list << i18n("ICMP protocol field", "Identificator: %1").arg( id() );
  //list << i18n("ICMP protocol field", "Sequence: %1").arg( sequence() );

  return list;
}
