/***************************************************************************
                          filterelem.cpp  -  description
                             -------------------
    begin                : Die Sep 23 2003
    copyright            : (C) 2003 by Eggert Ehmke
    email                : eggert.ehmke@berlin.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qregexp.h>
#include <klocale.h>
#include <kdebug.h>
#include "filterelem.h"

FilterRecord::FilterRecord ():
  _type (from), _expression (contains), _CaseSensitive (false), _RegExp (false)
{}

FilterRecord::FilterRecord (filterType type, expressionType expression, const QString& match, bool caseSensitive, bool regExp):
  _type (type),
  _expression (expression),
  _match (match),
  _CaseSensitive (caseSensitive),
  _RegExp (regExp)
{}

QString FilterRecord::toString () const
{
  QString name;
  switch (_type)
  {
    case from:
      name += i18n ("From");
      break;
    case subject:
      name += i18n ("Subject");
      break;
    case header:
      name += i18n ("Header");
      break;
    case size:
      name += i18n ("Size");
      break;
    case to:
      name += i18n ("To");
      break;
  }
  name += " ";
  switch (_expression)
  {
    case contains:
      name += i18n ("Contains");
      break;
    case contains_not:
      name += i18n ("Doesn't contain");
      break;
    case equals:
      name += i18n ("Equals");
      break;
    case equals_not:
      name += i18n ("Doesn't equal");
      break;
    case greater:
      name += i18n ("Greater");
      break;
    case less:
      name += i18n ("Less");
      break;
  }
  name += " ";
  name += _match;
  return name;
}

FilterElem::FilterElem():
  _secondCondition (noSecondCondition),
  _counter (0)
{
}

FilterElem::FilterElem (const FilterRecord& record):
  _filter1 (record), _secondCondition (noSecondCondition), _counter (0)
{
}

FilterElem::FilterElem (const FilterElem& elem):
  _filter1 (elem._filter1),
  _filter2 (elem._filter2),
  _secondCondition (elem._secondCondition),
  _counter (elem._counter)
{
}

FilterElem::~FilterElem()
{
}

void FilterElem::setSecondCondition (secondCondition scnd, const FilterRecord& filter)
{
  _secondCondition = scnd;
  _filter2 = filter;
}

QString FilterElem::toString () const
{
  QString name;
  name.sprintf ("%06d_", _counter);
  name += _filter1.toString();
  if (_secondCondition != FilterElem::noSecondCondition)
  {
    name += " ";
    switch (_secondCondition)
    {
      case condAnd:
        name += i18n ("And");
        break;
      case condOr:
        name += i18n ("Or");
        break;
      default:
        break;
    }
    name += " " + _filter2.toString();
  }
  return name;
}

bool FilterRecord::matches (ShowRecordElem* pElem)
{
  QString text;
  int msgsize = 0;
  switch (_type)
  {
    case subject:
      text = pElem->subject();
      break;
    case from:
      text = pElem->from();
      break;
    case to:
      text = pElem->to();
      break;
    case header:
      text = pElem->header();
      break;
    case size:
      {
        msgsize = pElem->size();
        int value = _match.toInt();
        switch (_expression)
        {
          case equals:
            return (msgsize == value);
            break;
          case equals_not:
            return (msgsize != value);
            break;
          case greater:
            return (msgsize > value);
            break;
          case less:
            return (msgsize < value);
            break;
          default:
            // contains does not make sense for size
            return false;
        }
        return false;
      }
      break;
    default:
      return false;
  }

  switch (_expression)
  {
    case contains:
      if (_RegExp)
      {
        return (QRegExp (_match, _CaseSensitive).search(text) >= 0);
      }
      else
      {
        return (text.contains (_match, _CaseSensitive) > 0);
      }
      break;
    case contains_not:
      if (_RegExp)
      {
        return (QRegExp (_match, _CaseSensitive).search(text) < 0);
      }
      else
      {
        return (text.contains (_match, _CaseSensitive) == 0);
      }
      break;
    case equals:
      if (_RegExp)
      {
        return (QRegExp (_match, _CaseSensitive).exactMatch (text));
      }
      else
      {
        if (_CaseSensitive)
        {
          return (text == _match);
        }
        else
        {
          return (text.upper() == _match.upper());
        }
      }
      break;
    case equals_not:
      if (_RegExp)
      {
        return !(QRegExp (_match, _CaseSensitive).exactMatch (text));
      }
      else
      {
        if (_CaseSensitive)
        {
          return (text != _match);
        }
        else
        {
          return (text.upper() != _match.upper());
        }
      }
      break;
    case greater:
      if (_CaseSensitive)
      {
        return (text > _match);
      }
      else
      {
        return (text.upper() > _match.upper());
      }
      break;
    case less:
      if (_CaseSensitive)
      {
        return (text < _match);
      }
      else
      {
        return (text.upper() < _match.upper());
      }
      break;
    default:
      return false;
  }

  return false;
}

bool FilterElem::matches (ShowRecordElem* pElem)
{
  bool result = _filter1.matches (pElem);
  switch (_secondCondition)
  {
    case noSecondCondition:
      return result;
      break;
    case condAnd:
      if (result)
        return _filter2.matches (pElem);
      else
        return false;
      break;
    case condOr:
      if (result)
        return true;
      else
        return _filter2.matches (pElem);
  }
  return false;
}

unsigned int FilterElem::getCounter( )
{
  return _counter;
}

void FilterElem::incCounter( )
{
  _counter++;
}

void FilterElem::setCounter( unsigned int number )
{
  _counter = number;
}
