/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef PROPERTYVALUE_H
#define PROPERTYVALUE_H

#include <QVariant>

/** \brief A data type with a QVariant and some additional information.
  *
  * It doesn't provide much intellegence. Think of it not so much like a class,
  * but more like just a struct. You have to set all members manually to give
  * them useful values.
  *
  * \li #internalValue is a QVariant that holds the "real" value.
  * \li #type holds information about the type of data in #internalValue.
  * \li #formatedValue holds a well formated and localised version of #internalValue
  *     ready for displaying.
  * \li #formatedValueAlignment holds information about the alignment of #formatedValue.
  * \li #toolTip holds a tooltip.
  * \li #whatsThis holds a "What's this" help text.
  *
  * The data type provides the copy constructor and the assignment operator, so you can safly
  * use it like any other data type. The data type is made available to QMetaType and
  * is this way available in for example QVariant. If you want to use
  * it in \e queued signal and slot connections, do something like
  * <tt>int id = qRegisterMetaType<VersionNumber>();</tt> This will register
  * the type also for this use case. <tt>id</tt> will contain the
  * type identifier used by QMetaObject. */
class PropertyValue
{

  public:

    PropertyValue ();

    PropertyValue (const PropertyValue &value); // copy constructor

    virtual ~PropertyValue();

    PropertyValue & operator=(const PropertyValue & value);

    /** \return \e true if all 3 members are equal. Otherwise \e false. */
    bool operator==(const PropertyValue value) const;

    /** \return \e false if all 3 members are equal. Otherwise \e true. */
    bool operator!=(const PropertyValue value) const;

    /** Typecast to QString. */
    operator QString() const;

    // typedefs

    /** Enum for determinating the type of value that this object holds.
    *   \li \e PropertyValue::value: This object holds a correct value.
    *   \li \e PropertyValue::error: This object holds an error.
    *   \li \e PropertyValue::unset: The value of this object is unset. */
    enum propertyValueType {
      value,
      error,
      unset
    };

    // members

    /** This member holds the "real" value.
    *
    *   <b> Default value: </b> \e none */
    QVariant internalValue;

    /** Provides information about the "type" of data in #internalValue.
    *
    *   <b> Default value: </b> \e PropertyValue::unset */
    propertyValueType type;

    /** Provides a well formated and localised version of #internalValue
    *   ready for displaying.
    *
    *   <b> Default value: </b> \e none */
    QString formatedValue;

    /** Provides information about the alignment that should be used for
    *   #formatedValue. It's of type Qt::Alignment (see
    *   http://doc.trolltech.com/main-snapshot/qt.html#AlignmentFlag-enum
    *   for details).
    *
    *   <b> Default value: </b><i>(Qt::AlignLeft | Qt::AlignVCenter)</i> */
    Qt::Alignment formatedValueAlignment;

    /** Provides a tool tip for the value.
    *
    *   <b> Default value: </b> \e none */
    QString toolTip;

    /** Provides a text for the "What's this" help function.
    *
    *   <b> Default value: </b> \e none */
    QString whatsThis;

  private:
    /** Internally used to copy another object to \e this object. */
    void helper_copyFromHere (const PropertyValue &value);
};

Q_DECLARE_METATYPE(PropertyValue)

#endif
