/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "console_reader.h"

console_reader::console_reader(const QPointer<QObject> parent) : QObject(parent)
{
  internal_splitBehavior = QString::KeepEmptyParts;
  //set up m_process
  m_process.setOutputChannelMode(KProcess::MergedChannels); //merge stderr into stdout
  QObject::connect(&m_process,
                    SIGNAL(readyRead()),
                    this,
                    SLOT(read_console_output()));
}

console_reader::~console_reader()
{
  m_process.kill();
  m_process.waitForFinished();
}

QString console_reader::QByteArray_to_QString(QByteArray & m_byteArray)
{
  return QString::fromLocal8Bit(m_byteArray);
}

void console_reader::read_console_output()
{
  // variables
  QByteArray temp_bytearray;
  bool newLineAtTheEnd;
  QStringList m_output_lines;

  // code

  /* Append the process output data to our internal buffer: m_uncomplete_line.
  * Because readAllStandardOutput() delivers a Bytearray with the data
  * who's actually available, that's not always exactly 1 line, but maybe
  * a half line or various (and a half) lines - because of that,
  * we need this buffer. */
  temp_bytearray = m_process.readAllStandardOutput();
  m_uncomplete_line.append(QByteArray_to_QString(temp_bytearray));
  // attention: QByteArray_to_QString could modify the bytearray. Don't use the bytearray any more!
  /* Use a unified way for "new line": First replace "\x0D\x0A" by "\x0A",
  *  than replace also all single "\x0D" by "\x0A". Doing it in this order
  *  garanties that we have later exactly as many line breaks as before!
  *
  *  I'm not sure if C++ doesn't interpretate \n on Windows platform sometimes
  *  as (13, 10) instead of (10), so using hex numbers I make sure that
  *  that isn't relevant. */
  m_uncomplete_line.replace("\x0D\x0A", "\x0A");
  m_uncomplete_line.replace("\x0D", "\x0A");

  // now process the data in our internal buffer
  newLineAtTheEnd = m_uncomplete_line.endsWith(10);
  m_output_lines=m_uncomplete_line.split(10, internal_splitBehavior);
  if (m_output_lines.isEmpty()) {
    // the internal buffer is made empty (possibly there where \n characters...):
    m_uncomplete_line="";
  } else {
    if (! newLineAtTheEnd) {
      /* In this case, the last line in the buffer isn't yet completely written by
      the command line program. So this last line stays in the buffer waiting for
      being completed the next time when the process writes data. (The
      other - complete - lines, saved in m_output_lines, will be processed.) */
      m_uncomplete_line=m_output_lines.takeLast(); /* removes the last item from the
      list and returns it -> here saving it to the buffer. Works only if m_output_lines
      isn't empty - for this, you need to put all this block in the "else" block
      from  if (m_output_lines.isEmpty())... */
    } else {
      m_uncomplete_line="";  // in this case, the last line is completely written. So
      // the buffer is made empty, because all the data will be processed now.
    };
    if (! m_output_lines.isEmpty()) {
      interpretate_console_output(m_output_lines);  // process the data // attention:
      // interpretate_console_output could modify m_output_lines (isn't const). Don't
      // use m_output_lines any more!
    };
  };
}
