/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mainwindow.h"

#include "settings_general_dialog.h"
#include "stationdirectorytree.h"
#include <QDockWidget>
#include <KApplication>
#include <KAction>
#include <KLocale>
#include <KActionCollection>
#include <KTipDialog>
#include <KStatusBar>
#include <KStandardDirs>

MainWindow::MainWindow(QWidget *parent) : KXmlGuiWindow(parent)
{
  //initialise central widget
  m_streamlistwidget = new stationlistWidget(this);
  setCentralWidget(m_streamlistwidget);

  //setup actions
  KStandardAction::preferences(this, SLOT(display_global_settings_dialog()),
                                actionCollection());

  KStandardAction::quit(kapp, SLOT(quit()), actionCollection());

  KAction * newStreamAction = new KAction(this);
  newStreamAction->setText(i18nc("@action", "&New stream"));
  newStreamAction->setIcon(KIcon("document-new"));
  newStreamAction->setShortcut(QKeySequence::New);
  connect(newStreamAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(addNewStation()));
  actionCollection()->addAction("newStream", newStreamAction);
  m_streamlistwidget->contextMenu.addAction(newStreamAction);

  KAction * deleteStreamAction = new KAction(this);
  deleteStreamAction->setText(i18nc("@action", "&Delete stream"));
  deleteStreamAction->setIcon(KIcon("edit-delete"));
  deleteStreamAction->setShortcut(Qt::SHIFT + Qt::Key_Delete);
  deleteStreamAction->setDisabled(true);
  connect(deleteStreamAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(deleteStation()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           deleteStreamAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           deleteStreamAction, SLOT(setEnabled(bool)));
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           deleteStreamAction, SLOT(setEnabled(bool)));
  actionCollection()->addAction("deleteStream", deleteStreamAction);
  m_streamlistwidget->contextMenu.addAction(deleteStreamAction);

  KAction * selectAllAction = KStandardAction::selectAll(m_streamlistwidget, SLOT(selectAll()),
                                                          actionCollection());
  m_streamlistwidget->contextMenu.addAction(selectAllAction);

  KAction * recordAction = new KAction(this);
  recordAction->setText(i18nc("@action", "&Record"));
  recordAction->setIcon(KIcon("media-record"));
  // TODO: use a appropriate value and accept multimedia keys also
  recordAction->setShortcut(Qt::CTRL + Qt::Key_R);
  recordAction->setDisabled(true);
  connect(recordAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(record()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           recordAction, SLOT(setDisabled(bool)));
  // TODO: enable only when the stream is really stoped...
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           recordAction, SLOT(setEnabled(bool)));
  // TODO: enable only when there are also streams that are really stoped...
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           recordAction, SLOT(setEnabled(bool)));
  actionCollection()->addAction("record", recordAction);
  m_streamlistwidget->contextMenu.addAction(recordAction);

  KAction * stopRecordAction = new KAction(this);
  stopRecordAction->setText(i18nc("@action", "&Stop"));
  stopRecordAction->setIcon(KIcon("media-playback-stop"));
  QList<QKeySequence> stopRecordAction_tempShortcutList;
  stopRecordAction_tempShortcutList.append(Qt::Key_S);
  stopRecordAction_tempShortcutList.append(Qt::CTRL + Qt::Key_S);
  stopRecordAction_tempShortcutList.append(Qt::META + Qt::Key_V);
  // TODO: use a appropriate value and accept multimedia keys also
  stopRecordAction->setShortcuts(stopRecordAction_tempShortcutList);
  stopRecordAction->setDisabled(true);
  connect(stopRecordAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(stopRecord()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           stopRecordAction, SLOT(setDisabled(bool)));
  // TODO: enable only when the streams are really running:
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           stopRecordAction, SLOT(setEnabled(bool)));
  // TODO: enable only when the streams are really running:
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           stopRecordAction, SLOT(setEnabled(bool)));
  actionCollection()->addAction("stopRecord", stopRecordAction);
  m_streamlistwidget->contextMenu.addAction(stopRecordAction);

  KAction * streamSettingsAction = new KAction(this);
  streamSettingsAction->setText(i18nc("@action properties of a stream)", "&Properties"));
  //stopRecordAction->setIcon(KIcon("media-playback-stop")); // TODO: use a appropriate value
  streamSettingsAction->setShortcut(Qt::ALT + Qt::Key_Return);
  // TODO: use a appropriate value and accept multimedia keys also
  streamSettingsAction->setDisabled(true);
  connect(streamSettingsAction, SIGNAL(triggered(bool)),
           m_streamlistwidget, SLOT(displayStreamSettings()));
  connect(m_streamlistwidget, SIGNAL(noneSelected(bool)),
           streamSettingsAction, SLOT(setDisabled(bool)));
  connect(m_streamlistwidget, SIGNAL(oneSelected(bool)),
           streamSettingsAction, SLOT(setEnabled(bool)));
  // TODO: maybe make this possible later:
  connect(m_streamlistwidget, SIGNAL(multipleSelected(bool)),
           streamSettingsAction, SLOT(setDisabled(bool)));
  actionCollection()->addAction("streamSettings", streamSettingsAction);
  m_streamlistwidget->contextMenu.addAction(streamSettingsAction);

  KStandardAction::tipOfDay(this, SLOT(displayTipOfDay()), actionCollection());

  //setup GUI
  setupGUI();

  statusBar()->insertPermanentItem(QString(), 0);
  statusBar()->insertPermanentItem(QString(), 1);
  connect(m_streamlistwidget->stationlistmodel(),
           SIGNAL(bandwidthChanged()),
           this,
           SLOT(actualize_bandwidth_in_statusBar()));
  connect(m_streamlistwidget->stationlistmodel(),
           SIGNAL(numberOfActiveStreamsChanged()),
           this,
           SLOT(actualize_numberOfActiveStreams_in_statusBar()));
  actualize_bandwidth_in_statusBar();  // initialization
  actualize_numberOfActiveStreams_in_statusBar();  // initialization

  QDockWidget *m_streamDirectory = new QDockWidget(
    i18nc("@title:window Dock name of the directory of radio stations",
          "Shoutcast radiostation directory"),
    this);
  m_streamDirectory->setObjectName("theRadiostationDirectory");  // to avoid a warning on stdout
  m_streamDirectory->setAllowedAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea |
    Qt::BottomDockWidgetArea);
  stationDirectoryTree *mainwidget_of_dock = new stationDirectoryTree(
    m_streamDirectory,
    KStandardDirs::locate("appdata", "action-favorite-genres-amarok.svgz"));
  m_streamDirectory->setWidget(mainwidget_of_dock);
  addDockWidget(Qt::LeftDockWidgetArea, m_streamDirectory);

  KTipDialog::showTip(this, "kradioripper/tips");
}

MainWindow::~MainWindow()
{
}

void MainWindow::actualize_numberOfActiveStreams_in_statusBar()
{
  statusBar()->changeItem(
    i18ncp("@info:status",
            "Recording %1 stream.",
            "Recording %1 streams.",
            m_streamlistwidget->stationlistmodel()->numberOfActiveStreams()),
    0);
}

void MainWindow::actualize_bandwidth_in_statusBar()
{
  statusBar()->changeItem(
    i18ncp("@info:status The unit is kbit (=1000 bit) instead of Kibit (=1024 bit). "
              "See http://en.wikipedia.org/wiki/Binary_prefix for details.",
            "%1 kbit/s",
            "%1 kbit/s",
            m_streamlistwidget->stationlistmodel()->bandwidth()),
    1);
}

bool MainWindow::queryClose()
{
  return m_streamlistwidget->queryClose();
}

void MainWindow::saveProperties(KConfigGroup & m_configGroup)
{
  m_streamlistwidget->saveProperties(m_configGroup);
}

void MainWindow::readProperties(const KConfigGroup & m_configGroup)
{
  m_streamlistwidget->readProperties(m_configGroup);
}

void MainWindow::display_global_settings_dialog()
{
  // if the dialog object yet exists, we can display it directly and return.
  if (KConfigDialog::showDialog("settings_general_dialog")) {
    return;
  };

  // else we have to construct and configure it before displaying it and return
  settings_general_dialog * dialog = new settings_general_dialog(this,
                                                                  "settings_general_dialog");
  dialog->show();
}

void MainWindow::displayTipOfDay()
{
  KTipDialog::showTip(this, "kradioripper/tips", true);
}
