/*
 *  Copyright (c) 2003 Patrick Julien  <freak@codepimps.org>
 *  Copyright (c) 2004 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KoColorSpaceRegistry.h"

#include <QStringList>
#include <QDir>

#include <kcomponentdata.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kservice.h>
#include <kservicetypetrader.h>

#include <lcms.h>

#include "KoPluginLoader.h"
#include "KoColorSpace.h"
#include "KoBasicHistogramProducers.h"

#include "colorprofiles/KoLcmsColorProfile.h"
#include "colorspaces/KoAlphaColorSpace.h"
#include "colorspaces/KoLabColorSpace.h"
#include "colorspaces/KoRgbU16ColorSpace.h"
#include "colorspaces/KoRgbU8ColorSpace.h"

struct KoColorSpaceRegistry::Private {
    QMap<QString, KoColorProfile * > profileMap;
    QMap<QString, KoColorSpace * > csMap;
    typedef QList<KisPaintDeviceAction *> PaintActionList;
    QMap<QString, PaintActionList> paintDevActionMap;
    KoColorSpace *alphaCs;
    static KoColorSpaceRegistry *singleton;
};

KoColorSpaceRegistry *KoColorSpaceRegistry::Private::singleton = 0;

KoColorSpaceRegistry* KoColorSpaceRegistry::instance()
{
    if(KoColorSpaceRegistry::Private::singleton == 0)
    {
        KoColorSpaceRegistry::Private::singleton = new KoColorSpaceRegistry();
        KoColorSpaceRegistry::Private::singleton->init();
    }
    return KoColorSpaceRegistry::Private::singleton;
}


void KoColorSpaceRegistry::init()
{
    // prepare a list of the profiles
    KGlobal::mainComponent().dirs()->addResourceType("kis_profiles",
                                                     "data", "krita/profiles/");

    QStringList profileFilenames;
    profileFilenames += KGlobal::mainComponent().dirs()->findAllResources("kis_profiles", "*.icm");
    profileFilenames += KGlobal::mainComponent().dirs()->findAllResources("kis_profiles", "*.ICM");
    profileFilenames += KGlobal::mainComponent().dirs()->findAllResources("kis_profiles", "*.ICC");
    profileFilenames += KGlobal::mainComponent().dirs()->findAllResources("kis_profiles", "*.icc");

    QDir dir("/usr/share/color/icc/", "*.icc;*.ICC;*.icm;*.ICM");

    QStringList filenames = dir.entryList();

    for (QStringList::iterator it = filenames.begin(); it != filenames.end(); ++it) {
        profileFilenames += dir.absoluteFilePath(*it);
    }

    dir.setPath(QDir::homePath() + "/.color/icc/");
    filenames = dir.entryList();

    for (QStringList::iterator it = filenames.begin(); it != filenames.end(); ++it) {
        profileFilenames += dir.absoluteFilePath(*it);
    }

    // Set lcms to return NUll/false etc from failing calls, rather than aborting the app.
    cmsErrorAction(LCMS_ERROR_SHOW);

    // Load the profiles
    if (!profileFilenames.empty()) {
        KoColorProfile * profile = 0;
        for ( QStringList::Iterator it = profileFilenames.begin(); it != profileFilenames.end(); ++it ) {
            profile = new KoLcmsColorProfile(*it);
            Q_CHECK_PTR(profile);

            profile->load();
            if (profile->valid()) {
                d->profileMap[profile->name()] = profile;
            }
        }
    }

    KoColorProfile *labProfile = new KoLcmsColorProfile(cmsCreateLabProfile(NULL));
    addProfile(labProfile);
    add(new KoLabColorSpaceFactory());
    KoHistogramProducerFactoryRegistry::instance()->add(
                new KoBasicHistogramProducerFactory<KoBasicU16HistogramProducer>
                (KoID("LABAHISTO", i18n("L*a*b* Histogram")), lab16()));
    KoColorProfile *rgbProfile = new KoLcmsColorProfile(cmsCreate_sRGBProfile());
    addProfile(rgbProfile);
    add(new KoRgbU16ColorSpaceFactory());

    add(new KoRgbU8ColorSpaceFactory());
    KoHistogramProducerFactoryRegistry::instance()->add(
                new KoBasicHistogramProducerFactory<KoBasicU8HistogramProducer>
                (KoID("RGB8HISTO", i18n("RGB8 Histogram")), rgb8()) );



    // Create the default profile for grayscale, probably not the best place to but that, but still better than in a grayscale plugin
    // .22 gamma grayscale or something like that. Taken from the lcms tutorial...
    LPGAMMATABLE Gamma = cmsBuildGamma(256, 2.2);
    cmsHPROFILE hProfile = cmsCreateGrayProfile(cmsD50_xyY(), Gamma);
    cmsFreeGamma(Gamma);
    KoColorProfile *defProfile = new KoLcmsColorProfile(hProfile);
    addProfile(defProfile);

    // Create the built-in colorspaces
    d->alphaCs = new KoAlphaColorSpace(this);

    KoPluginLoader::PluginsConfig config;
    config.whiteList = "ColorSpacePlugins";
    config.blacklist = "ColorSpacePluginsDisabled";
    config.group = "koffice";
    KoPluginLoader::instance()->load("KOffice/ColorSpace","[X-Pigment-Version] == 1", config);
}

KoColorSpaceRegistry::KoColorSpaceRegistry() : d(new Private())
{
}

KoColorSpaceRegistry::~KoColorSpaceRegistry()
{
    delete d;
}

KoColorProfile *  KoColorSpaceRegistry::profileByName(const QString & name) const
{
    if (d->profileMap.find(name) == d->profileMap.end()) {
        return 0;
    }

    return d->profileMap[name];
}

QList<KoColorProfile *>  KoColorSpaceRegistry::profilesFor(const QString &id)
{
    return profilesFor(value(id));
}

QList<KoColorProfile *>  KoColorSpaceRegistry::profilesFor(KoColorSpaceFactory * csf)
{
    QList<KoColorProfile *>  profiles;

    QMap<QString, KoColorProfile * >::Iterator it;
    for (it = d->profileMap.begin(); it != d->profileMap.end(); ++it) {
        KoColorProfile *  profile = it.value();
        if(csf->profileIsCompatible(profile))
        {
//         if (profile->colorSpaceSignature() == csf->colorSpaceSignature()) {
            profiles.push_back(profile);
        }
    }
    return profiles;
}

void KoColorSpaceRegistry::addProfile(KoColorProfile *p)
{
      if (p->valid()) {
          d->profileMap[p->name()] = p;
      }
}

void KoColorSpaceRegistry::addPaintDeviceAction(KoColorSpace* cs,
        KisPaintDeviceAction* action) {
    d->paintDevActionMap[cs->id()].append(action);
}

QList<KisPaintDeviceAction *>
KoColorSpaceRegistry::paintDeviceActionsFor(KoColorSpace* cs) {
    return d->paintDevActionMap[cs->id()];
}

KoColorSpace * KoColorSpaceRegistry::colorSpace(const QString &csID, const QString &pName)
{
    QString profileName = pName;

    if(profileName.isEmpty())
    {
        KoColorSpaceFactory *csf = value(csID);

        if(!csf)
            return 0;

        profileName = csf->defaultProfile();
    }

    QString name = csID + "<comb>" + profileName;

    if (d->csMap.find(name) == d->csMap.end()) {
        KoColorSpaceFactory *csf = value(csID);
        if(!csf)
        {
            kDebug() <<"Unknown color space type :" << csf;
            return 0;
        }

        KoColorProfile *p = profileByName(profileName);
        if(!p && !profileName.isEmpty())
        {
            kDebug() <<"Profile not found :" << profileName;
            return 0;
        }
        KoColorSpace *cs = csf->createColorSpace(this, p);
        if(!cs)
            return 0;

        d->csMap[name] = cs;
    }

    if(d->csMap.contains(name))
        return d->csMap[name];
    else
        return 0;
}


KoColorSpace * KoColorSpaceRegistry::colorSpace(const QString &csID, const KoColorProfile *profile)
{
    if( profile )
    {
        KoColorSpace *cs = colorSpace( csID, profile->name());

        if(!cs)
        {
            // The profile was not stored and thus not the combination either
            KoColorSpaceFactory *csf = value(csID);
            if(!csf)
            {
                kDebug() <<"Unknown color space type :" << csf;
                return 0;
            }

            cs = csf->createColorSpace(this, const_cast<KoColorProfile *>(profile));
            if(!cs )
                return 0;

            QString name = csID + "<comb>" + profile->name();
            d->csMap[name] = cs;
        }

        return cs;
    } else {
        return colorSpace( csID, "");
    }
}

KoColorSpace * KoColorSpaceRegistry::alpha8()
{
   return d->alphaCs;
}

KoColorSpace * KoColorSpaceRegistry::rgb8(const QString &profileName)
{
    return colorSpace(KoRgbU8ColorSpace::colorSpaceId(), profileName);
}

KoColorSpace * KoColorSpaceRegistry::rgb8(KoColorProfile * profile)
{
    return colorSpace(KoRgbU8ColorSpace::colorSpaceId(), profile);
}

KoColorSpace * KoColorSpaceRegistry::rgb16(const QString &profileName)
{
    return colorSpace(KoRgbU16ColorSpace::colorSpaceId(), profileName);
}

KoColorSpace * KoColorSpaceRegistry::rgb16(KoColorProfile * profile)
{
    return colorSpace(KoRgbU16ColorSpace::colorSpaceId(), profile);
}

KoColorSpace * KoColorSpaceRegistry::lab16(const QString &profileName)
{
    return colorSpace(KoLabColorSpace::colorSpaceId(), profileName);
}

KoColorSpace * KoColorSpaceRegistry::lab16(KoColorProfile * profile)
{
    return colorSpace(KoLabColorSpace::colorSpaceId(), profile);
}

QList<KoID> KoColorSpaceRegistry::colorModelsList() const
{
    QList<KoID> ids;
    QList<KoColorSpaceFactory*> factories = values();
    foreach(KoColorSpaceFactory* factory, factories)
    {
        if(!ids.contains(factory->colorModelId()))
        {
            ids << factory->colorModelId();
        }
    }
    return ids;
}

QList<KoID> KoColorSpaceRegistry::colorDepthList(const KoID& colorModelId) const
{
    QList<KoID> ids;
    QList<KoColorSpaceFactory*> factories = values();
    foreach(KoColorSpaceFactory* factory, factories)
    {
        if(!ids.contains(factory->colorDepthId()) && factory->colorModelId() == colorModelId )
        {
            ids << factory->colorDepthId();
        }
    }
    return ids;
}

QString KoColorSpaceRegistry::colorSpaceId(const KoID& colorModelId, const KoID& colorDepthId)
{
    QList<KoColorSpaceFactory*> factories = values();
    foreach(KoColorSpaceFactory* factory, factories)
    {
        if(factory->colorModelId() == colorModelId && factory->colorDepthId() == colorDepthId )
        {
            return factory->id();
        }
    }
    return "";
}

#include "KoColorSpaceRegistry.moc"
