/***************************************************************************
                                QSSegment.h                              
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef QSSEGMENT_H
#define QSSEGMENT_H

#include"qscoord.h"
#include"qsgattr.h"
#include"qsmatrix.h"

class QSDrv;
class QSCurve;
class QSAxes;

/**
  * \brief Defines an appearance of QSCurve  Base class
  *
  * Draws a single element of QSCurve. It can be reimplemented to draw Bars, Vectors, etc, see QSCurve::setPObject.
  * startDrawing() is always called before the drawing begins, endDrawing() is called after drawing is finished.
  * When drawing initPass() is called and next drawSegment() method is called for each datapoint, finally
  * endPass() in invoked and this operation is repeated a few times ( as requested by startDraw() return value )..
  * @author Kamil Dobkowski
  */
class QSSegment {

  public:
   /**
     * Constructor.
     */
   QSSegment();
   /**
     * Destructor
     */
   virtual ~QSSegment();
   /**
     * Returns the requested number of passes.
     */
   virtual int startDraw( const QSCurve * );
   /**
     * Inits pass
     */
   virtual void initPass( int pass );
   /**
     * Draws a single point of the data
     */
   virtual void drawSegment( int nr, const QSPt2f& point, const QSPt2f& delta, const QSGLine& l, const QSGFill& f, const QSGArrow& a1, const QSGArrow& a2 ) = 0;
   /**
     * Ends pass
     */
   virtual void endPass() {}
   /**
     * Ends drawing
     */
   virtual void stopDraw() {}

  protected:
    QSDrv *m_drv;
    const QSCurve *m_parent;
  };

//-------------------------------------------------------------//

/**
  * \brief Defines an appearance of QSCurve. Draws an ordinary curve
  *
  * @author Kamil Dobkowski
  */
class QSSLines : public QSSegment  {

  public:

     QSSLines();
     virtual ~QSSLines();
     virtual int startDraw( const QSCurve *parent );
     virtual void initPass( int pass );
     virtual void drawSegment( int nr, const QSPt2f& point, const QSPt2f& delta, const QSGLine& l, const QSGFill& f, const QSGArrow& a1, const QSGArrow& a2 );
     virtual void endPass();
     virtual void stopDraw();

  private:

    QSGLine m_curr_line;
};

//-------------------------------------------------------------//

/**
  * \brief Defines an appearance of QSCurve. Draws area plots and ribbons
  * @author Kamil Dobkowski
  */
class QSSPolys : public QSSegment  {

  public:

     enum Style { Area = 0, Ribbon = 1 };

     QSSPolys( Style style );
     virtual ~QSSPolys();
     virtual int startDraw( const QSCurve *parent );
     virtual void initPass( int pass );
     virtual void drawSegment( int nr, const QSPt2f& point, const QSPt2f& delta, const QSGLine& l, const QSGFill& f, const QSGArrow& a1, const QSGArrow& a2 );
     virtual void endPass();
     virtual void stopDraw();

  private:
	
    Style m_style;
    double m_zero_level;
    QSPt2f m_prev1;
    QSPt2f m_prev2;
    int m_pass;
    QSGLine m_curr_line;
    QSGFill m_curr_fill;
};

//-------------------------------------------------------------//

/**
  * \brief Defines an appearance of QSCurve. Draws simpl bars.
  * @author Kamil Dobkowski
  */
class QSSBars : public QSSegment  {

  public:
	
     QSSBars();
     virtual ~QSSBars();
     virtual int startDraw( const QSCurve *parent );
     virtual void initPass( int pass );
     virtual void drawSegment( int nr, const QSPt2f& point, const QSPt2f& delta, const QSGLine& l, const QSGFill& f, const QSGArrow& a1, const QSGArrow& a2 );
     virtual void endPass();
     virtual void stopDraw();

  private:

    double m_zero_level;
    double m_prevx;
    QSPt2f m_prev;
    QSGLine m_curr_line;
    QSGFill m_curr_fill;
    void draw_bar( int nr, const QSPt2f& pos );
};

//-------------------------------------------------------------//

/**
  * \brief Defines an appearance of QSCurve. Draws vectors, ellipses, rectangles, flux.
  * @author Kamil Dobkowski
  */
class QSSFigures: public QSSegment  {

  public:
	
   enum Style { Vectors = 0, Rectangles = 1, Ellipses = 2, Flux = 3 };

   QSSFigures( Style style );
   virtual ~QSSFigures();
   virtual int startDraw( const QSCurve *parent );
   virtual void initPass( int pass );
   virtual void drawSegment( int nr, const QSPt2f& point, const QSPt2f& delta, const QSGLine& l, const QSGFill& f, const QSGArrow& a1, const QSGArrow& a2 );
   virtual void endPass();
   virtual void stopDraw();

  private:
	int m_style;
  };

//-------------------------------------------------------------//

/**
  * \brief Defines an appearance of QSCurve. Draws stairs plots.
  * @author Kamil Dobkowski
  */
class QSSStairs : public QSSegment  {

  public:

     enum Style { Left, Middle, Right };

     QSSStairs( Style style );
     virtual ~QSSStairs();
     virtual int startDraw( const QSCurve *parent );
     virtual void initPass( int pass );
     virtual void drawSegment( int nr, const QSPt2f& point, const QSPt2f& delta, const QSGLine& l, const QSGFill& f, const QSGArrow& a1, const QSGArrow& a2 );
     virtual void endPass();
     virtual void stopDraw();

  private:
    Style m_style;
    QSPt2f m_prev;
    QSPt2f m_prev_prev;
    QSGLine curr_line;
    void get_stair( const QSPt2f& pos, QSPt2f pos[3] );
};

#endif








