/***************************************************************************
                                 qgraphicsldata.cpp
                             -------------------
    begin                : 01-January-2001

    copyright            : (C) 2001 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qsgraphicaldata.h"

//-------------------------------------------------------------//

QSGraphicalData::QSGraphicalData( QObject *parent, QSGraphicalData *parentObject, const char * name )
:QSData( parent, parentObject, name )
 {
  m_parent_base_object = parentObject;
  m_channels_count = 0;
  m_channels = NULL;
  m_settings.lines = NULL;
  m_settings.fills = NULL;
  m_settings.fonts = NULL;
  m_settings.points = NULL;
  m_settings.lines_count = 0;
  m_settings.fills_count = 0;
  m_settings.fonts_count = 0;
  m_settings.points_count = 0;
 }

//-------------------------------------------------------------//

QSGraphicalData::~QSGraphicalData()
 {
  int i;
  for ( i=0; i< m_channels_count; i++ ) delete m_channels[i];
  delete[] m_settings.lines;
  delete[] m_settings.fills;
  delete[] m_settings.fonts;
  delete[] m_settings.points;
 }

//-------------------------------------------------------------//

void QSGraphicalData::setTitle( const QString& new_title )
 {
  if ( m_title_str != new_title ) {
  	parametersChanging();
  	m_title_str = new_title;
	emit sigTitleChanged( m_title_str );
  	parametersChanged();
  	}
 }
//-------------------------------------------------------------//

void QSGraphicalData::initChannelTable( int channels_num )
 {
  m_channels_count = channels_num;
  m_channels = new QSMatrix*[m_channels_count];
  for ( int i=0; i<m_channels_count; i++ ) m_channels[i] = NULL;
 }

//-------------------------------------------------------------//

void QSGraphicalData::initAttributeTables( int fonts_num, int fills_num,  int lines_num, int points_num )
 {
  m_settings.fonts_count = fonts_num;
  m_settings.fills_count = fills_num;
  m_settings.lines_count = lines_num;
  m_settings.points_count = points_num;
  m_settings.fonts  = new QSGFont[fonts_num];
  m_settings.fills  = new QSGFill[fills_num];
  m_settings.lines  = new QSGLine[lines_num];
  m_settings.points = new QSGPoint[points_num];
 }

//-------------------------------------------------------------//

bool QSGraphicalData::setMatrix( int channel, QSMatrix *m )
 {
  if ( channel >= 0 && channel < m_channels_count ) {
                dataChanging(channel);
                delete m_channels[channel];
                m_channels[channel] = m;
		if ( m ) m->setDataObject( this, channel );
                dataChanged(channel);
                return true;
               }
  return false;
 }

//-------------------------------------------------------------//

QSMatrix *QSGraphicalData::matrix( int channel ) const
 {
  return channel < m_channels_count ? m_channels[channel] : NULL;
 }

//-------------------------------------------------------------//

QSMatrix *QSGraphicalData::takeMatrix( int channel, QSMatrix *newMatrix )
 {
 if ( channel >= 0 && channel < m_channels_count && m_channels[channel] != newMatrix ) {
                dataChanging(channel);
                QSMatrix *result = m_channels[channel];
		if ( result ) result->setDataObject( NULL, -1 );
                m_channels[channel] = newMatrix;
		if ( newMatrix ) newMatrix->setDataObject( this, channel );
                dataChanged(channel);
                return result;
               }
  return NULL;
 }

//-------------------------------------------------------------//

void QSGraphicalData::deleteMatrix( int channel )
 {
  if ( channel >=0 && channel < m_channels_count && m_channels[channel] ) {
         dataChanging(channel);
         delete m_channels[channel];
         m_channels[channel] = NULL;
         dataChanged(channel);
        }
 }

//-------------------------------------------------------------//

bool QSGraphicalData::isChannel( int channel ) const
 {
   if (channel<m_channels_count )
        if ( m_channels[channel] != NULL )
              if ( m_channels[channel]->cols() > 0 &&
                   m_channels[channel]->rows() > 0  ) return true;
   return false;
 }

//-------------------------------------------------------------//

void QSGraphicalData::setFont( int e, const QSGFont &font )
 {
  if ( e >= 0 && e < m_settings.fonts_count && m_settings.fonts[e] != font ) {
         parametersChanging();
         m_settings.fonts[e] = font;
         parametersChanged();
        }
 }

//-------------------------------------------------------------//

void QSGraphicalData::setFill( int e, const QSGFill &fill )
 {
  if ( e >= 0 && e < m_settings.fills_count && m_settings.fills[e] != fill  ) {
         parametersChanging();
         m_settings.fills[e] = fill;
         parametersChanged();
        }
 }

//-------------------------------------------------------------//

void QSGraphicalData::setLine( int e, const QSGLine &line )
 {
   if ( e >= 0 && e < m_settings.lines_count && m_settings.lines[e] != line  ) {
         parametersChanging();
         m_settings.lines[e] = line;
         parametersChanged();
        }
 }

//-------------------------------------------------------------//

void QSGraphicalData::setPoint( int e, const QSGPoint &point )
 {
   if ( e >= 0 && e < m_settings.points_count && m_settings.points[e] != point ) {
         parametersChanging();
         m_settings.points[e] = point;
         parametersChanged();
        }
 }

//-------------------------------------------------------------//

void QSGraphicalData::parametersChanging()
 {
  if ( m_parent_base_object ) m_parent_base_object->parametersChanging();
 }

//-------------------------------------------------------------//

void QSGraphicalData::parametersChanged()
 {
  if ( m_parent_base_object ) m_parent_base_object->parametersChanged();
 }

//-------------------------------------------------------------//

QSGFont QSGraphicalData::fontFromData( QSMatrix *source, int row, int start_col, const QSGFont& default_value )
 {
  QSGFont result = default_value;
  if ( source && source->rows() > row ) {
	int cols = source->cols() - start_col;
	if ( cols>0 && !source->string(row,start_col+0).isEmpty() ) result.family  = source->string(row,start_col+0);
        if ( cols>1 && !source->string(row,start_col+1).isEmpty() ) result.size    = (int )source->value(row,start_col+1);
        if ( cols>2 && !source->string(row,start_col+2).isEmpty() ) result.bold    = (bool )source->value(row,start_col+2);
        if ( cols>3 && !source->string(row,start_col+3).isEmpty() ) result.italic  = (bool )source->value(row,start_col+3);
        result.color = colorFromData( source, row, start_col+4, result.color );
	}
  return result;
 }

//-------------------------------------------------------------//

QSGFill QSGraphicalData::fillFromData( QSMatrix *source, int row, int start_col, const QSGFill& default_value )
  {
  QSGFill result = default_value;
  if ( source && source->rows() > row ) {
	int cols = source->cols() - start_col;
	if ( cols>0 && !source->string(row,start_col+0).isEmpty() ) result.style  = (QSGFill::Style )source->value(row,start_col+0);
        result.color = colorFromData( source, row, start_col+1, result.color );
	}
  return result;
  }

//-------------------------------------------------------------//

QSGLine QSGraphicalData::lineFromData( QSMatrix *source, int row, int start_col, const QSGLine& default_value )
  {
  QSGLine result = default_value;
  if ( source && source->rows() > row ) {	
	int cols = source->cols() - start_col;
	if ( cols>0 && !source->string(row,start_col+0).isEmpty() ) result.style  = (QSGLine::Style )source->value(row,start_col+0);
	if ( cols>1 && !source->string(row,start_col+1).isEmpty() ) result.width  = (int )source->value(row,start_col+1);
        result.color = colorFromData( source, row, start_col+2, result.color );
	}
  return result;
  }

//-------------------------------------------------------------//

QSGPoint QSGraphicalData::pointFromData( QSMatrix *source, int row, int start_col, const QSGPoint& default_value )
  {
  QSGPoint result = default_value;
  if ( source && source->rows() > row ) {	
	int cols = source->cols() - start_col;
	if ( cols>0 && !source->string(row,start_col+0).isEmpty() ) result.style  = (QSGPoint::Style )source->value(row,start_col+0);
	if ( cols>1 && !source->string(row,start_col+1).isEmpty() ) result.fill   = (QSGPoint::Fill  )source->value(row,start_col+1);
	if ( cols>2 && !source->string(row,start_col+2).isEmpty() ) result.size   = (int )source->value(row,start_col+2);
        result.color = colorFromData( source, row, start_col+3, result.color );
	}
   return result;
  }

//-------------------------------------------------------------//

QSGArrow QSGraphicalData::arrowFromData( QSMatrix *source, int row, int start_col, const QSGArrow& default_value )
  {
   QSGArrow result = default_value;
   if ( source && source->rows() > row ) {	
   	int cols = source->cols() - start_col;
   	if ( cols>0 && !source->string(row,start_col+0).isEmpty() ) result.style  = (QSGArrow::Style )source->value(row,start_col+0);
   	if ( cols>1 && !source->string(row,start_col+1).isEmpty() ) result.size  = (int )source->value(row,start_col+1);
	}
   return result;
  }

//-------------------------------------------------------------//

QSGColor QSGraphicalData::colorFromData( QSMatrix *source, int row, int start_col, const QSGColor& default_value )
  {
   QSGColor result = default_value;
   if ( source && source->rows() > row ) {
	int cols = source->cols() - start_col;
        if ( cols>0 && !source->string(row,start_col+0).isEmpty() ) result.r = (unsigned char )source->value(row,start_col+0);
        if ( cols>1 && !source->string(row,start_col+1).isEmpty() ) result.g = (unsigned char )source->value(row,start_col+1);
        if ( cols>2 && !source->string(row,start_col+2).isEmpty() ) result.b = (unsigned char )source->value(row,start_col+2);
        if ( cols>3 && !source->string(row,start_col+3).isEmpty() ) result.a = (unsigned char )source->value(row,start_col+3);	
	}
   return result;
  }

//-------------------------------------------------------------//

QSGraphicalData::ColumnType QSGraphicalData::fontColumnFormat( int column )
 {
  switch( column ) {
	case 0: return ColumnFontFamily;
	case 1: return ColumnSize;
	case 2: return ColumnFontBold;
	case 3: return ColumnFontItalic;
	default: return colorColumnFormat( column-4 );
	}
 }

//-------------------------------------------------------------//

QSGraphicalData::ColumnType QSGraphicalData::fillColumnFormat( int column )
 {
  switch( column ) {
	case 0: return ColumnFillStyle;
	default: return colorColumnFormat( column-1 );
	}
 }

//-------------------------------------------------------------//

QSGraphicalData::ColumnType QSGraphicalData::lineColumnFormat( int column )
 {
  switch( column ) {
	case 0: return ColumnLineStyle;
	case 1: return ColumnWidth;
	default: return colorColumnFormat( column-2 );
	}
 }

//-------------------------------------------------------------//

QSGraphicalData::ColumnType QSGraphicalData::pointColumnFormat( int column )
 {
  switch( column ) {
	case 0: return ColumnPointStyle;
	case 1: return ColumnPointFill;
	case 2: return ColumnSize;
	default: return colorColumnFormat( column-3 );
	}
 }

//-------------------------------------------------------------//

QSGraphicalData::ColumnType QSGraphicalData::arrowColumnFormat( int column )
 {
  switch( column ) {
	case 0: return ColumnArrowStyle;
	case 1: return ColumnSize;
	default: return ColumnUnknown;
	}
 }

//-------------------------------------------------------------//

QSGraphicalData::ColumnType QSGraphicalData::colorColumnFormat( int column )
 {
  switch( column ) {
	case 0: return ColumnRed;
	case 1: return ColumnGreen;
	case 2: return ColumnBlue;
	case 3: return ColumnAlpha;
	default: return ColumnUnknown;
	}
 }

//-------------------------------------------------------------//

void QSGraphicalData::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSData::loadStateFromStream( stream, factory );
  QString data;
  // load fonts
  for( int i=0; i<fontCount(); i++ ) { stream >> data; setFont( i, toQSGFont(data) ); }
  // load fills
  for( int i=0; i<fillCount(); i++ ) { stream >> data; setFill( i, toQSGFill(data) ); }
  // load lines
  for( int i=0; i<lineCount(); i++ ) { stream >> data; setLine( i, toQSGLine(data) ); }
  // load points
  for( int i=0; i<pointCount(); i++ ) { stream >> data; setPoint( i, toQSGPoint(data) ); }
 }

//-------------------------------------------------------------//

void QSGraphicalData::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSData::saveStateToStream( stream, factory );
  // save fonts
  for( int i=0; i<fontCount(); i++ ) stream << toQString(font(i));
  // save fills
  for( int i=0; i<fillCount(); i++ ) stream << toQString(fill(i));
  // save lines
  for( int i=0; i<lineCount(); i++ ) stream << toQString(line(i));
  // save points
  for( int i=0; i<pointCount(); i++ ) stream << toQString(point(i));
 }




