/***************************************************************************
                                   qdata.h
                             -------------------
    version              :
    begin                :

    copyright            : (C) 2000 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef QSDATA_H
#define QSDATA_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include"qsgattr.h"
#include"qsmatrix.h"
#include"qsserializable.h"
#include<qobject.h>
#include<qstring.h>
#include<math.h>

//-------------------------------------------------------------------------------------------------------------//

/**
  * \brief Object which contains data ( list of matrices )
  *
  * Object, which contains data - a list of matrices, see setMatrix(), matrix(). This class is
  * inherited by all dataset types: QSSurface, QSCurve, etc. and axes : QSAxes2D, QSAxes3D etc.
  * For example an ordinary XYPlot ( QSCurve ) contains XVector, YVector, DX Vector, DYVector. and draws all succesive
  * ( x, y ,dx, dy ) points as a curve with error bars.  This object can own child QSData objects, for example: axes
  * contains curves ( but curves have no furher child object ) - see child(), childCount(), childIndex(). This is only an
  * interface class and does not contain implementation of matrix list.
  * If you are going to change data ( change value of some matrix element, etc ) you have to notify QSData object using
  * dataChanging(), dataChanged() before and after the change, and it notifies outside world about this fact using
  * sigDataChanged() .
  * @see QSMatrix
  * @author Kamil Dobkowski
  */
class QSData : public QObject, public QSSerializable
 {
  Q_OBJECT

  public:
       /**
         * Column type
         */
       enum ColumnType {
		ColumnUnknown = 0,
		ColumnX,
		ColumnY,
		ColumnZ,
		ColumnV,
		ColumnDX,
		ColumnDY,
		ColumnSize,
		ColumnWidth,
		ColumnLineStyle,
		ColumnFillStyle,
		ColumnPointStyle,
		ColumnPointFill,
		ColumnArrowStyle,
		ColumnFontFamily,
		ColumnFontSize,
		ColumnFontBold,
		ColumnFontItalic,
		ColumnRed,
		ColumnGreen,
		ColumnBlue,
		ColumnAlpha,
		ColumnByte,
		ColumnInt,
		ColumnBool,
		ColumnString
		};
      /**
        * Constructor
        */
      QSData( QObject *parent=NULL, QSData *parentObject=NULL, const char *name=0 );
       /**
	 * Destructor. It should delete all child object as well.
	 */
      virtual ~QSData();
	/**
	*  The same as setName() but emits nameChanged( const char *newName ).
	*/
      virtual void setObjectName( const char *name );
	/**
	  * The same as name()
	  */
      const char *objectName() const { return name(); }
      /**
        * Sets a new matrix as the given channel in the plot.
        */
      virtual bool setMatrix( int channel, QSMatrix *m );
      /**
        * Returns matrix 'channel' if any, or NULL.
        */
      virtual QSMatrix *matrix( int channel ) const;
      /**
        * Gives ownership of the matrix to caller and
	* sets data to newMatrix.
        */
      virtual QSMatrix *takeMatrix( int channel, QSMatrix *newMatrix = NULL );
      /**
        * Deletes the choosen matrix.
        */
      virtual void deleteMatrix( int channel );
      /**
        * Returns the total number of channels.
        */
      virtual int channelCount() const;
       /**
         * This function is called before data is changed by 'setMatrix'
         * method. You should also call it by hand, when you are going to change
         * the data in any matrix..
         * It informs derived classes to stop a drawing process if it is in progress at the
         * moment.
         * Default implementation calls dataChanging( this, channel ),
         */
      virtual void dataChanging( int channel = -1 );
      /**
        * This function is called after data has changed by 'setMatrix'
        * method. You should also call it by hand, when you have changed
        * the data in any matrix.
        * It informs derived classes to mark old max and min data values as invalid,
        * and informs them that they should repaint themselves.
	* Default implementation calls dataChanging( this, channel ),
        */
      virtual void dataChanged( int channel = -1 );
	/**
	  * Returns the parent object.
	  */
	virtual QSData *parentObject() const { return m_parent_object; }
      /**
        * Number of child QSData objects
        */
      virtual int childCount() const { return 0; }
      /**
        * Returns a child object
        */
      virtual QSData *child( int index ) const { return NULL; }
      /**
        * Returns a child index.
        */
      virtual int childIndex( QSData *child ) const;
	/**
	  * Returns a variable name ( 'x', 'y', 'z', 'v' etc ) for use in formulas.
	  */
      virtual QString channelVariable( int channel ) const { return QString::null; }
      /**
        * Metadata - type of the given column. It should be reimplemented in derived classes.
	* Used to create editors in sheet
        */	
      virtual ColumnType columnType( int channel, int column ) const { return ColumnUnknown; }
      /**
        * Restores all QObject properties. Loads all matrices
        */
	virtual void loadStateFromStream( QDataStream& stream, QSObjectFactory *factory );
	/**
	  * Saves all QObjects properties. Saves all matrices.
	  */
	virtual void saveStateToStream( QDataStream& stream, QSObjectFactory *factory );

   signals:	
	/**
	  * Name of this object has changed
	  */
	void sigNameChanged( const char *name );
	/**
	  * Name of this object has changed
	  */
	void sigNameChanged( QSData *thisOrChildObject, const char *name );
      /**
        * Data has changed in this or any child object.
        */
      void sigDataChanged( QSData *thisOrChildObject, int channel );
      /**
        * A new child object was added to the child list.
        */
      void sigChildAdded( QSData *newChild );
      /**
        * Child object is about to be removed from the list.
        */
      void sigChildRemoved( QSData *removedChild );
      /**
        * Index of some child object was changed ( the list was reordered ).
        */
      void sigChildOrder();
	/**
	  * sigChildAdded() + sigChildRemoved() + sigChildOrder()
	  */
	void sigChildListChanged();
      /**
        * Ups ! It is emmited from destructor of this object.
        */
      void sigDeleted( QSData *thisObject );

   protected:
      /**
        * Does nothing. This function collects all 'dataChanging()' calls from this object and child objects.
        * if parent is not NULL calls parent->dataChanging( this, channel );
        */
      virtual void dataChanging( QSData *object, int channel = -1 );
      /**
	* Emits sigDataChanged(). This function collects all 'dataChanged()' calls from this object and child objects.
        * and if parent is not NULL calls parent->dataChanged( this, channel );
        */
      virtual void dataChanged( QSData *object, int channel = -1 );
      /**
        *
        */
       virtual void nameChanged( QSData *object, const char *newName );
	
      QSData *m_parent_object;
 };

#endif


