/***************************************************************************
                                qscurve.h
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef QSCURVE_H
#define QSCURVE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include"qsaxes.h"
#include"qsplot.h"
#include"qscoord.h"
#include"qssegment.h"

//-------------------------------------------------------------------//



/**
  * \brief Dataset: XY plot.
  *
  * Base class for parametric curves. The appearance of QSCurve can be customized by setting your own
  * QSSegment object, see setPObject() . Data format is described in QSCurve::Channels .See also QSAxes::plotAdd() .
  * @author Kamil Dobkowski
  */
class QSCurve : public QSPlot2D
 {
  Q_OBJECT
	Q_PROPERTY( int type READ type WRITE setType )
	Q_PROPERTY( double fixedDX READ fixedDX WRITE setFixedDX )
	Q_PROPERTY( double fixedDY READ fixedDY WRITE setFixedDY )
	Q_PROPERTY( double percentDX READ percentDX WRITE setPercentDX )
	Q_PROPERTY( double percentDY READ percentDY WRITE setPercentDY )
	Q_PROPERTY( double zeroLevelX READ zeroLevelX WRITE setZeroLevelX )
	Q_PROPERTY( double zeroLevelY READ zeroLevelY WRITE setZeroLevelY )
	Q_PROPERTY( QString arrow1 READ arrow1Property WRITE setArrow1Property )
	Q_PROPERTY( QString arrow2 READ arrow2Property WRITE setArrow2Property )
	
  public:

        /**
          * Descriptive names of data channels. See QSData::setMatrix()
	  * All channels must be column vectors with the same number of rows.
	  * Values from the same row from all matrices define a one data point.
	  * XVector contains the x coordinate, YVector contains the y coordinate, DX
	  * vector contain the dx coordinate ( used for vectors, error bars etc ),
	  * DY vector contains the dy coordinate, and other vectors contain other attributes.
          */
        enum Channels   {
                          XVector  = 0,
                          YVector  = 1,
                          DXVector = 2,
                          DYVector = 3,
                          LineStyles = 4,
                          FillStyles = 5,
			  PointStyles = 6,
			  ArrowStyles = 7,
			  ErrorLineStyles = 8,
			  XLineStyles = 9,
			  YLineStyles = 10
                        };


        /**
          * Types of the series.
          * @see #setPlotType
          */
	enum SeriesType {
			 Lines = 0,
			 Area,
			 Ribbon,
			 Bars,
			 Vectors,
			 Flux,
			 Rectangles,
			 Ellipses,
			 LeftStairs,
			 MiddleStairs,
			 RightStairs,
			 User
			};
	
       /**
         * Constructor.
	 */
	QSCurve( QSAxes* parent, const char * name=0 );
       /**
         * Destructor.
         */
       ~QSCurve();
        /**
          * Sets a new plot type. See 'SeriesType'
          */
        void setType( int type );
        /**
          * Returns the plot type.
          */
        int type() const { return m_type; }
	/**
	  * Delta acts as a DX,DY channel with fixed values.
	  * Used when drawing vectors, error-bars, rectangle-type plots.
	  * Total delta value is calculated as follows ( example for DX ):
	  * DX[i] = DXVector[i] + FixedDelta.X + PercentDelta.X*DXVector[i]/100.0;
	  * Error is assumed to be +-delta.
	  */
	void setFixedDelta( const QSPt2f& delta );
	/**
          * Sets a new fixed delta value.
          */
	void setFixedDX( double value );
	/**
          * Sets a new fixed delta value.
          */
	void setFixedDY( double value );
	/**
	  * Delta PDX[i] = PercentDelta.X*YVector[i]/100.0.
	  * Total delta value is calculated as follows ( example for DX ):
	  * DX[i] = DXVector[i] + FixedDelta.X + PercentDelta.X*DXVector[i]/100.0;
	  */	
	void setPercentDelta( const QSPt2f& delta );
	/**
          * Sets a percent delta
	  */	
	void setPercentDX( double value );
	/**
          * Sets a percent delta
	  */	
	void setPercentDY( double value );
	/**
	  * Returns a fixed delta value.
	  */
	QSPt2f fixedDelta() const { return m_fdelta; }
	/**
	  * Returns a fixed delta value.
	  */
	double fixedDX() const { return m_fdelta.x; }
	/**
	  * Returns a fixed delta value.
	  */
	double fixedDY() const { return m_fdelta.y; }
	/**
	  * Returns a delta value.
	  */
	QSPt2f percentDelta() const { return m_pdelta; }
	/**
	  * Returns a delta value.
	  */
	double percentDX() const { return m_pdelta.x; }
	/**
	  * Returns a delta value.
	  */
	double percentDY() const { return m_pdelta.y; }
        /**
          * Sets a zero level. It is the y value from which, for example,
          * sample lines starts.
          */
        void setZeroPoint( const QSPt2f& p );
        /**
          * Sets zero level.
          */
        void setZeroLevelX( double x );
        /**
          * Sets zero level.
          */
        void setZeroLevelY( double y );	
        /**
          * Returns the zero-level.
          */
        QSPt2f zeroPoint() const { return m_zero; }
	/**
          * Returns the zero-level.
          */
	double zeroLevelX() const { return m_zero.x; }
	/**
          *  Returns the zero-level.
          */
	double zeroLevelY() const { return m_zero.y; }
        /**
          * Sets a new plot object. Normally there is no need to use it.
          * Pointer is consumed and will be deleted when not needed.
          * Remember that also 'User' plot type must be set.
          */
        void setPObject( QSSegment *po );
        /**
          * Returns the plot object or NULL.
          */
        QSSegment *pObject() const { return m_po; }
	/**
          * This is usable only for vectors or errorbars.
          */
	void setArrow1( const QSGArrow& astyle );
	/**
          * This is usable only for vectors or errorbars.
          */
	void setArrow2( const QSGArrow& astyle );
	/**
          * Sets arrow property
          */
	void setArrow1Property( const QString& value ) { setArrow1(toQSGArrow(value)); }
	/**
          * Sets arrow property
          */
	void setArrow2Property( const QString& value ) { setArrow2(toQSGArrow(value)); }
        /**
          * Returns arrow1
          */
	QSGArrow arrow1() const { return m_arrow1; }
        /**
          * Returns arrow2
          */
	QSGArrow arrow2() const { return m_arrow2; }
	/**
          * Returns arrow1 property
          */
	QString arrow1Property() const { return toQString(arrow1()); }
	/**
	  * Returns arrow2 property
	  */
	QString arrow2Property() const { return toQString(arrow2()); }


	virtual QString posInfo( QSPt2f& pos );
        virtual QSPt2f legendItemSize( QSDrv *drv );
        virtual void drawLegendItem( const QSPt2f& pos, QSDrv *drv );
	virtual void loadStateFromStream( QDataStream& stream, QSObjectFactory *factory );
	virtual void saveStateToStream( QDataStream& stream, QSObjectFactory *factory );
	virtual QString channelVariable( int channel ) const;	
        virtual ColumnType columnType( int channel, int column ) const;
         /**
	   * See QSGraphicalData::setFill()
	   */
	 enum FillElement {
	 	BaseFill	= 0
	 	};
	 	
	/**
	  * See QSGraphicalData::setLine()
	  */
	 enum LineElement {
	 	BaseLine 	= 0,
	 	ErrorLine 	= 1,
	 	XLine		= 2,
	 	YLine		= 3
	 	};
	 	
        /**
	  * See QSGraphicalData::setPoint()
	  */
	 enum PointElement {
	 	PointMark	= 0
	 	};

      protected:

        virtual void dataChanged( int channel = -1 );
	virtual void allocRuntimeData();
      	virtual void freeRuntimeData();
      	virtual bool getAxisRange( QSAxis *axis, double& min, double& max );
      	virtual bool start();
      	virtual bool step();
      	virtual void end();

     private:
        SeriesType   m_type;
        QSPt2f       m_zero;
        QSPt2f	     m_fdelta;
        QSPt2f	     m_pdelta;
        QSGArrow     m_arrow1;
	QSGArrow     m_arrow2;
        QSSegment   *m_po;
        double	     m_xmin, m_xmax;
        double       m_ymin, m_ymax;
        bool         m_evalid;   // extremes valid
	struct       curve_runtime_data *d;
	
	enum Stage {
        	     StartDrawLines	 = 0,
        	     DrawLines		 = 1,
        	     StartDrawSeries	 = 2,
        	     DrawSeries       	 = 3,
        	     StartDrawErrorbars  = 4,
        	     DrawErrorbars	 = 5,
        	     StartDrawPointMarks = 6,
        	     DrawPointMarks	 = 7,
        	     Stop		 = 8 };
        	
        void create_po(); 	
	void get_values( int index );
	void get_attributes( int index );
	void start_draw_errorbars();
	void draw_errorbars();
	void start_draw_series();
	void draw_series();
	void start_draw_lines();
	void draw_lines();
	void start_draw_pointmarks();
	void draw_pointmarks();
 };



#endif






























































