/***************************************************************************
                                 qsaxes2d.cpp
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "qsaxes2d.h"
#include "qsaxis.h"
#include "qsdrvqt.h"
#include <math.h>

struct axes2d_runtime_data {
   };

//-------------------------------------------------------------//

QSAxes2D::QSAxes2D(QObject* parent, const char * name)
:QSAxes(parent,&t,name)
 {
  t.matrixI( t.T );
  m_drv    = NULL;

#define CHANNELS_NUM 0
#define FONTS_NUM    0
#define FILLS_NUM    2
#define LINES_NUM    0
#define POINTS_NUM   0

  initChannelTable( CHANNELS_NUM );
  initAttributeTables( FONTS_NUM, FILLS_NUM, LINES_NUM, POINTS_NUM );
  default_settings();
  m_is_graphics_active = false;
 }

//-------------------------------------------------------------//

QSAxes2D::~QSAxes2D()
 {
 }

//--------------------------------------------------------------//

void QSAxes2D::drawAxis( QSAxis *axis )
 {
  int axis_number = axisIndex(axis);
  if ( axis->visible() )
     if ( axis->type() == QSAxis::XAxisType ||
	  axis->type() == QSAxis::YAxisType ) {

          ///////////////////////////
	  //      draw axis line   //
	  ///////////////////////////

	  QSPt2f axis_p1( 0.0, 0.0 );
	  QSPt2f axis_p2( 1.0, 1.0 );
	  double pos = axis->position();
	  if ( axis->defaultPosition()  ) pos = 0.0;
	  if ( axis->oppositePosition() ) pos = 1.0 - pos;
	  if ( axis->type() == QSAxis::XAxisType ) axis_p1.y = axis_p2.y = pos;
	  if ( axis->type() == QSAxis::YAxisType ) axis_p1.x = axis_p2.x = pos;

          // little hack for better look
          QSPt2f cp1 = t.world2DToCanvas( axis_p1 );
	  QSPt2f cp2 = t.world2DToCanvas( axis_p2 );
	  if ( axis->type() == QSAxis::XAxisType && axis_p1.y > 0.5 ) { cp1.y -= 1.0; cp2.y -= 1.0; }
	  if ( axis->type() == QSAxis::YAxisType && axis_p1.x < 0.5 ) { cp1.x -= 1.0; cp2.x -= 1.0; }
		
	  // draw line
          m_drv->setLine( axis->line(QSAxis::AxisLine) );
	  if ( axis->reversed() ) m_drv->drawArrow( cp2, cp1, axis->arrow1(), axis->arrow2() );
		             else m_drv->drawArrow( cp1, cp2, axis->arrow1(), axis->arrow2() );

	  ///////////////////////////////////
	  //  draw tic marks and labels    //
	  ///////////////////////////////////

	  // tic direction & lenght & line
	  QSPt2f tic_direction( 0.0, 0.0 );
	  if ( axis->type() == QSAxis::XAxisType ) tic_direction.y = 1.0;
	  if ( axis->type() == QSAxis::YAxisType ) tic_direction.x = 1.0;
	  if ( axis->ticsOuter()        ) tic_direction.set(-tic_direction.x,-tic_direction.y);
	  if ( axis->oppositePosition() ) tic_direction.set(-tic_direction.x,-tic_direction.y);
	  double tic_len_x = 0.015 * m_csize.y / m_csize.x;
	  double tic_len_y = 0.015;
	  QSPt2f tic_length( tic_direction.x*tic_len_x, tic_direction.y*tic_len_y );
	  QSGLine tic_line = axis->line(QSAxis::AxisLine); tic_line.style = QSGLine::Solid;

	  // tic label align
	  int align = 0;
          if ( axis->type() == QSAxis::XAxisType ) align = AlignHCenter | ( axis->oppositePosition() ? AlignBottom : AlignTop );
	  if ( axis->type() == QSAxis::YAxisType ) align = AlignVCenter | ( axis->oppositePosition() ? AlignLeft : AlignRight );

	  // draw tic marks and labels
	  int tic_nr = 0;
          QSPt2f label_pos;
	  QSPt2f tic_pos = axis_p1; // x ( for x axis ) or y ( for y axis ) will be set in the future.
	  list<QSAxisTic>::const_iterator curr = axis->tics()->begin();
	  list<QSAxisTic>::const_iterator last = axis->tics()->end();
          m_drv->setCurrentElement( QSAxes::GridCategory, axis_number );
	  if ( !m_really_fast ) // a little hack for paintSkeleton
	  while( curr != last ) {
	  	// modify align for rotated labels
	  	int tics_align = align;
          	if ( tics_align & AlignHCenter ) {
			if ( curr->m_angle > 0 ) tics_align = tics_align & ~AlignHCenter | AlignLeft;  // clear AlignHCenter and set AlignLeft
			if ( curr->m_angle < 0 ) tics_align = tics_align & ~AlignHCenter | AlignRight; // clear AlignHCenter and set AlignRight
			}

		// tic mark
	  	if ( axis->type() == QSAxis::XAxisType ) tic_pos.x = curr->m_pos; // p1-p2 vertical line
	  	if ( axis->type() == QSAxis::YAxisType ) tic_pos.y = curr->m_pos; // p1-p2 horizontal line
		if ( axis->ticsVisible() && curr->m_major ) {
			m_drv->setLine( tic_line );
			m_drv->drawLine2( tic_pos, tic_pos+tic_length );
			}

		// tic_label
		double shift = (tic_nr%2) ? axis->ticLabelPos2() : axis->ticLabelPos1();
		QSPt2f label_shift( tic_direction.x*shift, tic_direction.y*shift );
		if ( axis->ticsOuter() ) label_pos = tic_pos + tic_length + label_shift;
				    else label_pos = tic_pos - label_shift;

		m_drv->setFont( curr->m_font );
		m_drv->drawRText2( label_pos, curr->m_angle, curr->m_label, tics_align );

		if ( curr->m_major ) tic_nr++;
		curr++;
		}

	  ///////////////////////
	  //    draw title     //
	  ///////////////////////
	
	  // Y axes have rotated titles
	  int angle = 0;
          if ( axis->type() == QSAxis::YAxisType ) if ( axis->oppositePosition() ) angle = 90; else angle = -90;

	  // position axis at axis->titlePosition()
	  double shift_direction = axis->oppositePosition() ? 1.0 : -1.0;
	  if ( axis->type() == QSAxis::XAxisType ) { label_pos.x = axis->titlePosition(); label_pos.y += shift_direction * axis->titleDistance(); }
          if ( axis->type() == QSAxis::YAxisType ) { label_pos.y = axis->titlePosition(); label_pos.x += shift_direction * axis->titleDistance(); }

          // draw title ( if isn't paintg only skeleton )
	  m_drv->setCurrentElement( QSAxes::AxisCategory, axis_number );
	  m_drv->setFont( axis->font(QSAxis::TitleFont) );
          if ( !m_really_fast ) m_drv->drawRTextBox2( label_pos, angle, axis->title(), align );	
	}
 }

/*
	  QSPt2f label_max_size;
		// size of the label
		QSPt2f curr_label_size = m_drv->rTextSize( curr->m_angle, curr->m_label );
		label_max_size.x = QMAX( label_max_size.x, curr_label_size.x );
		label_max_size.y = QMAX( label_max_size.y, curr_label_size.y );	
*/
/*
	  // move title away from axis on distance label_max_size + axis->titleDistance
          if ( axis->type() == QSAxis::YAxisType ) shift_direction = -shift_direction;
	  if ( axis->oppositePosition()          ) shift
	  QSPt2f clabel_pos = t.world2DToCanvas( label_pos );
	  if ( axis->type() == QSAxis::XAxisType ) label_pos.y += label_max_size.y;
          if ( axis->type() == QSAxis::YAxisType ) label_pos.x += label_max_size.x;
*/
// 	  QSPt2f middle = t.world2DToCanvas( QSPt2f(axis->titlePosition(),axis->titlePosition()) );
//          label_pos = label_pos + label_max_size;

          	//QSPt2f cp1 = t.world2DToCanvas( axis_p1 );
	  	//QSPt2f cp2 = t.world2DToCanvas( axis_p2 ) + d;

	  //if ( axis->type() == QSAxis::XAxisType ) d.y = -ticlen( 0, true );
	  //if ( axis->type() == QSAxis::YAxisType ) d.x =  ticlen( 0, true );

//-------------------------------------------------------------//

void QSAxes2D::drawGrid( QSAxis *axis, bool major )
 {
  if ( axis->visible() )
     if ( axis->type() == QSAxis::XAxisType ||
	  axis->type() == QSAxis::YAxisType ) {
		list<QSAxisTic>::const_iterator curr = axis->tics()->begin();
		list<QSAxisTic>::const_iterator last = axis->tics()->end();
		while ( curr != last ) {
			if ( curr->m_major == major ) {
				QSPt2f p1( 0.0, 0.0 );
				QSPt2f p2( 1.0, 1.0 );
				if ( axis->type() == QSAxis::XAxisType ) p1.x = p2.x = curr->m_pos;
	                	if ( axis->type() == QSAxis::YAxisType ) p1.y = p2.y = curr->m_pos;
                                m_drv->setLine( curr->m_line );
				m_drv->drawLine2( p1, p2 );
				}
			curr++;
			}
		}
 }


//-------------------------------------------------------------//

double QSAxes2D::ticlen( int, bool major )
 {
  return m_csize.y*(major?10.0:6.0)/1000.0;
 }

//-------------------------------------------------------------//

void QSAxes2D::allocRuntimeData()
 {
  QSAxes::allocRuntimeData();
  //d = new axes2d_runtime_data();
 }

//-------------------------------------------------------------//

void QSAxes2D::freeRuntimeData()
 {
  //delete d; d = NULL;
  QSAxes::freeRuntimeData();
 }

//-------------------------------------------------------------//

void QSAxes2D::setFrameWidth( int width )
 {
  if ( width < 0 ) width = 0;
  SET_PROPERTY( m_framew, width );
 }

//-------------------------------------------------------------//

void QSAxes2D::axisRangesCalculated()
 {
  init_2dtr();
  // activate graphics
  assert( !m_is_graphics_active );
  m_is_graphics_active = true;
  m_drv->startDrawing();	
  draw_axes();
  QSAxes::axisRangesCalculated();
 }


//-------------------------------------------------------------//

void QSAxes2D::stop()
 {
  if ( m_is_graphics_active ) {
	m_is_graphics_active = false;
	m_drv->stopDrawing();
	}	
  QSAxes::stop();
 }

//-------------------------------------------------------------//

void QSAxes2D::init_2dtr()
 {
   // left-top corner
   QSPt2f tf( m_cpos.x, m_cpos.y );

   // right bottom corner
   QSPt2f bf( m_cpos.x+m_csize.x,
              m_cpos.y+m_csize.y );

   // the axis frame
   double framew = m_drv->toPixels(frameWidth());
   // for better look on a raster devices
   if ( frameWidth() > 0 && framew < 1.0 ) framew = 1.0;
   tf.x += framew;
   tf.y += framew;
   /*
   bf.x -= framew;
   bf.y -= framew;
   */

   t.matrixI( t.T );

   // ( 0,0 ) is in bottom,left corner.
   t.applyS( t.T, 1.0, -1.0 );
   t.applyT( t.T, 0.0,  1.0  );

   // scale (0,0,1,1) rectangle to (tf, bf) rectangle
   t.applyViewport( t.T, tf.x, tf.y, bf.x-tf.x, bf.y-tf.y );

   // remember inversion for further use
   t.inv( t.IT, t.T );

   // Clip rectangle
   // added + 1 to bf
   t.setClipRect( tf.x, tf.y, bf.x, bf.y );
   }

//-------------------------------------------------------------//

void QSAxes2D::draw_axes()
 {
  /*
  QSPt2f p1;
  QSPt2f p2;
  //t.getClipRect( &p1.x, &p1.y, &p2.x, &p2.y );
  */

  m_drv->setLine( QSGLine::invisibleLine );
  QSPt2f p1 = m_cpos;
  QSPt2f p2 = m_cpos + m_csize;

   // the axis frame
   double framew = m_drv->toPixels(frameWidth());
   // for better look on a raster devices
   if ( frameWidth() > 0 && framew < 1.0 ) framew = 1.0;
   QSPt2f canvas_p1( p1.x+framew, p1.y+framew );
   QSPt2f canvas_p2( p2.x+framew, p2.y+framew );

  // frame
  if ( m_framew > 0 ) {
  	m_drv->setFill( fill(FrameFill) );
  	draw_rect( p1, QSPt2f(canvas_p2.x,canvas_p1.y) ); // top edge
  	draw_rect( QSPt2f(canvas_p1.x,p2.y), canvas_p2 ); // bottom edge
  	draw_rect( p1, QSPt2f(canvas_p1.x,canvas_p2.y) ); // left edge
  	draw_rect( QSPt2f(p2.x,canvas_p1.y), canvas_p2 ); // right edge
        }

  // interior
  QSGFill f = fill(RectFill);
  if ( !m_transparent && f.style == QSGFill::Transparent ) f = QSGFill();
  m_drv->setFill( f );
  draw_rect( canvas_p1, p2 );
 }

//-------------------------------------------------------------//

void QSAxes2D::draw_rect( const QSPt2f& p1, const QSPt2f& p2 )
 {
   QSPt2f p[2];

   p[0].set( QMIN(p1.x,p2.x), QMIN(p1.y,p2.y) );
   p[1].set( QMAX(p1.x,p2.x), QMAX(p1.y,p2.y) );

   if ( p[0].x == p[1].x ) p[1].x += 1.0;
   if ( p[0].y == p[1].y ) p[1].y += 1.0;

   QSPt2f pts[4];
   pts[0] = p[0];
   pts[1] = QSPt2f( p[1].x, p[0].y );
   pts[2] = p[1];
   pts[3] = QSPt2f( p[0].x, p[1].y );

   m_drv->drawPoly( pts, 4 );
 }

//-------------------------------------------------------------//

void QSAxes2D::default_settings()
 {
  m_framew = 1;
  QSGFill f;
  f.color.set( 0, 0, 0 );
  m_settings.fills[FrameFill] = f;

  m_settings.fills[RectFill] = QSGFill::transparentFill;
 }

//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

void QSAxes2D::paintPlot( QPainter *painter, double dpi, bool blocking, bool transparent )
 {
  if ( painter ) {			
	  if ( m_internal_state == Busy ) stop();
	  QSDrvQt qtdrv;
          qtdrv.setProjection( &t );
          qtdrv.setDC(painter,dpi,false); // don't delete QPainter
	  start( &qtdrv, blocking, transparent );
	 }
 }

//-------------------------------------------------------------//

void QSAxes2D::drawPlot( QSDrv *init_drv, bool blocking, bool transparent )
 {
   if ( init_drv ) {
	 if ( m_internal_state == Busy ) stop();
	 init_drv->setProjection( &t );
	 start( init_drv, blocking, transparent );
	} 
 }

//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//

void QSAxes2D::initMappings( QSDrv *drv )
 {
  QSAxes::initMappings( drv );
  if ( !state() ) {
  	allocRuntimeData();
  	init_2dtr();
  	freeRuntimeData();
  	drv->startDrawing();
  	drv->stopDrawing();
  	}
 }



//-------------------------------------------------------------//

QString QSAxes2D::posInfo( QSPt2f& pos )
 {
  QString result = QSAxes::posInfo( pos );
  return result;
 }

//-------------------------------------------------------------//

QSPt3f QSAxes2D::mixedToCanvas( const QSPt3f& pos, CoordinateSystem in_coords[3], double dpi, QSAxis *xAxis, QSAxis *yAxis, QSAxis *zAxis ) const
 {
  QSPt3f result( 0.0, 0.0, 0.0 );

  switch( in_coords[0] ) {
	case mmCoord:		result.x = QSCoord::mmToPixels( pos.x, dpi ); 		    break;
	case normCoord:    	result.x = normalizedXToCanvas( pos.x );      		    break;
	case worldCoord:	result.x = t.world2DToCanvasX( pos.x );       		    break;
	case dataCoord:    	result.x = t.world2DToCanvasX( xAxis->dataToWorld(pos.x) ); break;
	}
  switch( in_coords[1] ) {
	case mmCoord:		result.y = QSCoord::mmToPixels( pos.y, dpi ); 		    break;
	case normCoord:		result.y = normalizedYToCanvas( pos.y );      		    break;
	case worldCoord:	result.y = t.world2DToCanvasY( pos.y );       		    break;
	case dataCoord:		result.y = t.world2DToCanvasY( yAxis->dataToWorld(pos.y) ); break;
	}
  switch( in_coords[2] ) {
	case mmCoord:		result.z = QSCoord::mmToPixels( pos.z, dpi ); 		    break;
	case normCoord:    	break; // no normalized z
	case worldCoord:	result.z = t.world2DToCanvasZ( pos.z );       		    break;
	case dataCoord:		result.z = t.world2DToCanvasZ( zAxis->dataToWorld(pos.z) ); break;
	}

  return result;
 }

//-------------------------------------------------------------//

QSPt3f QSAxes2D::canvasToMixed( const QSPt3f& pos, CoordinateSystem out_coords[3], double dpi, QSAxis *xAxis, QSAxis *yAxis, QSAxis *zAxis ) const
 {
  QSPt3f result;
  switch( out_coords[0] ) {
	case mmCoord:	   result.x = QSCoord::pixelsToMM( pos.x, dpi ); 	       break;
	case normCoord:    result.x = canvasToNormalizedX( pos.x );      	       break;
	case worldCoord:   result.x = t.canvasXToWorld2D( pos.x );       	       break;
	case dataCoord:    result.x = xAxis->worldToData( t.canvasXToWorld2D(pos.x) ); break;
	}
  switch( out_coords[1] ) {
	case mmCoord:	   result.y = QSCoord::pixelsToMM( pos.y, dpi ); 	       break;
	case normCoord:    result.y = canvasToNormalizedY( pos.y );      	       break;
	case worldCoord:   result.y = t.canvasYToWorld2D( pos.y );       	       break;
	case dataCoord:    result.y = yAxis->worldToData( t.canvasYToWorld2D(pos.y) ); break;
	}
  switch( out_coords[2] ) {
	case mmCoord:	   result.z = QSCoord::pixelsToMM( pos.z, dpi ); 	       break;
	case normCoord:    break; // no normalized z
        case worldCoord:   result.z = t.canvasZToWorld2D( pos.z );       	       break;
	case dataCoord:    result.z = zAxis->worldToData( t.canvasZToWorld2D(pos.z) ); break;
	}
  return result;
 }

//-------------------------------------------------------------//

void QSAxes2D::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSAxes::loadStateFromStream( stream, factory );
 }

//-------------------------------------------------------------//

void QSAxes2D::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSAxes::saveStateToStream( stream, factory );
 }




