/***************************************************************************
                                 ksprojectxml.cpp
                             -------------------
    begin                : Sun Feb 13 2000
    copyright            : (C) 2000 by Kamil Dobkowski
    email                : kamildobk@friko.onet.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ksprojectxml.h"
#include "ksmatrix.h"
#include "widgets/qsconsole.h"
#include "widgets/qsaxis.h"
#include "widgets/qsaxes2d.h"
#include "widgets/qsaxes3d.h"
#include "widgets/qsplot.h"
#include "widgets/qscurve.h"
#include "widgets/qsimage.h"
#include "widgets/qscontour.h"
#include "widgets/qssurface.h"
#include "widgets/qsfigure.h"
#include "widgets/qsclegend.h"
#include "formula/mpformula.h"
#include "ksdatasymbolfactory.h"
#include "qscobjects.h"
#include "ksglobalmatrixlist.h"
#include "ksworkbook.h"
#include "kmatplotshell.h"


#include <qmessagebox.h>
#include <qvaluestack.h>
#include <qprinter.h>
#include <qintdict.h>
#include <qregexp.h>
#include <qmap.h>
#include <qxml.h>



//
// Funkcja do zapisywania stringow
//

typedef QMap<QString,int> attribute_names;

#define MIMETYPE  "application/x-kmatplot"

// some tag-name ( not all )
#define KMATPLOT "kmatplot"
#define CANVAS	 "canvas"
#define PAGE	 "page"

#define AXES2D		"axes2d"
#define AXES3D		"axes3d"
#define CURVE		"curve"
#define IMAGE		"image"
#define CONTOUR		"contour"
#define GCONTOUR	"gridded-contour"
#define NGCONTOUR	"non-gridded-contour"
#define SURFACE		"surface"
#define FIGURE		"figure"

#define KMATPLOT  "kmatplot"
#define CANVAS	  "canvas"
#define PAGE	  "page"



// reading // writing:
class ksprojectxml_runtime_data : public QXmlDefaultHandler {

    public:

    	  ksprojectxml_runtime_data();
    	 ~ksprojectxml_runtime_data();

         QFile  *f;
	 KSWorkbook   *workbook;
         KMatplotShell *shell;

///////////////////////////////// reading ////////////////////////////////////////////////

	 enum ParserState { ParseKMatplot, ParseCanvas, ParseSheets, ParseSheet, ParsePage,
	 		    ParseAxes2d, ParseAxes3d, ParseAxis, ParseAxesBackground, ParseAxesTitle,
	 		    ParseMatrix, ParseMatrixRef, ParseMatrixColRef, ParseMatrixString, ParseMatrixFormula,
	 		    ParseCurve, ParseImage, ParseContour, ParseSurface, ParseFigure,
			    ParseGradient,
	 		    ParseCLegend, ParseCLabel, ParseCArrow, ParseCRect, ParseCGroup  };

	QValueStack<ParserState> parser_state;
	QString chars;
	QString error_string;
	void set_error_string( const QString& string );
	int initial_sheets;

	QSPage  *curr_page;
        KSSheet *curr_sheet;
	QSAxes   *curr_axes;
	QSAxis   *curr_axis;
	QSPlot    *curr_plot;

	QSCurve   *curr_curve;
	QSImage   *curr_image;
	QSContour *curr_contour;
	QSSurface *curr_surface;
	QSFigure  *curr_figure;

	QSMatrix *curr_matrix;
	int curr_row;

	QSGFill curr_gradient_fill[5];
	int curr_gradient_type;

	QSCLegend *curr_clegend;
	QSCLabel  *curr_clabel;
	QSCArrow  *curr_carrow;
	QSCRect   *curr_crect;
	QValueStack<QSCGroup*> curr_cgroup;

	QSAxes2D *curr_axes2d;
	QSAxes3D *curr_axes3d;	

	
	 virtual bool startDocument();
	 virtual bool startElement( const QString&, const QString&, const QString&, const QXmlAttributes& );
    	 virtual bool endElement( const QString&, const QString&, const QString& );
	 virtual bool characters( const QString& ch );
	 virtual bool endDocument();
	 virtual bool warning ( const QXmlParseException & exception );
	 virtual bool error ( const QXmlParseException & exception );
	 virtual bool fatalError ( const QXmlParseException & exception ); 	

	 bool read_sheets_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_sheet_stag( const QString& name, const QXmlAttributes& attr );

	 bool read_kmatplot_stag( const QString& name, const QXmlAttributes& attr );	
	 bool read_canvas_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_page_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_axis_background_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_axis_title_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_axis_stag( const QString& name, const QXmlAttributes& attr );	

	 bool read_curve_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_image_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_contour_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_surface_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_figure_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_gradient_stag( const QString& name, const QXmlAttributes& attr );

	 bool read_axes_2d_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_axes_3d_stag( const QString& name, const QXmlAttributes& attr );	
	 bool read_clegend_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_clabel_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_carrow_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_crect_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_cgroup_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_cobject_stag( QSCObject *object, const QString& name, const QXmlAttributes& attr );
	 bool read_some_object_stag( QSCObjectCollection *collection, const QString& name, const QXmlAttributes& attr );
		
	 bool read_matrix_stag( const QString& name, const QXmlAttributes& attr );
	 bool read_matrix_ref_stag( const QString& name, const QXmlAttributes& attr );
         bool read_matrix_col_ref_stag( const QString& name, const QXmlAttributes& attr );
         bool read_matrix_string_stag( const QString& name, const QXmlAttributes& attr );
         bool read_matrix_formula_stag( const QString& name, const QXmlAttributes& attr );

         bool read_sheets_etag( const QString& name );
         bool read_sheet_etag( const QString& name );

	 bool read_kmatplot_etag( const QString& name );	
	 bool read_canvas_etag( const QString& name );
	 bool read_page_etag( const QString& name );
	 bool read_axis_background_etag( const QString& name );
	 bool read_axis_title_etag( const QString& name );
	 bool read_axis_etag( const QString& name );
	
	 bool read_curve_etag( const QString& name );
	 bool read_image_etag( const QString& name );
	 bool read_contour_etag( const QString& name );
	 bool read_surface_etag( const QString& name );
	 bool read_figure_etag( const QString& name );
	 bool read_gradient_etag( const QString& name );

	 bool read_clegend_etag( const QString& name );
	 bool read_clabel_etag( const QString& name );
	 bool read_carrow_etag( const QString& name );
	 bool read_crect_etag( const QString& name );
	 bool read_axes_2d_etag( const QString& name );
	 bool read_axes_3d_etag( const QString& name );
	 bool read_cgroup_etag( const QString& name );
	
	 bool read_matrix_etag( const QString& name );
         bool read_matrix_ref_etag( const QString& name );
 	 bool read_matrix_col_ref_etag( const QString& name );
 	 bool read_matrix_string_etag( const QString& name );
 	 bool read_matrix_formula_etag( const QString& name );

	 bool read_axes( QSAxes *a, const QString& name, const QXmlAttributes& attr );
         bool read_plot( QSPlot *p, const QString& name, const QXmlAttributes& attr );
         bool read_plot_3d( QSPlot3D *p, const QString& name, const QXmlAttributes& attr );
	 bool read_base( QSGraphicalData *b, const QString& name, const QXmlAttributes& attr );
         bool read_data( QSData *d, const QString& name, const QXmlAttributes& attr );
	
	 QSPt2 read_pt2( const QXmlAttributes& attr );
	 QSPt3 read_pt3( const QXmlAttributes& attr );
	 QSPt3f read_pt3f( const QXmlAttributes& attr );	 	
	 QSGFill  read_fill( const QXmlAttributes& attr );
	 QSGLine  read_line( const QXmlAttributes& attr );
	 QSGFont  read_font( const QXmlAttributes& attr );
	 QSGArrow read_arrow( const QXmlAttributes& attr );
	 QSGPoint read_point( const QXmlAttributes& attr );
	 int element_number( const QXmlAttributes& attr );
	 int sheet_number( int sheet ) { return sheet+initial_sheets; }
	 void create_matrix( QSData *d, const QString& name, const QXmlAttributes& attr );
	 QSAxes *cobject_parent_axes( const QXmlAttributes& attr  );
	 QObject *cobject_parent( const QXmlAttributes& attr  );
         void cobject_set_default_axes( QSCObject *object, const QXmlAttributes& attr );
         bool remove_default_axes( QSAxes *axes );
	

///////////////////////////////// writing ////////////////////////////////////////////////
	
	 int indent;
	
	 void xml( const QString& line );
	 void write_data( QSData *data );
	 void write_base( QSGraphicalData *base );
	 void write_axes( QSAxes *axes );
	 void write_dataset( QSPlot *plot, int stack_order );
         void write_plot_3d( QSPlot3D *plot_3d );
	 void write_some_object( QSCObject *object );
	 void write_object( QSCObject *object );
	 void write_group( QSCGroup *object );
	 void write_matrix( QSMatrix *m, int channel_nr );
	 void write_line( const QSGLine& l, int element );
	 void write_fill( const QSGFill& f, int element );
	 void write_font( const QSGFont& f, int element );
	 void write_point( const QSGPoint& p, int element );
	 void write_arrow( const QSGArrow& a, int element );
	 void write_point( const QSPt3f& p, const QString& name );
	 void write_point( const QSPt3& p, const QString& name );
	 void write_point( const QSPt2& p, const QString& name );
	 void write_gradient( const QSGGradient& g );
	 void write_global_matrix_list( KSSheetList *sheets );
	 void write_global_matrix( int sheetNumber );

//////////////////////// utilities //////////////////////////////////////////////////
	
	 attribute_names etype;
         attribute_names page_size;
	 attribute_names orientation;
         attribute_names line_style;
         attribute_names fill_style;
         attribute_names point_style;
	 attribute_names point_fill_style;
         attribute_names arrow_style;
         attribute_names axis_scale;
 	 attribute_names axis_type;
	 attribute_names series_type;
	 attribute_names gradient_type;
	 attribute_names column_type;
	
	 QString xml_from_string( QString text );
	 QString xml_to_string( QString text );
	 QString get_attr_name( attribute_names& attr, int value );
	 int get_attr_value( attribute_names& attr, const QString& name );	
	};

//-----------------------------------------------------------//

KSProjectXML::KSProjectXML( KMatplotShell *shell )
 {
  m_shell = shell;
  d = NULL;
 }

//-----------------------------------------------------------//

KSProjectXML::~KSProjectXML()
 {
  free_runtime_data();
 }

//-----------------------------------------------------------//

void KSProjectXML::alloc_runtime_data()
 {
  d = new ksprojectxml_runtime_data();
  d->shell = m_shell;
  d->workbook = m_shell->workbook();
  d->initial_sheets = m_shell->workbook()->sheets()->childCount();
 }

//-----------------------------------------------------------//

void KSProjectXML::free_runtime_data()
 {
  delete d; d = NULL;
 }

//-----------------------------------------------------------//

ksprojectxml_runtime_data::ksprojectxml_runtime_data()
 {
   indent = 0;

   page_size.insert( "A4", QPrinter::A4 );
   page_size.insert( "B5", QPrinter::B5 );
   page_size.insert( "Letter", QPrinter::Letter );
   page_size.insert( "Legal", QPrinter::Legal );
   page_size.insert( "Executive", QPrinter::Executive );
   page_size.insert( "A0", QPrinter::A0 );
   page_size.insert( "A1", QPrinter::A1 );
   page_size.insert( "A2", QPrinter::A2 );
   page_size.insert( "A3", QPrinter::A3 );
   page_size.insert( "A5", QPrinter::A5 );
   page_size.insert( "A6", QPrinter::A6 );
   page_size.insert( "A7", QPrinter::A7 );
   page_size.insert( "A8", QPrinter::A8 );
   page_size.insert( "A9", QPrinter::A9 );
   page_size.insert( "B0", QPrinter::B0 );
   page_size.insert( "B1", QPrinter::B1 );
   page_size.insert( "B10", QPrinter::B10 );
   page_size.insert( "B2", QPrinter::B2 );
   page_size.insert( "B3", QPrinter::B3 );
   page_size.insert( "B4", QPrinter::B4 );
   page_size.insert( "B6", QPrinter::B6 );
   page_size.insert( "B7", QPrinter::B7 );
   page_size.insert( "B8", QPrinter::B8 );
   page_size.insert( "B9", QPrinter::B9 );
   page_size.insert( "C5E", QPrinter::C5E );
   page_size.insert( "Comm10E", QPrinter::Comm10E );
   page_size.insert( "DLE", QPrinter::DLE );
   page_size.insert( "Folio", QPrinter::Folio );
   page_size.insert( "Ledger", QPrinter::Ledger );
   page_size.insert( "Tabloid", QPrinter::Tabloid );
   page_size.insert( "NPageSize", QPrinter::NPageSize );


   orientation.insert( "Portrait", QPrinter::Portrait );
   orientation.insert( "Landscape", QPrinter::Landscape );

   etype.insert( "double", EDouble );
   etype.insert( "float", EFloat  );
   etype.insert( "long", ELong   );
   etype.insert( "unsigned short", EUShort );
   etype.insert( "short", EShort  );
   etype.insert( "unsigned char", EUChar  );

   fill_style.insert( "transparent", QSGFill::Transparent );
   fill_style.insert( "solid", QSGFill::Solid );
   fill_style.insert( "hlines", QSGFill::Horiz );
   fill_style.insert( "vlines", QSGFill::Vert );
   fill_style.insert( "crossing", QSGFill::Cross );
   fill_style.insert( "fdiag", QSGFill::BDiag );
   fill_style.insert( "bdiag", QSGFill::FDiag );
   fill_style.insert( "diagcross", QSGFill::DiagCross );
   fill_style.insert( "half", QSGFill::Half );

   line_style.insert( "invisible", QSGLine::Invisible );
   line_style.insert( "solid", QSGLine::Solid );
   line_style.insert( "dash", QSGLine::Dash );
   line_style.insert( "dash-dot", QSGLine::DashDot );
   line_style.insert( "dot", QSGLine::Dot );
   line_style.insert( "dash-dot-dot", QSGLine::DashDotDot );

   point_style.insert( "invisible", QSGPoint::Invisible );
   point_style.insert( "circle", QSGPoint::Circle );
   point_style.insert( "rectangle", QSGPoint::Rect );
   point_style.insert( "triangle", QSGPoint::Triangle );
   point_style.insert( "diamond", QSGPoint::Diamond );
   point_style.insert( "cross", QSGPoint::Cross );
   point_style.insert( "plus", QSGPoint::Plus );
   point_style.insert( "hline", QSGPoint::HLine );
   point_style.insert( "vline", QSGPoint::VLine );

   point_fill_style.insert( "transparent", QSGPoint::Transparent );
   point_fill_style.insert( "opaque", QSGPoint::Opaque );
   point_fill_style.insert( "filled", QSGPoint::Filled );

   arrow_style.insert( "none", QSGArrow::None );
   arrow_style.insert( "normal", QSGArrow::Arrow );
   arrow_style.insert( "filled", QSGArrow::FArrow );
   arrow_style.insert( "narrow", QSGArrow::NArrow );
   arrow_style.insert( "reversed", QSGArrow::RArrow );
   arrow_style.insert( "reversed-filled", QSGArrow::RFArrow );
   arrow_style.insert( "reversed-narrow", QSGArrow::RNArrow );
   arrow_style.insert( "rectangle", QSGArrow::Rect );
   arrow_style.insert( "diamond", QSGArrow::Diamond );
   arrow_style.insert( "circle", QSGArrow::Circle );
   arrow_style.insert( "line", QSGArrow::Line );
   arrow_style.insert( "fdiag", QSGArrow::FDiagLine );
   arrow_style.insert( "bdiag", QSGArrow::BDiagLine );

   axis_type.insert( "X", QSAxis::XAxisType );
   axis_type.insert( "Y", QSAxis::YAxisType );
   axis_type.insert( "Z", QSAxis::ZAxisType );
   axis_type.insert( "V", QSAxis::VAxisType );
   axis_type.insert( "?", QSAxis::UnknownAxisType );

   axis_scale.insert( "linear", QSAxis::LinearScale );
   axis_scale.insert( "log", QSAxis::LogScale );

   series_type.insert( "lines", QSCurve::Lines );
   series_type.insert( "area", QSCurve::Area );
   series_type.insert( "ribbon", QSCurve::Ribbon );
   series_type.insert( "bars", QSCurve::Bars );
   series_type.insert( "vectors", QSCurve::Vectors );
   series_type.insert( "flux", QSCurve::Flux );
   series_type.insert( "rectangles", QSCurve::Rectangles );
   series_type.insert( "ellipses", QSCurve::Ellipses );
   series_type.insert( "left-stairs", QSCurve::LeftStairs );
   series_type.insert( "middle-stairs", QSCurve::MiddleStairs );
   series_type.insert( "right-stairs", QSCurve::RightStairs );

   gradient_type.insert( "default", QSGGradient::GDefault );
   gradient_type.insert( "gray", QSGGradient::GGray );
   gradient_type.insert( "strange", QSGGradient::GStrange );
   gradient_type.insert( "two-colors", QSGGradient::GTwoColors );
   gradient_type.insert( "five-colors", QSGGradient::GFiveColors );

   column_type.insert("unknown",KSSheet::ColumnUnknown );
   column_type.insert("x",KSSheet::ColumnX );
   column_type.insert("y",KSSheet::ColumnY );
   column_type.insert("z",KSSheet::ColumnZ );
   column_type.insert("v",KSSheet::ColumnV );
   column_type.insert("dx",KSSheet::ColumnDX );
   column_type.insert("dy",KSSheet::ColumnDY );
 }

//-----------------------------------------------------------//

ksprojectxml_runtime_data::~ksprojectxml_runtime_data()
 {
 }


//-----------------------------------------------------------//

int ksprojectxml_runtime_data::get_attr_value( attribute_names& attr, const QString& name )
 {
  if ( attr.contains(name) ) return attr[name];
  return attr.begin().data();
 }

//-----------------------------------------------------------//

QString ksprojectxml_runtime_data::get_attr_name( attribute_names& attr, int value )
 {
  attribute_names::Iterator it;
  for( it = attr.begin(); it != attr.end(); it++ ) {
         if ( it.data() == value ) return it.key();
         }
  return attr.begin().key();
 }


//-----------------------------------------------------------//
//-----------------------------------------------------------//
//-----------------------------------------------------------//
//-----------------------------------------------------------//
//-----------------------------------------------------------//

bool KSProjectXML::saveToFile( QFile& f )
 {
  QString l;

  //os.precision(9);
  //os.setEncoding( QTextStream::UnicodeUTF8 );

  alloc_runtime_data();
  d->f = &f;

  // write header
  xml( "<?xml version='1.0' encoding='UTF-8'?>" );
  xml( "<!DOCTYPE kmatplot>" );
  xml( "<kmatplot mime='"MIMETYPE"' editor='KMatplot' version='0.3' url='http://kmatplot.sourceforge.net' program-author='K. Dobkowski'>" );
  d->indent++;

  // write canvas header
  l = "<canvas page-size='%1' orientation='%2' dpi='%3'>";
  l = l.arg( d->get_attr_name( d->page_size,   d->workbook->printer()->pageSize()) );
  l = l.arg( d->get_attr_name( d->orientation, d->workbook->printer()->orientation()) );
  l = l.arg( 72.0 );
  xml( l );
  d->indent++;

  // write sheets
  d->write_global_matrix_list( m_shell->workbook()->sheets() );


  for( int page_nr=0; page_nr<d->workbook->pageCount(); page_nr++ ) {
	d->curr_page = d->workbook->page(page_nr);
  	l = "<page number='%1' title='%2'>";
	l = l.arg( page_nr );
	l = l.arg( d->xml_from_string(d->curr_page->title()) );
	xml( l );
  	d->indent++;

  	// write objects	
  	for( int object_nr=0; object_nr<d->curr_page->objects()->count(); object_nr++ ) {
		d->write_some_object( d->curr_page->objects()->object(object_nr) );
		}

  	d->indent--;
  	xml("</page>");
        }

  d->indent--;
  xml("</canvas>");

  d->indent--;
  xml( "</kmatplot>");

  free_runtime_data();
  return true;
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_global_matrix_list( KSSheetList *sheets )
 {
  QString l = "<sheets number='%1'>";
  l = l.arg( sheets->childCount() );
  xml( l );
  indent++;
  for( int sheet_nr=0; sheet_nr<sheets->childCount(); sheet_nr++ ) {	
	write_global_matrix( sheet_nr );
	}
  indent--;
  xml("</sheets>");
  // no write_data
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_global_matrix( int sheetNumber )
 {
  KSSheet *sheet = workbook->sheets()->child(sheetNumber);
  QString l = "<sheet number='%1' name='%2'>";
  l = l.arg( sheetNumber );
  l = l.arg( sheet->objectName() );
  xml( l );
  indent++;
  // write column types
  const KSSheet::ColumnInfo *columns = sheet->columnInfo();
  KSSheet::ColumnInfo::ConstIterator curr_col;
  for( curr_col = columns->begin(); curr_col != columns->end(); ++curr_col ) {
        l = "<column number='%1' title='%2' type='%3'/>";
	l = l.arg( curr_col.key() );
  	l = l.arg( xml_from_string(curr_col.data().title) );
  	l = l.arg( get_attr_name(column_type,curr_col.data().type) );
	xml(l);
	}
  write_data( sheet );
  indent--;
  xml("</sheet>");
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_axes( QSAxes *axes )
 {
  QString l;
  QSAxes2D *axes_2d = dynamic_cast<QSAxes2D *>(axes);
  QSAxes3D *axes_3d = dynamic_cast<QSAxes3D *>(axes);
  QString type = axes_2d ? AXES2D : AXES3D;
  // position and type
  l = "<%1>";
  l = l.arg( type );
  xml( l ); indent++;

  // area
  l = "<area xmm='%1' ymm='%2' wmm='%3' hmm='%4'/>";
  l = l.arg(axes->posMM().x);
  l = l.arg(axes->posMM().y);
  l = l.arg(axes->sizeMM().x);
  l = l.arg(axes->sizeMM().y);
  xml( l );

  // draw in background
  l = "<draw-in-background enabled='%1'/>";
  l = l.arg( axes->drawInBackground() );
  xml( l );

  // title
  xml( "<title-attributes>");
  indent++;
  indent--;
  xml( "</title-attributes>");

  // background
  xml("<background-attributes>");
  indent++;
  write_fill( axes->background(), -1 );
  indent--;
  xml("</background-attributes>");

  // write 2d axes properties
  if ( axes_2d ) {
         l = "<frame width='%1'/>";
         l = l.arg( axes_2d->frameWidth() );
         xml( l );
  	}

  // write 3d axes
  if ( axes_3d ) {

  	// OpenGL
  	l = "<opengl enabled='%1' enable-transparency='%2' global-transparency='%3' shade-walls='%4' auto-stroke='%5' auto-stroke-lightness='%6'/>";
  	l = l.arg( axes_3d->openGL() );
  	l = l.arg( axes_3d->glTransparency() );
  	l = l.arg( axes_3d->glGlobalTransparency() );
  	l = l.arg( axes_3d->glShadeWalls() );
  	l = l.arg( axes_3d->glMeshAutoStroke() );
  	l = l.arg( axes_3d->glAutoStrokeLightness() );
  	xml( l );
  	
  	// walls
  	l = "<wall-thickness xy='%1' xz='%2' yz='%3'/>";
  	l = l.arg( axes_3d->xyWallThickness() );
  	l = l.arg( axes_3d->xzWallThickness() );
	l = l.arg( axes_3d->yzWallThickness() );
  	xml( l );

	// edges
  	l = "<edge-length x='%1' y='%2' z='%3'/>";
  	l = l.arg( axes_3d->xEdgeLength() );
  	l = l.arg( axes_3d->yEdgeLength() );
	l = l.arg( axes_3d->zEdgeLength() );
  	xml( l );
  	
  	// viewpoint
  	l = "<viewpoint azimuth='%1' elevation='%2' distance='%3'/>";
  	l = l.arg( axes_3d->azimuth() );
  	l = l.arg( axes_3d->elevation() );
  	l = l.arg( axes_3d->distance() );
  	xml( l );
  	
  	// light
  	l = "<light enabled='%1' azimuth='%2' elevation='%3' ambient-intensity='%4' directed-intensity='%5'/>";
  	l = l.arg( axes_3d->light() );
  	l = l.arg( axes_3d->lightAzimuth() );
  	l = l.arg( axes_3d->lightElevation() );
  	l = l.arg( axes_3d->ambientLight() );
  	l = l.arg( axes_3d->directLight() );
  	xml( l );
  	
  	// projection
  	l = "<perspective enabled='%1' focus-distance='%2'/>";
  	l = l.arg( axes_3d->perspective() );
  	l = l.arg( axes_3d->focusDistance() );
  	xml( l );

  	//autoscale
  	l = "<autoscale enabled='%1'/>";
  	l = l.arg( axes_3d->autoscale() );
  	xml( l );	
  	}

  // axes
  for( int axis_nr=0; axis_nr<axes->axisCount(); axis_nr++ ) {
	QSAxis *axis = axes->axis(axis_nr);
  	l = "<axis type='%1' visible='%2' scrollable='%3'>";
  	l = l.arg( get_attr_name( axis_type, axis->type() ) );
	l = l.arg( axis->visible() );
	l = l.arg( axis->scrollable() );
  	xml( l );
  	indent++;

  	write_arrow( axis->arrow1(), 1 );
  	write_arrow( axis->arrow2(), 2 );
  	l = "<display position='%1' default='%2' opposite='%3' />";
  	l = l.arg( axis->position() );
  	l = l.arg( axis->defaultPosition() );
  	l = l.arg( axis->oppositePosition() );
  	xml( l );
  	l = "<range min='%1' max='%2' rounded='%3'/>";
  	l = l.arg( axis->min() );
  	l = l.arg( axis->max() );
  	l = l.arg( axis->roundRangeToTicStep() );
  	xml( l );
  	l = "<scale type='%1' base='%2' reversed='%3'/>";
  	l = l.arg( get_attr_name(axis_scale,axis->scaleType()) );
  	l = l.arg( axis->scaleBase() );
  	l = l.arg( axis->reversed() );
  	xml( l );
  	l = "<grid-step major='%1' minor='%2'/>";
  	l = l.arg( axis->majorGridStep() );
  	l = l.arg( axis->minorGridStep() );
  	xml( l );
  	l = "<tics format='%1' visible='%2' outer='%3' angle='%4' pos1='%5' pos2='%6'/>";
  	l = l.arg( xml_from_string(axis->ticsFormat()) );
  	l = l.arg( axis->ticsVisible() );
  	l = l.arg( axis->ticsOuter() );
  	l = l.arg( axis->ticsAngle() );
  	l = l.arg( axis->ticLabelPos1() );
  	l = l.arg( axis->ticLabelPos2() );
	xml( l );
	l = "<title-position pos='%1' distance='%2'/>";
	l = l.arg( axis->titlePosition() );
	l = l.arg( axis->titleDistance() );	
	xml( l );

	for( int i=0; i<3; i++ ) {
		l = "<remembered-view number='%1' min='%2' max='%3' scale='%4' base='%5' reversed='%6' rounded='%7'/>";
		l = l.arg( i );
		l = l.arg( axis->rememberedViews[i].min );
		l = l.arg( axis->rememberedViews[i].max );
		l = l.arg( get_attr_name(axis_scale,axis->rememberedViews[i].scale) );
		l = l.arg( axis->rememberedViews[i].base );
		l = l.arg( axis->rememberedViews[i].reversed );
		l = l.arg( axis->rememberedViews[i].round );
		xml( l );
		}

        write_base( axis );

  	indent--;
  	xml("</axis>");
  	}

  // write base object
  write_base( axes );
     	
  // write datasets
  for( int dataset_nr=0; dataset_nr<axes->plotCount(); dataset_nr++ ) {
	write_dataset( axes->plot(dataset_nr), dataset_nr );	
	}

  // write group object
  write_group( axes->shadowObject() );
	  	
  indent--;
  l = "</%1>";
  l = l.arg( type );
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_dataset( QSPlot *plot, int stack_order )
 {
  QString l;
  QString type;
  QSCurve   *cr = dynamic_cast<QSCurve*>(plot);
  QSImage   *im = dynamic_cast<QSImage*>(plot);
  QSContour *ct = dynamic_cast<QSContour*>(plot);
  QSSurface *sf = dynamic_cast<QSSurface*>(plot);
  QSFigure  *fg = dynamic_cast<QSFigure*>(plot);

  QSGriddedContour *gct = dynamic_cast<QSGriddedContour*>(plot);
  QSNonGriddedContour *ngct = dynamic_cast<QSNonGriddedContour*>(plot);

  if ( cr ) type = CURVE;
  else if ( im ) type = IMAGE;
  else if ( gct ) type = GCONTOUR;
  else if ( ngct ) type = NGCONTOUR;
  else if ( sf ) type = SURFACE;
  else if ( fg ) type = FIGURE;
  else type = "unknown";

  l = "<%1 stack-order='%2'>";
  l = l.arg( type );
  l = l.arg( stack_order );
  xml( l );
  indent++;

  // legend item
  l = "<legend-item visible='%1'/>";
  l = l.arg( plot->legendItemVisible() );
  xml( l );

  // default axes
  l = "<default-axis x-axis='%1' y-axis='%2' z-axis='%3' v-axis='%4'/>";
  l = l.arg( plot->parentAxes()->axisIndex(plot->defaultAxis(QSAxis::XAxisType)) );
  l = l.arg( plot->parentAxes()->axisIndex(plot->defaultAxis(QSAxis::YAxisType)) );
  l = l.arg( plot->parentAxes()->axisIndex(plot->defaultAxis(QSAxis::ZAxisType)) );
  l = l.arg( plot->parentAxes()->axisIndex(plot->defaultAxis(QSAxis::VAxisType)) );
  xml( l );

  // gradient
  write_gradient( plot->gradient() );

  // curve
  if ( cr ) {
  	l = "<series type='%1'/>";
  	l = l.arg( get_attr_name(series_type,cr->type()) );
  	xml( l );
  	
  	l = "<delta fixed-x='%1' fixed-y='%2' percent-x='%3' percent-y='%4'/>";
  	l = l.arg( cr->fixedDelta().x );
  	l = l.arg( cr->fixedDelta().y );
  	l = l.arg( cr->percentDelta().x );
  	l = l.arg( cr->percentDelta().y );
  	xml( l );  	
  	
  	l = "<zero x='%1' y='%2'/>";
  	l = l.arg( cr->zeroPoint().x );
 	l = l.arg( cr->zeroPoint().y );
 	xml( l );
 	  	
  	write_arrow( cr->arrow1(), 1 );
  	write_arrow( cr->arrow2(), 2 );
  	}
  else
  // image
  if ( im ) {
  	l = "<raw-mode enabled='%1'/>";
  	l = l.arg( im->rawMode() );
        xml( l );
        l = "<use-gradient enabled='%1'/>";
  	l = l.arg( im->useGradient() );
  	xml( l );;
  	}
  else
  // contour
  if ( ct ) {
  	l = "<filled enabled='%1'/>";
  	l = l.arg( ct->contourFills() );
  	xml( l );
  	
  	l = "<lines enabled='%1'/>";
  	l = l.arg( ct->contourLines() );
  	xml( l );
  	
  	l = "<labels enabled='%1' spacing='%2'/>";
  	l = l.arg( ct->contourLabels() );
	l = l.arg( ct->labelSpacing() );
  	xml( l );
  	}
  else
  // surface
  if ( sf ) {
	l = "<grid x-step='%1' y-step='%2'/>";
	l = l.arg( sf->xGridStep() );
	l = l.arg( sf->yGridStep() );
	xml( l );

        write_plot_3d( sf );
  	}
  else
  // figure
  if ( fg ) {
  	l = "<vertex-compare accuracy='%1'/>";
  	l = l.arg( fg->vertexCompareAccuracy() );
  	xml( l );
	write_plot_3d( fg );
  	}

  write_base( plot );
  indent--;
  l = "</%1>";
  l = l.arg( type );
  xml( l );
 }
//-----------------------------------------------------------//
  	
void ksprojectxml_runtime_data::write_plot_3d( QSPlot3D *plot_3d )
 {
  QString l;
  //autodivide
  l = "<autodivide enabled='%1'/>";
  l = l.arg( plot_3d->autoDivide() );
  xml( l );

  // colored
  l = "<colored enabled='%1'/>";
  l = l.arg( plot_3d->colored() );
  xml( l );

  // topBottom
  l = "<topbottom enabled='%1'/>";
  l = l.arg( plot_3d->topBottom() );
  xml( l );

  l = "<edges autocolor='%1'/>";
  l = l.arg( plot_3d->edgeAutoColor() );
  xml( l );
 }

//-----------------------------------------------------------//
  	
void ksprojectxml_runtime_data::write_base( QSGraphicalData *base )
 {
  QString l = "<title text='%1'/>";
  l = l.arg( xml_from_string(base->title()) );
  xml( l );

  for( int line_nr=0; line_nr<base->lineCount(); line_nr ++ )
  	write_line( base->line(line_nr), line_nr );
  	
  for( int fill_nr=0; fill_nr<base->fillCount(); fill_nr ++ )
  	write_fill( base->fill(fill_nr), fill_nr );
  	
  for( int font_nr=0; font_nr<base->fontCount(); font_nr ++ )
  	write_font( base->font(font_nr), font_nr );
  	
  for( int point_nr=0; point_nr<base->pointCount(); point_nr ++ )
  	write_point( base->point(point_nr), point_nr );
  	
  write_data( base );
 }

//-----------------------------------------------------------//
  	
void ksprojectxml_runtime_data::write_data( QSData *data )
 {
  for( int channel_nr=0; channel_nr<data->channelCount(); channel_nr++ )
  	write_matrix( data->matrix(channel_nr), channel_nr );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_gradient( const QSGGradient& g )
 {
  QString l = "<gradient type='%1'>";
  l = l.arg( get_attr_name( gradient_type, g.type() )  );
  xml( l );
  indent++;
  write_fill( g.fill(0), 0 );
  write_fill( g.fill(1), 1 );
  write_fill( g.fill(2), 2 );
  write_fill( g.fill(3), 3 );
  write_fill( g.fill(4), 4 );
  indent--;
  xml("</gradient>");
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_some_object( QSCObject *object )
 {
  QSCLegend *legend    = dynamic_cast<QSCLegend *>(object);
  QSCLabel  *label     = dynamic_cast<QSCLabel*>(object);
  QSCArrow  *arrow     = dynamic_cast<QSCArrow*>(object);
  QSCRect   *rectangle = dynamic_cast<QSCRect*>(object);
  QSCGroup  *group     = dynamic_cast<QSCGroup*>(object);
  QString l;

  // axes
  if ( object->isAxesShadow() ) {
	write_axes( object->parentAxes() );
	}
  // legend
  else if ( legend ) {
	xml( "<legend>" );
	indent++;
	l = "<align frame='%1'/>";
	l = l.arg( legend->align() );
	xml( l );
	l = "<format columns='%1'/>";
	l = l.arg( legend->columns() );
	xml( l );  	
  	write_point( legend->pos(), "pos"  );
  	write_point( legend->coord(), "coord-system" );
  	write_point( legend->shadowPos(), "shadow-shift" );
  	write_font( legend->font(), 1 );
	write_fill( legend->fill(), 1 );
	write_line( legend->frame(), 1 );
	write_fill( legend->shadowFill(), 2 );
	write_object( legend );
	indent--;
	xml( "</legend>" );
  }
  // label
  else if ( label ) {
	xml( "<label>" );
	indent++;
  	l = "<text>%1</text>";
  	l = l.arg( xml_from_string(label->text()) );
  	xml( l );
  	l = "<angle deg='%1'/>";
  	l = l.arg( label->angle() );
	xml( l );
	l = "<align text='%1' frame='%2'/>";
	l = l.arg( label->textAlign() );
	l = l.arg( label->frameAlign() );
	xml( l ); 	
 	write_point( label->pos(), "pos"  );
  	write_point( label->coord(), "coord-system" );
  	write_point( label->shadowPos(), "shadow-shift" );
  	write_font( label->font(), 1 );
	write_fill( label->fill(), 1 );
	write_line( label->frame(), 1 );
	write_fill( label->shadowFill(), 2 );
	write_object( label );	
	indent--;
	xml( "</label>" );
  }
  // arrow
  else if ( arrow ) {
	xml( "<arrow>" );
	indent++;
	write_point( arrow->originPos(), "begin"  );
	write_point( arrow->endPos(), "end"  );	
  	write_point( arrow->originCoord(), "coord-system-begin" );
  	write_point( arrow->endCoord(), "coord-system-end" );
  	write_line( arrow->line(), 1 );  	
  	write_arrow( arrow->originArrow(), 1 );
  	write_arrow( arrow->endArrow(), 2 );
  	write_point( arrow->originPoint(), 1 );
  	write_point( arrow->endPoint(), 2 );
	write_object( arrow );		
	indent--;
	xml( "</arrow>" );
  }
  // rectangle
  else if ( rectangle ) {  	
	xml( "<rectangle>" );
	indent++;
  	l = "<type ellipse='%1'/>";
  	l = l.arg( rectangle->ellipse() );
  	xml( l );
	write_point( rectangle->originPos(), "begin"  );
	write_point( rectangle->endPos(), "end"  );	
  	write_point( rectangle->originCoord(), "coord-system-begin" );
  	write_point( rectangle->endCoord(), "coord-system-end" );
  	write_point( rectangle->shadowPos(), "shadow-shift" );
  	write_fill( rectangle->fill(), 1 );
	write_line( rectangle->frame(), 1 );
	write_fill( rectangle->shadowFill(), 2 );
	write_object( rectangle );	 	
	indent--;
	xml("</rectangle>");	
  }
  // group
  else if ( group ) {
	xml( "<group>" );
	indent++;
	write_group( group );
	indent--;
	xml( "</group>" );
	}
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_object( QSCObject *object )
 {
  QString l = "<axes-bindings  x-axis='%1' y-axis='%2' z-axis='%3'/>";
  l = l.arg( object->defaultXAxis() );
  l = l.arg( object->defaultYAxis() );
  l = l.arg( object->defaultZAxis() );
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_group( QSCGroup *group )
 {
  for ( int i=0; i<group->objects()->count(); i++ ) {
	write_some_object( group->objects()->object(i) );
	}
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_line( const QSGLine& line, int element )
 {
  QString l = "<line element-number='%1' style='%2' width='%3' color='%4'/>";
  l = l.arg(element);
  l = l.arg(get_attr_name(line_style,line.style));
  l = l.arg(line.width);
  l = l.arg(toQString(line.color));
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_fill( const QSGFill& f, int element )
 {
  QString l = "<fill element-number='%1' style='%2' color='%3'/>";
  l = l.arg(element);
  l = l.arg(get_attr_name(fill_style,f.style));
  l = l.arg(toQString(f.color));
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_font( const QSGFont& f, int element )
 {
  QString l = "<font element-number='%1' family='%2' size='%3' bold='%4' italic='%5' color='%6'/>";
  l = l.arg(element);
  l = l.arg(f.family);
  l = l.arg(f.size);
  l = l.arg(f.bold);
  l = l.arg(f.italic);
  l = l.arg(toQString(f.color));
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_point( const QSGPoint& p, int element )
 {
  QString l = "<point element-number='%1' style='%2' fill='%3' size='%4' color='%5'/>";
  l = l.arg(element);
  l = l.arg(get_attr_name(point_style,p.style));
  l = l.arg(get_attr_name(point_fill_style,p.fill));
  l = l.arg(p.size);
  l = l.arg(toQString(p.color));
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_arrow( const QSGArrow& a, int element )
 {
  QString l = "<dart element-number='%1' style='%2' size='%3' />";
  l = l.arg(element);
  l = l.arg(get_attr_name(arrow_style,a.style));
  l = l.arg(a.size);
  xml( l );
 }
//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_point( const QSPt3f& p, const QString& name )
 {
  QString l = "<%1 x='%2' y='%3' z='%4'/>";
  l = l.arg( name );
  l = l.arg( p.x );
  l = l.arg( p.y );
  l = l.arg( p.z );
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_point( const QSPt3& p, const QString& name )
 {
  QString l = "<%1 x='%2' y='%3' z='%4'/>";
  l = l.arg( name );
  l = l.arg( p.x );
  l = l.arg( p.y );
  l = l.arg( p.z );
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_point( const QSPt2& p, const QString& name )
 {
  QString l = "<%1 x='%2' y='%3'/>";
  l = l.arg( name );
  l = l.arg( p.x );
  l = l.arg( p.y );
  xml( l );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::write_matrix( QSMatrix *m, int channel )
 {
  if ( m == NULL ) return;
  KSMatrix        *ks_matrix          = dynamic_cast<KSMatrix *>(m);
  KSMatrixString  *ks_matrix_string   = dynamic_cast<KSMatrixString  *>(m);
  KSMatrixFormula *ks_matrix_formula  = dynamic_cast<KSMatrixFormula *>(m);
  KSMatrixWorksheetCellRange *ks_matrix_ref  = dynamic_cast<KSMatrixWorksheetCellRange *>(m);

  // Write subclass shell
  if ( ks_matrix ) {
	 QString l = "<matrix title='%1' rows='%2' cols='%3' element-type='%4' data-channel='%5'>";
	 l = l.arg( xml_from_string(ks_matrix->name()) );
  	 l = l.arg( ks_matrix->rows() );
  	 l = l.arg( ks_matrix->cols() );
  	 l = l.arg( get_attr_name(etype,ks_matrix->type()) );
  	 l = l.arg( channel );
  	 xml( l );
  	 indent++;
	
 	 QCString e; 	
 	 for( int row=0;row<ks_matrix->rows();row++ ) {
  	 	for( int i=0; i<indent; i++ ) f->putch( '\t' );
  	 		f->writeBlock("<row>",5);
  	 		for( int col=0;col<ks_matrix->cols();col++ ) {
  	 			e.sprintf("%.9g ",ks_matrix->value(row,col));
  	 			f->writeBlock( e.data(), e.size()-1 );
  	 			}
  	 	f->writeBlock("</row>",6);
  	 	f->putch( '\n' );
  		} 	
  	indent--;
  	xml("</matrix>");
	}
  else
  if ( ks_matrix_ref ) {
	// don't write ref channel as it is always 0
        QString l = "<matrixref title='%1' sheet='%1' row-from='%2' row-step='%3' row-to='%4' col-from='%5' col-step='%6' col-to='%7' transposition='%8' data-channel='%9'/>";
        l = l.arg( xml_from_string(ks_matrix_ref->name()) );
        l = l.arg( ks_matrix_ref->worksheet() );
	l = l.arg( ks_matrix_ref->rowFrom() );
	l = l.arg( ks_matrix_ref->rowStep() );
	l = l.arg( ks_matrix_ref->rowTo() );
	l = l.arg( ks_matrix_ref->colFrom() );
	l = l.arg( ks_matrix_ref->colStep() );
	l = l.arg( ks_matrix_ref->colTo() );
	l = l.arg( ks_matrix_ref->transposition() );
        l = l.arg( channel );
	xml( l );
        }
  else
  if ( ks_matrix_string ) {
	QString l = "<matrix-string title='%1' rows='%2' cols='%3' data-channel='%4'>";
	l = l.arg( xml_from_string(ks_matrix_string->name()) );
	l = l.arg( ks_matrix_string->rows() );
	l = l.arg( ks_matrix_string->cols() );
	l = l.arg( channel );
	xml( l );
  	indent++;
		
 	for( int row=0;row<ks_matrix_string->rows();row++ )
		for( int col=0;col<ks_matrix_string->cols();col++ )
			if ( !ks_matrix_string->string(row,col).isEmpty() ) {
				l = "<e row='%1' col='%2' string='%3'/>";
				l = l.arg( row );
				l = l.arg( col );
				l = l.arg( xml_from_string(ks_matrix_string->string(row,col)) );
                        	xml( l );
  	 			}
	
  	indent--;
	xml( "</matrix-string>" );
	}
  else
  if ( ks_matrix_formula ) {
 	QString l = "<matrix-formula title='%1' formula='%2' transposition='%3' data-channel='%4'/>";
	l = l.arg( xml_from_string(ks_matrix_formula->name()) );
	l = l.arg( xml_from_string(ks_matrix_formula->formula()) );
	l = l.arg( ks_matrix_formula->transposition() );
	l = l.arg( channel );
	xml( l );
	}

 }
  	
//-----------------------------------------------------------//

void ksprojectxml_runtime_data::xml( const QString& line )
 {
  assert( indent >= 0 );
  for ( int i=0; i<indent; i++ ) f->putch( '\t' );
  QCString chars = line.utf8();
  f->writeBlock( chars.data(), chars.size()-1 );
  f->putch( '\n' );

  //(*s) << line << endl;
 }

//-----------------------------------------------------------//

void KSProjectXML::xml( const QString& line )
 {
  d->xml( line );
 }

//-----------------------------------------------------------//

QString ksprojectxml_runtime_data::xml_from_string( QString text )
 {
  text.replace( QRegExp("&"), "&amp;" );
  text.replace( QRegExp("'"), "&apos;" );
  text.replace( QRegExp(">"), "&gt;" );
  text.replace( QRegExp("<"), "&lt;" );
  text.replace( QRegExp("\""), "&quot;" );
  text.replace( QRegExp("\n"), "&#10;" );
  return text;
 }

//-----------------------------------------------------------//

QString ksprojectxml_runtime_data::xml_to_string( QString text )
 {
  text.replace( QRegExp("\\&amp;"), "&" );
  text.replace( QRegExp("\\&apos;"), "'" );
  text.replace( QRegExp("\\&gt;"), ">" );
  text.replace( QRegExp("\\&lt;"), "<" );
  text.replace( QRegExp("\\&quot;"), "\"" );
  text.replace( QRegExp("&#10;"), "\n" );
  return text;
 }












//-----------------------------------------------------------//
//-----------------------------------------------------------//
//-----------------------------------------------------------//
//-----------------------------------------------------------//
























bool KSProjectXML::loadFromFile( QFile& f  )
 {
  m_shell->workbook()->commandHistory()->clear();
  alloc_runtime_data();
  d->f = &f;
  QXmlInputSource src( f );
  QXmlSimpleReader reader;

  reader.setContentHandler( d );
  reader.setErrorHandler( d );
  bool result = reader.parse( src );

  free_runtime_data();
  return result;
 }
//-----------------------------------------------------------//

void ksprojectxml_runtime_data::set_error_string( const QString& string )
 {
  error_string += string;
  //cout << "##################### ERROR: " << string << " ####################" << endl;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::warning ( const QXmlParseException & )
 {
  error_string += QObject::tr("<br/> Warning: ") + errorString();
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::error ( const QXmlParseException & )
 {
  error_string += QObject::tr("<br/> <b>Error:</b> ") + errorString();
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::fatalError ( const QXmlParseException & )
 {
  error_string += QObject::tr("<br/> <b>Fatal error:</b> ") + errorString();
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::startDocument()
 {
  parser_state.clear();
  parser_state.push( ParseKMatplot );
  error_string = QString::null;
  return TRUE;
 }

//-----------------------------------------------------------//

// There is a pair of functions of each object which is loaded !!!!
// To add a new object:
// - add a new ksprojectxml_runtime_data::ParserState value
// - add a new pair of funtions to handle start and end elements.
// - add expression to the top level read_?_stag function which starts parsing the given element.
// remeber that a begining tag belongs to the higher level read_tag_function
// <page number='1'>
// 	<arrow stack-order='1'>  // WARNING: read_page_stag - read_page_stag must handle 'arrow' and turn 'ParserState' to 'ParseArrow'
//		<begin ...       // read_arrow_stag
//		<end ...         // read_arrow_stag
//	</arrow>                 // read_arrow_etag
//
bool ksprojectxml_runtime_data::startElement( const QString&, const QString& name, const QString&, const QXmlAttributes& attr )
 {
//  cout << " Start tag " << (const char *)name << endl;
  bool result = TRUE;
  switch( parser_state.top() ) {
	case ParseSheets:		result =  read_sheets_stag( name, attr ); break;
	case ParseSheet:		result =  read_sheet_stag( name, attr ); break;
	case ParseMatrix:		result =  read_matrix_stag( name, attr ); break;
	case ParseMatrixRef:		result =  read_matrix_ref_stag( name, attr ); break;
	case ParseMatrixColRef:		result =  read_matrix_col_ref_stag( name, attr ); break;
	case ParseMatrixString:		result =  read_matrix_string_stag( name, attr ); break;
	case ParseMatrixFormula:	result =  read_matrix_formula_stag( name, attr ); break;	
  	case ParseKMatplot:		result =  read_kmatplot_stag( name, attr ); break;
  	case ParseCanvas:		result =  read_canvas_stag( name, attr ); break;
  	case ParsePage:			result =  read_page_stag( name, attr ); break;
	case ParseAxes2d:		result =  read_axes_2d_stag( name, attr ); break;
	case ParseAxes3d:		result =  read_axes_3d_stag( name, attr ); break;
	case ParseAxis:			result =  read_axis_stag( name, attr ); break;
	case ParseAxesBackground:	result =  read_axis_background_stag( name, attr ); break;
	case ParseAxesTitle:		result =  read_axis_title_stag( name, attr ); break;
	case ParseCurve:		result =  read_curve_stag( name, attr ); break;
	case ParseImage:		result =  read_image_stag( name, attr ); break;
	case ParseContour:		result =  read_contour_stag( name, attr ); break;
	case ParseSurface:		result =  read_surface_stag( name, attr ); break;
	case ParseFigure:		result =  read_figure_stag( name, attr ); break;
	case ParseGradient:		result =  read_gradient_stag( name, attr ); break;
	case ParseCLegend:		result =  read_clegend_stag( name, attr ); break;
	case ParseCLabel:		result =  read_clabel_stag( name, attr ); break;
	case ParseCArrow:		result =  read_carrow_stag( name, attr ); break;
	case ParseCRect:		result =  read_crect_stag( name, attr ); break;
	case ParseCGroup:		result =  read_cgroup_stag( name, attr ); break;
	}
  chars = QString::null;
//  cout << " End tag " << (const char *)name << endl;
  return result;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::endElement( const QString&, const QString& name, const QString& )
 {
  bool result = TRUE;
  switch( parser_state.top() ) {
	case ParseSheets:		result =  read_sheets_etag( name ); break;
	case ParseSheet:		result =  read_sheet_etag( name ); break;
	case ParseMatrix:		result =  read_matrix_etag( name ); break;
	case ParseMatrixRef:		result =  read_matrix_ref_etag( name ); break;
	case ParseMatrixColRef:		result =  read_matrix_col_ref_etag( name ); break;
	case ParseMatrixString:		result =  read_matrix_string_etag( name ); break;
	case ParseMatrixFormula:	result =  read_matrix_formula_etag( name ); break;
  	case ParseKMatplot:		result =  read_kmatplot_etag( name ); break;
  	case ParseCanvas:		result =  read_canvas_etag( name ); break;
  	case ParsePage:			result =  read_page_etag( name ); break;
	case ParseAxes2d:		result =  read_axes_2d_etag( name ); break;
	case ParseAxes3d:		result =  read_axes_3d_etag( name ); break;
	case ParseAxis:			result =  read_axis_etag( name ); break;
	case ParseAxesBackground:	result =  read_axis_background_etag( name ); break;
	case ParseAxesTitle:		result =  read_axis_title_etag( name ); break;
	case ParseCurve:		result =  read_curve_etag( name ); break;
	case ParseImage:		result =  read_image_etag( name ); break;
	case ParseContour:		result =  read_contour_etag( name ); break;
	case ParseSurface:		result =  read_surface_etag( name ); break;
	case ParseFigure:		result =  read_figure_etag( name ); break;
	case ParseGradient:		result =  read_gradient_etag( name ); break;
	case ParseCLegend:		result =  read_clegend_etag( name ); break;
	case ParseCLabel:		result =  read_clabel_etag( name ); break;
	case ParseCArrow:		result =  read_carrow_etag( name ); break;
	case ParseCRect:		result =  read_crect_etag( name ); break;
	case ParseCGroup:		result =  read_cgroup_etag( name ); break;
	}
  chars = QString::null;
  return result;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::characters( const QString& new_chars )
 {
  chars += new_chars;
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::endDocument()
 {
  shell->refreshSettings();
  parser_state.clear();

  if ( error_string != QString::null ) QSConsole::write( " XML parser returned :<br/><br/>" + error_string );
	
  // object is auto delete list, so each remaining object
  // is deleted
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_kmatplot_stag( const QString& name,const  QXmlAttributes& attr )
 {
  if ( name == "kmatplot" ) {
 	 if ( attr.value("mime") != MIMETYPE ) set_error_string( QObject::tr("<br> <br> KMatplot(XML): Incorrect MIME-TYPE ") );
 	}
  else
  if ( name == "canvas" ) {
  	 parser_state.push( ParseCanvas );
  	 QPrinter *p = workbook->printer();
  	 QPrinter::PageSize    ps = (QPrinter::PageSize)get_attr_value( page_size, attr.value("page-size") );
  	 QPrinter::Orientation po = (QPrinter::Orientation)get_attr_value( orientation,attr.value("orientation") );
  	 p->setPageSize( ps );
  	 p->setOrientation( po );
  	 workbook->setPrinter( p );	
  	} else {
  	 set_error_string( QObject::tr("<br> XML(kmatplot): Unexpected tag: ")+name );
  	}
  return TRUE;
 }	

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_kmatplot_etag( const QString& name )
 {
  if ( name == "kmatplot" ) parser_state.pop();
  return TRUE;
 }
 	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_canvas_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "page" ) {
  	 parser_state.push( ParsePage );
	 curr_page = new QSPage( workbook );
	 curr_page->setTitle( xml_to_string(attr.value("title")) );
	 workbook->pageAdd( curr_page );
  	}
  else if ( name == "sheets" ) {
        int sheet_number = attr.value("number").toInt();
	for ( int curr_sheet=0; curr_sheet<sheet_number; curr_sheet++ ) {
		workbook->sheets()->sheetAdd( new KSSheet( workbook->sheets() ) );
		}
	parser_state.push( ParseSheets );
	}
  else {
  	 set_error_string( QObject::tr("<br> XML(canvas): Unexpected tag: ")+name);
  	}
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_canvas_etag( const QString& name  )
 {
  if ( name == "canvas" ) parser_state.pop();
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_sheets_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "sheet" ) {
	 curr_sheet = workbook->sheets()->child(sheet_number(attr.value("number").toInt()));
	 curr_sheet->setObjectName( attr.value("name") );
	 parser_state.push( ParseSheet );
	}
  else set_error_string( QObject::tr("<br> XML(sheets): Unexpected tag: ")+name );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_sheets_etag( const QString& name  )
 {
  if ( name == "sheets" ) parser_state.pop();
  return TRUE; 	
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_sheet_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "column" ) {
	 if ( curr_sheet )
	 curr_sheet->setColumnData( attr.value("number").toInt(),
				    (KSSheet::ColumnType )get_attr_value(column_type,attr.value("type")),
				    xml_to_string(attr.value("title")) );
	}
  else read_data( curr_sheet, name, attr );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_sheet_etag( const QString& name  )
 {
  if ( name == "sheet" ) {
	parser_state.pop();
	curr_sheet = NULL;
	}
  return TRUE;	
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_some_object_stag( QSCObjectCollection *collection, const QString& name, const QXmlAttributes& )
 {
  if ( name == AXES2D ) {
  	parser_state.push( ParseAxes2d );
  	curr_axes = curr_axes2d = new QSAxes2D();
  	collection->add( curr_axes2d->shadowObject() );
  	}
  else if ( name == AXES3D ) {
  	parser_state.push( ParseAxes3d );
  	curr_axes = curr_axes3d = new QSAxes3D();
  	collection->add( curr_axes3d->shadowObject() );
  	} 	
  else if ( name == "legend" ) {
  	parser_state.push( ParseCLegend );
	curr_clegend = new QSCLegend( NULL );
  	collection->add( curr_clegend );  		
  	}
  else if ( name == "label" ) {
  	parser_state.push( ParseCLabel );
	curr_clabel = new QSCLabel( "" );
  	collection->add( curr_clabel );	
  	}
  else if ( name == "arrow" ) {
  	parser_state.push( ParseCArrow );
	curr_carrow = new QSCArrow();
  	collection->add( curr_carrow );
  	}	
  else if ( name == "rectangle" ) {
  	parser_state.push( ParseCRect );
	curr_crect = new QSCRect();
  	collection->add( curr_crect );	
  	}
  else if ( name == "group" ) {
  	parser_state.push( ParseCGroup );
	curr_cgroup.push( new QSCGroup() );	
  	collection->add( curr_cgroup.top() );	
	}
  else return false;

  return true;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_page_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( read_some_object_stag( curr_page->objects(), name, attr ) ) {
	}
  else set_error_string( QObject::tr("<br> XML(page): Unexpected tag: ")+name );
  	
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_page_etag( const QString& name )
 {
  if ( name == "page" ) {
	parser_state.pop();
  	curr_page = NULL;
	}

  return TRUE;
 }


//-----------------------------------------------------------//
/*
QSAxes *ksprojectxml_runtime_data::cobject_parent_axes( const QXmlAttributes& attr )
 {
  int parent_stack_order = attr.value("parent-axes").toInt();
  QSCObject *o = objects[parent_stack_order];
  return ( o ? o->parentAxes() : NULL );
 }

//-----------------------------------------------------------//

QObject *ksprojectxml_runtime_data::cobject_parent( const QXmlAttributes& )
 {
  return NULL;
 }
	
//-----------------------------------------------------------//

void ksprojectxml_runtime_data::cobject_set_default_axes( QSCObject *object, const QXmlAttributes& attr )
 {
  if ( object->parentAxes() ) {
	object->setDefaultAxis( object->parentAxes()->axis(attr.value("x-axis").toInt()) );
	object->setDefaultAxis( object->parentAxes()->axis(attr.value("y-axis").toInt()) );
	object->setDefaultAxis( object->parentAxes()->axis(attr.value("z-axis").toInt()) );
	}
 } 	
*/
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axes_2d_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_axes2d ) return TRUE;

  if ( name == "frame" ) {
  	curr_axes2d->setFrameWidth( attr.value("width").toInt() );
  	}
  else return read_axes( curr_axes2d, name, attr );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axes_2d_etag( const QString& name )
 {
  if ( name == AXES2D )  {
  	parser_state.pop();
	// we read all axes from file so we don't need default axes
        remove_default_axes( curr_axes2d );
	curr_axes = curr_axes2d = NULL;
  	}
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axes_3d_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_axes3d ) return TRUE;

  if ( name == "opengl" ) {
	curr_axes3d->setOpenGL( (bool )attr.value("enabled").toInt() );
	curr_axes3d->glSetTransparency( (bool )attr.value("enable-transparency").toInt() );
	curr_axes3d->glSetGlobalTransparency( attr.value("global-transparency").toInt() );
	curr_axes3d->glSetShadeWalls( (bool )attr.value("shade-walls").toInt() );
	curr_axes3d->glSetMeshAutoStroke( (bool )attr.value("auto-stroke").toInt() );
	curr_axes3d->glSetAutoStrokeLightness( attr.value("auto-stroke-lightness").toInt() );
  	}
  else if ( name == "wall-thickness" ) {
  	curr_axes3d->setWallThickness( attr.value("xy").toDouble(),
				       attr.value("xz").toDouble(),
				       attr.value("yz").toDouble() );
  	}
  else if ( name == "edge-length" ) {
  	curr_axes3d->setEdgeLength( attr.value("x").toDouble(),
				    attr.value("y").toDouble(),
				    attr.value("z").toDouble() );
  	}

  else if ( name == "viewpoint" ) {
  	curr_axes3d->setAzimuth( attr.value("azimuth").toInt() );
  	curr_axes3d->setElevation( attr.value("elevation").toInt() );
  	curr_axes3d->setDistance( attr.value("distance").toInt() );
  	}
  else if ( name == "perspective" ) {
  	curr_axes3d->setPerspective( attr.value("enabled").toInt() );
  	curr_axes3d->setFocusDistance( attr.value("focus-distance").toInt() );
  	}
  else if ( name == "light" ) {
  	curr_axes3d->setLight( attr.value("enabled").toInt() );
  	curr_axes3d->setLightAzimuth( attr.value("azimuth").toInt() );
  	curr_axes3d->setLightElevation( attr.value("elevation").toInt() );
  	curr_axes3d->setDirectLight( attr.value("directed-intensity").toInt() );
  	curr_axes3d->setAmbientLight( attr.value("ambient-intensity").toInt() );
  	}
  else if ( name == "autoscale" ) {
  	curr_axes3d->setAutoscale( (bool )attr.value("enabled").toInt() );
  	}	
  else return read_axes( curr_axes3d, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axes_3d_etag( const QString& name )
 {
  if ( name == AXES3D ) {
  	parser_state.pop();
	remove_default_axes( curr_axes3d );
  	curr_axes = curr_axes3d = NULL;
  	}
  return TRUE;
 }


//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axes( QSAxes *axes, const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "area" ) {
  	 axes->setPosMM( QSPt2f( attr.value("xmm").toDouble(), attr.value("ymm").toDouble() ) );
  	 axes->setSizeMM( QSPt2f( attr.value("wmm").toDouble(), attr.value("hmm").toDouble() ) );
  	}
  else if ( name == "axis" ) {
  	parser_state.push( ParseAxis );
  	curr_axes = axes;
	curr_axis = new QSAxis( (QSAxis::AxisType )get_attr_value(axis_type,attr.value("type")), curr_axes );
	curr_axis->setVisible( (bool )attr.value("visible").toInt() );
	curr_axis->setScrollable( (bool )attr.value("scrollable").toInt() );
	curr_axes->axisAdd( curr_axis );
  	}
  else if ( name == "draw-in-background" ) {
	axes->setDrawInBackground( (bool )attr.value("enabled").toInt() );
	}
  else if ( name == "title-attributes" ) {
  	curr_axes = axes;
  	parser_state.push( ParseAxesTitle );
  	}
  else if ( name == "background-attributes" ) {
  	curr_axes = axes;
  	parser_state.push( ParseAxesBackground );
  	}
  else if ( name == CURVE ) {
  	parser_state.push( ParseCurve );
        curr_plot = curr_curve = new QSCurve( axes ); axes->plotAdd( curr_curve );
  	}
  else if ( name == IMAGE ) {
  	parser_state.push( ParseImage );
  	curr_plot = curr_image = new QSImage( axes ); axes->plotAdd( curr_image );
  	}
  else if ( name == CONTOUR || name == GCONTOUR ) {
  	parser_state.push( ParseContour );
  	curr_plot = curr_contour = new QSGriddedContour( axes ); axes->plotAdd( curr_contour );
  	}
  else if ( name == NGCONTOUR ) {
  	parser_state.push( ParseContour );
  	 curr_plot = curr_contour = new QSNonGriddedContour( axes ); axes->plotAdd( curr_contour );
  	}	
  else if ( name == SURFACE ) {
  	parser_state.push( ParseSurface );
  	curr_plot = curr_surface = new QSSurface( axes ); axes->plotAdd( curr_surface );
  	}
  else if ( name == FIGURE ) {
  	parser_state.push( ParseFigure );
	curr_plot = curr_figure = new QSFigure( axes ); axes->plotAdd( curr_figure );
  	}
  else if ( read_some_object_stag( axes->shadowObject()->objects(), name, attr ) ) {
	}
  else return read_base( axes, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axis_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_axis ) return TRUE;

  if ( name == "display" ) {
  	curr_axis->setPosition( attr.value("position").toDouble() );
	curr_axis->setDefaultPosition( (bool )attr.value("default").toInt() );
	curr_axis->setOppositePosition( (bool )attr.value("opposite").toInt() );
  	}
  else if ( name == "range" ) {
  	curr_axis->setRange( attr.value("min").toDouble(), attr.value("max").toDouble() );
  	curr_axis->setRoundRangeToTicStep( (bool )attr.value("rounded").toInt() );
  	}
  else if ( name == "scale" ) {
  	curr_axis->setScale( (QSAxis::AxisScale )get_attr_value( axis_scale, attr.value("type") ), attr.value("base").toDouble() );
  	curr_axis->setReversed( (bool )attr.value("reversed").toInt() );
  	}
  else if ( name == "grid-step" ) {
  	curr_axis->setGridStep( attr.value("major").toDouble(), attr.value("minor").toDouble() );
  	}
  else if ( name == "tics" ) {
  	curr_axis->setTicsFormat( xml_to_string(attr.value("format")) );
  	curr_axis->setTicsVisible( (bool )attr.value("visible").toInt() );
  	curr_axis->setTicsOuter( (bool )attr.value("outer").toInt() );
  	curr_axis->setTicsAngle( (bool )attr.value("angle").toInt() );
	curr_axis->setTicLabelPos1( attr.value("pos1").toDouble() );
	curr_axis->setTicLabelPos2( attr.value("pos2").toDouble() );
  	}
  else if ( name == "title-position" ) {
	curr_axis->setTitlePosition( attr.value("pos").toDouble() );
	curr_axis->setTitleDistance( attr.value("distance").toDouble() );
	}
  else if ( name == "dart" ) {
  	if ( element_number( attr ) == 1 )
  		 curr_axis->setArrow1( read_arrow( attr ) );
  	    else curr_axis->setArrow2( read_arrow( attr ) );
  		
  	}
  else if ( name == "remembered-view" ) {
	int nr = attr.value("number").toInt();
	if ( nr>=0 && nr<3 ) {
		curr_axis->rememberedViews[nr].min = attr.value("min").toDouble();
		curr_axis->rememberedViews[nr].max = attr.value("max").toDouble();
		curr_axis->rememberedViews[nr].scale = (QSAxis::AxisScale )get_attr_value( axis_scale, attr.value("scale") );
		curr_axis->rememberedViews[nr].base = attr.value("base").toDouble();
		curr_axis->rememberedViews[nr].reversed = (bool )attr.value("reversed").toInt();
                curr_axis->rememberedViews[nr].round = (bool )attr.value("rounded").toInt();
		}
	}
  else read_base( curr_axis, name, attr );
	
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_axis_etag( const QString& name )
 {
  if ( name == "axis" ) {
  	parser_state.pop();
  	curr_axis = NULL;
  	curr_axes = NULL;
  	}
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::remove_default_axes( QSAxes *axes )
// axis object when created always contains a set of default axes ( they are first on the list -
// getAxisOfType() returns always first axis of the given type )  we have all axes written
// to the file and want the axis set to be exactly the same as written so after we read axes from a
// file, we have to delete those default ones.
 {
  axes->axisDelete( axes->axisOfType(QSAxis::XAxisType) );
  axes->axisDelete( axes->axisOfType(QSAxis::YAxisType) );
  axes->axisDelete( axes->axisOfType(QSAxis::ZAxisType) );
  axes->axisDelete( axes->axisOfType(QSAxis::VAxisType) );
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_axis_background_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_axes ) return TRUE;
  if ( name == "fill" ) {
         curr_axes->setBackground( read_fill( attr ) );
  	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): Background-attributes - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_axis_title_stag( const QString& name, const QXmlAttributes& )
 {
  if ( !curr_axes ) return TRUE;
  set_error_string( QObject::tr("<br> KMatplot(XML): Title-attributes - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_axis_background_etag( const QString& name )
 {
  if ( name == "background-attributes" ) {
  	parser_state.pop();
  	curr_axes = NULL;
  	}
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_axis_title_etag( const QString& name )
 {
  if ( name == "title-attributes" ) {
  	parser_state.pop();
  	curr_axes = NULL;
  	}
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_curve_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_curve ) return TRUE;

  if ( name == "series" ) {
  	curr_curve->setType( (QSCurve::SeriesType )get_attr_value(series_type,attr.value("type")) );
  	}
  else if ( name == "delta" ) {
  	curr_curve->setFixedDelta( QSPt2f( attr.value("fixed-x").toDouble(),
  					   attr.value("fixed-y").toDouble() ) );
  	curr_curve->setPercentDelta( QSPt2f( attr.value("percent-x").toDouble(),
  					     attr.value("percent-y").toDouble() ) );
  	}
  else if ( name == "zero" ) {
  	curr_curve->setZeroPoint( QSPt2f( attr.value("x").toDouble(),
  					  attr.value("y").toDouble() ) );
  	}
  else if ( name == "dart" ) {
  	if ( element_number( attr ) == 1 )
  		 curr_curve->setArrow1( read_arrow( attr ) );
  	    else curr_curve->setArrow2( read_arrow( attr ) );
  		
  	}
  else return read_plot( curr_curve, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_curve_etag( const QString& name )
 {
  if ( name == CURVE ) {
  	parser_state.pop();
  	curr_plot = curr_curve =  NULL;
  	}
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_image_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_image ) return TRUE;

  if ( name == "raw-mode" ) {
  	curr_image->setRawMode( (bool )attr.value("enabled").toInt() );
  	}
  else
  if ( name == "use-gradient" ) {
  	curr_image->setUseGradient( (bool )attr.value("enabled").toInt() );
  	}
  else return read_plot( curr_image, name, attr );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_image_etag( const QString& name )
 {
  if ( name == IMAGE ) {
  	parser_state.pop();
  	curr_plot = curr_image = NULL;
  	}
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_contour_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_contour ) return TRUE;
  	
  if ( name == "filled" ) {
  	curr_contour->setContourFills( (bool )attr.value("enabled").toInt() );
  	}
  else if ( name == "lines" ) {
	curr_contour->setContourLines( (bool )attr.value("enabled").toInt() );
  	}
  else if ( name == "labels" ) {
  	curr_contour->setContourLabels( (bool )attr.value("enabled").toInt() );
  	curr_contour->setLabelSpacing( attr.value("spacing").toDouble() );
  	}
/*
  else if ( name == "grid-labels" ) {
  	curr_contour->setContourGridLabelsFormat( xml_to_string( attr.value("format") ) );
  	}
*/
  else return read_plot( curr_contour, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_contour_etag( const QString& name )
 {
  if ( name == CONTOUR || name == GCONTOUR || name == NGCONTOUR ) {
  	parser_state.pop();
  	curr_plot = curr_contour = NULL;
  	}
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_surface_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_surface ) return TRUE;
  if ( name == "labels" ) {
	curr_surface->setXGridStep( attr.value("x-step").toInt() );
	curr_surface->setYGridStep( attr.value("y-step").toInt() );
	}
  return read_plot_3d( curr_surface, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_surface_etag( const QString& name )
 {
  if ( name == SURFACE ) {
  	parser_state.pop();
  	curr_plot = curr_surface = NULL;
  	}
  return TRUE;
 }
 	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_figure_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_figure ) return TRUE;
  if ( name == "vertex-compare" ) {
  	curr_figure->setVertexCompareAccuracy( attr.value("accuracy").toInt() );
  	}
  else return read_plot_3d( curr_figure, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_figure_etag( const QString& name )
 {
  if ( name == FIGURE ) {
  	parser_state.pop();
  	curr_plot = curr_figure = NULL;
  	}
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_plot_3d( QSPlot3D *p, const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "colored" ) {
	p->setColored( (bool )attr.value("enabled").toInt() );
  	}
  else if ( name == "topbottom" ) {
  	p->setTopBottom( (bool )attr.value("enabled").toInt() );
  	}
  else if ( name == "autodivide" ) {
  	p->setAutoDivide( (bool )attr.value("enabled").toInt() );
  	}
  else if ( name == "edges" ) {
	p->setEdgeAutoColor( attr.value("autocolor").toInt() );
	}
  else return read_plot( p, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_plot( QSPlot *p, const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "legend-item" ) {
  	p->setLegendItemVisible( (bool )attr.value("visible").toInt() );
  	}
  else if ( name == "default-axis" ) {
	// ups we have four initial axes, which we remove after reading an axes object.
	// our new read axes starts from 4 then.
  	if ( !attr.value("x-axis").isEmpty() ) p->setDefaultAxis( p->parentAxes()->axis(attr.value("x-axis").toInt()+4) );
  	if ( !attr.value("y-axis").isEmpty() ) p->setDefaultAxis( p->parentAxes()->axis(attr.value("y-axis").toInt()+4) );
  	if ( !attr.value("z-axis").isEmpty() ) p->setDefaultAxis( p->parentAxes()->axis(attr.value("z-axis").toInt()+4) );
  	if ( !attr.value("v-axis").isEmpty() ) p->setDefaultAxis( p->parentAxes()->axis(attr.value("v-axis").toInt()+4) );
  	}
  else if ( name == "gradient" ) {
	parser_state.push( ParseGradient );
	curr_gradient_type = get_attr_value( gradient_type, attr.value("type") );
	return TRUE;
	}
  else return read_base( p, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_base( QSGraphicalData *b, const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "fill" ) {
  	b->setFill( element_number(attr), read_fill(attr) );
  	}
  else if ( name == "font" ) {
	b->setFont( element_number(attr), read_font(attr) );
  	}
  else if ( name == "line" ) {
  	b->setLine( element_number(attr), read_line(attr) );
  	}
  else if ( name == "point" ) {
  	b->setPoint( element_number(attr), read_point(attr) );
  	}
  else if ( name == "title" ) {
  	b->setTitle( xml_to_string(attr.value("text")) );
  	}
  else return read_data( b, name, attr );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_data( QSData *d, const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "matrix" ) {
  	parser_state.push( ParseMatrix );
  	create_matrix( d, name, attr );
  	}
  else if ( name == "matrixref" ) {
         parser_state.push( ParseMatrixRef );
         create_matrix( d, name, attr );
	}
  else if ( name == "matrixcolref" ) {
	 parser_state.push( ParseMatrixColRef );
         create_matrix( d, name, attr );
	}
  else if ( name == "matrix-string" ) {
	parser_state.push( ParseMatrixString );
	create_matrix( d, name, attr );
	}
  else if ( name == "matrix-formula" ) {
	parser_state.push( ParseMatrixFormula );
	create_matrix( d, name, attr );
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): QSData - Unexpected tag ") + name );
  return TRUE;
 }
 	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_clegend_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_clegend ) return TRUE;

  if ( name == "align" ) {
  	curr_clegend->setAlign( attr.value("frame").toInt() );
  	}
  else if ( name == "format" ) {
  	curr_clegend->setColumns( attr.value("columns").toInt() );
  	}
  else if ( name == "pos" ) {
  	curr_clegend->setPos( read_pt3f(attr) );
  	}
  else if ( name == "coord-system" ) {
  	curr_clegend->setCoord( read_pt3(attr) );
  	}
  else if ( name == "shadow-shift" ) {
  	curr_clegend->setShadowPos( read_pt2(attr) );
  	}
  else if ( name == "font" ) {
  	curr_clegend->setFont( read_font(attr) );
  	}
  else if ( name == "fill" ) {
  	if ( element_number( attr ) == 1 ) curr_clegend->setFill( read_fill(attr) );
  				      else curr_clegend->setShadowFill( read_fill(attr) );
  	}
  else if ( name == "line" ) {
  	curr_clegend->setFrame( read_line(attr) );
  	}
  else read_cobject_stag( curr_clegend, name, attr );
  	
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_clegend_etag( const QString& name )
 {
  if ( name == "legend" ) {
  	parser_state.pop();
  	curr_clegend = NULL;
  	}
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_clabel_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_clabel ) return TRUE;

  if ( name == "text" ) {
  	// see end tag
  	}
  else if ( name == "angle" ) {
  	curr_clabel->setAngle( attr.value("deg").toInt() );
  	}
  else if ( name == "align" ) {
  	curr_clabel->setFrameAlign( attr.value("frame").toInt() );
  	curr_clabel->setTextAlign( attr.value("text").toInt() );
  	}
  else if ( name == "pos" ) {
  	curr_clabel->setPos( read_pt3f(attr) );
  	}
  else if ( name == "coord-system" ) {
  	curr_clabel->setCoord( read_pt3(attr) );
  	}
  else if ( name == "shadow-shift" ) {
  	curr_clabel->setShadowPos( read_pt2(attr) );
  	}
  else if ( name == "font" ) {
  	curr_clabel->setFont( read_font(attr) );
  	}
  else if ( name == "fill" ) {
  	if ( element_number( attr ) == 1 ) curr_clabel->setFill( read_fill(attr) );
  				      else curr_clabel->setShadowFill( read_fill(attr) );
  	}
  else if ( name == "line" ) {
  	curr_clabel->setFrame( read_line(attr) );
  	}
  else read_cobject_stag( curr_clabel, name, attr );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_clabel_etag( const QString& name )
 {
  if ( name == "text" ) {
  	 curr_clabel->setText( chars );
  	}
  else if ( name == "label" ) {
  	parser_state.pop();
  	curr_clabel = NULL;
  	}
  	
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_carrow_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_carrow ) return TRUE;

  if ( name == "begin" ) {
  	curr_carrow->setOriginPos( read_pt3f(attr) );	
  	}
  else if ( name == "end" ) {
  	curr_carrow->setEndPos( read_pt3f(attr) );	  	
  	}
  else if ( name == "coord-system-begin" ) {
  	 curr_carrow->setOriginCoord( read_pt3(attr) );	
  	}
  else if ( name == "coord-system-end" ) {
  	 curr_carrow->setEndCoord( read_pt3(attr) );	
  	}
  else if ( name == "dart" ) {
  	if ( element_number( attr ) == 1 ) curr_carrow->setOriginArrow( read_arrow(attr) );
  				      else curr_carrow->setEndArrow( read_arrow(attr) );
  	}
  else if ( name == "point" ) {
	if ( element_number( attr ) == 1 ) curr_carrow->setOriginPoint( read_point(attr) );
  				      else curr_carrow->setEndPoint( read_point(attr) );  	
  	}
  else if ( name == "line" ) {
  	curr_carrow->setLine( read_line(attr) );
  	}
  else read_cobject_stag( curr_carrow, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_carrow_etag( const QString& name )
 {
  if ( name == "arrow" ) {
  	parser_state.pop();
  	curr_carrow = NULL;
  	}
  	
  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_crect_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( !curr_crect ) return TRUE;

  if ( name == "type" ) {
  	curr_crect->setEllipse( (bool )attr.value("ellipse").toInt() );
  	}
  else if ( name == "begin" ) {
  	curr_crect->setOriginPos( read_pt3f(attr) );	
  	}
  else if ( name == "end" ) {
  	curr_crect->setEndPos( read_pt3f(attr) );	  	
  	}
  else if ( name == "coord-system-begin" ) {
  	 curr_crect->setOriginCoord( read_pt3(attr) );	
  	}
  else if ( name == "coord-system-end" ) {
  	 curr_crect->setEndCoord( read_pt3(attr) );	
  	}
  else if ( name == "shadow-shift" ) {
  	curr_crect->setShadowPos( read_pt2(attr) );
  	}
  else if ( name == "fill" ) {
  	if ( element_number( attr ) == 1 ) curr_crect->setFill( read_fill(attr) );
  				      else curr_crect->setShadowFill( read_fill(attr) );
  	}
  else if ( name == "line" ) {
  	curr_crect->setFrame( read_line(attr) );
  	}
  else read_cobject_stag( curr_crect, name, attr );

  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_crect_etag( const QString& name )
 {
  if ( name == "rectangle" ) {
  	parser_state.pop();
  	curr_crect = NULL;
  	}
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_cgroup_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( read_some_object_stag( curr_cgroup.top()->objects(), name, attr ) ) {
	}
  else read_cobject_stag( curr_cgroup.top(), name, attr );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_cgroup_etag( const QString& name )
 {
  if ( name == "group" ) {
  	parser_state.pop();
	curr_cgroup.pop();
  	}
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_cobject_stag( QSCObject *object, const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "axes-bindings" ) {
	object->setDefaultXAxis( attr.value("x-axis").toInt() );
	object->setDefaultYAxis( attr.value("y-axis").toInt() );
	object->setDefaultZAxis( attr.value("z-axis").toInt() );		
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): QSCObject - unexpected tag ") + name );

  return TRUE;
 }
  	
//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_stag( const QString& , const QXmlAttributes&  )
 {
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_etag( const QString& name )
 {		
  if ( name == "row" ) {
  	if ( curr_matrix && curr_row < curr_matrix->rows()  ) {
        	double value;
        	QTextIStream line(&chars);
        	for( int col=0; col<curr_matrix->cols(); col++ ) {
                 	value = 0.0; line >> value;
			curr_matrix->setValue( curr_row, col, value );
                	}
       		}  	
  	curr_row += 1;
  	}
  else if ( name == "matrix" ) {
  	curr_matrix = NULL;
  	curr_row = 0;
  	parser_state.pop();
  	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): Matrix - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_ref_stag( const QString& , const QXmlAttributes&  )
 {
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_ref_etag( const QString& name )
 {
  if ( name == "matrixref" ) {
	curr_matrix = NULL;
	parser_state.pop();
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): MatrixRef - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_formula_stag( const QString& , const QXmlAttributes&  )
 {
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_formula_etag( const QString& name )
 {
  if ( name == "matrix-formula" ) {
	curr_matrix = NULL;
	parser_state.pop();
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): MatrixFormula - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_col_ref_stag( const QString& , const QXmlAttributes&  )
 {
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_col_ref_etag( const QString& name )
 {
  if ( name == "matrixcolref" ) {
	curr_matrix = NULL;
	parser_state.pop();
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): MatrixColRef - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_string_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "e" ) {
	curr_matrix->setString( attr.value("row").toInt(), attr.value("col").toInt(), xml_to_string(attr.value("string")) );
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): MatrixString - unexpected tag ") + name );
  return TRUE;
 }

//-----------------------------------------------------------//
	
bool ksprojectxml_runtime_data::read_matrix_string_etag( const QString& name )
 {
  if ( name == "matrix-string" ) {
	curr_matrix = NULL;
	parser_state.pop();
	}
  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_gradient_stag( const QString& name, const QXmlAttributes& attr )
 {
  if ( name == "fill" ) {
 	int f_number = element_number( attr );
	curr_gradient_fill[QMIN(4,QMAX(f_number,0))] = read_fill( attr );
	}
  else set_error_string( QObject::tr("<br> KMatplot(XML): Gradient - unexpected tag ") + name );

  return TRUE;
 }

//-----------------------------------------------------------//

bool ksprojectxml_runtime_data::read_gradient_etag( const QString& name )
 {
  if ( name == "gradient" ) {
  	parser_state.pop();
  	curr_plot->setGradient( QSGGradient( curr_gradient_fill[0],
                                             curr_gradient_fill[1],
                                             curr_gradient_fill[2],
                                                       curr_gradient_fill[3],
                                                       curr_gradient_fill[4],
						       (QSGGradient::Type )curr_gradient_type) );
	}

  return TRUE;
 }
 	
//-----------------------------------------------------------//

int ksprojectxml_runtime_data::element_number( const QXmlAttributes& attr )
 {
  return attr.value("element-number").toInt();
 }

//-----------------------------------------------------------//

QSGFill ksprojectxml_runtime_data::read_fill( const QXmlAttributes& attr )
 {
  QSGFill result;
  result.style = (QSGFill::Style )get_attr_value( fill_style, attr.value("style") );
  result.color = toQSGColor( attr.value("color") );
  return result;
 }

//-----------------------------------------------------------//

QSGLine ksprojectxml_runtime_data::read_line( const QXmlAttributes& attr )
 {
  QSGLine result;
  result.style = (QSGLine::Style )get_attr_value( line_style, attr.value("style") );
  result.width = attr.value("width").toInt();
  result.color = toQSGColor( attr.value("color") );
  return result;
 }

//-----------------------------------------------------------//

QSGFont ksprojectxml_runtime_data::read_font( const QXmlAttributes& attr )
 {
  QSGFont result;
  result.family = xml_to_string( attr.value("family") );
  result.size   = attr.value("size").toInt();
  result.bold   = (bool )attr.value("bold").toInt();
  result.italic = (bool )attr.value("italic").toInt();
  result.color  = toQSGColor( attr.value("color") );
  return result;
 }

//-----------------------------------------------------------//

QSGArrow ksprojectxml_runtime_data::read_arrow( const QXmlAttributes& attr )
 {
  QSGArrow result;
  result.style = (QSGArrow::Style )get_attr_value( arrow_style, attr.value("style") );
  result.size = attr.value("size").toInt();
  return result;
 }

//-----------------------------------------------------------//

QSGPoint ksprojectxml_runtime_data::read_point( const QXmlAttributes& attr )
 {
  QSGPoint result;
  result.style = (QSGPoint::Style )get_attr_value( point_style, attr.value("style") );
  result.fill  = (QSGPoint::Fill )get_attr_value( point_fill_style, attr.value("fill") );
  result.size  = attr.value("size").toInt();
  result.color = toQSGColor( attr.value("color") );
  return result;
 }

//-----------------------------------------------------------//

QSPt2 ksprojectxml_runtime_data::read_pt2( const QXmlAttributes& attr )
 {
  return QSPt2( attr.value("x").toInt(), attr.value("y").toInt() );
 }

//-----------------------------------------------------------//

QSPt3 ksprojectxml_runtime_data::read_pt3( const QXmlAttributes& attr )
 {
  return QSPt3( attr.value("x").toInt(), attr.value("y").toInt(), attr.value("z").toInt() );
 }

//-----------------------------------------------------------//

QSPt3f ksprojectxml_runtime_data::read_pt3f( const QXmlAttributes& attr )
 {
  return QSPt3f( attr.value("x").toDouble(), attr.value("y").toDouble(), attr.value("z").toDouble() );
 }

//-----------------------------------------------------------//

void ksprojectxml_runtime_data::create_matrix( QSData *d, const QString& name, const QXmlAttributes& attr )
 {
  if ( d == NULL ) {
	curr_matrix = NULL;
	set_error_string( QObject::tr("KMatplot(create_matrix): No data object allocated - skipping matrix.<br>") );
	return;
	}

  QString title = xml_to_string( attr.value("title") );
  int channel = attr.value("data-channel").toInt();

  if ( name == "matrix" ) {
  	int rows = attr.value("rows").toInt();
  	int cols = attr.value("cols").toInt();
  	KSMatrix *new_matrix = KSMatrix::create( (EType )get_attr_value(etype,attr.value("element-type")) );
  	new_matrix->resize( rows, cols );
  	curr_matrix = new_matrix;
	}
   else if ( name == "matrixref" ) {
	 KSMatrixWorksheetCellRange *new_matrix = new KSMatrixWorksheetCellRange( workbook );
         new_matrix->setWorksheet( sheet_number(attr.value("sheet").toInt()) );

	 new_matrix->setRowFrom( attr.value("row-from").toInt() );
	 new_matrix->setRowStep( attr.value("row-step").toInt() );
	 new_matrix->setRowTo( attr.value("row-to").toInt() );
	 new_matrix->setColFrom( attr.value("col-from").toInt() );
	 new_matrix->setColStep( attr.value("col-step").toInt() );
	 new_matrix->setColTo( attr.value("col-to").toInt() );
	 new_matrix->setTransposition( (bool )attr.value("transposition").toInt() );
  	curr_matrix = new_matrix;
	}
   else if ( name == "matrixcolref" ) {
	 KSMatrixWorksheetCellRange *new_matrix = new KSMatrixWorksheetCellRange( workbook );
	 new_matrix->setWorksheet( sheet_number(attr.value("sheet").toInt()) );

	 new_matrix->setColumn( attr.value("column").toInt() );
	 new_matrix->setTransposition( (bool )attr.value("transposition").toInt() );
  	 curr_matrix = new_matrix;
	}
   else if ( name == "matrix-string" ) {
  	int rows = attr.value("rows").toInt();
  	int cols = attr.value("cols").toInt();
	KSMatrixString *new_matrix = new KSMatrixString();
	new_matrix->resize( rows, cols );
	curr_matrix = new_matrix;
	}
   else if ( name == "matrix-formula" ) {
	KSMatrixFormula *new_matrix = new KSMatrixFormula();
   	new_matrix->setFormula( xml_to_string( attr.value("formula") ) );
	new_matrix->setTransposition( (bool )attr.value("transposition").toInt() );	
	curr_matrix = new_matrix;
	}
   if ( curr_matrix ) {
	curr_matrix->setName(title);
   	curr_matrix->setDataObject( d, channel );
   	d->setMatrix( channel, curr_matrix );

	// setting matrix failed - nonexistient channel ?
        if ( d->matrix(channel) != curr_matrix ) {
		 delete curr_matrix;
		 curr_matrix = NULL;
		}

	// recalculate formula matrix -> add output to console
	if ( dynamic_cast<KSMatrixFormula*>(curr_matrix) ) {
		KSMatrixFormula *formula_matrix = dynamic_cast<KSMatrixFormula*>(curr_matrix);
  		MPFactoryList factory( new KSDataSymbolFactory( workbook, formula_matrix->dataObject() ) );
  		MPFormulaError error;
  		formula_matrix->init( error, &factory );
		if ( error.hasError() ) QSConsole::write( QObject::tr("Error parsing formula :")+formula_matrix->formula()+"<br>"+error.message() );
		}
	}

   curr_row = 0;
  }

//-----------------------------------------------------------//






















