/***************************************************************************
                                kscommands.cpp
                             -------------------                                         
    begin                :
    copyright            : (C) 2001 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include"kscommands.h"
#include"ksglobalmatrixlist.h"
#include"ksobjectfactory.h"
#include"ksmatrix.h"
#include"widgets/qscobject.h"
#include"widgets/qsaxes.h"
#include"widgets/qsplot.h"
#include<qobject.h>
#include<qvariant.h>

//---------------------------------------------------------------------------------------------//

KSCmdAddCObject::KSCmdAddCObject( QSCObject *object, QSCObjectCollection *parent )
: KSCommand( QObject::tr("New object") )
 {
  m_objects.setAutoDelete( TRUE );
  m_objects.append( object );
  m_parent = parent;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddCObject::KSCmdAddCObject( QSCObjectCollection *objects, QSCObjectCollection *parent )
: KSCommand( QObject::tr("New objects") )
 {
  m_objects.setAutoDelete( TRUE );
  for( int i=0; i<objects->count(); i++ ) m_objects.append( objects->object(i) );
  m_parent = parent;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddCObject::~KSCmdAddCObject()
 {
 }	

//---------------------------------------------------------------------------------------------//

bool KSCmdAddCObject::execute()
// improve this to keep relative order of object
 {
  if ( !m_parent ) {
	setError( QObject::tr("No page selected") );
	return false;
	}
  for( int i=0; i<(int )m_objects.count(); i++ ) {
  	m_parent->add( m_objects.at(i) );
	}
  // object is owned by collection now
  m_objects.setAutoDelete( false );
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddCObject::unexecute()
 {
  for( int i=0; i<(int )m_objects.count(); i++ ) {
	m_parent->remove( m_parent->find(m_objects.at(i)) );
	}
  // take ownership	
  m_objects.setAutoDelete( true );
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

//---------------------------------------------------------------------------------------------//

KSCmdRemoveCObject::KSCmdRemoveCObject( QSCObject *object )
: KSCommand( QObject::tr("Remove object") )
 {
  m_parent = object->collection();
  m_objects.setAutoDelete( false );
  m_objects.append( object );
  m_positions.insert( ( m_parent ? m_parent->find(object) : 0 ), object );
 }

//---------------------------------------------------------------------------------------------//

//
// All objects in collection must have the same parent collection
//
KSCmdRemoveCObject::KSCmdRemoveCObject( QSCObjectCollection *objects )
: KSCommand( QObject::tr("Remove objects") )
 {
  m_parent = NULL;
  if ( objects->count() ) m_parent = objects->object(0)->collection();
  m_objects.setAutoDelete( false );
  for( int i=0; i<objects->count(); i++ ) {
	m_objects.append( objects->object(i) );
	m_positions.insert( ( m_parent ? m_parent->find(objects->object(i)) : 0 ), objects->object(i) );	
	}
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemoveCObject::~KSCmdRemoveCObject()	
 {
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemoveCObject::execute()
 {
  if ( !m_parent ) {
	setError( QObject::tr("No object selected") );
	return false;
	}	
  for( int i=0; i<(int )m_objects.count(); i++ ) {
	m_objects.at(i)->stop();
	m_parent->remove( m_parent->find(m_objects.at(i)) );
	}
  // take ownership	
  m_objects.setAutoDelete( true );
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemoveCObject::unexecute()
 {
  QMap<int,QSCObject*>::Iterator it;
  // map is sorted by position - insert all objects at previous positions
  for( it = m_positions.begin(); it != m_positions.end(); ++it ) {
	m_parent->insert( it.key(), it.data() );
	}
  // object is owned by collection now
  m_objects.setAutoDelete( false );
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

//
// objects still belongs to the caller.
// All objects in collection 'objects' must have the same parent collection
// but it needn't be 'objects' itself.
//
KSCmdObjectOrder::KSCmdObjectOrder( CommandType t, QSCObjectCollection *objects, QSWorkbook * )
: KSCommand( QObject::tr("Order object(s)") )
 {
  m_type = t;
  // copy collection to interanal buffer
  if ( objects && objects->count() > 0 ) {
  	m_collection = objects->object(0)->collection();
  	for ( int i=0; i<objects->count(); i++ ) {
		m_objects.insert( m_collection->find(objects->object(i)), objects->object(i) );
		}
	}
 }

//---------------------------------------------------------------------------------------------//

KSCmdObjectOrder::~KSCmdObjectOrder()
 {
 }

//---------------------------------------------------------------------------------------------//
	
bool KSCmdObjectOrder::execute()
 {
  if ( m_objects.count() == 0 ) {
	setError( QObject::tr("No object selected") );
	return false;
	}
  int min_pos = 0;
  int max_pos = m_collection->count()-1;
  QMap<int,QSCObject*>::Iterator it;
  switch( m_type ) {
	case ToFront:	for( it = m_objects.begin(); it != m_objects.end(); ++it )  {
				it.data()->toFront();
				}
			break;
	case ToBack: 	it = m_objects.end();
			while( it != m_objects.begin() ) {
 				--it;
				it.data()->toBack();
				}
			break;
	case Raise:     it = m_objects.end();
			while( it != m_objects.begin() ) {
				--it;
				it.data()->reorder( QMIN(it.key()+1,max_pos) );
				max_pos = max_pos - 1;
				}
			break;
	case Lower:     for( it =m_objects.begin(); it != m_objects.end(); ++it ) {
				it.data()->reorder( QMAX(it.key()-1,min_pos) );
				min_pos = min_pos + 1;
				}
			break;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdObjectOrder::unexecute()
 {
  QMap<int,QSCObject*>::Iterator it;
  // map is sorted by position - move all objects to previous positions
  switch( m_type ) {
	case ToFront:
	case Raise:	for( it = m_objects.begin(); it != m_objects.end(); ++it ) {
				it.data()->reorder( it.key() );
				}
			break;
	case ToBack:
	case Lower:	it = m_objects.end();
			while( it != m_objects.begin() ) {
				--it; it.data()->reorder( it.key() );
				}		
			break;
	}
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

/**
  * Dont mix rotate and setBox command .Object can be only rotated or only moved/resized.
  */
KSCmdTransformCObjects::KSCmdTransformCObjects( QSDrv *drv )
: KSCommand(QObject::tr("Transform object(s)"))
 {
  m_drv = drv;
 }

//---------------------------------------------------------------------------------------------//

KSCmdTransformCObjects::~KSCmdTransformCObjects()
 {
  delete m_drv;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdTransformCObjects::addObject( QSCObject *object )
 {
  m_drv->startDrawing();
  m_old_data.insert( object, TransformData(object->box(m_drv),object->angle()) );
  m_drv->stopDrawing();
 }

//---------------------------------------------------------------------------------------------//

QSRectf KSCmdTransformCObjects::objectRect( QSCObject *object )
 {
  return m_old_data[object].m_rect;
 }

//---------------------------------------------------------------------------------------------//

int KSCmdTransformCObjects::objectAngle( QSCObject *object )
 {
  return m_old_data[object].m_angle;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdTransformCObjects::commit()
 {
  m_drv->startDrawing();
  TransformDataList::Iterator element;
  for( element = m_old_data.begin(); element != m_old_data.end(); ++element ) {
	m_new_data.insert( element.key(), TransformData(element.key()->box(m_drv),element.key()->angle()) );
	}
  m_drv->stopDrawing();
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdTransformCObjects::execute()
 {
  m_drv->startDrawing();
  TransformDataList::Iterator element;
  for( element = m_new_data.begin(); element != m_new_data.end(); ++element ) {
	element.key()->setAngle(element.data().m_angle);
	element.key()->setBox(element.data().m_rect,m_drv);
	}
  for( element = m_old_data.begin(); element != m_old_data.end(); ++element ) element.key()->forceUpdate();
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdTransformCObjects::unexecute()
 {
  m_drv->startDrawing();
  TransformDataList::Iterator element;
  for( element = m_old_data.begin(); element != m_old_data.end(); ++element ) {
	element.key()->setAngle(element.data().m_angle);
	element.key()->setBox(element.data().m_rect,m_drv);
	}
  m_drv->stopDrawing();
  for( element = m_old_data.begin(); element != m_old_data.end(); ++element ) element.key()->forceUpdate();
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdCObjectGroup::KSCmdCObjectGroup( QSCObjectCollection *objects, QSCObjectCollection *selection )
: KSCommand( QObject::tr("Group objects") )
 {
  m_group = NULL;
  m_parent = NULL;
  m_selection = selection;
  if ( objects->count() ) m_parent = objects->object(0)->collection();
  m_objects.setAutoDelete( false );
  for( int i=0; i<objects->count(); i++ ) {
	m_objects.append(objects->object(i));
	m_positions.insert( ( m_parent ? m_parent->find(objects->object(i)) : 0 ),  objects->object(i) );
	}
 }

//---------------------------------------------------------------------------------------------//

KSCmdCObjectGroup::~KSCmdCObjectGroup()
 {
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdCObjectGroup::execute()
 {
  if ( !m_parent ) {
	setError( QObject::tr("No page selected") );
	return false;	
	}
  if ( m_objects.count() < 2 ) {
	setError( QObject::tr("No object to group") );
	return false;
	}

  m_parent->setAutoUpdates( false );
  if ( m_selection ) m_selection->setAutoUpdates( false );

  m_group = new QSCGroup();
  for( unsigned int i=0; i<m_objects.count(); i++ ) {
	m_parent->remove( m_parent->find(m_objects.at(i)) );
	m_group->objects()->add( m_objects.at(i) );
	}
  m_parent->add( m_group );

  if ( m_selection ) {
	m_selection->clear();
	m_selection->add( m_group );
	}

  m_parent->setAutoUpdates( true );
  if ( m_selection ) m_selection->setAutoUpdates( true );

  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdCObjectGroup::unexecute()
 {
  m_parent->setAutoUpdates( false );
  if ( m_selection ) m_selection->setAutoUpdates( false );

  m_parent->remove( m_parent->find(m_group) );
  if ( m_selection ) m_selection->clear();
  QMap<int,QSCObject*>::Iterator it;
  // map is sorted by position - insert all objects at previous positions
  for( it = m_positions.begin(); it != m_positions.end(); ++it ) {
	m_group->objects()->remove( m_group->objects()->find(it.data()) );
	m_parent->insert( it.key(), it.data() );
	if ( m_selection ) m_selection->add( it.data() );
	}
  delete m_group; m_group = NULL;

  m_parent->setAutoUpdates( true );
  if ( m_selection ) m_selection->setAutoUpdates( true );
 }


//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdCObjectUngroup::KSCmdCObjectUngroup( QSCGroup *group, QSCObjectCollection *selection )
: KSCommand( QObject::tr("Ungroup objects") )
 {
  m_group = group;
  m_parent = NULL;
  m_selection = selection;
  m_objects.setAutoDelete( false );
  if ( m_group ) {
	m_parent = m_group->collection();
  	for( int i=0; i<group->objects()->count(); i++ ) m_objects.append( m_group->objects()->object(i) );
	}
 }

//---------------------------------------------------------------------------------------------//

KSCmdCObjectUngroup::~KSCmdCObjectUngroup()
 {
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdCObjectUngroup::execute()
 {
  if ( !m_group ) {
	setError( QObject::tr("No group selected") );
	return false;	
	}
  m_parent->setAutoUpdates( false );
  if ( m_selection ) m_selection->setAutoUpdates( false );

  if ( m_selection ) m_selection->clear();
  m_parent->remove( m_parent->find(m_group) );
  // map is sorted by position - insert all objects at previous positions
  for( unsigned int i=0; i<m_objects.count(); i++ ) {
	m_group->objects()->remove( m_group->objects()->find(m_objects.at(i)) );
	m_parent->add( m_objects.at(i) );
 	if ( m_selection ) m_selection->add(m_objects.at(i));
	}
  delete m_group; m_group = NULL;

  m_parent->setAutoUpdates( true );
  if ( m_selection ) m_selection->setAutoUpdates( true );

  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdCObjectUngroup::unexecute()
 {
  m_parent->setAutoUpdates( false );
  if ( m_selection ) m_selection->setAutoUpdates( false );

  m_group = new QSCGroup();
  for( unsigned int i=0; i<m_objects.count(); i++ ) {
	m_parent->remove(m_parent->find(m_objects.at(i)));
	m_group->objects()->add(m_objects.at(i));
	}
  m_parent->add( m_group );
  if ( m_selection ) {
	m_selection->clear();
	m_selection->add( m_group );
	}

  m_parent->setAutoUpdates( true );
  if ( m_selection ) m_selection->setAutoUpdates( true );

 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//



KSCmdAddDataset::KSCmdAddDataset( QSPlot *plot )
: KSCommand(QObject::tr("Add dataset"))
 {
  m_plot = plot;
  m_axes = plot->parentAxes();
  m_delete_plot = true;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddDataset::~KSCmdAddDataset()
 {
  if ( m_delete_plot ) delete m_plot;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdAddDataset::execute()
 {
  m_axes->plotAdd( m_plot );
  m_delete_plot = false;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddDataset::unexecute()
 {
  m_axes->plotRemove( m_plot );
  m_delete_plot = true;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdAddDatasetGroup::KSCmdAddDatasetGroup()
: KSCommand(QObject::tr("Add datasets"))
 {
  m_plots.setAutoDelete( TRUE );
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddDatasetGroup::~KSCmdAddDatasetGroup()
 {
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddDatasetGroup::addDataset( QSPlot *plot )
 {
  if ( plot ) {
	m_plots.append( plot );
	m_axes = plot->parentAxes();
	}
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdAddDatasetGroup::execute()
 {
  QSPlot *plot;
  for ( plot=m_plots.first(); plot!=0; plot=m_plots.next() ) m_axes->plotAdd( plot );	
  m_plots.setAutoDelete( FALSE );
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddDatasetGroup::unexecute()
 {
  QSPlot *plot;
  for ( plot=m_plots.first(); plot!=0; plot=m_plots.next() ) m_axes->plotRemove( plot );	
  m_plots.setAutoDelete( TRUE );
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdRemoveDataset::KSCmdRemoveDataset( QSPlot *plot )
: KSCommand(QObject::tr("Remove dataset"))
 {
  m_plot = plot;
  m_axes = plot ? plot->parentAxes() : NULL;
  m_delete_plot = false;
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemoveDataset::~KSCmdRemoveDataset()
 {
  if ( m_delete_plot ) delete m_plot;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemoveDataset::execute()
 {
  if ( !m_plot ) {
	setError(QObject::tr("No dataset selected"));
	return false;
	}
  m_pos = m_axes->plotIndex(m_plot);
  m_axes->plotRemove(m_plot);
  m_delete_plot = true;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemoveDataset::unexecute()
 {
  m_axes->plotInsert(m_pos,m_plot);
  m_delete_plot = false;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdDatasetOrder::KSCmdDatasetOrder( CommandType t, QSPlot *plot )
:KSCommand( QObject::tr("Order datasets") )
 {
  m_type = t;
  m_plot = plot;
  m_axes = plot ? plot->parentAxes() : NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdDatasetOrder::~KSCmdDatasetOrder()
 {
 }

//---------------------------------------------------------------------------------------------//
	
bool KSCmdDatasetOrder::execute()
 {
  if ( !m_plot ) {
	setError(QObject::tr("No dataset selected"));
	return false;
	}
  m_pos = m_axes->plotIndex(m_plot);
  switch( m_type ) {
	case ToFront:	m_axes->plotToFront(m_plot); break;
	case ToBack:    m_axes->plotToBack(m_plot); break;
	case Raise:     m_axes->plotRaise(m_plot); break;
	case Lower:     m_axes->plotLower(m_plot); break;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdDatasetOrder::unexecute()
 {
  switch( m_type ) {
	case ToFront:  m_axes->plotReorder(m_pos,m_plot); break;
	case ToBack:   m_axes->plotReorder(m_pos,m_plot); break;
	case Raise:    m_axes->plotLower(m_plot); break;
	case Lower:    m_axes->plotRaise(m_plot); break;
	}
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdAddAxis::KSCmdAddAxis( QSAxis *axis )
:KSCommand(QObject::tr("Add axis") )
 {
  m_axis = axis;
  m_axes = axis->parentAxes();
  m_delete_axis = true;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddAxis::~KSCmdAddAxis()
 {
  if ( m_delete_axis ) delete m_axis;
 }	

//---------------------------------------------------------------------------------------------//

bool KSCmdAddAxis::execute()
 {
  m_axes->axisAdd( m_axis );
  m_delete_axis = false;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddAxis::unexecute()
 {
  m_axes->axisRemove( m_axis );
  m_delete_axis = true;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdRemoveAxis::KSCmdRemoveAxis( QSAxis *axis )
: KSCommand(QObject::tr("Remove axis"))
 {
  m_axis = axis;
  m_axes = axis ? axis->parentAxes() : NULL;
  m_delete_axis = false;
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemoveAxis::~KSCmdRemoveAxis()
 {
  if ( m_delete_axis ) delete m_axis;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemoveAxis::execute()
 {
  if ( !m_axis ) {
	setError(QObject::tr("No axis selected"));
	return false;
	}
  m_pos = m_axes->axisIndex(m_axis);
  if ( m_axes->axisRemove(m_axis) ) {
  	m_delete_axis = true;
	} else {
	setError(QObject::tr("Can't remove the last axis of a given type !"));
	return false;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemoveAxis::unexecute()
 {
  m_axes->axisInsert( m_pos, m_axis );
  m_delete_axis = false;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdAxisOrder::KSCmdAxisOrder( CommandType t, QSAxis *axis )
:KSCommand( QObject::tr("Order axes") )
 {
  m_type = t;
  m_axis = axis;
  m_axes = axis ? axis->parentAxes() : NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAxisOrder::~KSCmdAxisOrder()
 {
 }

//---------------------------------------------------------------------------------------------//
	
bool KSCmdAxisOrder::execute()
 {
  if ( !m_axis ) {
	setError(QObject::tr("No axis selected"));
	return false;
	}
  m_pos = m_axes->axisIndex(m_axis);
  switch( m_type ) {
	case ToFront:	m_axes->axisToFront(m_axis); break;
	case ToBack:    m_axes->axisToBack(m_axis); break;
	case Raise:     m_axes->axisRaise(m_axis); break;
	case Lower:     m_axes->axisLower(m_axis); break;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAxisOrder::unexecute()
 {
  switch( m_type ) {
	case ToFront:  m_axes->axisReorder(m_pos,m_axis); break;
	case ToBack:   m_axes->axisReorder(m_pos,m_axis); break;
	case Raise:    m_axes->axisLower(m_axis); break;
	case Lower:    m_axes->axisRaise(m_axis); break;
	}
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdAddPage::KSCmdAddPage( QSPage *page, QSWorkbook *workbook )
: KSCommand(QObject::tr("Add page"))
 {
  m_page = page;
  m_workbook = workbook;
  m_delete_page = true;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddPage::~KSCmdAddPage()	
 {
  if ( m_delete_page ) delete m_page;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdAddPage::execute()
 {
  m_workbook->pageAdd( m_page );
  m_delete_page = false;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddPage::unexecute()
 {
  m_workbook->pageRemove( m_workbook->pageFind(m_page) );
  m_delete_page = true;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdRemovePage::KSCmdRemovePage( QSPage *page, QSWorkbook *workbook )
: KSCommand(QObject::tr("Remove page"))
 {
  m_page = page;
  m_workbook = workbook;
  m_delete_page = false;
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemovePage::~KSCmdRemovePage()	
 {
  if ( m_delete_page ) delete m_page;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemovePage::execute()
 {
  if (m_workbook->pageFind(m_page)<0 ) {
	setError(QObject::tr("No page selected"));
	return false;
	}

  m_pos = m_workbook->pageFind(m_page);
  m_workbook->pageRemove(m_pos);
  m_delete_page = true;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemovePage::unexecute()
 {
  m_workbook->pageInsert(m_pos,m_page);
  m_delete_page = false;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdPageOrder::KSCmdPageOrder( CommandType t, QSPage *page, QSWorkbook *workbook )
:KSCommand( QObject::tr("Order pages") )
 {
  m_type = t;
  m_page = page;
  m_workbook = workbook;
 }

//---------------------------------------------------------------------------------------------//

KSCmdPageOrder::~KSCmdPageOrder()
 {
 }

//---------------------------------------------------------------------------------------------//
	
bool KSCmdPageOrder::execute()
 {
  if ( !m_page ) {
	setError(QObject::tr("No page selected"));
	return false;
	}
  m_pos = m_workbook->pageFind(m_page);
  switch( m_type ) {
	case ToFront:	m_workbook->pageToFront(m_pos); break;
	case ToBack:    m_workbook->pageToBack(m_pos); break;
	case Raise:     m_workbook->pageRaise(m_pos); break;
	case Lower:     m_workbook->pageLower(m_pos); break;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdPageOrder::unexecute()
 {
  int page_index = m_workbook->pageFind(m_page);
  switch( m_type ) {
	case ToFront:  m_workbook->pageReorder(m_pos,page_index); break;
	case ToBack:   m_workbook->pageReorder(m_pos,page_index); break;
	case Raise:    m_workbook->pageLower(page_index); break;
	case Lower:    m_workbook->pageRaise(page_index); break;
	}
 }


//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdAddSheet::KSCmdAddSheet( KSSheet *sheet, KSSheetList *sheetList )
: KSCommand(QObject::tr("Add sheet"))
 {
  m_sheet = sheet;
  m_sheet_list = sheetList;
  m_delete_sheet = true;
 }

//---------------------------------------------------------------------------------------------//

KSCmdAddSheet::~KSCmdAddSheet()
 {
  if ( m_delete_sheet ) delete m_sheet;
 }

//---------------------------------------------------------------------------------------------//
	
bool KSCmdAddSheet::execute()
 {
  m_sheet_list->sheetAdd( m_sheet );
  m_delete_sheet = false;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdAddSheet::unexecute()
 {
  m_sheet_list->sheetRemove( m_sheet_list->sheetFind(m_sheet) );
  m_delete_sheet = true;
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdRemoveSheet::KSCmdRemoveSheet( KSSheet *sheet, KSSheetList *sheetList  )
: KSCommand(QObject::tr("Remove sheet"))
 {
  m_sheet = sheet;
  m_sheet_list = sheetList;
  m_delete_sheet = false;
  m_cell_range_list.setAutoDelete(FALSE);
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemoveSheet::~KSCmdRemoveSheet()
 {
  if ( m_delete_sheet ) delete m_sheet;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemoveSheet::execute()
 {
  if ( !m_sheet ) {
	setError(QObject::tr("No sheet selected"));
	return false;
	}

  // nullify all pointers
  m_cell_range_list.clear();
  for( int i=0; i<m_sheet->cellRangeCount(); i++ ) {
	KSMatrixWorksheetCellRange *curr_matrix = m_sheet->cellRange(i);
	curr_matrix->setWorksheet( -1 );
	m_cell_range_list.append( curr_matrix );
	}

  int m_pos = m_sheet_list->sheetFind(m_sheet);
  m_sheet_list->sheetRemove( m_pos );
  m_delete_sheet = true;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemoveSheet::unexecute()
 {
  m_sheet_list->sheetInsert( m_pos, m_sheet );

  // restore all pointers
  int sheet_index = m_sheet_list->sheetFind(m_sheet);
  for( unsigned int i=0; i<m_cell_range_list.count(); i++ ) {
	m_cell_range_list.at(i)->setWorksheet(sheet_index);
	}
  m_cell_range_list.clear();

  m_delete_sheet = false;
 }


//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdSheetOrder::KSCmdSheetOrder( CommandType t, KSSheet *sheet, KSSheetList *sheetList )
:KSCommand( QObject::tr("Order sheets") )
 {
  m_type = t;
  m_sheet = sheet;
  m_sheet_list = sheetList;
 }

//---------------------------------------------------------------------------------------------//

KSCmdSheetOrder::~KSCmdSheetOrder()
 {
 }

//---------------------------------------------------------------------------------------------//
	
bool KSCmdSheetOrder::execute()
 {
  if ( !m_sheet ) {
	setError(QObject::tr("No sheet selected"));
	return false;
	}
  m_pos = m_sheet_list->sheetFind(m_sheet);
  switch( m_type ) {
	case ToFront:	m_sheet_list->sheetToFront(m_pos); break;
	case ToBack:    m_sheet_list->sheetToBack(m_pos); break;
	case Raise:     m_sheet_list->sheetRaise(m_pos); break;
	case Lower:     m_sheet_list->sheetLower(m_pos); break;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSheetOrder::unexecute()
 {
  int sheet_index = m_sheet_list->sheetFind(m_sheet);
  switch( m_type ) {
	case ToFront:  m_sheet_list->sheetReorder(m_pos,sheet_index); break;
	case ToBack:   m_sheet_list->sheetReorder(m_pos,sheet_index); break;
	case Raise:    m_sheet_list->sheetLower(sheet_index); break;
	case Lower:    m_sheet_list->sheetRaise(sheet_index); break;
	}
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdSetData::KSCmdSetData( QSData *dataObject, int channel, QSMatrix *newMatrix )
: KSCommand( QObject::tr("Set data" ) )
 {
  m_data_object = dataObject;
  m_channel = channel;
  m_new_matrix = newMatrix;
  m_delete_new_matrix = true;
 }

//---------------------------------------------------------------------------------------------//

KSCmdSetData::~KSCmdSetData()
 {
  if ( m_delete_new_matrix ) delete m_new_matrix;
			else delete m_old_matrix;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdSetData::execute()
 {
  m_old_matrix = m_data_object->takeMatrix( m_channel, m_new_matrix );
  m_delete_new_matrix = false;
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetData::unexecute()
 {
  m_new_matrix = m_data_object->takeMatrix( m_channel, m_old_matrix );
  m_delete_new_matrix = true;
 }



//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdMatrixTranspose::KSCmdMatrixTranspose(QSMatrix *matrix)
 : KSCommand(QObject::tr("Transpose"))
 {
  m_matrix = matrix;
 }

//---------------------------------------------------------------------------------------------//

KSCmdMatrixTranspose::~KSCmdMatrixTranspose()
 {
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdMatrixTranspose::execute()
 {
  m_matrix->dataChanging();
  bool result = m_matrix->transpose();
  m_matrix->dataChanged();
  if ( !result ) {
	setError(QObject::tr("Can't transpose this type of matrix !"));
	return false;
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdMatrixTranspose::unexecute()
 {
  m_matrix->dataChanging();
  m_matrix->transpose();
  m_matrix->dataChanged();	
 }


//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdSetProperties::KSCmdSetProperties( QObject *object )
: KSCommand( "Set properties" )
 {
 m_object = object;
 }

//---------------------------------------------------------------------------------------------//

KSCmdSetProperties::~KSCmdSetProperties()
 {
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetProperties::setProperty( const char *name, const QVariant& value )
 {
  if ( !m_old_properties.contains(name) ) m_old_properties.insert( name, m_object->property(name) );
  m_new_properties.insert( name, value );
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdSetProperties::execute()
 {
  PropertyList::Iterator property;
  for( property = m_new_properties.begin(); property != m_new_properties.end(); ++property ) {
	m_object->setProperty( property.key(), property.data() );	
	}
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetProperties::unexecute()
 {
  PropertyList::Iterator property;
  for( property = m_old_properties.begin(); property != m_old_properties.end(); ++property ) {
	m_object->setProperty( property.key(), property.data() );	
	}
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdSetGraphicalProperties::KSCmdSetGraphicalProperties( QSGraphicalData *object )
:KSCmdSetProperties( object )
 {
  m_object = object;
 }

//---------------------------------------------------------------------------------------------//

KSCmdSetGraphicalProperties::~KSCmdSetGraphicalProperties()
 {
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetGraphicalProperties::setFont( int element, const QSGFont& font )
 {
  m_old_fonts.insert( element, m_object->font(element) );
  m_new_fonts.insert( element, font );
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetGraphicalProperties::setFill( int element, const QSGFill& fill )
 {
  m_old_fills.insert( element, m_object->fill(element) );
  m_new_fills.insert( element, fill );
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetGraphicalProperties::setLine( int element, const QSGLine& line )
 {
  m_old_lines.insert( element, m_object->line(element) );
  m_new_lines.insert( element, line );
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetGraphicalProperties::setPoint( int element, const QSGPoint& point )
 {
  m_old_points.insert( element, m_object->point(element) );
  m_new_points.insert( element, point );
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdSetGraphicalProperties::execute()
 {
  if ( KSCmdSetProperties::execute() ) {
  	QMap<int,QSGFont>::Iterator font;
  	for( font = m_new_fonts.begin(); font != m_new_fonts.end(); ++font ) m_object->setFont( font.key(), font.data() );
  	QMap<int,QSGFill>::Iterator fill;
  	for( fill = m_new_fills.begin(); fill != m_new_fills.end(); ++fill ) m_object->setFill( fill.key(), fill.data() );
  	QMap<int,QSGLine>::Iterator line;
  	for( line = m_new_lines.begin(); line != m_new_lines.end(); ++line ) m_object->setLine( line.key(), line.data() );
  	QMap<int,QSGPoint>::Iterator point;
  	for( point = m_new_points.begin(); point != m_new_points.end(); ++point ) m_object->setPoint( point.key(), point.data() );
  	return true;
	}
  return false;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetGraphicalProperties::unexecute()
 {
  KSCmdSetProperties::unexecute();
  	QMap<int,QSGFont>::Iterator font;
  	for( font = m_old_fonts.begin(); font != m_old_fonts.end(); ++font ) m_object->setFont( font.key(), font.data() );
  	QMap<int,QSGFill>::Iterator fill;
  	for( fill = m_old_fills.begin(); fill != m_old_fills.end(); ++fill ) m_object->setFill( fill.key(), fill.data() );
  	QMap<int,QSGLine>::Iterator line;
  	for( line = m_old_lines.begin(); line != m_old_lines.end(); ++line ) m_object->setLine( line.key(), line.data() );
  	QMap<int,QSGPoint>::Iterator point;
  	for( point = m_old_points.begin(); point != m_old_points.end(); ++point ) m_object->setPoint( point.key(), point.data() );
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//


//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdSetRanges::KSCmdSetRanges( QSAxes *axes )
: KSCommand(QObject::tr("Set axis range"))
 {
  m_axes = axes;
  m_new_ranges.setAutoDelete( TRUE );
  m_old_ranges.setAutoDelete( TRUE );
  save_data( &m_old_ranges );
 }

//---------------------------------------------------------------------------------------------//

KSCmdSetRanges::~KSCmdSetRanges()
 {
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetRanges::commit()
 {
  save_data( &m_new_ranges );
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetRanges::save_data( viewList *list )
 {
  for ( int axis_nr=0; axis_nr<m_axes->axisCount(); axis_nr++ ) {
	axis_remembered_view_t *view = new axis_remembered_view_t();
	QSAxis *axis = m_axes->axis(axis_nr);
	view->min = axis->min();
	view->max = axis->max();
	view->base = axis->scaleBase();
	view->scale = axis->scaleType();
	view->round = axis->roundRangeToTicStep();
	view->reversed = axis->reversed();
	list->insert(axis,view);
	}
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdSetRanges::execute()
 {
  set_data( &m_new_ranges );
  return true;
 }

 //---------------------------------------------------------------------------------------------//

void KSCmdSetRanges::unexecute()
 {
  set_data( &m_old_ranges );
 }

//---------------------------------------------------------------------------------------------//

void KSCmdSetRanges::set_data( viewList *list )
 {
  for ( int axis_nr=0; axis_nr<m_axes->axisCount(); axis_nr++ ) {
	QSAxis *axis = m_axes->axis(axis_nr);	
	if ( list->find(axis) ) {
		axis_remembered_view_t *view = list->find(axis);
		axis->setRange( view->min, view->max );
		axis->setScaleType( view->scale );
		axis->setScaleBase( view->base );
		axis->setReversed( view->reversed );
		axis->setRoundRangeToTicStep( view->round );
		}
	}
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

// create command ( current data is remembered ) and modify directly data - first execute does nothing,
// because data is already modified
KSCmdChangeData::KSCmdChangeData( QSMatrix *matrix, int rowFrom, int colFrom, int rowTo, int colTo )
: KSCommand(QObject::tr("Change data"))
 {
  m_matrix = matrix;
  m_row_from = rowFrom;
  m_col_from = colFrom;
  m_row_to = rowTo;
  m_col_to = colTo;
  m_old_data = copy_modified_range();
  m_new_data = NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdChangeData::~KSCmdChangeData()
 {
  delete m_old_data;
  delete m_new_data;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdChangeData::execute()
 {
  m_matrix->dataChanging();
  if  ( m_new_data ) {
	m_old_data = copy_modified_range();
	apply_buffer( m_new_data );
	delete m_new_data; m_new_data = NULL;
	}
  m_matrix->dataChanged();
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdChangeData::unexecute()
 {
  m_matrix->dataChanging();
  if ( m_old_data ) {
	m_new_data = copy_modified_range();
	apply_buffer( m_old_data );
	delete m_old_data; m_old_data = NULL;
	}
  m_matrix->dataChanged();
 }

//---------------------------------------------------------------------------------------------//

QSMatrix *KSCmdChangeData::copy_modified_range()
 {
  QSMatrix *result = KSObjectFactory::cloneMatrix( m_matrix, false );
  result->resize( m_row_to-m_row_from+1, m_col_to-m_col_from+1 );
  result->copyRange( 0, 0, m_matrix, m_row_from, m_col_from, m_row_to, m_col_to );
  return result;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdChangeData::apply_buffer( QSMatrix *buffer )
 {
  m_matrix->copyRange( m_row_from, m_col_from, buffer, 0, 0, buffer->rows()-1, buffer->cols()-1 );
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdChangeValue::KSCmdChangeValue( QSMatrix *matrix, int row, int col, const QString& new_value )
: KSCommand(QObject::tr("Change value"))
 {
  m_matrix = matrix;
  m_row = row;
  m_col = col;
  m_new_value = new_value;
  m_old_value = m_matrix->string(m_row,m_col);
 }

//---------------------------------------------------------------------------------------------//

KSCmdChangeValue::~KSCmdChangeValue()
 {
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdChangeValue::execute()
 {
  m_matrix->dataChanging();
  m_matrix->setString(m_row,m_col,m_new_value);
  m_matrix->dataChanged();
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdChangeValue::unexecute()
 {
  m_matrix->dataChanging();
  m_matrix->setString(m_row,m_col,m_old_value);
  m_matrix->dataChanged();
 }



//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdRemoveRows::KSCmdRemoveRows( QSMatrix *matrix, int fromRow, int toRow )
: KSCommand( QObject::tr("Remove rows") )
 {
  m_matrix = matrix;
  m_from_row = fromRow;
  m_to_row = toRow;
  m_old_data = NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemoveRows::~KSCmdRemoveRows()
 {
  delete m_old_data;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemoveRows::execute()
 {
  m_matrix->dataChanging();
  // rememberd removed data
  m_old_data = KSObjectFactory::cloneMatrix( m_matrix );
  m_old_data->resize( m_to_row-m_from_row+1, m_matrix->cols() );
  m_old_data->copyRange( 0, 0, m_matrix, m_from_row, 0, m_to_row, m_matrix->cols()-1 );
  //shift rows up
  m_matrix->copyRange( m_from_row, 0, m_matrix, m_to_row+1, 0, m_matrix->rows()-1, m_matrix->cols()-1 );
  // resize matrix
  m_matrix->resize( m_matrix->rows()-m_to_row+m_from_row-1, m_matrix->cols() );
  m_matrix->dataChanged();
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemoveRows::unexecute()
 {
  m_matrix->dataChanging();
  // resize matrix
  m_matrix->resize( m_matrix->rows() +m_to_row-m_from_row+1, m_matrix->cols() );
  // shift rows down
  m_matrix->copyRange( m_to_row+1, 0, m_matrix, m_from_row, 0, m_matrix->rows()-1, m_matrix->cols()-1 );
  // restore previous data
  m_matrix->copyRange( m_from_row, 0, m_old_data, 0, 0, m_old_data->rows()-1, m_old_data->cols()-1 );
  delete m_old_data; m_old_data = NULL;
  m_matrix->dataChanged();
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdRemoveCols::KSCmdRemoveCols( QSMatrix *matrix, int fromCol, int toCol )
: KSCommand( QObject::tr("Remove columns") )
 {
  m_matrix = matrix;
  m_from_col = fromCol;
  m_to_col = toCol;
  m_old_data = NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdRemoveCols::~KSCmdRemoveCols()
 {
  delete m_old_data;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdRemoveCols::execute()
 {
  m_matrix->dataChanging();
  // rememberd removed data
  m_old_data = KSObjectFactory::cloneMatrix( m_matrix );
  m_old_data->resize( m_matrix->rows(), m_to_col-m_from_col+1 );
  m_old_data->copyRange( 0, 0, m_matrix, 0, m_from_col, m_matrix->rows()-1, m_to_col );
  //shift cols left
  m_matrix->copyRange( 0, m_from_col, m_matrix, 0, m_to_col+1, m_matrix->rows()-1, m_matrix->cols()-1 );
  // resize matrix
  m_matrix->resize( m_matrix->rows(), m_matrix->cols()-m_to_col+m_from_col-1 );
  m_matrix->dataChanged();
  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdRemoveCols::unexecute()
 {
  m_matrix->dataChanging();
  // resize matrix
  m_matrix->resize( m_matrix->rows(), m_matrix->cols() +m_to_col-m_from_col+1 );
  // shift cols right
  m_matrix->copyRange( 0, m_to_col+1, m_matrix, 0, m_from_col, m_matrix->rows()-1, m_matrix->cols()-1 );
  // restore previous data
  m_matrix->copyRange( 0, m_from_col, m_old_data, 0, 0, m_old_data->rows()-1, m_old_data->cols()-1 );
  delete m_old_data; m_old_data = NULL;
  m_matrix->dataChanged();
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//

KSCmdInsertRow::KSCmdInsertRow( QSMatrix *matrix, int atPos )
: KSCommand(QObject::tr("Insert row"))
 {
  m_matrix = matrix;
  m_at_pos = atPos;
  m_old_data = NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdInsertRow::~KSCmdInsertRow()
 {
  delete m_old_data;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdInsertRow::execute()
 {
  m_matrix->dataChanging();
  // rememberd the last row in the case 'resize' method is not implemented
  m_old_data = KSObjectFactory::cloneMatrix( m_matrix );
  m_old_data->resize( 1, m_matrix->cols() );
  m_old_data->copyRange( 0, 0, m_matrix, m_matrix->rows()-1, 0, m_matrix->rows()-1, m_matrix->cols() );
  // resize matrix
  m_matrix->resize( m_matrix->rows()+1, m_matrix->cols() );
  // shift rows down
  m_matrix->copyRange( m_at_pos+1, 0, m_matrix, m_at_pos, 0, m_matrix->rows()-1, m_matrix->cols()-1 );
  m_matrix->dataChanged();
  // clear new row
  if ( m_at_pos < m_matrix->rows() )
  for( int i=0; i<m_matrix->cols(); i++ ) m_matrix->setString( m_at_pos, i, QString::null );

  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdInsertRow::unexecute()
 {
  m_matrix->dataChanging();
  // shift rows up
  m_matrix->copyRange( m_at_pos, 0, m_matrix, m_at_pos+1, 0, m_matrix->rows()-1, m_matrix->cols()-1 );
  // resize matrix
  m_matrix->resize( m_matrix->rows()-1, m_matrix->cols() );
  // restore row in the case 'resize' is not implemented
  m_matrix->copyRange( m_matrix->rows()-1, 0, m_old_data, 0, 0, m_old_data->rows()-1, m_old_data->cols()-1 );
  delete m_old_data; m_old_data = NULL;
  m_matrix->dataChanged();
 }

//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//
//---------------------------------------------------------------------------------------------//


KSCmdInsertCol::KSCmdInsertCol( QSMatrix *matrix, int atPos )
: KSCommand(QObject::tr("Insert col"))
 {
  m_matrix = matrix;
  m_at_pos = atPos;
  m_old_data = NULL;
 }

//---------------------------------------------------------------------------------------------//

KSCmdInsertCol::~KSCmdInsertCol()
 {
  delete m_old_data;
 }

//---------------------------------------------------------------------------------------------//

bool KSCmdInsertCol::execute()
 {
  m_matrix->dataChanging();
  // rememberd the last col for the case 'resize' method is not implemented
  m_old_data = KSObjectFactory::cloneMatrix( m_matrix );
  m_old_data->resize( m_matrix->rows(), 1 );
  m_old_data->copyRange( 0, 0, m_matrix, 0, m_matrix->cols()-1, m_matrix->rows()-1, m_matrix->cols()-1 );
  // resize matrix
  m_matrix->resize( m_matrix->rows(), m_matrix->cols()+1 );
  // shift cols right
  m_matrix->copyRange( 0, m_at_pos+1, m_matrix, 0, m_at_pos, m_matrix->rows()-1, m_matrix->cols()-1 );
  m_matrix->dataChanged();
  // clear new col
  if ( m_at_pos < m_matrix->cols() )
  for( int i=0; i<m_matrix->rows(); i++ ) m_matrix->setString( i, m_at_pos, QString::null );

  return true;
 }

//---------------------------------------------------------------------------------------------//

void KSCmdInsertCol::unexecute()
 {
  m_matrix->dataChanging();
  // shift cols left
  m_matrix->copyRange( 0, m_at_pos, m_matrix, 0, m_at_pos+1, m_matrix->rows()-1, m_matrix->cols()-1 );
  // resize matrix
  m_matrix->resize( m_matrix->rows(), m_matrix->cols()-1 );
  // restore row in the case 'resize' is not implemented
  m_matrix->copyRange( 0, m_matrix->cols()-1, m_old_data, 0, 0, m_old_data->rows()-1, m_old_data->cols()-1 );
  delete m_old_data; m_old_data = NULL;
  m_matrix->dataChanged();
 }

//--------------------------------------------------------------------//
//--------------------------------------------------------------------//
//--------------------------------------------------------------------//
//--------------------------------------------------------------------//

struct KSCmdRemoveSheetCols::private_data {
	KSSheet::ColumnData *columns;
	};

KSCmdRemoveSheetCols::KSCmdRemoveSheetCols( KSSheet *sheet, int fromCol, int toCol )
: KSCmdRemoveCols( sheet->matrix(0), fromCol, toCol )
 {
  m_sheet = sheet;
  m_data = new private_data();
  m_data->columns = new KSSheet::ColumnData[toCol-fromCol+1];
 }

//--------------------------------------------------------------------//

KSCmdRemoveSheetCols::~KSCmdRemoveSheetCols()
 {
  delete[] m_data->columns;
  delete m_data;
 }

//--------------------------------------------------------------------//

bool KSCmdRemoveSheetCols::execute()
 {
  for( int i=m_from_col; i<=m_to_col; i++ ) m_data->columns[i-m_from_col] = m_sheet->columnData(i);

  int removed_cols = m_to_col-m_from_col+1;

  // shift column headers to the left
  for( int col=m_from_col; col<m_matrix->cols()-removed_cols; col++ ) {
	KSSheet::ColumnData cdata = m_sheet->columnData(col+removed_cols);
	cdata.column = col;
	m_sheet->setColumnData( cdata );
	}

  /*
  -- not finished
  // shift to left all column pointers
  for( int i=0; i<m_sheet->cellRangeCount(); i++ ) {
	KSWorksheetCellRange *m = m_sheet->cellRange(i);
	if ( m->colFrom() == m->colTo() && m->colFrom() > m_at_pos ) {
		m->setColFrom( m->colFrom()-1 );
		m->setColTo( m->colTo()-1 );
		}
	}
  */
  return KSCmdRemoveCols::execute();
 }

//--------------------------------------------------------------------//

void KSCmdRemoveSheetCols::unexecute()
 {
  KSCmdRemoveCols::unexecute();
  m_matrix->dataChanging();
  int removed_cols = m_to_col-m_from_col+1;
  // shift columns to the right
  for( int col=m_matrix->cols()-1; col>=m_to_col; col-- ) {
	KSSheet::ColumnData cdata = m_sheet->columnData(col-removed_cols);
	cdata.column = col;
	m_sheet->setColumnData( cdata );
	}

  for( int i=m_from_col; i<=m_to_col; i++ ) m_sheet->setColumnData(m_data->columns[i-m_from_col]);
  m_matrix->dataChanged();
 }

//--------------------------------------------------------------------//
//--------------------------------------------------------------------//
//--------------------------------------------------------------------//
//--------------------------------------------------------------------//
//--------------------------------------------------------------------//

struct KSCmdInsertSheetCol::private_data {
	KSSheet::ColumnData column;
	};

KSCmdInsertSheetCol::KSCmdInsertSheetCol( KSSheet *sheet, int atPos )
: KSCmdInsertCol( sheet->matrix(0), atPos )
 {
  m_sheet = sheet;
  m_data = new private_data();
 }

//--------------------------------------------------------------------//

KSCmdInsertSheetCol::~KSCmdInsertSheetCol()
 {
  delete m_data;
 }

//--------------------------------------------------------------------//

bool KSCmdInsertSheetCol::execute()
 {
  m_data->column = m_sheet->columnData(m_matrix->cols()-1);

  // shift all comlumn headers to the right
  for( int col=m_matrix->cols()-1; col>=m_at_pos+1; col-- ) {
	KSSheet::ColumnData cdata = m_sheet->columnData(col-1);
	cdata.column = col;
	m_sheet->setColumnData( cdata );
	}
  m_sheet->removeColumnData( m_at_pos );
  /*
  // shift to right all column pointers
  for( int i=0; i<m_sheet->cellRangeCount(); i++ ) {
	KSWorksheetCellRange *m = m_sheet->cellRange(i);
	if ( m->colFrom() == m->colTo() && m->colFrom() >= m_at_pos ) {
		m->setColFrom( m->colFrom()+1 );
		m->setColTo( m->colTo()+1 );
		}
	}
  */
  return KSCmdInsertCol::execute();
 }

//--------------------------------------------------------------------//

void KSCmdInsertSheetCol::unexecute()
 {
  KSCmdInsertCol::unexecute();
  m_matrix->dataChanging();
  // shift all column headers to the left
  for( int col=m_at_pos; col<m_matrix->cols()-1; col++ ) {
	KSSheet::ColumnData cdata = m_sheet->columnData(col+1);
	cdata.column = col;
	m_sheet->setColumnData( cdata );
	}
  /*
  // shift to left all column pointers
  for( int i=0; i<m_sheet->cellRangeCount(); i++ ) {
	KSWorksheetCellRange *m = m_sheet->cellRange(i);
	if ( m->colFrom() == m->colTo() && m->colFrom() > m_at_pos ) {
		m->setColFrom( m->colFrom()-1 );
		m->setColTo( m->colTo()-1 );
		}
	}
  */
  m_sheet->setColumnData( m_data->column );	
  m_matrix->dataChanged();
 }

//--------------------------------------------------------------------//
