/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Set of functions to deal with internationalization (translation).
 */
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "interface.h"
#include "main.h"
#include "callbacks.h"
#include "keyboard.h"
#include "velocity.h"
#include "fluidness.h"
#include "translation.h"

/**********************************************************************
 * Variables
 */
extern GtkWidget *window_main_;
extern GtkWidget *window_top10_;

Lang_Name_Code *lang;
gint lang_num = 0;

/**********************************************************************
 * Set the code of the current language defined by environmental 
 * variables and apply it for the GTK interface
 */
void
trans_set_language_env (gchar * code)
{
	if (g_setenv ("LANGUAGE", code, 1) == FALSE)
		if (g_setenv ("LANG", code, 1) == FALSE)
			g_warning ("no language environment variable could be set.");
	main_preferences_set_string ("interface", "language", code);
	gtk_set_locale ();
}

/**********************************************************************
 * Initialize 'lang' and 'code' accordingly to 'language_set',
 * which was defined in "languages.h".
 */
void
trans_init_lang_name_code ()
{
	gint i;
	gchar delimiter[] = "\n";
	gchar tmp_str[LANG_NAME_MAX_LEN];
	gchar **temp_lang = NULL;
	const gchar *languages_set = LANG_SET;

	/* 
	 * Get the number of configured languages
	 */
	temp_lang = g_strsplit (languages_set, delimiter, -1);
	i = 0;
	while (temp_lang[i] != NULL)
		i++;
	g_assert (i > 0);
	lang = g_new (Lang_Name_Code, i);
	lang_num = i;

	delimiter[1] = '(';
	for (i = 0; i < lang_num; i++)
	{
		/* Initialize 'lang'
		 */
		strncpy (tmp_str, temp_lang[i], LANG_NAME_MAX_LEN - 1);
		tmp_str[LANG_NAME_MAX_LEN] = '\0';
		g_strdelimit (tmp_str, delimiter, '\0');
		strcpy (lang[i].name, g_strstrip (tmp_str));

		/* Initialize 'code'
		 */
		strcpy (tmp_str, strchr (temp_lang[i], '('));
		*(strrchr (tmp_str, ')')) = '\0';
		g_assert (strlen (tmp_str + 1) < 6);
		strcpy (lang[i].code, tmp_str + 1);

		/* Initialize 'cd'
		 */
		if (lang[i].code[0] == 'C')
		{
			lang[i].cd[0] = 'e';
			lang[i].cd[1] = 'n';
		}
		else
		{
			lang[i].cd[0] = lang[i].code[0];
			lang[i].cd[1] = lang[i].code[1];
		}
		lang[i].cd[2] = '\0';
	}
	g_strfreev (temp_lang);
}

gchar *
trans_get_code (gint i)
{
	if (i >= lang_num || i < 0)
		return (NULL);
	return (lang[i].cd);
}

gboolean
trans_lang_is_available (gchar * test)
{
	gint i;

	for (i = 0; i < lang_num; i++)
		if (g_str_equal (lang[i].code, test))
			break;
	return (i == lang_num ? FALSE : TRUE);
}

/**********************************************************************
 * Define if we may put a stop mark at the end of "phrases".
 */
gboolean
trans_lang_has_stopmark ()
{
	gboolean stopmark;
	gchar *hlp;

	hlp = main_preferences_get_string ("interface", "language");
	stopmark = g_str_has_prefix (hlp, "ur") ||
		   g_str_has_prefix (hlp, "bn");
	g_free (hlp);

	return (!stopmark);
}

/**********************************************************************
 * Private auxiliar function
 */
static gboolean
trans_lang_get_similar (gchar * test)
{
	gint i;
	gchar aux_code_2[3];

	if (g_str_equal (test, "C"))
		return TRUE;

	strncpy (aux_code_2, test, 2);
	aux_code_2[2] = '\0';

	for (i = 0; i < lang_num; i++)
	{
		if (strstr (lang[i].code, aux_code_2))
		{
			g_free (test);
			test = g_strdup (lang[i].code);
			break;
		}
	}
	if (i == lang_num && g_str_has_prefix (test, "en"))
	{
		g_free (test);
		test = g_strdup ("C");
		return (TRUE);
	}
	return (i == lang_num ? FALSE : TRUE);
}

/**********************************************************************
 * Get the current locale and change it if necessary
 */
void
trans_init_language_env ()
{
	gchar *tmp_code = NULL;
	gboolean lang_ok = TRUE;
	gint i;

	trans_init_lang_name_code ();

	/*
	 * If the language is already set in preferences, just use it
	 */
	if (main_preferences_exist ("interface", "language"))
		tmp_code = main_preferences_get_string ("interface", "language");
	else
	{
		/* 
		 * Read the current locale
		 */
#ifdef G_OS_UNIX
		i = 0;
		lang_ok = FALSE;
		while ((tmp_code = g_strdup (g_get_language_names ()[i])))
		{
			if (tmp_code[0] == 'C')
			{
				lang_ok = (i == 0 ? TRUE : FALSE);
				break;
			}
			lang_ok = trans_lang_is_available (tmp_code);
			if (lang_ok == TRUE)
				break;
			g_free (tmp_code);
			lang_ok = FALSE;
			i++;
		}
		if (lang_ok == FALSE)
		{
			i = 0;
			while ((tmp_code = g_strdup (g_get_language_names ()[i])))
			{
				if (tmp_code[0] == 'C')
				{
					lang_ok = (i == 0 ? TRUE : FALSE);
					break;
				}
				lang_ok = trans_lang_get_similar (tmp_code);
				if (lang_ok == TRUE)
					break;
				g_free (tmp_code);
				lang_ok = FALSE;
				i++;
			}
		}
#else
		tmp_code = g_win32_getlocale ();
		lang_ok = trans_lang_is_available (tmp_code);
		if (lang_ok == FALSE)
			lang_ok = trans_lang_get_similar (tmp_code);
#endif
	}
	if (tmp_code == NULL)
		tmp_code = g_strdup ("xx");
	/*
	 * If even a similar is not available, then use the neutral international one
	 */
	if (lang_ok == FALSE)
	{
		g_free (tmp_code);
		tmp_code = g_strdup ("eo");
		g_message ("as your prefered locales aren't available,");
		g_message ("we are using this neutral one: \"%s\"", tmp_code);
		g_message ("if you don't agree, try to change it at the main menu... ;-)");
	}

	trans_set_language_env (tmp_code);
	g_free (tmp_code);
}

/**********************************************************************
 * Inserts the list of available languages in the 'combo_language'.
 */
void
trans_set_combo_language ()
{
	gint i;
	gint i_env;
	gchar *tmp_code;
	GList *languages = NULL;
	GtkWidget *wg;

	tmp_code = main_preferences_get_string ("interface", "language");
	if (strcmp (tmp_code, "eo_XX") == 0)
	{
		main_preferences_set_string ("interface", "language", "eo");
		g_free (tmp_code);
		tmp_code = g_strdup ("eo");
	}

	for (i = 0, i_env = -1; i < lang_num; i++)
	{
		if (strcmp (lang[i].code, tmp_code) != 0)
			languages = g_list_insert (languages, lang[i].name, -1);
		else
			i_env = i;
	}

	if (i_env == -1)
		g_error ("set_combo_language() ==> the locale \"%s\" is not available!", tmp_code);
	else
		languages = g_list_insert (languages, lang[i_env].name, 0);

	wg = lookup_widget (window_main_, "combo_language");
	gtk_combo_set_popdown_strings (GTK_COMBO (wg), languages);

	g_free (tmp_code);
	g_list_free (languages);
}

/**********************************************************************
 * Get the current language defined by the preference's key and set in
 * the 'combo_language'.
 */
gchar *
trans_get_current_language ()
{
	gchar *tmp_code;
	gint i;

	tmp_code = main_preferences_get_string ("interface", "language");
	for (i = 0; i < lang_num; i++)
		if (strcmp (lang[i].code, tmp_code) == 0)
		{
			g_free (tmp_code);
			return (lang[i].name);
		}
	g_free (tmp_code);
	return ("??");
}

/**********************************************************************
 * Update the current language used accordingly to that selected in the
 * 'combo_language'
 */
void
trans_change_language ()
{
	gint i;
	gchar *selection = NULL;
	GtkWidget *wg;

	wg = lookup_widget (window_main_, "entry_language");
	selection = gtk_editable_get_chars (GTK_EDITABLE (wg), 0, -1);

	for (i = 0; i < lang_num; i++)
		if (strcmp (lang[i].name, selection) == 0)
			break;

	if (i == lang_num)
	{
		g_warning ("change_language() --> couldn't find the language: %s", selection);
		g_free (selection);
		return;
	}
	g_free (selection);

	trans_set_language_env (lang[i].code);
	g_message ("selected language code --> %s", lang[i].code);

	/* Reset everything up */
	callbacks_shield_set (TRUE);
	gtk_widget_destroy (window_main_);
	window_main_ = create_window_main ();
	wg = lookup_widget (window_main_, "entry_keyboard");
	gtk_entry_set_text (GTK_ENTRY (wg), keyb_get_name ());
	trans_set_combo_language ();
	main_set_auto_state ();
	gtk_widget_show (window_main_);
	callbacks_shield_set (FALSE);

	velo_reset_dict ();
	fluid_reset_paragraph ();

	if (window_top10_ != NULL)
	{
		gtk_widget_destroy (window_top10_);
		window_top10_ = NULL;
	}
}

/**********************************************************************
 * Find a file whose language prefix is similar to the current one
 */
FILE *
trans_lang_get_similar_file (const gchar * file_end)
{
	gint i;
	gchar *tmp_code;
	gchar *tmp_path = NULL;
	FILE *fh = NULL;

	tmp_code = main_preferences_get_string ("interface", "language");
	for (i = 0; i < lang_num && fh == NULL; i++)
	{
		if (strcmp (lang[i].code, tmp_code) == 0)
			continue;
		if (lang[i].code[0] == tmp_code[0] && lang[i].code[1] == tmp_code[1])
		{
			g_free (tmp_path);
			tmp_path =
				g_strconcat (main_get_data_path (), lang[i].code, file_end, NULL);
			fh = (FILE *) g_fopen (tmp_path, "r");
		}
	}
	g_free (tmp_code);
	g_free (tmp_path);
	i--;
	if (fh)
		g_message ("applying similar file: %s%s", lang[i].code, file_end);
	return (fh);
}

/**********************************************************************
 * Find a file whose language prefix is similar to the current one, returnig
 * its name
 */
gchar *
trans_lang_get_similar_file_name (const gchar * file_end)
{
	gint i;
	gchar *tmp_code;
	gchar *tmp_path = NULL;

	tmp_code = main_preferences_get_string ("interface", "language");
	for (i = 0; i < lang_num; i++)
	{
		if (strcmp (lang[i].code, tmp_code) == 0)
			continue;
		if (lang[i].code[0] == tmp_code[0] && lang[i].code[1] == tmp_code[1])
		{
			g_free (tmp_path);
			tmp_path =
				g_strconcat (main_get_data_path (), lang[i].code, file_end, NULL);
			if (g_file_test (tmp_path, G_FILE_TEST_IS_REGULAR))
				break;
		}
	}
	if (tmp_path == NULL)
		tmp_path = g_strconcat (main_get_data_path (), "C", file_end, NULL);
	g_free (tmp_code);
	return (tmp_path);
}

/**********************************************************************
 * Reads the text of a file to be presented at some dialog,
 * accordingly to the environment language.
 * 'text': buffer where the text is copied into.
 * 'max_text_len': the size of the buffer.
 * 'file_end': how the file name ends
 */
gchar *
trans_read_text (const gchar * file_end)
{
	gchar *tmp_name = NULL;
	gchar *tmp_path = NULL;
	gchar *tmp_code;
	gint i, j, tmpc;
	FILE *fh;

	tmp_code = main_preferences_get_string ("interface", "language");
	tmp_name = g_strconcat (tmp_code, file_end, NULL);

	/*
	 * Try at HOME
	 */
	tmp_path = g_strconcat (main_get_user_dir (), tmp_name, NULL);
	fh = (FILE *) g_fopen (tmp_path, "r");

	/*
	 * Try at PACKAGE_DATA
	 */
	if (fh == NULL)
	{
		g_free (tmp_path);
		tmp_path = g_strconcat (main_get_data_path (), tmp_name, NULL);
		fh = (FILE *) g_fopen (tmp_path, "r");
	}

	/*
	 * Try other "flavors" of the same language
	 */
	if (fh == NULL && strlen (tmp_code) > 1)
		fh = trans_lang_get_similar_file (file_end);

	/*
	 * Default to C
	 */
	if (fh == NULL && strcmp (tmp_code, "C") != 0)
	{
		g_message
			("trans_read_text() --> couldn't open the data file: %s\n"
			 " So, we have to apply the default one: C%s", tmp_name, file_end);
		main_preferences_set_string ("interface", "language", "C");
		gchar *text = trans_read_text (file_end);
		main_preferences_set_string ("interface", "language", tmp_code);
		g_free (tmp_code);
		g_free (tmp_path);
		g_free (tmp_name);
		return text;
	}

	if (fh == NULL)
		g_error ("trans_read_text() --> couldn't open the data file:\n %s", tmp_name);

	g_free (tmp_code);
	g_free (tmp_path);
	g_free (tmp_name);

	/*
	 * Process the file
	 */
	gsize bufsize = 16;
	gchar *buf = g_malloc (bufsize);
	gsize pos = 0;
	while (1)
	{
		gsize max = bufsize - pos - 1; // -1 for terminating zero
		gsize ct = fread (buf + pos, 1, max, fh);
		if (ct == 0)
			break;
		pos += ct;
		if (ct == max)
		{
			bufsize *= 2;
			buf = g_realloc (buf, bufsize);
		}
	}
	buf[pos] = 0;
	return buf;
}
