/* $Id: mainwindow.c,v 1.130 2007/11/18 20:10:39 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <libintl.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gmodule.h>
#ifndef __MINGW32__
#  include <netinet/in.h>
#  include <netdb.h>
#else
#  include <winsock2.h>
#endif
#ifdef HAVE_LIBGNUTLS
#  include <gnutls/gnutls.h>
#  include <gnutls/x509.h>
#  include <gnutls/openpgp.h>
#endif

#include "kildclient.h"
#include "perlscript.h"

#include "ansi.h"


/******************
 * Initialization *
 ******************/
World     *currentWorld         = NULL;
GtkWidget *wndMain              = NULL;
GList     *open_worlds          = NULL;
gboolean   window_has_focus     = TRUE;
gint       worlds_with_new_text = 0;
gboolean   debug_matches        = 0;


/*************************
 * File global variables *
 *************************/
static GtkWidget *ntbkWorlds;


/***********************
 * Function prototypes *
 ***********************/
static gboolean update_world_with_focus(gpointer data);
static gint     attach_world_gui(WorldGUI *gui);
static void     detach_world_gui(WorldGUI *gui);
static gchar   *format_time_string(World *world, int ctime, int itime);
static void     format_hms_time(GString *str, int seconds);
/* Glade callbacks */
G_MODULE_EXPORT gboolean exit_cb(GtkWidget *widget,
                                 GdkEvent *event,
                                 gpointer data);
G_MODULE_EXPORT gboolean window_focus_in_cb(GtkWidget     *widget,
                                            GdkEventFocus *evt,
                                            gpointer       data);
G_MODULE_EXPORT gboolean window_focus_out_cb(GtkWidget     *widget,
                                             GdkEventFocus *evt,
                                             gpointer       data);
G_MODULE_EXPORT void     notebook_page_changed_cb(GtkNotebook     *notebook,
                                                  GtkNotebookPage *page,
                                                  guint            page_num,
                                                  gpointer         data);
G_MODULE_EXPORT void     notebook_page_reordered_cb(GtkNotebook *notebook,
                                                    GtkWidget   *page,
                                                    guint        page_num,
                                                    gpointer     data);
/* Menus */
G_MODULE_EXPORT void     menu_world_open_cb(GtkWidget *widget,
                                            gpointer data);
G_MODULE_EXPORT void     open_new_world_cb(GtkWidget *widget,
                                           gpointer data);
G_MODULE_EXPORT void     edit_this_world_cb(GtkWidget *widget,
                                            gpointer data);
G_MODULE_EXPORT void     menu_statistics_cb(GtkWidget *widget,
                                            gpointer data);

G_MODULE_EXPORT void     menu_edit_open_cb(GtkWidget *widget,
                                           gpointer   data);
G_MODULE_EXPORT void     menu_cut_activate_cb(GtkWidget *widget,
                                              gpointer   data);
G_MODULE_EXPORT void     menu_copy_activate_cb(GtkWidget *widget,
                                               gpointer   data);
G_MODULE_EXPORT void     menu_paste_activate_cb(GtkWidget *widget,
                                                gpointer   data);
G_MODULE_EXPORT void     menu_delete_activate_cb(GtkWidget *widget,
                                                 gpointer   data);
G_MODULE_EXPORT void     menu_find_activate_cb(GtkWidget *widget,
                                               gpointer   data);

G_MODULE_EXPORT void     menu_input_clear_cb(GtkWidget *widget,
                                             gpointer data);
G_MODULE_EXPORT void     menu_input_prev_cb(GtkWidget *widget,
                                            gpointer data);
G_MODULE_EXPORT void     menu_input_next_cb(GtkWidget *widget,
                                            gpointer data);
G_MODULE_EXPORT void     menu_input_find_prev_cb(GtkWidget *widget,
                                                 gpointer data);
G_MODULE_EXPORT void     menu_input_find_next_cb(GtkWidget *widget,
                                                 gpointer data);


G_MODULE_EXPORT void     menu_preferences_open_cb(GtkWidget *widget,
                                                  gpointer data);
G_MODULE_EXPORT void     menu_disable_triggers_cb(GtkWidget *widget,
                                                  gpointer data);
G_MODULE_EXPORT void     menu_disable_aliases_cb(GtkWidget *widget,
                                                 gpointer data);
G_MODULE_EXPORT void     menu_disable_macros_cb(GtkWidget *widget,
                                                gpointer data);
G_MODULE_EXPORT void     menu_disable_timers_cb(GtkWidget *widget,
                                                gpointer data);
G_MODULE_EXPORT void     mnuDebugMatches_activate_cb(GtkWidget *widget,
                                                     gpointer data);
G_MODULE_EXPORT void     menu_edit_default_world_cb(GtkWidget *widget,
                                                    gpointer data);

G_MODULE_EXPORT void     menu_manual_cb(GtkWidget *widget,
                                        gpointer data);
G_MODULE_EXPORT void     about_menu_cb(GtkWidget *widget,
                                       gpointer data);



GtkWidget*
create_main_window(WorldGUI *gui)
{
  GtkWidget *window;
  GladeXML  *gladexml;
  GtkWidget *mnuPreferences;
  GtkWidget *mnuReconnect;
  GtkWidget *mnuDisconnect;
  GtkWidget *mnuConnectAnother;
  GtkWidget *mnuSave;
  GtkWidget *mnuClose;
  GtkWidget *mnuPrevious;
  GtkWidget *mnuNext;
  GtkWidget *mnuQuit;

  gladexml = glade_xml_new(get_kildclient_installed_file("wndmain.glade"),
                           "wndMain", NULL);
  window = glade_xml_get_widget(gladexml, "wndMain");
  glade_xml_signal_autoconnect(gladexml);
#ifndef __WIN32__
  gtk_window_set_icon_from_file(GTK_WINDOW(window),
                                SYSDATADIR "/pixmaps/kildclient.png",
                                NULL);
  gtk_window_set_default_icon_from_file(SYSDATADIR "/pixmaps/kildclient.png",
                                        NULL);
#endif

  /* User interface */
  ntbkWorlds = glade_xml_get_widget(gladexml, "ntbkWorlds");
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(ntbkWorlds),
                           globalPrefs.tab_position);

  attach_world_gui(gui);

  gtk_widget_grab_focus(GTK_WIDGET(gui->cmbEntry));

  /* This signal needs pointers to the objects */
  mnuPreferences = glade_xml_get_widget(gladexml, "mnuPreferences");
  g_signal_connect(G_OBJECT(mnuPreferences), "activate",
                   G_CALLBACK(show_preferences_dialog_cb), NULL);

  /* These signals apparently don't work from Glade */
  mnuReconnect = glade_xml_get_widget(gladexml, "mnuReconnect");
  g_signal_connect(G_OBJECT(mnuReconnect), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->reconnect()");
  mnuDisconnect = glade_xml_get_widget(gladexml, "mnuDisconnect");
  g_signal_connect(G_OBJECT(mnuDisconnect), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->dc()");
  mnuConnectAnother = glade_xml_get_widget(gladexml, "mnuConnectAnother");
  g_signal_connect(G_OBJECT(mnuConnectAnother), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->connectother()");
  mnuSave = glade_xml_get_widget(gladexml, "mnuSave");
  g_signal_connect(G_OBJECT(mnuSave), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->save()");
  mnuClose = glade_xml_get_widget(gladexml, "mnuClose");
  g_signal_connect(G_OBJECT(mnuClose), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->close()");
  mnuPrevious = glade_xml_get_widget(gladexml, "mnuPrevious");
  g_signal_connect(G_OBJECT(mnuPrevious), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->prev()");
  mnuNext = glade_xml_get_widget(gladexml, "mnuNext");
  g_signal_connect(G_OBJECT(mnuNext), "activate",
                   G_CALLBACK(menu_perl_run_cb), "$world->next()");
  mnuQuit = glade_xml_get_widget(gladexml, "mnuQuit");
  g_signal_connect(G_OBJECT(mnuQuit), "activate",
                   G_CALLBACK(menu_perl_run_cb), "quit");

  return window;
}


gboolean
exit_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  if (confirm_quit()) {
    gtk_main_quit();
  }

  /* Never let it propagate the event, since gtk_main_quit() deals with
     exiting. */
  return TRUE;
}


gboolean
window_focus_in_cb(GtkWidget     *widget,
                   GdkEventFocus *evt,
                   gpointer       data)
{
  if (!currentWorld)
    return FALSE;

  currentWorld->has_focus = TRUE;
  if (currentWorld->has_unread_text) {
    gtk_label_set_text(currentWorld->gui->lblNotebook,
                       currentWorld->name);
    currentWorld->has_unread_text = FALSE;
    --worlds_with_new_text;
  }

  if (currentWorld->selection_start || currentWorld->selection_end) {
    simo_combo_box_select_region(currentWorld->gui->cmbEntry,
                                 currentWorld->selection_start,
                                 currentWorld->selection_end);
  }

  if (currentWorld->connected) {
    execute_hook(currentWorld, "OnGetFocus", NULL);
  }

  gtk_window_set_title(GTK_WINDOW(wndMain), "KildClient");
  if (globalPrefs.urgency_hint) {
    gtk_window_set_urgency_hint(GTK_WINDOW(wndMain), FALSE);
  }
  window_has_focus = TRUE;

  return FALSE;
}


gboolean
window_focus_out_cb(GtkWidget     *widget,
                    GdkEventFocus *evt,
                    gpointer       data)
{
  if (!currentWorld)
    return FALSE;

  currentWorld->has_focus = FALSE;

  simo_combo_box_get_selection_bounds(currentWorld->gui->cmbEntry,
                                      &currentWorld->selection_start,
                                      &currentWorld->selection_end);

  if (currentWorld->connected) {
    execute_hook(currentWorld, "OnLoseFocus", NULL);
  }

  window_has_focus = FALSE;
  adjust_window_title();

  return FALSE;
}


void
adjust_window_title(void)
{
  if (worlds_with_new_text == 0) {
    gtk_window_set_title(GTK_WINDOW(wndMain), "KildClient");
  } else if (worlds_with_new_text == 1) {
    gtk_window_set_title(GTK_WINDOW(wndMain), "(*) KildClient");
  } else {
    gchar *title;
    title = g_strdup_printf("(*)[%d] KildClient",
                            worlds_with_new_text);
    gtk_window_set_title(GTK_WINDOW(wndMain), title);
    g_free(title);
  }

  if (globalPrefs.urgency_hint) {
    gtk_window_set_urgency_hint(GTK_WINDOW(wndMain), worlds_with_new_text);
  }
}


void
notebook_page_changed_cb(GtkNotebook     *notebook,
                         GtkNotebookPage *page,
                         guint            page_num,
                         gpointer         data)
{
  World *old_world = NULL;
  gint   old_page;

  old_page = gtk_notebook_get_current_page(notebook);
  if (old_page != - 1) {
    old_world = g_list_nth_data(open_worlds, old_page);
  }

  if (old_world) {
    old_world->has_focus      = FALSE;
    old_world->is_current_tab = FALSE;

    if(old_world->connected) {
      execute_hook(old_world, "OnLoseFocus", NULL);
    }
  }

  /* Updates currentWorld, handles focus, etc.
     Called as idle because the change may result from a call to a Perl
     function, and this changes the current Perl interpreter. */
  g_idle_add(update_world_with_focus, GINT_TO_POINTER(page_num));
}


void
notebook_page_reordered_cb(GtkNotebook *notebook,
                           GtkWidget   *widget,
                           guint        new_pos,
                           gpointer     data)
{
  World    *world;
  WorldGUI *gui;
  gint      old_pos;
  GList    *sucessor;

  gui = (WorldGUI *) g_object_get_data(G_OBJECT(widget), "gui");
  world = gui->world;

  old_pos = g_list_index(open_worlds, world);

  sucessor = g_list_nth(open_worlds, new_pos);
  if (old_pos < new_pos) {
    sucessor = g_list_next(sucessor);
  }

  open_worlds = g_list_remove(open_worlds, world);
  open_worlds = g_list_insert_before(open_worlds, sucessor, world);
}


static
gboolean
update_world_with_focus(gpointer data)
{
  gint n = GPOINTER_TO_INT(data);

  /* Now we make it point to the new world */
  currentWorld = (World *) g_list_nth_data(open_worlds, n);
  world_for_perl = currentWorld;
  if (currentWorld) {
    currentWorld->has_focus      = window_has_focus;
    currentWorld->is_current_tab = TRUE;
    if (currentWorld->has_unread_text) {
      gtk_label_set_text(currentWorld->gui->lblNotebook,
                         currentWorld->name);
      currentWorld->has_unread_text = FALSE;
      --worlds_with_new_text;
    }

    PERL_SET_CONTEXT(currentWorld->perl_interpreter);

    if (currentWorld->connected) {
      execute_hook(currentWorld, "OnGetFocus", NULL);
    }
  }

  /* Do not run again */
  return FALSE;
}


void
open_new_world_cb(GtkWidget *widget, gpointer data)
{
  open_new_world(NULL, NULL, FALSE, NULL);
}


gboolean
open_new_world(WorldGUI    *gui,
               const gchar *name,
               gboolean     can_offline,
               gboolean    *offline_selected)
{
  World    *world = NULL;
  gint      page = 0;
  gint      action;
  gchar    *errmsg = NULL;
  World    *prev_world = NULL;
  gboolean  our_gui = FALSE;

  if (!gui) {
    gui = world_gui_new(TRUE);
    page = attach_world_gui(gui);
    our_gui = TRUE;
  } else {
    prev_world = gui->world;
  }

  if (name) {
    world = get_world_from_name(name);
  } else {
    world = get_world_to_connect();
  }

  if (world) {
    world->gui = gui;
    gui->world = world;
    ansitextview_update_color_tags(gui, gui->world);
    while (1) {
      if (connect_to(world, &errmsg)) {
        set_focused_world(page);
        world->has_focus      = TRUE;
        world->is_current_tab = TRUE;
        gtk_widget_grab_focus(GTK_WIDGET(world->gui->cmbEntry));
        return TRUE;
      } else {
        action = disconnected_msgbox(errmsg, can_offline);
        g_free(errmsg);
        if (action == GTK_RESPONSE_CLOSE) {
          break;
        }
        if (action == RESP_OFFLINE) {
          *offline_selected = TRUE;
          free_world(world);
          gui->world = prev_world;
          ansitextview_update_color_tags(gui, gui->world);
          return TRUE;
        }
        if (action == RESP_OTHER) {
          free_world(world);
          world = get_world_to_connect();
          if (!world) {
            break;
          }
          world->gui = gui;
          gui->world = world;
          ansitextview_update_color_tags(gui, gui->world);
        }
      }
    }
  }

  if (our_gui) {
    detach_world_gui(gui);
    free_world_gui(gui);
  }
  return FALSE;
}


void
edit_this_world_cb(GtkWidget *widget, gpointer data)
{
  gboolean dummy;

  edit_world(&currentWorld, GTK_WINDOW(wndMain), &dummy, FALSE);

  gtk_label_set_text(currentWorld->gui->lblNotebook, currentWorld->name);
}


void
menu_statistics_cb(GtkWidget *widget, gpointer data)
{
  GladeXML  *gladexml;
  GtkWidget *dlgStatistics;
  GtkWidget *lblName;
  GtkWidget *lblHost;
  GtkWidget *lblPort;
  GtkWidget *lblConnTime;
  GtkWidget *lblIdleTime;
  GtkWidget *lblCompression;
  GtkWidget *lblRawBytes;
  GtkWidget *lblBytes;
  GtkWidget *lblCRatio;
  GtkWidget *lblLines;
  GtkWidget *lblTLS;
  gchar     *str;
  time_t     now;
  GString   *timestr;

  gladexml = glade_xml_new(get_kildclient_installed_file("kildclient.glade"),
                           "dlgStatistics", NULL);
  dlgStatistics  = glade_xml_get_widget(gladexml, "dlgStatistics");
  lblName        = glade_xml_get_widget(gladexml, "lblName");
  lblHost        = glade_xml_get_widget(gladexml, "lblHost");
  lblPort        = glade_xml_get_widget(gladexml, "lblPort");
  lblConnTime    = glade_xml_get_widget(gladexml, "lblConnTime");
  lblIdleTime    = glade_xml_get_widget(gladexml, "lblIdleTime");
  lblCompression = glade_xml_get_widget(gladexml, "lblCompression");
  lblRawBytes    = glade_xml_get_widget(gladexml, "lblRawBytes");
  lblBytes       = glade_xml_get_widget(gladexml, "lblBytes");
  lblCRatio      = glade_xml_get_widget(gladexml, "lblCRatio");
  lblLines       = glade_xml_get_widget(gladexml, "lblLines");
  lblTLS         = glade_xml_get_widget(gladexml, "lblTLS");

  /* Basic information */
  gtk_label_set_text(GTK_LABEL(lblName), currentWorld->name);
  if (currentWorld->host_ip) {
    str = g_strdup_printf("%s (%s)", currentWorld->host, currentWorld->host_ip);
  } else {
    str = g_strdup(currentWorld->host);
  }
  gtk_label_set_text(GTK_LABEL(lblHost), str);
  g_free(str);
  gtk_label_set_text(GTK_LABEL(lblPort), currentWorld->port);

  if (currentWorld->connected) {
    now = time(NULL);
    timestr = g_string_new(NULL);
    format_hms_time(timestr, difftime(now, currentWorld->connected_time));
    gtk_label_set_text(GTK_LABEL(lblConnTime), timestr->str);
    g_string_truncate(timestr, 0);
    format_hms_time(timestr, difftime(now, currentWorld->last_command_time));
    gtk_label_set_text(GTK_LABEL(lblIdleTime), timestr->str);
    g_string_free(timestr, TRUE);
  } else {
    gtk_label_set_text(GTK_LABEL(lblConnTime), _("Not connected"));
    gtk_label_set_text(GTK_LABEL(lblIdleTime), _("Not connected"));
  }

  /* Compression Information */
  if (currentWorld->zstream) {
    str = g_strdup_printf("MCCP version %d", currentWorld->mccp_ver);
    gtk_label_set_text(GTK_LABEL(lblCompression), str);
    g_free(str);
    str = g_strdup_printf("%ld", currentWorld->rawbytes);
    gtk_label_set_text(GTK_LABEL(lblRawBytes), str);
    g_free(str);
    str = g_strdup_printf("%ld", currentWorld->bytes);
    gtk_label_set_text(GTK_LABEL(lblBytes), str);
    g_free(str);
    str = g_strdup_printf("%.2f:1",
                          (double)currentWorld->bytes/(double)currentWorld->rawbytes);
    gtk_label_set_text(GTK_LABEL(lblCRatio), str);
    g_free(str);
  } else {
    str = g_strdup_printf("%ld", currentWorld->rawbytes);
    gtk_label_set_text(GTK_LABEL(lblBytes), str);
    g_free(str);
  }

  str = g_strdup_printf("%ld",
                        currentWorld->deleted_lines
                          + gtk_text_buffer_get_line_count(currentWorld->gui->txtBuffer));
  gtk_label_set_text(GTK_LABEL(lblLines), str);
  g_free(str);

  /* TLS information */
#ifdef HAVE_LIBGNUTLS
  if (currentWorld->use_tls) {
    GtkWidget               *lblTLSCipher;
    GtkWidget               *lblTLSMAC;
    GtkWidget               *lblTLSKX;
    GtkWidget               *lblTLSComp;
    GtkWidget               *lblTLSCertType;
    GtkWidget               *lblTLSCertHostname;
    GtkWidget               *lblTLSCertActivation;
    GtkWidget               *lblTLSCertExpiration;
    GtkWidget               *lblTLSCertFingerprint;
    gnutls_certificate_type  cert_type;
    const gnutls_datum      *cert_list;
    guint                    cert_list_size;
    gboolean                 hostname_matches = FALSE;
    time_t                   act_time;
    time_t                   exp_time;
    gchar                    digest[20];
    size_t                   digest_size;
    const gchar             *val;
    gchar                   *mval;
    gchar                    buf[MAX_BUFFER];
    gint                     i;

    val = gnutls_protocol_get_name(gnutls_protocol_get_version(currentWorld->tls_session));
    gtk_label_set_text(GTK_LABEL(lblTLS), val);

    lblTLSCipher = glade_xml_get_widget(gladexml, "lblTLSCipher");
    val = gnutls_cipher_get_name(gnutls_cipher_get(currentWorld->tls_session));
    gtk_label_set_text(GTK_LABEL(lblTLSCipher), val);

    lblTLSMAC = glade_xml_get_widget(gladexml, "lblTLSMAC");
    val = gnutls_mac_get_name(gnutls_mac_get(currentWorld->tls_session));
    gtk_label_set_text(GTK_LABEL(lblTLSMAC), val);

    lblTLSKX = glade_xml_get_widget(gladexml, "lblTLSKX");
    val = gnutls_kx_get_name(gnutls_kx_get(currentWorld->tls_session));
    gtk_label_set_text(GTK_LABEL(lblTLSKX), val);

    lblTLSComp = glade_xml_get_widget(gladexml, "lblTLSComp");
    val = gnutls_compression_get_name(gnutls_compression_get(currentWorld->tls_session));
    gtk_label_set_text(GTK_LABEL(lblTLSComp), val);

    lblTLSCertType = glade_xml_get_widget(gladexml, "lblTLSCertType");
    cert_type = gnutls_certificate_type_get(currentWorld->tls_session);
    val = gnutls_certificate_type_get_name(cert_type);
    gtk_label_set_text(GTK_LABEL(lblTLSCertType), val);

    /* Certificate/key information */
    lblTLSCertHostname    = glade_xml_get_widget(gladexml, "lblTLSCertHostname");
    lblTLSCertActivation  = glade_xml_get_widget(gladexml, "lblTLSCertActivation");
    lblTLSCertExpiration  = glade_xml_get_widget(gladexml, "lblTLSCertExpiration");
    lblTLSCertFingerprint = glade_xml_get_widget(gladexml, "lblTLSCertFingerprint");
    cert_list = gnutls_certificate_get_peers(currentWorld->tls_session,
                                             &cert_list_size);
    if (cert_type == GNUTLS_CRT_X509) {
      GtkWidget      *lblTLSCertSubDN;
      GtkWidget      *lblTLSCertIssDN;
      gnutls_x509_crt crt;
      size_t          size;

      if (cert_list_size > 0) {   /* Note that we consider only the first */
        gnutls_x509_crt_init(&crt);
        gnutls_x509_crt_import(crt, &cert_list[0], GNUTLS_X509_FMT_DER);

        hostname_matches = gnutls_x509_crt_check_hostname(crt,
                                                          currentWorld->host);

        act_time = gnutls_x509_crt_get_activation_time(crt);
        exp_time = gnutls_x509_crt_get_expiration_time(crt);

        digest_size = sizeof(digest);
        gnutls_x509_crt_get_fingerprint(crt, GNUTLS_DIG_MD5,
                                        digest, &digest_size);

        size = sizeof(buf);
        gnutls_x509_crt_get_dn(crt, buf, &size);
        lblTLSCertSubDN = glade_xml_get_widget(gladexml, "lblTLSCertSubDN");
        gtk_label_set_text(GTK_LABEL(lblTLSCertSubDN), buf);

        size = sizeof(buf);
        gnutls_x509_crt_get_issuer_dn(crt, buf, &size);
        lblTLSCertIssDN = glade_xml_get_widget(gladexml, "lblTLSCertIssDN");
        gtk_label_set_text(GTK_LABEL(lblTLSCertIssDN), buf);
      }
    } else if (cert_type == GNUTLS_CRT_OPENPGP) {
      gnutls_openpgp_key crt;

      gnutls_openpgp_key_init(&crt);
      gnutls_openpgp_key_import(crt, &cert_list[0], GNUTLS_OPENPGP_FMT_RAW);

      hostname_matches = gnutls_openpgp_key_check_hostname(crt,
                                                           currentWorld->host);

      act_time = gnutls_openpgp_key_get_creation_time(crt);
      exp_time = gnutls_openpgp_key_get_expiration_time(crt);

      digest_size = sizeof(digest);
      gnutls_openpgp_key_get_fingerprint(crt, digest, &digest_size);
    }

    if (hostname_matches) {
      mval = g_strdup_printf("Matches '%s'", currentWorld->host);
    } else {
      mval = g_strdup_printf("Does not match '%s'", currentWorld->host);
    }
    gtk_label_set_text(GTK_LABEL(lblTLSCertHostname), mval);
    g_free(mval);

    now = time(NULL);
    strftime(buf, MAX_BUFFER, "%c", localtime(&act_time));
    if (act_time > now) {
      strcat(buf, _(": Not yet active"));
    }
    gtk_label_set_text(GTK_LABEL(lblTLSCertActivation), buf);
    strftime(buf, MAX_BUFFER, "%c", localtime(&exp_time));
    if (exp_time < now) {
      strcat(buf, _(": Expired"));
    }
    gtk_label_set_text(GTK_LABEL(lblTLSCertExpiration), buf);

    mval = buf;
    for (i = 0; i < digest_size; i++) {
      sprintf(mval, "%.2X:", (unsigned char) digest[i]);
      mval += 3;
    }
    mval[strlen(mval) - 1] = 0; /* Remove last : */
    gtk_label_set_text(GTK_LABEL(lblTLSCertFingerprint), buf);
  } else {
    gtk_label_set_text(GTK_LABEL(lblTLS), _("None"));
  }
#endif

  /* Run the dialog */
  gtk_dialog_run(GTK_DIALOG(dlgStatistics));
  gtk_widget_destroy(dlgStatistics);
  g_object_unref(gladexml);
}


void
menu_edit_default_world_cb(GtkWidget *widget, gpointer data)
{
  gboolean dummy;

  edit_world(&default_world, GTK_WINDOW(wndMain), &dummy, FALSE);
}


void
menu_manual_cb(GtkWidget *widget, gpointer data)
{
  gchar *manualurl;

#ifndef __WIN32__
  manualurl = g_strdup_printf("file://%s", MANUALDIR "/index.html");
#else
  manualurl = g_strdup_printf(get_kildclient_installed_file("manual\\html\\index.html"));
#endif
  menu_url_open(widget, manualurl);
}



void
about_menu_cb(GtkWidget *widget, gpointer data)
{
  GladeXML         *xml;
  static GtkWidget *dlgAbout = NULL;

  if (!dlgAbout) {
    xml = glade_xml_new(get_kildclient_installed_file("kildclient.glade"),
                        "dlgAbout", NULL);
    dlgAbout = glade_xml_get_widget(xml, "dlgAbout");
    gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dlgAbout), VERSION);

    glade_xml_signal_autoconnect(xml);
  }

  gtk_widget_show_all(dlgAbout);
  gtk_window_present(GTK_WINDOW(dlgAbout));
}


static
gint
attach_world_gui(WorldGUI *gui)
{
  gint page;

  page = gtk_notebook_append_page(GTK_NOTEBOOK(ntbkWorlds), gui->vbox,
                                  GTK_WIDGET(gui->hboxTab));
  gtk_widget_show_all(GTK_WIDGET(gui->hboxTab));
  gtk_notebook_set_tab_reorderable(GTK_NOTEBOOK(ntbkWorlds),
                                   gui->vbox,
                                   TRUE);

  gtk_widget_show_all(ntbkWorlds);

  return page;
}


static
void
detach_world_gui(WorldGUI *gui)
{
  gint old_page;
  gint new_page;

  old_page = gtk_notebook_page_num(GTK_NOTEBOOK(ntbkWorlds), gui->vbox);
  gtk_notebook_remove_page(GTK_NOTEBOOK(ntbkWorlds), old_page);

  new_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(ntbkWorlds));
  update_world_with_focus(GINT_TO_POINTER(new_page));
}


void
set_focused_world(gint page)
{
  gtk_notebook_set_current_page(GTK_NOTEBOOK(ntbkWorlds), page);
}


gint
get_focused_world(void)
{
  return gtk_notebook_get_current_page(GTK_NOTEBOOK(ntbkWorlds));
}


void
remove_world(World *world, gboolean remove_gui)
{
  open_worlds = g_list_remove(open_worlds, world);

  if (world->connected || world->connecting) {
    g_free(close_connection(world));
  }

  if (world->file) {
    save_world_to_file(world);
    save_command_history(world);
    save_permanent_variables(world);
  }

  if (remove_gui) {
    detach_world_gui(world->gui);
    free_world_gui(world->gui);
  }

  free_world(world);

  if (open_worlds == NULL) {
    gtk_main_quit();
  }
}


gboolean
remove_world_timer(World *world)
{
  remove_world(world, TRUE);

  /* Because it is called as an idle function */
  return FALSE;
}


void
set_notebook_tab_position(GtkPositionType pos)
{
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(ntbkWorlds), pos);
}


void
menu_world_open_cb(GtkWidget *widget,
                   gpointer data)
{
  GladeXML  *gladexml;
  GtkWidget *mnuDisconnect;

  gladexml = glade_get_widget_tree(widget);
  mnuDisconnect = glade_xml_get_widget(gladexml,
                                       "mnuDisconnect");

  if (currentWorld && (currentWorld->connected || currentWorld->connecting)) {
    gtk_widget_set_sensitive(mnuDisconnect, TRUE);
  } else {
    gtk_widget_set_sensitive(mnuDisconnect, FALSE);
  }
}


void
menu_edit_open_cb(GtkWidget *widget,
                  gpointer data)
{
  GladeXML     *gladexml;
  GtkWidget    *mnuCut;
  GtkWidget    *mnuCopy;
  GtkWidget    *mnuPaste;
  GtkWidget    *mnuDelete;
  GdkDisplay   *display;
  GtkClipboard *clipboard;
  gboolean      input_selected;
  gboolean      output_selected;

  /* Safety check */
  if (!currentWorld
      || !currentWorld->gui->cmbEntry || !currentWorld->gui->txtBuffer) {
    return;
  }

  gladexml  = glade_get_widget_tree(widget);
  mnuCut    = glade_xml_get_widget(gladexml, "mnuCut");
  mnuCopy   = glade_xml_get_widget(gladexml, "mnuCopy");
  mnuPaste  = glade_xml_get_widget(gladexml, "mnuPaste");
  mnuDelete = glade_xml_get_widget(gladexml, "mnuDelete");

  /* We can paste when there is text in the clipboard */

  display = gtk_widget_get_display(GTK_WIDGET(widget));
  clipboard = gtk_clipboard_get_for_display(display, GDK_SELECTION_CLIPBOARD);
  gtk_widget_set_sensitive(mnuPaste,
                           gtk_clipboard_wait_is_text_available(clipboard));

  /* We can cut and delete if there is text selected in the input box */
  input_selected
    = simo_combo_box_get_selection_bounds(currentWorld->gui->cmbEntry,
                                          NULL, NULL);
  gtk_widget_set_sensitive(mnuCut,    input_selected);
  gtk_widget_set_sensitive(mnuDelete, input_selected);

  /* We can copy if there is a selection in the input box or the output
     window */
  output_selected
    = gtk_text_buffer_get_selection_bounds(currentWorld->gui->txtBuffer,
                                           NULL, NULL);
  gtk_widget_set_sensitive(mnuCopy, input_selected || output_selected);
}


void
menu_cut_activate_cb(GtkWidget *widget,
                     gpointer data)
{
  simo_combo_box_cut_clipboard(currentWorld->gui->cmbEntry);
}


void
menu_copy_activate_cb(GtkWidget *widget,
                     gpointer data)
{
  GtkTextIter start;
  GtkTextIter end;

  if (gtk_text_buffer_get_selection_bounds(currentWorld->gui->txtBuffer,
                                           &start, &end)) {
    GdkDisplay   *display;
    GtkClipboard *clipboard;

    display = gtk_widget_get_display(GTK_WIDGET(widget));

    clipboard = gtk_clipboard_get_for_display(display,
                                              GDK_SELECTION_CLIPBOARD);
    gtk_text_buffer_copy_clipboard(currentWorld->gui->txtBuffer, clipboard);
    /* The selection is unmarked after the previous step (possibly a issue
       with the non-focusability of the text buffer). So we set it again. */
    gtk_text_buffer_select_range(currentWorld->gui->txtBuffer, &start, &end);
  } else {
    simo_combo_box_copy_clipboard(currentWorld->gui->cmbEntry);
  }
}


void
menu_paste_activate_cb(GtkWidget *widget,
                       gpointer data)
{
  simo_combo_box_paste_clipboard(currentWorld->gui->cmbEntry);
}


void
menu_delete_activate_cb(GtkWidget *widget,
                        gpointer data)
{
  simo_combo_box_delete_selection(currentWorld->gui->cmbEntry);
}


void
menu_find_activate_cb(GtkWidget *widget, gpointer data)
{
  GtkTextIter search_start;

  if (!currentWorld || !currentWorld->gui || !currentWorld->gui->search_box) {
    return;
  }

  g_object_set(G_OBJECT(currentWorld->gui->search_box),
               "no-show-all", FALSE,
               NULL);
  gtk_widget_show_all(currentWorld->gui->search_box);

  gtk_text_buffer_get_start_iter(currentWorld->gui->txtBuffer, &search_start);
  if (!currentWorld->gui->txtmark_search_start) {
    currentWorld->gui->txtmark_search_start
      = gtk_text_buffer_create_mark(currentWorld->gui->txtBuffer,
                                    NULL,
                                    &search_start,
                                    FALSE);
    currentWorld->gui->txtmark_next_search_start
      = gtk_text_buffer_create_mark(currentWorld->gui->txtBuffer,
                                    NULL,
                                    &search_start,
                                    FALSE);
  }

  gtk_entry_set_text(GTK_ENTRY(currentWorld->gui->txtSearchTerm), "");
  gtk_widget_grab_focus(currentWorld->gui->txtSearchTerm);
}


void
menu_findnext_activate_cb(GtkWidget *widget, gpointer data)
{
  if (!currentWorld || !currentWorld->gui || !currentWorld->gui->search_box) {
    return;
  }

  if (!currentWorld->gui->txtmark_search_start) {
    /* No search started yet. */
    menu_find_activate_cb(NULL, NULL);
  } else {
    find_next_cb(NULL, currentWorld->gui);
  }
}


void
menu_input_clear_cb(GtkWidget *widget,
                    gpointer data)
{
  if (currentWorld) {
    clear_button_cb(NULL, currentWorld->gui);
  }
}


void
menu_input_prev_cb(GtkWidget *widget, gpointer data)
{
  if (currentWorld) {
    prev_or_next_command(currentWorld, FALSE);
  }
}


void
menu_input_next_cb(GtkWidget *widget, gpointer data)
{
  if (currentWorld) {
    prev_or_next_command(currentWorld, TRUE);
  }
}


void
menu_input_find_prev_cb(GtkWidget *widget, gpointer data)
{
  if (currentWorld) {
    find_prev_or_next_command(currentWorld, FALSE);
  }
}


void
menu_input_find_next_cb(GtkWidget *widget, gpointer data)
{
  if (currentWorld) {
    find_prev_or_next_command(currentWorld, TRUE);
  }
}



void
menu_preferences_open_cb(GtkWidget *widget,
                         gpointer data)
{
  GladeXML  *gladexml;
  GtkWidget *mnuDisableTriggers;
  GtkWidget *mnuDisableAliases;
  GtkWidget *mnuDisableMacros;
  GtkWidget *mnuDisableTimers;

  gladexml = glade_get_widget_tree(widget);
  mnuDisableTriggers = glade_xml_get_widget(gladexml,
                                            "mnuDisTriggers");
  mnuDisableAliases  = glade_xml_get_widget(gladexml,
                                            "mnuDisAliases");
  mnuDisableMacros   = glade_xml_get_widget(gladexml,
                                            "mnuDisMacros");
  mnuDisableTimers   = glade_xml_get_widget(gladexml,
                                            "mnuDisTimers");

  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mnuDisableTriggers),
                                 currentWorld->disable_triggers);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mnuDisableAliases),
                                 currentWorld->disable_aliases);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mnuDisableMacros),
                                 currentWorld->disable_macros);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mnuDisableTimers),
                                 currentWorld->disable_timers);
}


void
menu_disable_triggers_cb(GtkWidget *widget,
                         gpointer data)
{
  currentWorld->disable_triggers =
    gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
}


void
menu_disable_aliases_cb(GtkWidget *widget,
                        gpointer data)
{
  currentWorld->disable_aliases =
    gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
}


void
menu_disable_macros_cb(GtkWidget *widget,
                       gpointer data)
{
  currentWorld->disable_macros =
    gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
}


void
menu_disable_timers_cb(GtkWidget *widget,
                       gpointer data)
{
  currentWorld->disable_timers =
    gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
}


void
mnuDebugMatches_activate_cb(GtkWidget *widget,
                   gpointer data)
{
  debug_matches = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
}


gboolean
update_times_cb(gpointer data)
{
  /* Updates the connection and idle timers in the status bar of the
     worlds */
  GList  *worldptr;
  World  *world;
  time_t  now;
  gchar  *timestr;

  now = time(NULL);

  worldptr = open_worlds;
  while (worldptr) {
    world = (World *) worldptr->data;

    if (world->connected
        && world->gui && world->gui->lblTime) {
      timestr = format_time_string(world,
                                   (int) difftime(now, world->connected_time),
                                   (int) difftime(now, world->last_command_time));
      gtk_label_set_text(world->gui->lblTime, timestr);
      g_free(timestr);
    }

    worldptr = worldptr->next;
  }

  return TRUE;
}


static
gchar *
format_time_string(World *world, int ctime, int itime)
{
  GString *str;

  str = g_string_new(NULL);

  if (world->ctime_type != NO) {
    g_string_append(str, "Conn: ");
    if (world->ctime_type == SEC) {
      g_string_append_printf(str, "%ds", ctime);
    } else {
      format_hms_time(str, ctime);
    }

    if (world->itime_type != NO) {
      g_string_append(str, ", ");;
    }
  }

  if (world->itime_type != NO) {
    g_string_append(str, "Idle: ");
    if (world->itime_type == SEC) {
      g_string_append_printf(str, "%ds", itime);
    } else {
      format_hms_time(str, itime);
    }
  }

  return g_string_free(str, FALSE);
}


static
void
format_hms_time(GString *str, int seconds)
{
  int hours;
  int minutes;

  hours   = (int) (seconds/3600);
  seconds %= 3600;
  minutes = (int) (seconds/60);
  seconds %= 60;

  if (hours) {
    g_string_append_printf(str, "%dh", hours);
  }
  if (minutes) {
    if (hours) {
      g_string_append_printf(str, "%02dm", minutes);
    } else {
      g_string_append_printf(str, "%dm", minutes);
    }
  }
  g_string_append_printf(str, "%02ds", seconds);
}
