/* $Id: ansitextview.c,v 1.25 2006/08/10 14:36:33 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <time.h>
#include <stdarg.h>
#include <sys/types.h>
#include <regex.h>
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "kcircularqueue.h"

#include "ansi.h"
#include "kildclient.h"
#include "perlscript.h"


/*************************
 * File global variables *
 *************************/
/* Values used to get the 6x6x6 color cube of xterm 256 color mode */
static guint8 xterm256color[] = { 0x00, 0x5f, 0x87, 0xaf, 0xd7, 0xff };
/* Values used for greyscales in xterm 256 color mode */
static guint8 xterm256gray[] = { 0x08, 0x12, 0x1c, 0x26,
                                 0x30, 0x3a, 0x44, 0x4e,
                                 0x58, 0x62, 0x6c, 0x76,
                                 0x80, 0x8a, 0x94, 0x9e,
                                 0xa8, 0xb2, 0xbc, 0xc6,
                                 0xd0, 0xda, 0xe4, 0xee };
/* UTF-8 codes for the characters used in line-drawing mode */
static gchar *linedrawingchars[] = { "\342\227\206", /* ` */
                                     "\342\226\222", /* a */
                                     "\342\220\211", /* b */
                                     "\342\220\214", /* c */
                                     "\342\220\215", /* d */
                                     "\342\220\212", /* e */
                                     "\302\260",     /* f */
                                     "\302\261",     /* g */
                                     "\342\220\244", /* h */
                                     "\342\220\213", /* i */
                                     "\342\224\230", /* j */
                                     "\342\224\220", /* k */
                                     "\342\224\214", /* l */
                                     "\342\224\224", /* m */
                                     "\342\224\274", /* n */
                                     "\342\216\272", /* o */
                                     "\342\216\273", /* p */
                                     "\342\224\200", /* q */
                                     "\342\216\274", /* r */
                                     "\342\216\275", /* s */
                                     "\342\224\234", /* t */
                                     "\342\224\244", /* u */
                                     "\342\224\264", /* v */
                                     "\342\224\254", /* w */
                                     "\342\224\202", /* x */
                                     "\342\211\244", /* y */
                                     "\342\211\245", /* z */
                                     "\317\200",     /* { */
                                     "\342\211\240", /* | */
                                     "\302\243",     /* } */
                                     "\302\267",     /* ~ */ };
/* ASCII characters replacing those of line-drawing mode when they are
   not available. */
static gchar *replacingchars[] = { "?", /* ` */
                                   "#", /* a */
                                   "?", /* b */
                                   "?", /* c */
                                   "?", /* d */
                                   "?", /* e */
                                   "o", /* f */
                                   "?", /* g */
                                   "?", /* h */
                                   "?", /* i */
                                   "+", /* j */
                                   "+", /* k */
                                   "+", /* l */
                                   "+", /* m */
                                   "+", /* n */
                                   "-", /* o */
                                   "-", /* p */
                                   "-", /* q */
                                   "_", /* r */
                                   "_", /* s */
                                   "|", /* t */
                                   "|", /* u */
                                   "-", /* v */
                                   "-", /* w */
                                   "|", /* x */
                                   "<", /* y */
                                   ">", /* z */
                                   "p", /* { */
                                   "?", /* | */
                                   "L", /* } */
                                   ".", /* ~ */ };



/***********************
 * Function prototypes *
 ***********************/
static void        ansitextview_append_string_internal(WorldGUI    *gui,
                                                       const gchar *str,
                                                       gboolean     add_nl);
static void        ansitextview_insert_ansi_string(WorldGUI    *gui,
                                                   GtkTextIter *pos,
                                                   const gchar *str,
                                                   gsize        len);
static void        ansitextview_text_added(WorldGUI *gui);
static void        parse_ansi_string(WorldGUI *gui);
static void        reset_ansi_state(TAState *state);
static void        ansitextview_verify_urls_and_add(WorldGUI     *gui,
                                                    GtkTextIter  *pos,
                                                    const guchar *text,
                                                    gsize         len);
static void        ansitextview_insert_vt100linedraw_string(WorldGUI     *gui,
                                                            GtkTextIter  *pos,
                                                            const guchar *text,
                                                            gsize         len);
static void        ansitextview_append_formatted(GtkTextBuffer  *buffer,
                                                 GtkTextIter    *iter,
                                                 const gchar    *text,
                                                 gsize           len,
                                                 TextAppearance *ta,
                                                 GtkTextTag     *extra_tag);
static GtkTextTag *get_tag_for_color(WorldGUI *gui,
                                     guint8 r, guint8 g, guint8 b,
                                     gboolean  is_back);
static GtkTextTag *get_tag_for_xterm256_color(WorldGUI *gui,
                                              gint      x256idx,
                                              gint      code_used);



void
ansitextview_append_string(WorldGUI    *gui,
                           const gchar *str)
{
  /*
   * Append str (a null-terminated string) to the Buffer, in the default
   * color.
   */

  ansitextview_append_string_internal(gui, str, FALSE);
}


void
ansitextview_append_string_nl(WorldGUI    *gui,
                              const gchar *str)
{
  /*
   * Append str (a null-terminated string) to the Buffer, in the default
   * color, and a newline.
   */

  ansitextview_append_string_internal(gui, str, TRUE);
}


void
ansitextview_append_stringf(WorldGUI    *gui,
                            const gchar *format,
                            ...)
{
  /*
   * Formats a string using printf syntax and outputs it.
   */
  va_list  ap;
  gchar   *str;

  va_start(ap, format);
  str = g_strdup_vprintf(format, ap);
  ansitextview_append_string_internal(gui, str, FALSE);
  g_free(str);
}


static
void
ansitextview_append_string_internal(WorldGUI    *gui,
                                    const gchar *str,
                                    gboolean     add_nl)
{
  /*
   * Does the bulk work of ansitextview_append_string() and
   * ansitextview_append_string_nl(): Those functions just call this one
   * specifying whether a newline is to be added or not.
   */
  GtkTextIter  end_iter;

  gtk_text_buffer_get_iter_at_mark(gui->txtBuffer, &end_iter,
                                   gui->txtmark_linestart);
  gtk_text_buffer_insert(gui->txtBuffer,
                         &end_iter,
                         str,
                         -1);
  if (add_nl) {
    gtk_text_buffer_insert(gui->txtBuffer,
                           &end_iter,
                           "\n",
                           1);
  }
  gtk_text_buffer_move_mark(gui->txtBuffer, gui->txtmark_linestart, &end_iter);
  ansitextview_text_added(gui);
}


void
ansitextview_append_string_with_fgcolor(WorldGUI    *gui,
                                        const gchar *str,
                                        gint         fg_color)
{
  /*
   * Append str (a null-terminated string) to the Buffer, using the
   * specified foreground color.
   */
  GtkTextIter end_iter;
  GtkTextIter line_end;

  gtk_text_buffer_get_iter_at_offset(gui->txtBuffer, &end_iter, -1);
  gtk_text_buffer_insert_with_tags(gui->txtBuffer,
                                   &end_iter,
                                   str,
                                   -1,
                                   gui->ta.ansi_fore_tags[fg_color],
                                   NULL);

  gtk_text_buffer_get_iter_at_offset(gui->txtBuffer,
                                     &line_end,
                                     -1);
  gtk_text_buffer_move_mark(gui->txtBuffer,
                            gui->txtmark_linestart,
                            &line_end);
  ansitextview_text_added(gui);
}


void
ansitextview_append_ansi_string(WorldGUI    *gui,
                                const gchar *str,
                                gsize        len)
{
  GtkTextIter end_iter;

  gtk_text_buffer_get_end_iter(gui->txtBuffer, &end_iter);
  ansitextview_insert_ansi_string(gui, &end_iter, str, len);
}


static
void
ansitextview_insert_ansi_string(WorldGUI    *gui,
                                GtkTextIter *pos,
                                const gchar *str,
                                gsize        len)
{
  /*
   * Append str (a null-terminated string) possibly with ANSI color
   * codes to the Buffer.
   */
  static guchar text_buffer[MAX_BUFFER];
  gint          text_buffer_pos;
  gint          original_pos;

  text_buffer_pos = 0;
  original_pos    = 0;
  while (str[original_pos] && original_pos < len) {
    switch (gui->ta.ansifsm_state) {
    case ANSIFSM_DATA:
      if (str[original_pos] == ANSI_ESC) {
        gui->ta.ansifsm_state = ANSIFSM_ESC;
      } else {
        text_buffer[text_buffer_pos++] = str[original_pos];
      }
      break;

    case ANSIFSM_ESC:
      if (str[original_pos] == '[') {
        gui->ta.ansifsm_state = ANSIFSM_ANSISEQ;
        if (text_buffer_pos) {
          ansitextview_verify_urls_and_add(gui,
                                           pos,
                                           text_buffer,
                                           text_buffer_pos);
          text_buffer_pos = 0;
        }
      } else if (str[original_pos] == '(') {
        gui->ta.ansifsm_state = ANSIFSM_ESCLEFTPAR;
      } else {
        /* Put the ESC back */
        text_buffer[text_buffer_pos++] = ANSI_ESC;
        text_buffer[text_buffer_pos++] = str[original_pos];
        gui->ta.ansifsm_state = ANSIFSM_DATA;
      }
      break;

    case ANSIFSM_ANSISEQ:
      /* Since there is no reason for a valid ANSI sequence that is longer
         than MAX_BUFFER, we simply cut it if it is too long. */
      if ((isdigit(str[original_pos]) || str[original_pos] == ';')
          && gui->ta.ansiseq_pos < MAX_BUFFER) {
        gui->ta.ansiseq_buffer[gui->ta.ansiseq_pos++] = str[original_pos];
      } else {
        gui->ta.ansifsm_state = ANSIFSM_DATA;
        if (str[original_pos] == 'm') {
          parse_ansi_string(gui);
        } else {
          /* Not recognized, ignored and cleared. */
          /* FIXME: Copy it back, being careful with buffer size. */
          gui->ta.ansiseq_pos = 0;
        }
      }
      break;

    case ANSIFSM_ESCLEFTPAR:
      if (str[original_pos] == '0' || str[original_pos] == 'B') {
        gui->ta.ansifsm_state = ANSIFSM_DATA;
        if (text_buffer_pos) {
          ansitextview_verify_urls_and_add(gui,
                                           pos,
                                           text_buffer,
                                           text_buffer_pos);
          text_buffer_pos = 0;
        }
        gui->ta.state.linedraw = (str[original_pos] == '0');
      } else {
        /* Put unrecognized sequence back */
        text_buffer[text_buffer_pos++] = ANSI_ESC;
        text_buffer[text_buffer_pos++] = '(';
        text_buffer[text_buffer_pos++] = str[original_pos];
        gui->ta.ansifsm_state = ANSIFSM_DATA;
      }
      break;
    }
    ++original_pos;
  }

  /* Add any remaining text */
  if (text_buffer_pos) {
    ansitextview_verify_urls_and_add(gui,
                                     pos,
                                     text_buffer,
                                     text_buffer_pos);
    text_buffer_pos = 0;
  }

  ansitextview_text_added(gui);
}


void
ansitextview_append_echoed_string(WorldGUI    *gui,
                                  const gchar *str,
                                  gsize        len)
{
  /*
   * Append str (a null-terminated string) possibly with ANSI color
   * codes to the Buffer, which does not come from the MUD, but was
   * echoed. The state of the ANSI parser is saved in case an
   * incomplete ANSI sequence was pending. The line is printed above
   * any incomplete line (prompt).
   */
  static guchar saved_buffer[MAX_BUFFER];
  TAState       saved_state;
  gint          saved_pos;
  ANSIFSMState  saved_fsmstate;

  GtkTextIter end_iter;
  gint        printed;
  gint        to_print;


  memcpy(saved_buffer, gui->ta.ansiseq_buffer, gui->ta.ansiseq_pos);
  saved_pos      = gui->ta.ansiseq_pos;
  saved_fsmstate = gui->ta.ansifsm_state;
  saved_state    = gui->ta.state;

  gui->ta.ansifsm_state  = ANSIFSM_DATA;
  gui->ta.ansiseq_pos    = 0;

  /* Due to static buffers, only MAX_BUFFER chars can be printed at a time. */
  printed = 0;
  gtk_text_buffer_get_iter_at_mark(gui->txtBuffer, &end_iter,
                                   gui->txtmark_linestart);
  while (printed < len) {
    to_print = (len - printed) > MAX_BUFFER ? MAX_BUFFER : (len - printed);
    ansitextview_insert_ansi_string(gui, &end_iter, str + printed, to_print);
    printed += to_print;
  }
  gtk_text_buffer_move_mark(gui->txtBuffer,
                            gui->txtmark_linestart, &end_iter);

  memcpy(gui->ta.ansiseq_buffer, saved_buffer, saved_pos);
  gui->ta.ansiseq_pos    = saved_pos;
  gui->ta.ansifsm_state  = saved_fsmstate;
  gui->ta.state          = saved_state;
}


static
void
ansitextview_text_added(WorldGUI *gui)
{
  GtkAdjustment *adjustment;
  gint           lines;
  static gchar   msg[MAX_BUFFER];
  int            n;

  /* Update count of lines */
  lines = gtk_text_buffer_get_line_count(gui->txtBuffer);
  if (gui->lblLines) {
    gint totallines = gui->world->deleted_lines + lines;
    if (totallines == 1) {
      gtk_label_set_text(gui->lblLines, _("1 line"));
    } else {
      sprintf(msg, _("%d lines"), totallines);
      gtk_label_set_text(gui->lblLines, msg);
    }
  }

  /* Add time that these lines were processed */
  if (gui->line_times) {
    int    existing;
    time_t now;

    existing = k_circular_queue_size(gui->line_times);
    now = time(NULL);

    if (existing) {
      /* This last line that existed probably had something added.
         Update its time, unless this is the first line ever. */
      k_circular_queue_nth(gui->line_times, time_t, existing - 1) = now;
    }

    /* Add new times */
    n = lines - existing;
    if (n > 0) {
      k_circular_queue_push_copies(gui->line_times, &now, n);
    }
  }

  /* Scroll to end if appropriate */
  adjustment = gtk_scrolled_window_get_vadjustment(gui->scrolled_win);
  if (adjustment->value != adjustment->upper - adjustment->page_size
      && !(gui->world && gui->world->scrollOutput)) {
    return;
  }

  gtk_text_view_scroll_to_mark(gui->txtView,
                               gui->txtmark_end,
                               0,
                               TRUE,
                               0.5,
                               1);
}


static
void
parse_ansi_string(WorldGUI *gui)
{
  gchar **sequences;
  gchar **curr_seq;
  int    val;
  /* xterm256 state: used to parse xterm's 256 color cube.
     xterm sequences: ESC[38;5;Xm and ESC[48;5;Xm, fore and back resp.
     When this variable is 0, no xterm sequence parsing is being done.
     When 38 or 48 is seen, the variable is set to -38 or -48, respectively.
     When it is negative, and -5 is seen, its sign is changed.
     When it is positive, the color value is read and parsed.
  */
  gint   xterm256_state = 0;

  /* ESC[m, this means reset to default. */
  if (gui->ta.ansiseq_pos == 0) {
    reset_ansi_state(&gui->ta.state);
    return;
  }

  /* Add NULL terminator. */
  gui->ta.ansiseq_buffer[gui->ta.ansiseq_pos] = 0;

  sequences = g_strsplit_set((gchar *) gui->ta.ansiseq_buffer, ";", -1);
  curr_seq  = sequences;
  while (*curr_seq) {
    val = atoi(*curr_seq);

    if (xterm256_state < 0) {
      if (val == XTERM256_SECOND_CODE) {
        xterm256_state = -xterm256_state;
      } else {
        /* Ignore the strange sequence. */
        xterm256_state = 0;
      }
      ++curr_seq;
      continue;
    } else if (xterm256_state > 0) {
      if (xterm256_state == XTERM256_SET_FORE) {
        gui->ta.state.foreground_color_tag
          = get_tag_for_xterm256_color(gui, val, xterm256_state);
      } else if (xterm256_state == XTERM256_SET_BACK) {
        gui->ta.state.background_color_tag
          = get_tag_for_xterm256_color(gui, val, xterm256_state);
      }
      xterm256_state = 0;
      ++curr_seq;
      continue;
    }

    if (val == 0) {
      reset_ansi_state(&gui->ta.state);
    } else if (val >= ANSI_FIRST_FG_COLOR && val <= ANSI_LAST_FG_COLOR) {
      gui->ta.state.fg_color = val - ANSI_FIRST_FG_COLOR;
      gui->ta.state.foreground_color_tag = NULL;
    } else if (val == ANSI_FG_DEFAULT) {
      gui->ta.state.fg_color = ANSI_DEFAULT_COLOR_IDX;
      gui->ta.state.foreground_color_tag = NULL;
    } else if (val == ANSI_SET_FG_BOLD) {
      gui->ta.state.fg_base_idx = ANSI_BOLD_BASE_IDX;
    } else if (val == ANSI_BOLD_OFF) {
      gui->ta.state.fg_base_idx = ANSI_NORMAL_BASE_IDX;
    } else if (val >= ANSI_FIRST_BG_COLOR && val <= ANSI_LAST_BG_COLOR) {
      gui->ta.state.bg_color = val - ANSI_FIRST_BG_COLOR;
      gui->ta.state.background_color_tag = NULL;
    } else if (val == ANSI_BG_DEFAULT) {
      gui->ta.state.bg_color = ANSI_DEFAULT_COLOR_IDX;
      gui->ta.state.background_color_tag = NULL;
    } else if (val == ANSI_SET_BG_LIGHT) {
      gui->ta.state.bg_base_idx = ANSI_BOLD_BASE_IDX;
    } else if (val == ANSI_BOLD_BG_OFF) {
      gui->ta.state.bg_base_idx = ANSI_NORMAL_BASE_IDX;
    } else if (val == ANSI_SET_UNDERLINE) {
      gui->ta.state.underline = PANGO_UNDERLINE_SINGLE;
    } else if (val == ANSI_SET_DBLUNDERLINE) {
      gui->ta.state.underline = PANGO_UNDERLINE_DOUBLE;
    } else if (val == ANSI_UNDERLINE_OFF) {
      gui->ta.state.underline = PANGO_UNDERLINE_NONE;
    } else if (val == ANSI_SET_STRIKE) {
      gui->ta.state.strike = TRUE;
    } else if (val == ANSI_STRIKE_OFF) {
      gui->ta.state.strike = FALSE;
    } else if (val == ANSI_SET_ITALICS) {
      gui->ta.state.italics = TRUE;
    } else if (val == ANSI_ITALICS_OFF) {
      gui->ta.state.italics = FALSE;
    } else if (val == ANSI_SET_REVERSE) {
      gui->ta.state.reverse = TRUE;
    } else if (val == ANSI_REVERSE_OFF) {
      gui->ta.state.reverse = FALSE;
    } else if (val == ANSI_SET_HIDDEN) {
      gui->ta.state.hidden = TRUE;
    } else if (val == ANSI_HIDDEN_OFF) {
      gui->ta.state.hidden = FALSE;
    } else if (val == XTERM256_SET_FORE || val == XTERM256_SET_BACK) {
      xterm256_state = -val;
    } /* else ignored */
    ++curr_seq;
  }
  g_strfreev(sequences);

  gui->ta.ansiseq_pos = 0;
}


static
void
reset_ansi_state(TAState *state)
{
  /* Resets state to defaults, when ESC[0m or ESC[m is seen. */
  state->fg_color    = state->bg_color    = ANSI_DEFAULT_COLOR_IDX;
  state->fg_base_idx = state->bg_base_idx = ANSI_NORMAL_BASE_IDX;
  state->foreground_color_tag = state->background_color_tag = NULL;
  state->underline = PANGO_UNDERLINE_NONE;
  state->strike    = FALSE;
  state->italics   = FALSE;
  state->reverse   = FALSE;
  state->hidden    = FALSE;
}


static
void
ansitextview_verify_urls_and_add(WorldGUI     *gui,
                                 GtkTextIter  *pos,
                                 const guchar *text,
                                 gsize         len)
{
  static regex_t    regex;
  static gboolean   re_compiled = FALSE;
  static guchar     ctext[MAX_BUFFER + 1];
  static regmatch_t match;
  int               search_start;

  if (gui->ta.state.linedraw) {
    ansitextview_insert_vt100linedraw_string(gui, pos, text, len);
    return;
  }

  if (!re_compiled) {
    regcomp(&regex,
            "(((https?|ftp)://)|www\\.)[-A-Za-z0-9\\./?=&+%:#_~@,]*[-A-Za-z0-9\\/?=&+%:#_~]",
            REG_EXTENDED | REG_ICASE);
    re_compiled = TRUE;
  }

  /* We need a NULL-terminated string */
  memcpy(ctext, text, len);
  ctext[len] = '\0';

  search_start = 0;
  while (search_start < len) {
    if (regexec(&regex, (char *) (ctext + search_start), 1, &match, 0) == 0) {
      /* Text before the URL */
      ansitextview_append_formatted(gui->txtBuffer,
                                    pos,
                                    (gchar *) (text + search_start),
                                    match.rm_so,
                                    &gui->ta,
                                    NULL);
      /* The URL */
      ansitextview_append_formatted(gui->txtBuffer,
                                    pos,
                                    (gchar *) (text + search_start + match.rm_so),
                                    match.rm_eo - match.rm_so,
                                    &gui->ta,
                                    gui->txttag_url);
      /* Continue looking */
      search_start += match.rm_eo;
    } else {
      /* Text after the URL */
      ansitextview_append_formatted(gui->txtBuffer,
                                    pos,
                                    (gchar *) (text + search_start),
                                    len - search_start,
                                    &gui->ta,
                                    NULL);
      break;
    }
  }
}


static
void
ansitextview_insert_vt100linedraw_string(WorldGUI     *gui,
                                         GtkTextIter  *pos,
                                         const guchar *text,
                                         gsize         len)
{
  gint i;

  worldgui_determine_supported_chars(gui);

  for (i = 0; i < len; ++i) {
    if (text[i] < '_' || text[i] > '~') {
      ansitextview_append_formatted(gui->txtBuffer,
                                    pos,
                                    (const gchar *) text + i,
                                    1,
                                    &gui->ta,
                                    NULL);
    } else {
      switch (text[i]) {
      case '_': /* A blank */
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      " ",
                                      1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Geometric Shapes */
      case '`':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_geom_shapes
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Block elements */
      case 'a':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_block
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Control Pictures */
      case 'b': case 'c': case 'd': case 'e': case 'h': case 'i':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_control
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Latin-1 Supplement */
      case 'f': case 'g': case '}': case '~':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_l1_supplement
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Box drawing */
      case 'j': case 'k': case 'l': case 'm': case 'n': case 'q':
      case 't': case 'u': case 'v': case 'w': case 'x':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_box_drawing
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Miscellaneous Technical */
      case 'o': case 'p': case 'r': case 's':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_misc_tech
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Mathematical Operators */
      case 'y': case 'z': case '|':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_math
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;

      /* Greek and Coptic */
      case '{':
        ansitextview_append_formatted(gui->txtBuffer,
                                      pos,
                                      gui->ta.sup_greek
                                       ? linedrawingchars[text[i] - '_' - 1]
                                       : replacingchars[text[i] - '_' - 1],
                                      -1,
                                      &gui->ta,
                                      NULL);
        break;
      }
    }
  }
}


static
void
ansitextview_append_formatted(GtkTextBuffer  *buffer,
                              GtkTextIter    *iter,
                              const gchar    *text,
                              gsize           len,
                              TextAppearance *ta,
                              GtkTextTag     *extra_tag)
{
  gint        start_offset;
  GtkTextIter start_iter;
  gint        fg_color_idx;
  gint        bg_color_idx;

  start_offset = gtk_text_iter_get_offset(iter);

  gtk_text_buffer_insert(buffer,
                         iter,
                         text,
                         len);

  gtk_text_buffer_get_iter_at_offset(buffer, &start_iter, start_offset);

  /* Foreground & background */
  fg_color_idx = ta->state.fg_base_idx + ta->state.fg_color;
  bg_color_idx = ta->state.bg_base_idx + ta->state.bg_color;

  if (ta->state.foreground_color_tag) {
    gtk_text_buffer_apply_tag(buffer,
                              ta->state.foreground_color_tag,
                              &start_iter,
                              iter);
  } else {
    gint idx_to_use;

    if (ta->state.reverse ^ ta->state.hidden) {
      if (bg_color_idx == ANSI_DEFAULT_COLOR_IDX) {
        idx_to_use = ANSI_REVERSE_DEFAULT_IDX;
      } else if (bg_color_idx == ANSI_DEFAULT_BOLD_COLOR_IDX) {
        idx_to_use = ANSI_REVERSE_BOLD_DEFAULT_IDX;
      } else {
        idx_to_use = bg_color_idx;
      }
    } else {
      idx_to_use = fg_color_idx;
    }
    gtk_text_buffer_apply_tag(buffer,
                              ta->ansi_fore_tags[idx_to_use],
                              &start_iter,
                              iter);
  }

  if (ta->state.background_color_tag) {
    gtk_text_buffer_apply_tag(buffer,
                              ta->state.background_color_tag,
                              &start_iter,
                              iter);
  } else {
    gint idx_to_use;

    if (!ta->state.reverse) {
      idx_to_use = bg_color_idx;
    } else {
      if (fg_color_idx == ANSI_DEFAULT_COLOR_IDX) {
        idx_to_use = ANSI_REVERSE_DEFAULT_IDX;
      } else if (fg_color_idx == ANSI_DEFAULT_BOLD_COLOR_IDX) {
        idx_to_use = ANSI_REVERSE_BOLD_DEFAULT_IDX;
      } else {
        idx_to_use = fg_color_idx;
      }
    }
    gtk_text_buffer_apply_tag(buffer,
                              ta->ansi_back_tags[idx_to_use],
                              &start_iter,
                              iter);
  }

  /* Special attributes */
  if (ta->state.underline == PANGO_UNDERLINE_SINGLE) {
    gtk_text_buffer_apply_tag(buffer, ta->underline_tag, &start_iter, iter);
  } else if (ta->state.underline == PANGO_UNDERLINE_DOUBLE) {
    gtk_text_buffer_apply_tag(buffer, ta->dblunderline_tag, &start_iter, iter);
  }
  if (ta->state.strike) {
    gtk_text_buffer_apply_tag(buffer, ta->strike_tag, &start_iter, iter);
  }
  if (ta->state.italics) {
    gtk_text_buffer_apply_tag(buffer, ta->italics_tag, &start_iter, iter);
  }

  /* Any extra tag passed. */
  if (extra_tag) {
    gtk_text_buffer_apply_tag(buffer, extra_tag, &start_iter, iter);
  }
}


void
ansitextview_get_char_size(WorldGUI *gui,
                           gint     *char_height,
                           gint     *char_width)
{
  PangoLayout    *layout;
  PangoRectangle  logical_rect;

  layout = gtk_widget_create_pango_layout(GTK_WIDGET(gui->txtView), "W");
  pango_layout_get_pixel_extents(layout, NULL, &logical_rect);
  *char_height = logical_rect.height;
  *char_width  = logical_rect.width;

  g_object_unref(layout);
}


void
ansitextview_get_size(WorldGUI *gui, guint *lines, guint *cols)
{
  GdkRectangle rect;
  gint         char_height;
  gint         char_width;

  /* Get the size of the text view */
  gtk_text_view_get_visible_rect(gui->txtView, &rect);

  ansitextview_get_char_size(gui, &char_height, &char_width);
  *lines = rect.height / char_height;
  *cols  = rect.width / char_width;
}


void
ansitextview_update_color_tags(WorldGUI *gui, World *world)
{
  int i;

  if (!world) {
    fprintf(stderr, "BUG: update_color_tag without world.\n");
    return;
  }

  /* First time called, the tags have not been created yet. */
  if (!gui->ta.ansi_fore_tags) {
    /* Allocate space for colors */
    gui->ta.ansi_fore_tags = g_new0(GtkTextTag *, ANSI_TABLE_SIZE);
    gui->ta.ansi_back_tags = g_new0(GtkTextTag *, ANSI_TABLE_SIZE);
    for (i = 0; i < ANSI_TABLE_SIZE; ++i) {
      gui->ta.ansi_fore_tags[i] = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                          NULL,
                                                          NULL);
      gui->ta.ansi_back_tags[i] = gtk_text_buffer_create_tag(gui->txtBuffer,
                                                          NULL,
                                                          NULL);
    }
  }

  /* Update normal foreground/background colors */
  for (i = 0; i < ANSI_N_COLORS; ++i) {
    g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[i]),
                 "foreground-gdk", &world->ansicolors[i],
                 NULL);
    g_object_set(G_OBJECT(gui->ta.ansi_back_tags[i]),
                 "background-gdk", &world->ansicolors[i],
                 NULL);
  }
  for (i = ANSI_BOLD_BASE_IDX; i < (ANSI_BOLD_BASE_IDX + ANSI_N_COLORS); ++i) {
    g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[i]),
                 "foreground-gdk", &world->ansicolors[i - ANSI_TABLE_OFFSET],
                 NULL);
    g_object_set(G_OBJECT(gui->ta.ansi_back_tags[i]),
                 "background-gdk", &world->ansicolors[i - ANSI_TABLE_OFFSET],
                 NULL);
  }

  /* Update default colors */
  g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[ANSI_DEFAULT_COLOR_IDX]),
               "foreground-gdk", world->deffore,
               NULL);
  g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[ANSI_BOLD_BASE_IDX + ANSI_DEFAULT_COLOR_IDX]),
               "foreground-gdk", world->defbold,
               NULL);

  g_object_set(G_OBJECT(gui->ta.ansi_back_tags[ANSI_DEFAULT_COLOR_IDX]),
               "background-gdk", world->defback,
               NULL);
  g_object_set(G_OBJECT(gui->ta.ansi_back_tags[ANSI_BOLD_BASE_IDX + ANSI_DEFAULT_COLOR_IDX]),
               "background-gdk", world->defboldback,
               NULL);

  /* Update default colors for reverse video */
  g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[ANSI_REVERSE_DEFAULT_IDX]),
               "foreground-gdk", world->defback,
               NULL);
  g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[ANSI_REVERSE_BOLD_DEFAULT_IDX]),
               "foreground-gdk", world->defboldback,
               NULL);

  g_object_set(G_OBJECT(gui->ta.ansi_back_tags[ANSI_REVERSE_DEFAULT_IDX]),
               "background-gdk", world->deffore,
               NULL);
  g_object_set(G_OBJECT(gui->ta.ansi_back_tags[ANSI_REVERSE_BOLD_DEFAULT_IDX]),
               "background-gdk", world->defbold,
               NULL);

  /* Update use bold state */
  for (i = ANSI_BOLD_BASE_IDX;
       i < (ANSI_BOLD_BASE_IDX + ANSI_N_COLORS + 1);
       ++i) {
    g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[i]),
                 "weight", (world->usebold
                            ? PANGO_WEIGHT_BOLD
                            : PANGO_WEIGHT_NORMAL),
                 NULL);
  }
  g_object_set(G_OBJECT(gui->ta.ansi_fore_tags[ANSI_REVERSE_BOLD_DEFAULT_IDX]),
               "weight", (world->usebold
                          ? PANGO_WEIGHT_BOLD
                          : PANGO_WEIGHT_NORMAL),
               NULL);

  /* Update default colors in the TextView */
  gtk_widget_modify_base(GTK_WIDGET(gui->txtView),
                         GTK_STATE_NORMAL,
                         world->defback);
  gtk_widget_modify_text(GTK_WIDGET(gui->txtView),
                         GTK_STATE_NORMAL,
                         world->deffore);
}


gboolean
ansitextview_prune_extra_lines(gpointer data)
{
  WorldGUI *gui = (WorldGUI *) data;
  gint      lines;

  if (!gui->world) {
    return TRUE;
  }

  lines = gtk_text_buffer_get_line_count(gui->txtBuffer);
  if (lines > gui->world->buffer_lines) {
    GtkTextIter start;
    GtkTextIter end;
    int         to_del;

    to_del = lines - gui->world->buffer_lines;

    gtk_text_buffer_get_start_iter(gui->txtBuffer, &start);
    gtk_text_buffer_get_iter_at_line(gui->txtBuffer, &end, to_del);
    gtk_text_buffer_delete(gui->txtBuffer, &start, &end);

    /* Remove times of the deleted lines */
    if (gui->line_times) {
      k_circular_queue_pop(gui->line_times, to_del);
      gui->world->deleted_lines += to_del;
    }
  }

  return TRUE;
}


guchar *
strip_ansi(const guchar *original, int len)
{
  /* This is to be called on complete strings, so parsing always
     starts in DATA mode */
  ANSIFSMState state = ANSIFSM_DATA;
  guchar *stripped;
  int     original_pos;
  int     stripped_pos;

  /* In the worst case, the result after processing will have the
     same size + a null string terminator. */
  stripped = g_malloc(len + 1);
  original_pos = 0;
  stripped_pos = 0;

  while (original_pos < len) {
    switch (state) {
    case ANSIFSM_DATA:
      if (original[original_pos] == ANSI_ESC) {
        state = ANSIFSM_ESC;
      } else {
        stripped[stripped_pos++] = original[original_pos];
      }
      break;

    case ANSIFSM_ESC:
      if (original[original_pos] == '[') {
        state = ANSIFSM_ANSISEQ;
      } else {
        stripped[stripped_pos++] = ANSI_ESC;        /* Add ESC again */
        stripped[stripped_pos++] = original[original_pos];
        state = ANSIFSM_DATA;
      }
      break;

    case ANSIFSM_ANSISEQ:
      if (isalpha(original[original_pos])) {
        state = ANSIFSM_DATA;
      }
      break;

    default:
      break;
    }
    ++original_pos;
  }

  stripped[stripped_pos] = '\0';

  return stripped;
}


static
GtkTextTag *
get_tag_for_color(WorldGUI *gui,
                  guint8 r, guint8 g, guint8 b,
                  gboolean  is_back)
{
  gint        rgb;
  GtkTextTag *tag;
  GHashTable *hash;
  GdkColor    color;

  hash = is_back ? gui->ta.rgb_back_tags : gui->ta.rgb_fore_tags;
  rgb = (r << 16) | (g << 8) | b;

  tag = g_hash_table_lookup(hash, GINT_TO_POINTER(rgb));
  if (tag) {
    return tag;
  }

  /* Create if it does not exist */
  color.red   = (r << 8) | r;
  color.green = (g << 8) | g;
  color.blue  = (b << 8) | b;
  if (!is_back) {
    tag = gtk_text_buffer_create_tag(gui->txtBuffer,
                                     NULL,
                                     "foreground-gdk", &color,
                                     NULL);
  } else {
    tag = gtk_text_buffer_create_tag(gui->txtBuffer,
                                     NULL,
                                     "background-gdk", &color,
                                     NULL);
  }

  gtk_text_tag_set_priority(tag, 0);
  g_hash_table_insert(hash, GINT_TO_POINTER(rgb), tag);
  return tag;
}


static
GtkTextTag *
get_tag_for_xterm256_color(WorldGUI *gui,
                           gint      x256idx,
                           gint      code_used)
{
  guint8 r, g, b;

  if (x256idx <= 7) {           /* Normal ANSI colors */
    if (code_used == XTERM256_SET_FORE) {
      return gui->ta.ansi_fore_tags[x256idx];
    } else if (code_used == XTERM256_SET_BACK) {
      return gui->ta.ansi_back_tags[x256idx];
    }
  }

  if (x256idx <= 15) {   /* Highlighted ANSI colors */
    if (code_used == XTERM256_SET_FORE) {
      return gui->ta.ansi_fore_tags[x256idx + ANSI_BOLD_BASE_IDX];
    } else if (code_used == XTERM256_SET_BACK) {
      return gui->ta.ansi_back_tags[x256idx + ANSI_BOLD_BASE_IDX];
    }
  }

  if (x256idx <= 231) {  /* 6x6x6 rgb color cube */
    guint8 ri, gi, bi;

    x256idx -= 16;
    ri = x256idx / 36;
    x256idx %= 36;
    gi = x256idx / 6;
    x256idx %= 6;
    bi = x256idx;

    r = xterm256color[ri];
    g = xterm256color[gi];
    b = xterm256color[bi];
  } else {
    x256idx -= 232;

    r = xterm256gray[x256idx];
    g = b = r;
  }

  return get_tag_for_color(gui, r, g, b, code_used == XTERM256_SET_BACK);
}
