/***************************************************************************
 *   Copyright (C) 2007-2009 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KGmailNotifier.                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "trayicon.h"

#include <QPainter>
#include <QToolTip>

#include <KColorScheme>
#include <KGlobalSettings>

// ------------------------------------------------------------------------------------------------------------------- //
//    Construction & Destruction                                                                                       //
// ------------------------------------------------------------------------------------------------------------------- //

TrayIcon::TrayIcon(QWidget *parent)
    : KSystemTrayIcon(parent)
{
    connect(this, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(slotActivated(QSystemTrayIcon::ActivationReason)));
}

// ------------------------------------------------------------------------------------------------------------------- //

TrayIcon::~TrayIcon()
{
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Public methods                                                                                                   //
// ------------------------------------------------------------------------------------------------------------------- //

void TrayIcon::setIcon(const QPixmap &icon, const bool showNumber, const unsigned int number)
{
    if (showNumber && number > 0) // Display number on systray
    {
        // The following code is inspired by Akregator's tray icon code
        // and used in order to provide a more consistent systray look
        int iconWidth = icon.width();
        if (iconWidth == 0)
            return;

        QString numberStr = QString::number(number);
        QFont f = KGlobalSettings::generalFont();
        f.setBold(true);

        // Scale the number to fit nicely inside the icon
        float pointSize = f.pointSizeF();
        QFontMetrics fm(f);
        int width = fm.width(numberStr);
        if (width > (iconWidth - 2))
        {
            pointSize *= float(iconWidth - 2) / float(width);
            f.setPointSizeF(pointSize);
        }
        
        // Compose a new icon
        QPixmap finalIcon = icon;
        QPainter p(&finalIcon);
        p.setFont(f);
        KColorScheme scheme(QPalette::Active, KColorScheme::View);

        // Create a lighter background frame for the number
        fm = QFontMetrics(f);
        QRect boundingRect = fm.tightBoundingRect(numberStr);
        boundingRect.setWidth(boundingRect.width() + 2);
        boundingRect.setHeight(boundingRect.height() + 2);
        boundingRect.moveTo((iconWidth - boundingRect.width()) / 2,
                            (iconWidth - boundingRect.height()) / 2);

        // Draw the frame
        p.setOpacity(0.7f);
        p.setBrush(scheme.background(KColorScheme::LinkBackground));
        p.setPen(Qt::NoPen);
        p.drawRoundedRect(boundingRect, 2.0, 2.0);

        // Draw the number
        p.setOpacity(1.0f);
        p.setBrush(Qt::NoBrush);
        p.setPen(Qt::darkBlue);
        p.drawText(finalIcon.rect(), Qt::AlignCenter, numberStr);

        // Diplay the final icon
        this->KSystemTrayIcon::setIcon(QIcon(finalIcon));
    }
    else
        this->KSystemTrayIcon::setIcon(icon);
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Private slots                                                                                                    //
// ------------------------------------------------------------------------------------------------------------------- //

void TrayIcon::slotActivated(QSystemTrayIcon::ActivationReason reason)
{
    if (reason == Trigger)
        emit clicked();
}

// ------------------------------------------------------------------------------------------------------------------- //

#include "trayicon.moc"
