/***************************************************************************
 *   Copyright (C) 2007, 2008 by Marcel Hasler                             *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "accountmanager.h"
#include "configuration.h"

#include <iostream>
using namespace std;

AccountManager::AccountManager()
	: QObject()
{
	// Set up
	m_atomParser = new AtomParser;
	m_lastMessageCount = 0;
	m_currentMessageCount = 0;

	m_jobTimeoutTimer = new QTimer();

	m_atomFileUrl = new KURL();
	m_atomFileUrl->setProtocol("https");
	m_atomFileUrl->setHost("mail.google.com");
	m_atomFileUrl->setPath("/mail/feed/atom");

	m_tmpFileUrl = new KURL();
	m_tmpFileUrl->setProtocol("file");
	m_tmpFileUrl->setPath("/tmp/atomtmp");

	this->updateConfiguration();
}


AccountManager::~AccountManager()
{
	if (m_atomParser)
		delete m_atomParser;

	if (m_fileCopyJob)
		delete m_fileCopyJob;

	if (m_atomFileUrl)
		delete m_atomFileUrl;

	if (m_tmpFileUrl)
		delete m_tmpFileUrl;

	if (m_jobTimeoutTimer)
		delete m_jobTimeoutTimer;
}


bool AccountManager::isValid()
{
	return (m_accountName != "" && m_accountPassword != "");
}


void AccountManager::checkMail()
{
	if (m_accountName == "" || m_accountPassword == "")
	{
		emit connectionError("failed");
		return;
	}

	// Start downloading the atom file and connect the slots for parsing
	m_fileCopyJob = KIO::file_copy(*m_atomFileUrl, *m_tmpFileUrl, -1, true, false, false);
	m_fileCopyJob->addMetaData("no-cache", "true"); // Does this really help?
	connect(m_fileCopyJob, SIGNAL(result(KIO::Job *)), this, SLOT(slotParseData(KIO::Job *)));

	// Start timer for connection timeout
	connect(m_jobTimeoutTimer, SIGNAL(timeout()), this, SLOT(slotAbortJob()));
	m_jobTimeoutTimer->start(m_connectionTimeout, true);
}


void AccountManager::updateConfiguration()
{
	m_accountName = Configuration().readEntry("Username");
	m_connectionTimeout = (Configuration().readNumEntry("CheckInterval") - 1) * 1000;
	m_accountPassword = Configuration().readEntry("Password");

	m_atomFileUrl->setUser(m_accountName);
	m_atomFileUrl->setPass(m_accountPassword);
}


//////////////////////
// Access functions //
//////////////////////
unsigned int AccountManager::messageCount() const
{
	return m_currentMessageCount;
}


bool AccountManager::hasNewMail() const
{
	if (m_currentMessageCount > m_lastMessageCount)
		return true;
	else
		return false;
}


const QString& AccountManager::lastSender() const
{
	return m_lastSender;
}


const QString& AccountManager::lastSubject() const
{
	return m_lastSubject;
}


///////////////////
// Private slots //
///////////////////
void AccountManager::slotAbortJob()
{
	m_jobTimeoutTimer->stop();
	m_fileCopyJob->kill();
	emit connectionError("timeout");
}


void AccountManager::slotParseData(KIO::Job *job)
{
	m_jobTimeoutTimer->stop();

	if (job->error())
	{
		emit connectionError("failed");
		return;
	}

	bool success = m_atomParser->parseFile(m_tmpFileUrl->path());
	if (success)
	{
		m_lastMessageCount = m_currentMessageCount;
		m_currentMessageCount = m_atomParser->messageCount();

		if (m_currentMessageCount == 0)
		{
			m_lastSender = "";
			m_lastSubject = "";
		}
		else
		{
			m_lastSender = m_atomParser->lastSender();
			m_lastSubject = m_atomParser->lastSubject();
		}
	}
	else
		 cout << "An error occured while parsing the atom feed!" << endl;

	KIO::file_delete(*m_tmpFileUrl, false);
	emit connectionSuccessful();
}


#include "accountmanager.moc"
