/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "browser/view.h"
#include "browser/listview.h"
#include "browser/treeview.h"
#include "browser/actions.h"

#include "kftpbookmarks.h"
#include "misc/config.h"
#include "kftpsession.h"
#include "ftpsocket.h"

#include "misc.h"

#include <klocale.h>
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <kcombobox.h>
#include <kaction.h>
#include <kmessagebox.h>
#include <ksslinfodlg.h>

#include <qdir.h>
#include <qlabel.h>
#include <qlayout.h>

using namespace KFTPGrabberBase;

namespace KFTPWidgets {

namespace Browser {

View::View(QWidget *parent, const char *name, KFTPClientThr *p_ftpClient, ::KFTPSession *session)
 : QWidget(parent, name), m_session(session)
{
  m_ftpClient = p_ftpClient;
  m_connTimer = new QTimer(this);

  // Create the GUI
  init();
  populateToolbar();
  
  // Hide/show the tree acoording to configuration
  setTreeVisible(KFTPCore::Config::showTree());

  // Let us be up to date with bookmark changes
  connect(FTPBookmarkManager, SIGNAL(update()), this, SLOT(updateBookmarks()));
  
  // Some other stuff
  connect(m_ftpClient, SIGNAL(loginComplete(bool)), this, SLOT(slotFtpUpdate()));
  connect(m_ftpClient, SIGNAL(disconnectDone()), this, SLOT(slotFtpUpdate()));
  connect(m_connTimer, SIGNAL(timeout()), this, SLOT(slotDurationUpdate()));
  
  // Monitor state changes
  connect(m_ftpClient, SIGNAL(stateChanged(KFTPNetwork::SocketState)), this, SLOT(slotSocketStateChanged(KFTPNetwork::SocketState)));
  connect(m_ftpClient, SIGNAL(logUpdate(int, const QString&)), this, SLOT(updateActions()));
  
  // Config updates to hide/show the tree
  connect(KFTPCore::Config::self(), SIGNAL(configChanged()), this, SLOT(slotConfigUpdate()));
}


View::~View()
{
}

void View::init()
{
  // Init actions
  m_actions = new Actions(this);
  m_actions->initActions();

  // Layout
  QVBoxLayout *layout = new QVBoxLayout(this);

  // Create the toolbars
  m_toolBarFirst = new KToolBar(this, "first toolbar", false, false);
  m_toolBarSecond = new KToolBar(this, "second toolbar", false, false);

  m_toolBarFirst->setEnableContextMenu(false);
  m_toolBarFirst->setMovingEnabled(false);
  m_toolBarFirst->setFullSize(true);

  m_toolBarSecond->setEnableContextMenu(false);
  m_toolBarSecond->setMovingEnabled(false);
  m_toolBarSecond->setFullSize(true);

  // Create the erase button
  m_toolBarSecond->insertButton(QApplication::reverseLayout() ? "clear_left" :"locationbar_erase", 0, SIGNAL(clicked()), this, SLOT(slotHistoryEraseClicked()), true);
  
  // Create the labels
  QLabel *pathLabel = new QLabel(i18n("Path: "), m_toolBarSecond);
  m_toolBarSecond->insertWidget(1, 35, pathLabel);

  // Create the history combo
  m_toolBarSecond->insertCombo(KFTPCore::Config::defLocalDir(), 2, true, SIGNAL(activated(const QString&)), this, SLOT(slotHistoryActivated(const QString&)), true);
  m_historyCombo = m_toolBarSecond->getCombo(2);
  m_historyCombo->setMaxCount(25);
  m_historyCombo->setSizeLimit(25);
  m_historyCombo->setDuplicatesEnabled(false);
  m_historyCombo->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed));

  // Do some more stuff
  m_toolBarSecond->setItemAutoSized(1, true);
  m_toolBarSecond->setStretchableWidget(m_historyCombo);
  m_toolBarSecond->updateRects(true);
  m_toolBarFirst->updateRects(true);

  // Create a splitter
  m_splitter = new QSplitter(this);
  m_splitter->setOpaqueResize(true);
  m_splitter->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

  // Create a status bar
  QHBoxLayout *statusLayout = new QHBoxLayout(this);

  m_connDurationMsg = new QLabel(this);
  m_connDurationMsg->setAlignment(AlignAuto | AlignVCenter | AlignHCenter | ExpandTabs);
  m_connDurationMsg->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
  m_connDurationMsg->setMinimumWidth(100);

  m_sslIcon = new QPushButton(this);
  m_sslIcon->setFlat(true);
  slotFtpUpdate();

  connect(m_sslIcon, SIGNAL(clicked()), this, SLOT(slotDisplayCertInfo()));

  m_statusMsg = new QLabel(this);
  m_statusMsg->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
  m_statusMsg->setText(i18n("Idle."));

  statusLayout->addWidget(m_statusMsg, 1);
  statusLayout->addWidget(m_connDurationMsg);
  statusLayout->addWidget(m_sslIcon);

  // Add toolbars to the layout
  layout->addWidget(m_toolBarFirst);
  layout->addWidget(m_toolBarSecond);
  layout->addWidget(m_splitter, 10);
  layout->addLayout(statusLayout);

  // Now add some stuff to the splitter
  m_list = new ListView(m_splitter, m_ftpClient);
  m_splitter->moveToLast(m_list);

  // Create the tree view
  m_tree = new TreeView(m_splitter);
  m_splitter->moveToFirst(m_tree);

  m_list->setTreeView(m_tree);
  m_list->openURL(KURL(KFTPCore::Config::defLocalDir()));
  m_list->setFilter(QString::null);
}

void View::slotConfigUpdate()
{
  setTreeVisible(KFTPCore::Config::showTree());
}

void View::setTreeVisible(bool visible)
{
  if (visible)
    m_tree->show();
  else
    m_tree->hide();
  
  m_actions->m_toggleTreeViewAction->setChecked(visible);
}

void View::populateToolbar()
{
  // Add the actions to the toolbar
  m_actions->m_siteChangeAction->plug(m_toolBarFirst);

  m_toolBarFirst->insertSeparator();

  m_actions->m_goUpAction->plug(m_toolBarFirst);
  m_actions->m_goBackAction->plug(m_toolBarFirst);
  m_actions->m_goForwardAction->plug(m_toolBarFirst);
  m_actions->m_reloadAction->plug(m_toolBarFirst);

  m_toolBarFirst->insertSeparator();

  m_actions->m_goHomeAction->plug(m_toolBarFirst);
  m_actions->m_createDirAction->plug(m_toolBarFirst);
  
  m_toolBarFirst->insertSeparator();
  
  m_actions->m_abortAction->plug(m_toolBarFirst);
  m_actions->m_toggleTreeViewAction->plug(m_toolBarFirst);
  
  m_toolBarFirst->insertSeparator();
  
  m_actions->m_moreActions->plug(m_toolBarFirst);
}

void View::updateActions()
{
  // Force action update
  m_actions->updateActions();
}

void View::updateBookmarks()
{
  // Repopulate bookmarks menu on updates
  m_actions->m_connectAction->popupMenu()->clear();
  FTPBookmarkManager->guiPopulateBookmarksMenu(m_actions->m_connectAction, QDomNode(), false, m_session);
}

void View::setCompanion(View *companion)
{
  // Set the list's companion
  if (!companion)
    m_list->setCompanion(0L);
  else
    m_list->setCompanion(companion->m_list);
}

void View::slotHistoryActivated(const QString &text)
{
  KURL newURL = m_list->m_curURL;
  newURL.setPath(text);

  m_list->openURL(newURL);
}

void View::slotHistoryEraseClicked()
{
  m_historyCombo->setCurrentText(QString::null);
  m_historyCombo->setFocus();
}

void View::slotDisplayCertInfo()
{
  KSSLInfoDlg *sslInfo = 0;
  
  if (m_ftpClient->getClient()->getProtocol() == "ftp" && m_ftpClient->getClient()->getConfig("using_ssl") == 1) {
    // Show the dialog describing current X509 certificate
    KFTPNetwork::FtpSocket *sock = static_cast<KFTPNetwork::FtpSocket*>(m_ftpClient->getClient());
    
    sslInfo = new KSSLInfoDlg(true, this);
    sslInfo->setup(*sock->SSLGetObject(),
                   sock->getClientInfoUrl().host(),
                   m_list->m_curURL.path());
  } else if (m_ftpClient->getClient()->getProtocol() == "sftp") {
    // FIXME how to display certificate description ?
    KMessageBox::information(this, i18n("This is a SSH encrypted connection. No certificate info is currently available."));
    return;
  } else {
    sslInfo = new KSSLInfoDlg(false, this);
  }
  
  sslInfo->exec();
}

void View::slotDurationUpdate()
{
  m_connDuration = m_connDuration.addSecs(1);
  m_connDurationMsg->setText(m_connDuration.toString("hh:mm:ss"));
}

void View::slotFtpUpdate()
{
  // Change encryption icon
  m_sslIcon->setIconSet(SmallIconSet(m_ftpClient->getClient()->isEncrypted() ? "encrypted" : "decrypted"));
  m_sslIcon->setEnabled(m_ftpClient->getClient()->isConnected());

  // Start or stop the duration timer
  if (m_ftpClient->getClient()->isConnected()) {
    m_connTimer->start(1000);
    m_connDuration.setHMS(0, 0, 0);
  } else {
    m_connTimer->stop();
    m_connDurationMsg->setText("");
  }
  
  // Reset selected charset to default
  KPopupMenu *menu = m_actions->m_changeEncodingAction->popupMenu();
  menu->setItemChecked(m_actions->m_defaultCharsetOption, true);
  menu->setItemChecked(m_actions->m_curCharsetOption, false);
  m_actions->m_curCharsetOption = m_actions->m_defaultCharsetOption;
}

void View::slotSocketStateChanged(KFTPNetwork::SocketState state)
{
  QString statusMsg;
  
  // Show the current state
  switch (state) {
    case KFTPNetwork::S_IDLE: statusMsg = i18n("Idle."); break;
    case KFTPNetwork::S_CONNECT: statusMsg = i18n("Connecting..."); break;
    case KFTPNetwork::S_LOGIN: statusMsg = i18n("Logging in..."); break;
    case KFTPNetwork::S_TRANSFER: statusMsg = i18n("Transferring..."); break;
    case KFTPNetwork::S_LIST: statusMsg = i18n("Fetching directory listing..."); break;
    case KFTPNetwork::S_SCAN: statusMsg = i18n("Scanning directories..."); break;
    case KFTPNetwork::S_REMOVE: statusMsg = i18n("Removing directories..."); break;
    default: statusMsg = QString::null; break;
  }
  
  m_statusMsg->setText(statusMsg);
  
  // Enable or disable the view
  if (state != KFTPNetwork::S_IDLE) {
    m_tree->setEnabled(false);
    m_list->setEnabled(false);
    m_toolBarSecond->setEnabled(false);
  } else {
    m_tree->setEnabled(true);
    m_list->setEnabled(true);
    m_toolBarSecond->setEnabled(true);
  }
  
  // Update actions
  updateActions();
}

}

}

#include "view.moc"
