/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPFILEDIRLISTVIEW_H
#define KFTPFILEDIRLISTVIEW_H

#include "kftpclientthread.h"
#include "widgets/listview.h"

#include <klistview.h>
#include <kdirlister.h>
#include <kfileitem.h>
#include <kurl.h>
#include <kurldrag.h>

#include <qvaluelist.h>

namespace KFTPWidgets {

namespace Browser {

class History
{
public:
    History();
    
    void newEntry(const KURL &url);
    KURL getCurrent();
    QString getSelected();

    void goBack();
    void goForward();

    QValueList<QPair<KURL, QString> > m_history;
    QValueList<QPair<KURL, QString> >::iterator m_historyPos;
};

class TreeView;
class ListViewItem;

typedef QPtrList<ListViewItem> ListItems;

/**
 * @author Jernej Kos
 */
class ListView : public KFTPWidgets::ListView
{
Q_OBJECT
friend class Actions;
friend class View;
friend class KFTPSession;
public:
    enum LocationInfo {
      Up,
      Back,
      Forward,
      Reload,
      Home
    };

    ListView(QWidget *parent = 0, KFTPClientThr *p_ftpClient = 0);
    ~ListView();

    void openURL(const KURL url, bool addHistory = true);
    void setCompanion(ListView *companion);
    void setTreeView(TreeView *tree);
    void setHomeURL(const KURL url);
    void setFilter(const QString &filter);

    void resetView(const KURL url, const QString &rootText = 0);

    void moveTo(LocationInfo location);
    bool canMoveTo(LocationInfo location);

    KURL::List getCurrentSelection();
    ListItems getCurrentSelectionLI();
    
    /* Offline mode methods */
    bool isOfflineMode() { return m_offlineMode; }
    void setOfflineUrl(const KURL &url) { m_offlineURL = url; }
private:
    KFTPClientThr *m_ftpClient; /* Ftp client if remote */
    KDirLister *m_dirLister; /* Dir lister if local  */

    ListView *m_companion; /* The other list view */
    TreeView *m_tree; /* The treeview to use (or not) */

    /* Position information */
    KURL m_homeURL;
    KURL m_curURL;
    KURL m_reqURL;
    History m_history;

    /* Filter */
    QString m_filter;
    bool m_addHistory;

    /* Offline mode */
    KURL m_offlineURL;
    FTPDirList *m_offlineList;
    bool m_offlineMode;

    bool canTransfer(ListView *caller = 0);
    bool canLocalActions();
    int transferMode();

    /* Drag & drop support */
    QTimer *m_dragOpenTimer;
    QListViewItem *m_dropItem;
    QDragObject *m_dragObject;
    
    /**
     * Reselects the folder that was previously entered by scrolling the current
     * listview.
     */
    void smartFolderReselect();
    
    QDragObject *dragObject();
    void startDrag();
    bool acceptDrag(QDropEvent *e);
    void contentsDragEnterEvent(QDragEnterEvent *e);
    void contentsDragMoveEvent(QDragMoveEvent *e);
    void contentsDragLeaveEvent(QDragLeaveEvent*);
    void contentsDropEvent(QDropEvent *e);
private slots:
    void slotDirListerNewItems(const KFileItemList &items);
    void slotDirListerDeleteItem(KFileItem *item);
    void slotDirListerClear();
    void slotDirListerRefreshItems(const KFileItemList &items);

    void slotFtpConnect(bool);
    void slotFtpDisconnect();
    void slotFtpFinished(CommandType cmdType);
    void slotFtpErrorHandler(KFTPNetwork::Error error);

    void slotListClicked();
    void slotTreeClicked(const KURL url);
    void slotItemRenamed(QListViewItem *item, const QString &str, int col);

    void slotContextMenu(QListViewItem*, const QPoint&);

    void slotDragTimeout();
};

class ListViewItem : public KListViewItem
{
friend class ListView;
friend class Actions;
public:
    ListViewItem(ListView *parent);
    ListViewItem(ListViewItem *parent);

    ListViewItem(ListView *parent, KFileItem *item);
    ListViewItem(ListViewItem *parent, KFileItem *item);

    ListViewItem(ListView *parent, const FTPDirectoryItem &item, const KURL &url);

    void readFromFileItem(KFileItem *item);
    void createColumns();

    int compare(QListViewItem *i, int col, bool asc) const;
    virtual void paintCell( QPainter *p, const QColorGroup &cg, int column, int width, int alignment);
private:
    FTPDirectoryItem m_dirObject;
    KURL m_url;
    bool m_directory;
    bool m_paintOffline;

    void *m_filePtr;
};

}

}

#endif
