/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include <math.h>

#include "kftpqueue.h"
#include "kftpbookmarks.h"
#include "kftpgrabber.h"
#include "kftpsystemtray.h"
#include "kftpqueueprocessor.h"
#include "kftpsession.h"

#include "misc/config.h"

#include <kmessagebox.h>
#include <klocale.h>
#include <kio/renamedlg.h>
#include <kdiskfreesp.h>
#include <kfileitem.h>
#include <kstaticdeleter.h>
#include <kservice.h>
#include <kuserprofile.h>
#include <kstandarddirs.h>
#include <krun.h>
#include <kmdcodec.h>

#include <qapplication.h>
#include <qregexp.h>
#include <qdir.h>
#include <qobjectlist.h>
#include <qfile.h>

namespace KFTPQueue {

QueueScannerThread::QueueScannerThread(KFTPSession *session, Transfer *item)
  : QThread(), m_session(session), m_item(item)
{
}

void QueueScannerThread::run()
{
  // First scan the folder
  scanFolder(m_item);

  // We are done, post event to notify the GUI
  QueueScannerThreadEvent *e = new QueueScannerThreadEvent(m_item);
  qApp->postEvent(m_session, e);
}

void QueueScannerThread::scanFolder(Transfer *parent)
{
  QDir fs(parent->getSourceUrl().path());
  fs.setFilter(QDir::Readable | QDir::Hidden | QDir::All);

  const QFileInfoList *p_list = fs.entryInfoList();
  QFileInfoListIterator i(*p_list);
  QFileInfo *file;

  while ((file = i.current()) != 0) {
    ++i;

    if (file->fileName() == "." || file->fileName() == "..") continue;
    if (KFTPCore::Config::skipEmptyFiles() && file->size() == 0) continue;

    // Spawn transfer
    KURL destUrl = parent->getDestUrl();
    destUrl.addPath(file->fileName());

    // This is needed, since QFileInfo works with uint for the filesize
    filesize_t realSize = KFileItem(KFileItem::Unknown, KFileItem::Unknown, file->absFilePath()).size();

    if (file->isDir()) {
      // Directory
      KFTPQueue::TransferDir *transfer = new KFTPQueue::TransferDir(parent);
      transfer->setSourceUrl(KURL(file->absFilePath()));
      transfer->setDestUrl(destUrl);
      transfer->addSize(realSize);
      transfer->setTransferType(parent->getTransferType());
      transfer->setId(KFTPQueue::Manager::self()->m_lastQID++);

      qApp->lock();
      emit KFTPQueue::Manager::self()->newTransfer(transfer);
      qApp->unlock();

      // Call this function in recursion
      scanFolder(transfer);

      if (KFTPCore::Config::skipEmptyDirs() && !transfer->children()) {
        qApp->lock();
        KFTPQueue::Manager::self()->removeTransfer(transfer);
        qApp->unlock();
      }
    } else {
      // Spawn a new child transfer
      if (KFTPCore::Config::enablePrioList() && KFTPCore::Config::skipNoQueue() && KFTPCore::Config::self()->getFilePriority(file->absFilePath()) == PRIO_SKIP)
        continue;

      if (KFTPCore::Config::skipEmptyFiles() && realSize == 0)
        continue;

      KFTPQueue::TransferFile *transfer = new KFTPQueue::TransferFile(parent);
      transfer->setSourceUrl(KURL(file->absFilePath()));
      transfer->setDestUrl(destUrl);
      transfer->addSize(realSize);
      transfer->setTransferType(parent->getTransferType());
      transfer->setId(KFTPQueue::Manager::self()->m_lastQID++);

      qApp->lock();
      KFTPQueue::Manager::self()->newTransfer(transfer);
      qApp->unlock();
    }
  }
}

OpenedFile::OpenedFile(TransferFile *transfer)
  : m_source(transfer->getSourceUrl()),
    m_dest(transfer->getDestUrl()),
    m_hash(QString::null)
{
  // Calculate the file's MD5 hash
  QFile file(m_dest.path());
  if (!file.open(IO_ReadOnly)) {
    return;
  }
  
  KMD5 context;
  if (context.update(file))
    m_hash = QString(context.hexDigest());
  file.close();
}

bool OpenedFile::hasChanged()
{
  // Compare the file's MD5 hash with stored value
  QFile file(m_dest.path());
  if (!file.open(IO_ReadOnly)) {
    return false;
  }
  
  QString tmp = QString::null;
  KMD5 context;
  if (context.update(file))
    tmp = QString(context.hexDigest());
  file.close();
  
  return tmp != m_hash;
}

Manager *Manager::m_self = 0;
static KStaticDeleter<Manager> staticManagerDeleter;

Manager *Manager::self()
{
  if (!m_self) {
    staticManagerDeleter.setObject(m_self, new Manager());
  }
  
  return m_self;
}

Manager::Manager()
  : m_topLevel(new QueueObject(this, QueueObject::Toplevel)),
    m_processingQueue(false)
{
  m_topLevel->setId(0);
  
  m_lastQID = 1;
  m_curDownSpeed = 0;
  m_curUpSpeed = 0;

  m_emitUpdate = true;

  // Create the queue processor object
  m_queueProc = new KFTPQueueProcessor(this);

  connect(m_queueProc, SIGNAL(queueComplete()), this, SLOT(slotQueueProcessingComplete()));
  connect(m_queueProc, SIGNAL(queueAborted()), this, SLOT(slotQueueProcessingAborted()));

  // Create the queue converter object
  m_converter = new KFTPQueueConverter(this);
}

Manager::~Manager()
{
  if (m_self == this)
    staticManagerDeleter.setObject(m_self, 0, false);
}

Transfer *Manager::findTransfer(long id)
{
  // First try the cache
  QueueObject *object = m_queueObjectCache[QString::number(id)];
  
  if (!object) {
    object = m_topLevel->findChildObject(id);
    m_queueObjectCache.insert(QString::number(id), object);
  }
  
  return static_cast<Transfer*>(object);
}

Site *Manager::findSite(KURL url)
{
  // Reset path
  url.setPath("/");
  
  if (url.isLocalFile())
    return NULL;
  
  // Find the appropriate site and if one doesn't exist create a new one
  QueueObject *i;
  QPtrList<QueueObject> sites = topLevelObject()->getChildrenList();
  
  for (i = sites.first(); i; i = sites.next()) {
    if (i->getType() == QueueObject::Site) {
      Site *site = static_cast<Site*>(i);
      
      if (site->getUrl() == url)
        return site;
    }
  }
  
  // The site doesn't exist, let's create one
  Site *site = new Site(topLevelObject(), url);
  site->setId(m_lastQID++);
  emit newSite(site);
  
  return site;
}

void Manager::insertTransfer(Transfer *transfer)
{
  // Set id
  transfer->setId(m_lastQID++);
  
  // Reparent transfer
  filesize_t size = transfer->getSize();
  transfer->addSize(-size);
  
  if (transfer->hasParentObject())
    transfer->parentObject()->delChildObject(transfer);
  
  if (transfer->parent())
    transfer->parent()->removeChild(transfer);

  Site *site = 0;
  
  switch (transfer->getTransferType()) {
    case Download: site = findSite(transfer->getSourceUrl()); break;
    case Upload: site = findSite(transfer->getDestUrl()); break;
    case FXP: site = findSite(transfer->getSourceUrl()); break;
  }
  
  site->insertChild(transfer);
  site->addChildObject(transfer);
  transfer->addSize(size);

  emit newTransfer(transfer);
  
  if (m_emitUpdate)
    emit queueUpdate();
}

void Manager::insertTransfer(KURLDrag *drag)
{
  // Decode the drag
  KIO::MetaData p_meta;
  KURL::List p_urls;
  KURLDrag::decode(drag, p_urls, p_meta);

  // TODO make support for local drops - eg. from konqueror, where
  // we get no meta data, so we must get the file info ourselves and
  // reject remote urls (or show a dialog to ask the user if he
  // wants to connect to the remote site)

  // Now we should add transfers for all URLs
  Transfer *lastTransfer = 0L;
  KURL::List::iterator end(p_urls.end());

  for (KURL::List::iterator i(p_urls.begin()); i != end; ++i) {
    QString p_data = p_meta[(*i).htmlURL().local8Bit()];
    QChar p_type = p_data.at(0);
    filesize_t p_size = p_data.section(':', 1, 1).toULongLong();
    KURL p_src = (*i);
    KURL p_dst = KURL(p_meta["DestURL"]);
    p_dst.addPath(p_src.fileName());

    // Skip where both files are local
    if (p_src.isLocalFile() && p_dst.isLocalFile())
      continue;

    // Determine transfer type
    TransferType p_trans = p_src.isLocalFile() ? Upload : (p_dst.isLocalFile() ? Download : FXP);

    lastTransfer = spawnTransfer(p_src, p_dst, p_size, p_type == 'D', p_trans, true, true, 0L, true);
  }

  // Execute the transfer if set in configuration
  if (!KFTPCore::Config::queueOnDND() && p_urls.count() == 1 && lastTransfer)
    lastTransfer->delayedExecute(100);
}

Transfer *Manager::spawnTransfer(KURL srcURL, KURL dstURL, filesize_t size, bool dir, TransferType t_type, bool ignoreSkip,
                                 bool insertToQueue, QObject *parent, bool noScan)
{
  if (!ignoreSkip && KFTPCore::Config::enablePrioList() && KFTPCore::Config::skipNoQueue() && KFTPCore::Config::self()->getFilePriority(srcURL.path()) == PRIO_SKIP)
    return 0L;

  // Should we lowercase the destination path ?
  if (KFTPCore::Config::lowercaseDownload() && t_type == Download) {
    dstURL.setPath(dstURL.directory() + "/" + dstURL.fileName().lower());
  }

  if (!parent)
    parent = this;

  Transfer *transfer = 0L;

  if (dir)
    transfer = new TransferDir(parent);
  else {
    transfer = new TransferFile(parent);
    transfer->addSize(size);
  }

  transfer->setSourceUrl(srcURL);
  transfer->setDestUrl(dstURL);
  transfer->setTransferType(t_type);

  if (insertToQueue) {
    insertTransfer(transfer);
  } else {
    transfer->setId(m_lastQID++);
    emit newTransfer(transfer);
  }

  if (dir && !noScan) {
    // This is a directory, we should scan the directory and add all files/dirs found
    // as parent of current object
    KFTPSession *session = FTPSessionManager->spawnRemoteSession(IGNORE_SIDE, srcURL, 0, true);
    session->scanDirectory(transfer);
  }

  return transfer;
}

void Manager::removeTransfer(Transfer *transfer)
{
  if (!transfer)
    return;
    
  transfer->abort();
  long id = transfer->getId();
  long sid = transfer->parentObject()->getId();
  
  // Remove transfer from cache
  m_queueObjectCache.remove(QString::number(id));

  // Should the site be removed as well ?
  QueueObject *site = 0;
  if (transfer->parentObject()->getType() == QueueObject::Site && transfer->parentObject()->getChildrenList().count() == 1)
    site = transfer->parentObject();

  // Signal destruction & delete transfer
  transfer->faceDestruction();
  delete transfer;
  
  if (site) {
    delete site;
    emit siteRemoved(sid);
  }

  emit transferRemoved(id);
  
  if (m_emitUpdate)
    emit queueUpdate();
}

void Manager::removeTransfer(long id)
{
  // Just remove the transfer we find via id
  removeTransfer(findTransfer(id));
}

void Manager::revalidateTransfer(Transfer *transfer)
{
  QueueObject *i = transfer;
  
  while (i) {
    if (i->parentObject() == topLevelObject())
      break;
      
    i = i->parentObject();
  }
  
  // We have the site
  Site *curSite = static_cast<Site*>(i);
  Site *site = 0;
  
  switch (transfer->getTransferType()) {
    case Download: site = findSite(transfer->getSourceUrl()); break;
    case Upload: site = findSite(transfer->getDestUrl()); break;
    case FXP: site = findSite(transfer->getSourceUrl()); break;
  }
  
  // If the sites don't match, reparent transfer
  if (site != curSite) {
    transfer->parentObject()->delChildObject(transfer);
    transfer->parent()->removeChild(transfer);

    site->insertChild(transfer);
    site->addChildObject(transfer);
    
    emit transferRemoved(transfer->getId());
    emit newTransfer(transfer);
    
    if (curSite->getChildrenList().count() == 0) {
      emit siteRemoved(curSite->getId());
      curSite->deleteLater();
    }
  }
}

void Manager::removeFailedTransfer(FailedTransfer *transfer)
{
  // Remove the transfer and signal removal
  m_failedTransfers.remove(transfer);
  emit failedTransferRemoved(transfer->getTransfer()->getId());
  
  delete transfer;
}

void Manager::clearFailedTransferList()
{
  // Clear the failed transfers list
  FailedTransfer *transfer;
  QPtrListIterator<KFTPQueue::FailedTransfer> i(m_failedTransfers);
  
  while ((transfer = i.current()) != 0) {
    ++i;
    removeFailedTransfer(transfer);
  }
}

void Manager::moveTransferUp(long id)
{
  Transfer *transfer = findTransfer(id);
  transfer->parentObject()->moveChildUp(transfer);

  if (m_emitUpdate)
    emit queueUpdate();
}

void Manager::moveTransferDown(long id)
{
  Transfer *transfer = findTransfer(id);
  transfer->parentObject()->moveChildDown(transfer);

  if (m_emitUpdate)
    emit queueUpdate();
}

bool Manager::canBeMovedUp(long id)
{
  Transfer *transfer = findTransfer(id);
  return transfer ? transfer->parentObject()->canMoveChildUp(transfer) : false;
}

bool Manager::canBeMovedDown(long id)
{
  Transfer *transfer = findTransfer(id);
  return transfer ? transfer->parentObject()->canMoveChildDown(transfer) : false;
}

void Manager::doEmitUpdate()
{
  m_curDownSpeed = 0;
  m_curUpSpeed = 0;

  topLevelObject()->removeMarkedTransfers();
  
  // Get download/upload speeds
  QueueObject *i;
  QPtrList<QueueObject> sites = topLevelObject()->getChildrenList();
  
  for (i = sites.first(); i; i = sites.next()) {
    QueueObject *t;
    QPtrList<QueueObject> list = i->getChildrenList();
    
    for (t = list.first(); t; t = list.next()) {
      KFTPQueue::Transfer *tmp = static_cast<Transfer*>(t);
      
      switch (tmp->getTransferType()) {
        case Download: m_curDownSpeed += tmp->getSpeed(); break;
        case Upload: m_curUpSpeed += tmp->getSpeed(); break;
        case FXP: {
          m_curDownSpeed += tmp->getSpeed();
          m_curUpSpeed += tmp->getSpeed();
          break;
        }
      }
    }
  }

  // Emit global update to all GUI objects
  emit queueUpdate();
}

void Manager::start()
{
  if (m_processingQueue)
    return;
    
  m_processingQueue = true;
    
  // Now, go trough all queued files and execute them - try to do as little server connects
  // as possible
  m_queueProc->startProcessing();
}

void Manager::abort()
{
  m_processingQueue = false;
  
  // Stop further queue processing
  m_queueProc->stopProcessing();

  emit queueUpdate();
}

void Manager::slotQueueProcessingComplete()
{
  m_processingQueue = false;
  
  // Queue processing is now complete
  if (KFTPCore::Config::showBalloons()) {
    QString p_finishedStr = i18n("All queued transfers have been completed.");
    s_sysTray->showBalloon(p_finishedStr);
  }

  emit queueUpdate();
}

void Manager::slotQueueProcessingAborted()
{
  m_processingQueue = false;
}

void Manager::clearQueue()
{
  QueueObject *i;
  QPtrList<QueueObject> sites = topLevelObject()->getChildrenList();
  
  for (i = sites.first(); i; i = sites.next()) {
    QueueObject *t;
    QPtrList<QueueObject> list = i->getChildrenList();
    
    for (t = list.first(); t; t = list.next())
      removeTransfer(static_cast<Transfer*>(t));
  }
}

int Manager::getTransferPercentage()
{
  return 0;
}

int Manager::getNumRunning(bool onlyDirs)
{
  int running = 0;
  
  QueueObject *i;
  QPtrList<QueueObject> sites = topLevelObject()->getChildrenList();
  
  for (i = sites.first(); i; i = sites.next()) {
    QueueObject *t;
    QPtrList<QueueObject> list = i->getChildrenList();
    
    for (t = list.first(); t; t = list.next()) {
      if (t->isRunning() && (!onlyDirs || t->isDir()))
        running++;
    }
  }
  
  return running;
}

int Manager::fileExistsAction(TransferFile *transfer,
                              TransferType type,
                              const KURL &src,
                              const KURL &dst,
                              const FTPEntry &srcStat,
                              const FTPEntry &dstStat )
{
  FileExistsActions *fa = NULL;
  
  // File infos
  filesize_t srcSize = 0;
  time_t srcTime = 0;
  
  filesize_t dstSize = 0;
  time_t dstTime = 0;
  
  switch (type) {
    case KFTPQueue::Download: {
      KFileItem info(KFileItem::Unknown, KFileItem::Unknown, dst);
      dstSize = info.size();
      dstTime = info.time(KIO::UDS_MODIFICATION_TIME);
      
      srcSize = srcStat.size;
      srcTime = srcStat.date;
      
      fa = KFTPCore::Config::self()->dActions();
      break;
    }
    case KFTPQueue::Upload: {
      KFileItem info(KFileItem::Unknown, KFileItem::Unknown, src);
      srcSize = info.size();
      srcTime = info.time(KIO::UDS_MODIFICATION_TIME);
      
      dstSize = dstStat.size;
      dstTime = dstStat.date;
      
      fa = KFTPCore::Config::self()->uActions();
      break;
    }
    case KFTPQueue::FXP: {
      srcSize = srcStat.size;
      srcTime = srcStat.date;
      
      dstSize = dstStat.size;
      dstTime = dstStat.date;
      
      fa = KFTPCore::Config::self()->fActions();
      break;
    }
  }
  
  // Now that we have all data, get the action and do it
  FEAction action = fa->getActionForSituation(srcSize, srcTime, dstSize, dstTime);
  
  switch (action) {
    case FE_SKIP_ACT: {
      if (transfer->hasParentTransfer() && transfer->parentTransfer()->isRunning()) {
        transfer->m_deleteMe = true;
        transfer->addActualSize(-transfer->m_size);
      }

      transfer->resetTransfer();
      emit transfer->transferComplete(transfer->m_id);

      Manager::self()->doEmitUpdate();
      return -1;
    }
    case FE_OVERWRITE_ACT: {
      return KIO::R_OVERWRITE;
    }
    case FE_RESUME_ACT: {
      return KIO::R_RESUME;
    }
    case FE_RENAME_ACT: {
      // TODO implement rename action
    }
    case FE_USER_ACT: {
      QString newDestPath;
      KIO::RenameDlg_Result r = KIO::open_RenameDlg(
        i18n("File Exists"),
        transfer->getSourceUrl().prettyURL(),
        transfer->getDestUrl().prettyURL(),
        (KIO::RenameDlg_Mode) (KIO::M_OVERWRITE | KIO::M_RESUME),
        newDestPath,
        srcSize,
        dstSize,
        srcTime,
        dstTime
      );

      switch (r) {
        case KIO::R_RENAME: {
          KURL url = transfer->getDestUrl();
          url.setPath(newDestPath);
          transfer->setDestUrl(url);
          return 0;
        }
        case KIO::R_CANCEL: {
          if (transfer->hasParentTransfer() && transfer->parentTransfer()->isRunning()) {
            transfer->m_deleteMe = true;
            transfer->addActualSize(-transfer->m_size);
          }

          transfer->resetTransfer();
          emit transfer->transferComplete(transfer->m_id);

          Manager::self()->doEmitUpdate();
          return -1;
        }
        default: return r;
      }
    }
  }
  
  return 0;
}

void Manager::openAfterTransfer(TransferFile *transfer)
{
  QString mimeType = KMimeType::findByURL(transfer->getDestUrl(), 0, true, true)->name();
  KService::Ptr offer = KServiceTypeProfile::preferredService(mimeType, "Application");
      
  if (!offer) {
    KMessageBox::error(0, i18n("Unable to find a suitable application to open this file!"));
    return;
  }
  
  QStringList params = KRun::processDesktopExec(*offer, KURL::List(transfer->getDestUrl()), false);
  KProcess *p = new KProcess(this);
  *p << params;
  
  connect(p, SIGNAL(processExited(KProcess*)), this, SLOT(slotEditProcessTerminated(KProcess*)));
  
  p->start();
  
  // Save the process
  m_editProcessList.insert(p->pid(), OpenedFile(transfer));
}

void Manager::slotEditProcessTerminated(KProcess *p)
{
  // A process has terminated, we should reupload
  OpenedFile file = m_editProcessList[p->pid()];
  
  // Only upload a file if it has been changed
  if (file.hasChanged()) {
    TransferFile *transfer = new TransferFile(KFTPQueue::Manager::self());
    transfer->setSourceUrl(file.destination());
    transfer->setDestUrl(file.source());
    transfer->setTransferType(KFTPQueue::Upload);
    transfer->addSize(KFileItem(KFileItem::Unknown, KFileItem::Unknown, file.destination()).size());
    insertTransfer(transfer);
    
    // Execute the transfer
    transfer->delayedExecute();
  }
  
  // Cleanup
  m_editProcessList.remove(p->pid());
  p->deleteLater();
}

}

#include "kftpqueue.moc"
