/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftpclientthread.h"
#include "socket.h"
#include "errorhandler.h"

#include <qapplication.h>

CommandParameter::CommandParameter()
{
}

CommandParameter::CommandParameter(const QString &string)
{
  m_string = string;
  m_type = PARAM_STRING;
}

CommandParameter::CommandParameter(int integer)
{
  m_int = integer;
  m_type = PARAM_INT;
}

CommandParameter::CommandParameter(const KURL &url)
{
  m_url = url;
  m_type = PARAM_URL;
}

CommandParameter::CommandParameter(KFTPNetwork::Socket *client)
{
  m_data = client;
  m_type = PARAM_CLIENT;
}

CommandParameter::CommandParameter(KFTPQueue::Transfer *transfer)
{
  m_data = transfer;
  m_type = PARAM_TRANSFER;
}

int KFTPThreadCommand::execute(KFTPNetwork::Socket *client)
{
  FTP_EXCEPTION_TRY {
    switch (m_cmdType) {
      case TCMD_GET: {
        client->get(m_params[0].asUrl(), m_params[1].asUrl());
        break;
      }
      case TCMD_PUT: {
        client->put(m_params[0].asUrl(), m_params[1].asUrl());
        break;
      }
      case TCMD_FXP: {
        if (client->getFeatures() & SF_FXP_TRANSFER && m_params[2].asClient()->getFeatures() & SF_FXP_TRANSFER) {
          client->fxpTransfer(m_params[0].asUrl(), m_params[1].asUrl(), m_params[2].asClient());
        }
        break;
      }
      case TCMD_LIST: {
        (void) client->dirList(m_params[0].asUrl());
        break;
      }
      case TCMD_CONNECT: {
        client->connect(m_params[0].asUrl());
        break;
      }
      case TCMD_DISCONNECT: {
        client->disconnect();
        break;
      }
      case TCMD_SYNC: break;
      case TCMD_REMOVE: {
        client->remove(m_params[0].asUrl());
        break;
      }
      case TCMD_RENAME: {
        client->rename(m_params[0].asUrl(), m_params[1].asUrl());
        break;
      }
      case TCMD_CHMOD: {
        client->chmod(m_params[0].asUrl(), m_params[1].asInt());
        break;
      }
      case TCMD_MKDIR: {
        client->mkdir(m_params[0].asUrl());
        break;
      }
      case TCMD_STAT: {
        client->stat(m_params[0].asUrl());
        break;
      }
      case TCMD_SCAN: {
        client->recursiveScan(m_params[0].asTransfer());
        break;
      }
      case TCMD_RAW: {
        client->rawCommand(m_params[0].asString());
        break;
      }
      default: {
        qDebug("[THREAD] Unknown command!");
        return -999;
      }
    }
  } FTP_EXCEPTION_CATCH;

  return 0;
}

KFTPThreadDispatcher::KFTPThreadDispatcher(QObject *parent)
  : QObject(parent)
{
}

void KFTPThreadDispatcher::startDispatchLoop()
{
  m_abortLoop = false;
  m_curCmd = TCMD_NONE;

  while (!m_abortLoop) {
    /* Sleep a while if there are no commands */
    if (m_commandQueue.count() == 0)
      sleep(100);

    // Do not attempt to execute anything if the client is busy
    if (!m_ftpClient->socket()->isBusy()) {
      /* Go trough the command queue and execute the commands */
      while (m_commandQueue.count() > 0) {
        KFTPThreadCommand cmd = m_commandQueue.first();
        m_curCmd = cmd.m_cmdType;
        
        int err_code = cmd.execute(m_ftpClient->socket());
        if (err_code == 0)
          finished(cmd.m_cmdType);

        m_curCmd = TCMD_NONE;
        m_commandQueue.pop_front();
      }
    }
  }
}

void KFTPThreadDispatcher::queueCommand(CommandType cmdType, ParameterList params)
{
  KFTPThreadCommand newCommand;
  newCommand.m_cmdType = cmdType;
  newCommand.m_params = params;

  /* Add command to command queue */
  m_commandQueue.append(newCommand);
}

/*
                                                    CLASS KFTPCLIENTHREAD
*/

KFTPClientThread::KFTPClientThread()
  : KFTPThreadDispatcher(0)
{
}

void KFTPClientThread::setClient(KFTPNetwork::SocketManager *client)
{
  m_ftpClient = client;
}

void KFTPClientThread::run()
{
  // Start the main thread dispatch look
  KFTPThreadDispatcher::startDispatchLoop();
}

void KFTPClientThread::sleep(unsigned long msecs)
{
  msleep(msecs);
}

/*
                                                    CLASS KFTPCLIENTTHR
*/
KFTPClientThr::KFTPClientThr(QObject *parent, const char *name)
  : QObject(parent, name)
{
  m_thread = new KFTPClientThread();
  m_client = new KFTPNetwork::SocketManager(this);
  m_client->setProtocol(KURL("ftp://localhost/"));
  m_thread->setClient(m_client);
  m_thread->start();

  connect(m_thread, SIGNAL(finished(CommandType)), this, SLOT(slotFinished(CommandType)));
}

void KFTPClientThr::setupSignals(KFTPNetwork::Socket *socket)
{
  connect(socket, SIGNAL(sigRawReply(const QString&)), this, SLOT(slotRawReply(const QString&)));
  connect(socket, SIGNAL(sigLogUpdate(int, const QString&)), this, SLOT(slotLogUpdate(int, const QString&)));
  connect(socket, SIGNAL(sigDisconnectDone()), this, SLOT(slotDisconnectDone()));
  connect(socket, SIGNAL(sigLoginComplete(bool)), this, SLOT(slotLoginComplete(bool)));
  connect(socket, SIGNAL(sigResumedOffset(filesize_t)), this, SLOT(slotResumedOffset(filesize_t)));
  connect(socket, SIGNAL(sigRetrySuccess()), this, SLOT(slotRetrySuccess()));
  connect(socket, SIGNAL(sigError(KFTPNetwork::Error)), this, SLOT(slotErrorHandler(KFTPNetwork::Error)));
  connect(socket, SIGNAL(sigStateChanged(KFTPNetwork::SocketState)), this, SLOT(slotStateChanged(KFTPNetwork::SocketState)));
}

KFTPClientThr::~KFTPClientThr()
{
  m_thread->terminate();
}

void KFTPClientThr::customEvent(QCustomEvent *e)
{
  if (e->type() == THR_EVENT_ID) {
    ThreadEvent *te = (ThreadEvent*) e;

    switch (te->type()) {
      case EVENT_FINISHED: {
        int *p = (int*) te->data();
        emit finished((CommandType) *p);
        break;
      }
      case EVENT_LOG: {
        while (m_lastLog.count() > 0) {
          QString tmp = m_lastLog.first();
          int sep = tmp.find(':');
          emit logUpdate(tmp.mid(0, sep).toInt(), tmp.mid(sep+1));

          m_lastLog.pop_front();
        }
        break;
      }
      case EVENT_RAW_REPLY: {
        while (m_lastReply.count() > 0) {
          emit rawReply(m_lastReply.first());

          m_lastReply.pop_front();
        }
        break;
      }
      case EVENT_ERRORHDL: {
        KFTPNetwork::Error *p = (KFTPNetwork::Error*) te->data();
        emit errorHandler(*p);
        break;
      }
      case EVENT_OFFSET: {
        filesize_t *p = (filesize_t*) te->data();
        emit resumedOffset(*p);
        break;
      }
      case EVENT_LOGIN: {
        bool *p = (bool*) te->data();
        emit loginComplete(*p);
        break;
      }
      case EVENT_DISCONNECT: {
        emit disconnectDone();
        break;
      }
      case EVENT_RETRYOK: {
        emit retrySuccess();
        break;
      }
      case EVENT_STATE_CH: {
        KFTPNetwork::SocketState *state = (KFTPNetwork::SocketState*) te->data();
        emit stateChanged(*state);
        break;
      }
      default: {
        qDebug("Unknown event type!");
        return;
      }
    }
  }
}

void KFTPClientThr::slotStateChanged(KFTPNetwork::SocketState state)
{
  static KFTPNetwork::SocketState tmp;
  tmp = state;
  
  ThreadEvent *e = new ThreadEvent(&tmp, EVENT_STATE_CH);
  qApp->postEvent(this, e);
}

void KFTPClientThr::slotResumedOffset(filesize_t bytes)
{
  static filesize_t tmp;
  tmp = bytes;

  ThreadEvent *e = new ThreadEvent(&tmp, EVENT_OFFSET);
  qApp->postEvent(this, e);
}

void KFTPClientThr::slotFinished(CommandType cmdType)
{
  static int ctype;
  ctype = (int) cmdType;

  ThreadEvent *e = new ThreadEvent(&ctype, EVENT_FINISHED);
  qApp->postEvent(this, e);
}

void KFTPClientThr::slotRawReply(const QString &reply)
{
  m_lastReply.append(reply);

  if (m_lastReply.count() == 1) {
    ThreadEvent *e = new ThreadEvent(0, EVENT_RAW_REPLY);
    qApp->postEvent(this, e);
  }
}

void KFTPClientThr::slotLogUpdate(int type, const QString &text)
{
  QString tmp;
  tmp.sprintf("%d:", type);
  tmp.append(text);
  m_lastLog.append(tmp);

  if (m_lastLog.count() == 1) {
    ThreadEvent *e = new ThreadEvent(0, EVENT_LOG);
    qApp->postEvent(this, e);
  }
}

void KFTPClientThr::slotErrorHandler(KFTPNetwork::Error error)
{
  static KFTPNetwork::Error tmp;
  tmp = error;
  
  ThreadEvent *e = new ThreadEvent(&tmp, EVENT_ERRORHDL);
  qApp->postEvent(this, e);
}

void KFTPClientThr::slotDisconnectDone()
{
  ThreadEvent *e = new ThreadEvent(0, EVENT_DISCONNECT);
  qApp->postEvent(this, e);
}

void KFTPClientThr::slotLoginComplete(bool success)
{
  static bool tmp;
  tmp = success;
  
  ThreadEvent *e = new ThreadEvent(&tmp, EVENT_LOGIN);
  qApp->postEvent(this, e);
}

void KFTPClientThr::slotRetrySuccess()
{
  ThreadEvent *e = new ThreadEvent(0, EVENT_RETRYOK);
  qApp->postEvent(this, e);
}

void KFTPClientThr::connectToHost(const KURL &url)
{
  // Change the protocol
  m_client->socket()->QObject::disconnect(this);
  m_client->setProtocol(url);
  setupSignals(m_client->socket());
  
  ParameterList params;
  params.append(CommandParameter(url));
  
  m_thread->queueCommand(TCMD_CONNECT, params);
}

void KFTPClientThr::disconnect()
{
  m_thread->queueCommand(TCMD_DISCONNECT, ParameterList());
}

void KFTPClientThr::get(const KURL &source, const KURL &destination)
{
  ParameterList params;
  params.append(CommandParameter(source));
  params.append(CommandParameter(destination));

  m_thread->queueCommand(TCMD_GET, params);
}

void KFTPClientThr::put(const KURL &source, const KURL &destination)
{
  ParameterList params;
  params.append(CommandParameter(source));
  params.append(CommandParameter(destination));

  m_thread->queueCommand(TCMD_PUT, params);
}

void KFTPClientThr::fxpTransfer(const KURL &source, const KURL &destination, KFTPClientThr *client)
{
  ParameterList params;
  params.append(CommandParameter(source));
  params.append(CommandParameter(destination));
  params.append(CommandParameter(client->getClient()));

  m_thread->queueCommand(TCMD_FXP, params);
}

void KFTPClientThr::remove(const KURL &url)
{
  ParameterList params;
  params.append(CommandParameter(url));

  m_thread->queueCommand(TCMD_REMOVE, params);
}

void KFTPClientThr::rename(const KURL &source, const KURL &destination)
{
  ParameterList params;
  params.append(CommandParameter(source));
  params.append(CommandParameter(destination));

  m_thread->queueCommand(TCMD_RENAME, params);
}

void KFTPClientThr::chmod(const KURL &url, int mode)
{
  ParameterList params;
  params.append(CommandParameter(url));
  params.append(CommandParameter(mode));

  m_thread->queueCommand(TCMD_CHMOD, params);
}

void KFTPClientThr::mkdir(const KURL &url)
{
  ParameterList params;
  params.append(CommandParameter(url));

  m_thread->queueCommand(TCMD_MKDIR, params);
}

void KFTPClientThr::dirList(const KURL &url)
{
  ParameterList params;
  params.append(CommandParameter(url));

  m_thread->queueCommand(TCMD_LIST, params);
}

void KFTPClientThr::stat(const KURL &url)
{
  ParameterList params;
  params.append(CommandParameter(url));

  m_thread->queueCommand(TCMD_STAT, params);
}

void KFTPClientThr::scanDirectory(KFTPQueue::Transfer *transfer)
{
  ParameterList params;
  params.append(CommandParameter(transfer));
  
  m_thread->queueCommand(TCMD_SCAN, params);
}

void KFTPClientThr::rawCommand(const QString &cmd)
{
  ParameterList params;
  params.append(CommandParameter(cmd));
  
  m_thread->queueCommand(TCMD_RAW, params);
}

#include "kftpclientthread.moc"
