/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "socket.h"
#include "ftpsocket.h"
#include "ftpdirectoryparser.h"

#include "misc/config.h"
#include "kftpotpgenerator.h"
#include "errorhandler.h"

#include <qapplication.h>
#include <qdir.h>

#include <errno.h>
#include <sys/ioctl.h>

#include <kresolver.h>
#include <ksocks.h>
#include <ksocketaddress.h>
#include <klocale.h>
#include <kio/job.h>
#include <kio/renamedlg.h>

#define TELNET_IAC      255
#define TELNET_IP       244
#define TELNET_SYNCH    242


/*
 * Variable sized buffer support [EXPERIMENTAL]
 *
 * If defined, the sizes of the buffers in ftpGet and ftpPut will
 * increase if the buffer fills up and decrease if it doesn't.
 * This gives up to 2x faster downloading and uploading (tested
 * locally).
 *
 * Comment the #define below if you experience any problems.
 */
#define VARIABLE_BUFFER

#define F_STAT(x) FTPDirectoryItem(ftpStat(KURL(x)))
#define F_STAT_FXP(x) FTPDirectoryItem(client->ftpStat(KURL(x)))

namespace KFTPNetwork {

FtpSocket::FtpSocket(QObject *parent)
 : Socket(parent, "ftp"),
   m_controlSSL(0),
   m_dataSSL(0),
   m_clientCert(0)
{
}

void FtpSocket::initConfig()
{
  Socket::initConfig();
  
  m_clientCert = 0L;
}

void FtpSocket::get(KURL source, KURL destination)
{
  if (ftpGet(source.path(), destination.path()) < 0)
    emit sigLogUpdate(4, i18n("File transfer has failed."));
}

void FtpSocket::put(const KURL &source, const KURL &destination)
{
  if (ftpPut(source.path(), destination.path()) < 0)
    emit sigLogUpdate(4, i18n("File transfer has failed."));
}

void FtpSocket::remove(const KURL &url)
{
  /* Discover if path is file or directory and call the right method */
  if (checkIsDir(url)) {
    m_stateInfo.enterSocketState(S_REMOVE, true);

    recursiveDelete(url);
    ftpRmdir(url.path());

    m_stateInfo.enterSocketState(S_IDLE, true);
    return;
  }

  ftpDelete(url.path());
}

void FtpSocket::rename(const KURL &source, const KURL &destination)
{
  ftpRename(source.path(), destination.path());
}

void FtpSocket::chmod(const KURL &url, int mode)
{
  ftpChmod(mode, url.path());
}

void FtpSocket::mkdir(const KURL &url)
{
  ftpMkdir(url.path());
}

void FtpSocket::fxpTransfer(const KURL &source, const KURL &destination, Socket *client)
{
  FtpSocket *socket = static_cast<FtpSocket*>(client);

  m_stateInfo.enterSocketState(S_TRANSFER);
  socket->m_stateInfo.enterSocketState(S_TRANSFER);

  int retr = ftpFxpTransfer(source.path(), destination.path(), socket);

  m_stateInfo.enterSocketState(S_IDLE);
  socket->m_stateInfo.enterSocketState(S_IDLE);

  if (retr < 0) {
    emit sigLogUpdate(4, i18n("FXP transfer failed."));
    emit socket->sigLogUpdate(4, i18n("FXP transfer failed."));
  } else {
    emit sigLogUpdate(3, i18n("FXP transfer completed."));
    emit socket->sigLogUpdate(3, i18n("FXP transfer completed."));
  }
}

void FtpSocket::stat(const KURL &url)
{
  emit sigStatResult(ftpStat(url));
}

int FtpSocket::getFeatures()
{
  return SF_FXP_TRANSFER | SF_RAW_COMMAND;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////  SSL ENCRYPTION  /////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

KSSL *FtpSocket::SSLNewControl(int fd)
{
  // Create a new KSSL object for the specified file descriptor
  if (m_controlSSL != 0L) {
    m_controlSSL->reInitialize();
  } else {
    m_controlSSL = new KSSL();
    m_controlSSL->TLSInit();
  }

  // Use the client certificate if required
  if (m_clientCert != 0L) {
    m_controlSSL->setClientCertificate(m_clientCert);
  }

  m_sslLastResult = (m_controlSSL->connect(fd) == 1) ? true : false;

  return m_controlSSL;
}

KSSL *FtpSocket::SSLNewData(int fd)
{
  // Create a new KSSL object for the specified file descriptor
  if (m_dataSSL != 0L) {
    m_dataSSL->reInitialize();
  } else {
    m_dataSSL = new KSSL();
    m_dataSSL->TLSInit();
  }

  // Use the client certificate if required
  if (m_clientCert != 0L) {
    m_dataSSL->setClientCertificate(m_clientCert);
  }

  m_sslLastResult = (m_dataSSL->connect(fd) == 1) ? true : false;

  return m_dataSSL;
}

KSSL *FtpSocket::SSLGetObject()
{
  // Return the current control KSSL object (if any)
  return m_controlSSL;
}

void FtpSocket::SSLSetClientCert(KSSLPKCS12 *cert)
{
  m_clientCert = cert;
}

void FtpSocket::SSLClose()
{
  // We won't be using SSL for this connection anymore
  if (m_controlSSL != 0L) {
    m_controlSSL->close();

    delete m_controlSSL;
    m_controlSSL = 0L;
  }

  if (m_dataSSL != 0L) {
    delete m_dataSSL;
    m_dataSSL = 0L;
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////  FTP PROTOCOL  ///////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int FtpSocket::ftpReadLine(char *buf, int max)
{
  if (max == 0)
    return 0;

  char read_buf[1024];
  char *c = 0L;
  int read_bytes = -1;
  QCString s_buf;
  QString tmp = "";

  if (m_readBuf.find('\n') == -1) {
    while (!c) {
      int b_read;
      memset(read_buf, 0, sizeof(read_buf));

      if (getConfig("using_ssl") == 1) {
        /* We are in TLS mode */
        if ((b_read = m_controlSSL->read(read_buf, sizeof(read_buf))) == -1) {
          return -1;
        }
      } else {
        /* Normal cleartext mode */
        if ((b_read = KSocks::self()->recv(m_control, read_buf, sizeof(read_buf), 0)) == -1) {
          return -1;
        }

        // The peer has closed the connection
        if (b_read == 0) {
          // We are disconnected
          Socket::disconnect();

          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_TIMEOUT);
          FTP_EXCEPTION;
        }
      }

      read_bytes += b_read;

      // Search for the \n (if it exists)
      c = strchr(read_buf, '\n');
      s_buf += read_buf;
    }

    tmp = s_buf;
  }

  if (m_readBuf.length() > 0)
    tmp.prepend(m_readBuf);

  int n_pos = tmp.find('\n');
  memset(buf, 0, max);
  strcpy(buf, tmp.mid(0, n_pos+1).ascii());
  read_bytes = strlen(buf);

  // Save the remaining bytes
  tmp.remove(0, n_pos+1);
  m_readBuf = tmp;

  return read_bytes;
}

char FtpSocket::ftpReadResponse()
{
  // Read the response into the response buffer and return the first
  // char of the buffer for error code recognition
  int size;
  char match[5];

  if ((size = ftpReadLine(m_responseBuf, 1024)) == -1) {
    return '\0';
  }

  if (m_responseBuf[3] == '-') {
    // Multiline text
    strncpy(match, m_responseBuf, 3);
    match[3] = ' ';
    match[4] = '\0';

    m_lastMultiResp = "";

    emit sigRawReply(m_responseBuf);
    emit sigLogUpdate(2, m_remoteEncoding->decode(m_responseBuf));

    while (strncmp(m_responseBuf, match, 4)) {
      // While the response code is the same as above, read the multiline
      // response from server

      if (ftpReadLine(m_responseBuf, 1024) == -1) {
        return '\0';
      }

      if (m_responseBuf[3] == '-' || m_responseBuf[0] == ' ') {
        emit sigRawReply(m_responseBuf);
        emit sigLogUpdate(2, m_remoteEncoding->decode(m_responseBuf));
        m_lastMultiResp += m_responseBuf;
      }
    }
  } else {
    // Single line response
  }

  // For log update
  emit sigRawReply(m_responseBuf);
  emit sigLogUpdate(0, m_remoteEncoding->decode(m_responseBuf));

  // Return the first character
  return m_responseBuf[0];
}

bool FtpSocket::ftpIsRespCode(const QString &code)
{
  // Get the current response code and compare
  QCString curResp;
  curResp += m_responseBuf[0];
  curResp += m_responseBuf[1];
  curResp += m_responseBuf[2];

  return code == QString(curResp);
}

int FtpSocket::ftpSendCommand(const QString &command, bool doRead)
{
  // Send a command to the server, if it fails try to retry
  // or reconnect

  QCString buffer(m_remoteEncoding->encode(command) + "\r\n");
  char response = '\0';

  // For log update
  emit sigLogUpdate(1, command);

  int retr;

  if (getConfig("using_ssl") == 1) {
    /* We are in TLS mode */
    retr = m_controlSSL->write(buffer.data(), buffer.length());
  } else {
    /* Normal mode */
    retr = KSocks::self()->write(m_control, buffer.data(), buffer.length());
  }

  if (doRead) {
    if (retr > 0)
      response = ftpReadResponse();
  
    if (response == '\0' || (ftpIsRespCode("421") && m_isLoggedIn)) {
      // If response is NULL(\0) or the response code is 421
      // we are disconnected/timed out
      Socket::disconnect();
  
      // Dispatch error message and throw an exception so the processing will stop
      m_errorHandler->dispatchError(EC_TIMEOUT);
      FTP_EXCEPTION;
    }
  }

  return true;
}

bool FtpSocket::ftpCanRead()
{
  // Check if there is anything to read on the control socket
  if (getConfig("using_ssl") == 1) {
    return m_controlSSL->pending() > 0;
  } else {
    int numBytes = 0;
    ioctl(m_control, FIONREAD, &numBytes);

    return numBytes > 0;
  }
}

void FtpSocket::ftpChangeTLSDataProt(char prot)
{
  if (getConfig("using_ssl") == 1) {
    QString cmd = "PROT ";
    cmd += prot;

    if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
      emit sigLogUpdate(4, i18n("Protection mode setup failed, fallback to unencrypted connection."));
      setConfig("prot_mode", 'C');
    }
  }
}

int FtpSocket::protoConnect()
{
  QMutexLocker locker(&m_connMutex);
  
  // Support decimal IPs
  QString addr = getClientInfoUrl().host();
  bool isDecimal;
  unsigned long decIP = addr.toULong(&isDecimal);
  if (isDecimal) {
    // It is a decimal formatted IP, let's convert
    QString hex;
    hex.setNum(decIP, 16);
    addr = "";

    addr += QString::number(hex.mid(0, 2).toInt(0, 16));
    addr += ".";

    addr += QString::number(hex.mid(2, 2).toInt(0, 16));
    addr += ".";

    addr += QString::number(hex.mid(4, 2).toInt(0, 16));
    addr += ".";

    addr += QString::number(hex.mid(6, 2).toInt(0, 16));
  }

  struct sockaddr_in sa;
  struct hostent *he;
  memset(&sa, 0, sizeof(sa));
  sa.sin_family = AF_INET;
  sa.sin_port = htons(getClientInfoUrl().port());
  int on = 1;

  int ret = inet_aton(addr.ascii(), &sa.sin_addr);
  if (ret == 0) {
    if ((he = ::gethostbyname(addr.ascii())) == NULL) {
      sigLogUpdate(4, i18n("Unknown host '%1'.").arg(addr));

      return -2;
    }

    memcpy((char *)&sa.sin_addr, he->h_addr, he->h_length);
  }

  m_socketAddress = sa;
  m_control = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (m_control < 0) {
    return -1;
  }

  if (setsockopt(m_control, SOL_SOCKET, SO_REUSEADDR, (char*) &on, sizeof(on)) == -1) {
    return -1;
  }
  
  // Set the socket timeout
  struct timeval tv;
  tv.tv_sec = KFTPCore::Config::controlTimeout();
  tv.tv_usec = 0;
  
  if (setsockopt(m_control, SOL_SOCKET, SO_RCVTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    return -1;
  }
  
  if (setsockopt(m_control, SOL_SOCKET, SO_SNDTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    return -1;
  }
  
  if (::connect(m_control, (struct sockaddr*) &sa, sizeof(sa)) < 0) {
    sigLogUpdate(4, i18n("Connection failed (%1).").arg(strerror(errno)));

    return -2;
  }
  
  // Clear the buffer
  m_readBuf = "";

  // If we are required to use implicit SSL, do so now
  if (getConfig("use_implicit") == 1) {
    SSLNewControl(m_control);
    setConfig("using_ssl", 0);

    if (m_sslLastResult != 1) {
      sigLogUpdate(4, i18n("Unable to establish implicit SSL connection."));
      return -5;
    } else {
      sigLogUpdate(3, i18n("Established implicit SSL connection."));
      setConfig("using_ssl", 1);
      setConfig("use_tls", 0);
    }
  } else {
    // Otherwise we are not yet in SSL mode
    setConfig("using_ssl", 0);
  }

  if (ftpReadResponse() != '2') {
    // Response code should be 2xx, but it is NOT
    qDebug("[ERROR] Unable to connect: %s", QString(m_responseBuf + 3).stripWhiteSpace().ascii());

    return -5;
  }

  return 1;
}

int FtpSocket::protoLogin()
{
  if (m_isLoggedIn) return 1;
  
  m_stateInfo.enterSocketState(S_LOGIN);

  // If both is empty, do anonymous
  if (!getClientInfoUrl().hasUser() && !getClientInfoUrl().hasPass()) {
    getClientInfoUrl().setUser("anonymous");
    getClientInfoUrl().setPass(KFTPCore::Config::anonMail());
  }

  // Shall we use TLS ?
  if (getConfig("use_tls") == 1) {
    if (ftpSendCommand("AUTH TLS") && m_responseBuf[0] != '2') {
      // Failed, disable TLS usage on this connection
      setConfig("use_tls", 0);
    } else {
      SSLNewControl(m_control);
      setConfig("using_ssl", 0);

      if (m_sslLastResult != 1)
        emit sigLogUpdate(4, i18n("Unable to establish TLS connection."));
      else
        setConfig("using_ssl", 1);
    }
  }
  
  // Do we have a valid SSL mode ? If not, default to P.
  if (getConfig("using_ssl") && getConfig("prot_mode") != 'C' && getConfig("prot_mode") != 'P') {
    setConfig("prot_mode", 'P');
  }

  // Actualy send the login commands
  ftpSendCommand("USER " + getClientInfoUrl().user());

  if (ftpIsRespCode("331")) {
    // Send the password
    if (!strncmp(m_responseBuf, "331 Response to otp-", 20) ||
        !strncmp(m_responseBuf, "331 Response to s/key", 21)) {
      // OTP: 331 Response to otp-md5 41 or4828 ext required for foo.
      QString tmp = m_responseBuf;
      tmp = tmp.section(' ', 3, 5);

      KFTPOTPGenerator g(tmp, getClientInfoUrl().pass());
      ftpSendCommand("PASS " + g.generateOTP());
    } else {
      ftpSendCommand("PASS " + getClientInfoUrl().pass());
    }
  }

  if (!ftpIsRespCode("230")) {
    // Login failed
    m_stateInfo.enterSocketState(S_IDLE);
    return -1;
  }

  /* Set some stuff if in TLS mode */
  if (getConfig("using_ssl") == 1) {
    ftpSendCommand("PBSZ 0");
    ftpChangeTLSDataProt(getConfig("prot_mode"));
  }

  // Now we should be logged in, switch to UNIX-like dir listing if this is
  // IIS 4
  if (ftpSendCommand("SYST") && m_responseBuf[0] == '2') {
    if (!strncmp(m_responseBuf, "215 Windows_NT version", 22)) {
      ftpSendCommand("SITE DIRSTYLE");

      if (!strncmp(m_responseBuf, "200 MSDOS-like directory output is on", 37)) {
        // It was UNIX before, repair what we broke =)
        ftpSendCommand("SITE DIRSTYLE");
      }
    }
  }

  // Send FEAT command to discover extended features ;)
  if (ftpSendCommand("FEAT") && m_responseBuf[0] == '2') {
      QStringList p_featList = QStringList::split("\n", m_lastMultiResp);
      QStringList::iterator featListEnd( p_featList.end() );
      for (QStringList::iterator i( p_featList.begin() );i != featListEnd; ++i) {
        QString p_feat = *i;
        p_feat = p_feat.stripWhiteSpace();

        // Check for FEAT output type
        if (p_feat.left(4) == "211-") {
          p_feat.remove(0, 4);
        }

        if (p_feat == "PRET") {
          // This server looks like a distributed FTP server, enable the
          // option if not already enabled so we will send the right commands
          // before transfers.
          setConfig("feat_pret", 1);
        } else if (p_feat == "SSCN") {
          // Server supports SSCN mode, if we are using SSL connection, we should
          // enable support
          setConfig("feat_sscn", 1);
        } else if (p_feat == "CPSV") {
          // Server supports CPSV (similar to SSCN)
          setConfig("feat_cpsv", 1);
        } else if (p_feat.startsWith("XCRC")) {
          // Server supports XCRC checks
          // FIXME Disabled xcrc until 0.7
          //setConfig("feat_xcrc", 1);
        } else if (p_feat.startsWith("SITE")) {
          // Let's see some SITE commands that are supported
          p_feat.remove(0, 5);
          QStringList siteList = QStringList::split(";", p_feat);

          QStringList::iterator siteListEnd( siteList.end() );
          for (QStringList::iterator j( siteList.begin() ); j != siteListEnd; ++j) {
            QString siteFeat = *j;
            siteFeat = siteFeat.stripWhiteSpace();

            if (siteFeat == "SET") {
              // This is a very usefull options, since SITE SET TRANSFERPROGRESS ON may be
              // implemented -- this is needed for showing status of FXP transfers
              setConfig("feat_site_set", 1);
            }
          }
        } else if (p_feat.startsWith("UTF8")) {
          // TODO enable utf8 and send it right after the feat is done
          setConfig("feat_utf8", 1);
        }
      }
  }

  // Check where we are
  if (!ftpSendCommand("PWD") || m_responseBuf[0] != '2') {
    // Ugh ?! Unable to send PWD command... wierd
    m_stateInfo.enterSocketState(S_IDLE);
    return -2;
  }

  // 257 "/home/default/path"
  m_defDir = QString(m_responseBuf);
  int first = m_defDir.find('"') + 1;
  m_defDir = m_defDir.mid(first, m_defDir.findRev('"') - first);

  m_stateInfo.enterSocketState(S_IDLE);
  m_isLoggedIn = true;
  return 1;
}

void FtpSocket::protoDisconnect()
{
  /* Free the SSL stuff */
  if (getConfig("using_ssl") == 1) {
    SSLClose();
  }

  if (m_control != 0) {
    close(m_control);
    m_control = 0;
  }

  m_isLoggedIn = false;
}

void FtpSocket::ftpCloseDataSockets()
{
  if (m_dataConn != 0) {
    if (getConfig("using_ssl") == 1 && m_dataSSL) {
      m_dataSSL->close();
    } else {
      shutdown(m_dataConn, 2);
      ::close(m_dataConn);
    }

    m_dataConn = 0;
  }

  if (m_data != 0) {
    ::close(m_data);
    m_data = 0;
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////  CONNECTION TYPE METHODS  ///////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool FtpSocket::ftpPASVConnection()
{
  KExtendedSocket kSocket;
  int ip[6];
  struct linger lng = { 1, 120 };

  // Can we actually do PASV - only via IPv4 !
  if (m_socketAddress.sin_family != AF_INET)
    return false;

  // Check if PASV mode is supported
  if (getConfig("can_pasv") == 0)
    return false;

  if (!ftpSendCommand("PASV") || !ftpIsRespCode("227")) {
    if (m_responseBuf[0] == '5') {
      // Unknown command - disable PASV mode
      setConfig("can_pasv", 0);
    }

    return false;
  }

  // Ok PASV command successfull - let's parse the result
  char *begin = strchr(m_responseBuf, '(');

  // Some stinky servers don't respect RFC and do it on their own
  if (!begin)
    begin = strchr(m_responseBuf, '=');

  if (!begin || (sscanf(begin, "(%d,%d,%d,%d,%d,%d)",&ip[0], &ip[1], &ip[2], &ip[3], &ip[4], &ip[5]) != 6 &&
                 sscanf(begin, "=%d,%d,%d,%d,%d,%d",&ip[0], &ip[1], &ip[2], &ip[3], &ip[4], &ip[5]) != 6)) {
    // Unable to parse ?
    return false;
  }

  // Convert to string
  QString host;
  int port;

  host.sprintf("%d.%d.%d.%d", ip[0], ip[1], ip[2], ip[3]);
  port = ip[4] << 8 | ip[5];

  // If the reported IP address is from a private IP range, this might be because the
  // remote server is not properly configured. So we just use the server's real IP instead
  // of the one we got (if the host is really local, then this should work as well).
  if (host.startsWith("192.168.") || host.startsWith("10.") || host.startsWith("172.16."))
    kSocket.setAddress(inet_ntoa(m_socketAddress.sin_addr), port);
  else
    kSocket.setAddress(host, port);

  kSocket.setSocketFlags(KExtendedSocket::noResolve);
  kSocket.setTimeout(10);

  if (kSocket.connect() < 0) {
    // Could not connect using PASV, despite the server told us it supports PASV earlier.
    // Maybe this was due to ports used for PASV was blocked/filtered, or the server simply
    // was lying about its capabilities (less likely). Updating configuration so PASV wont
    // be used anymore for this connection.
    int pasv_fails = getConfig("pasv_fails");
    
    // PASV has failed two times (and this is not a distributed server), it shall be disabled
    if (++pasv_fails >= 2 && getConfig("feat_pret") != 1) {
      sigLogUpdate(4, i18n("Passive mode has failed two times, disabling use of PASV!"));
      setConfig("can_pasv", 0);
    }
    
    setConfig("pasv_fails", pasv_fails);
    return false;
  }

  // Get the socket's file descriptor and set some opts
  int on = 1;
  m_data = kSocket.fd();

  if ((setsockopt(m_data, SOL_SOCKET, SO_REUSEADDR, (char*)&on, sizeof(on)) == -1) || (m_data < 0)) {
    return false;
  }
  if ((setsockopt(m_data, SOL_SOCKET, SO_KEEPALIVE, (char*)&on, sizeof(on)) < 0)) {
    qDebug("Keepalive not allowed");
  }
  if (setsockopt(m_data, SOL_SOCKET, SO_LINGER, (char*)&lng, sizeof(lng)) < 0) {
    qDebug("Unable to set linger mode");
  }
  
  // Set the socket timeout
  struct timeval tv;
  tv.tv_sec = 10;
  tv.tv_usec = 0;
  
  if (setsockopt(m_data, SOL_SOCKET, SO_RCVTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    ::close(m_data);
    return false;
  }
  
  if (setsockopt(m_data, SOL_SOCKET, SO_SNDTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    ::close(m_data);
    return false;
  }

  kSocket.release();
  return true;
}

bool FtpSocket::ftpEPSVConnection()
{
  KExtendedSocket kSocket;
  struct linger lng = { 1, 120 };

  // Check if EPSV mode is supported
  if (getConfig("can_epsv") == 0)
    return false;

  if (!ftpSendCommand("EPSV") || m_responseBuf[0] != '2') {
    if (m_responseBuf[0] == '5') {
      // Unknown command - disable EPSV mode
      setConfig("can_epsv", 0);
    }

    return false;
  }

  // 229 Entering Extended Passive Mode (|||55016|)
  char *begin = strchr(m_responseBuf, '(');
  int port;

  if (!begin || sscanf(begin, "(|||%d|)", &port) != 1) {
    // Unable to parse
    return false;
  }

  kSocket.setAddress(inet_ntoa(m_socketAddress.sin_addr), port);
  kSocket.setSocketFlags(KExtendedSocket::noResolve);
  kSocket.setTimeout(10);

  if (kSocket.connect() < 0) {
    return false;
  }

  // Get the socket's file descriptor and set some opts
  int on = 1;
  m_data = kSocket.fd();

  if ((setsockopt(m_data, SOL_SOCKET, SO_REUSEADDR, (char*)&on, sizeof(on)) == -1) || (m_data < 0)) {
    return false;
  }
  if (setsockopt(m_data, SOL_SOCKET, SO_KEEPALIVE, (char*)&on, sizeof(on)) < 0) {
    qDebug("Keepalive not allowed");
  }
  if (setsockopt(m_data, SOL_SOCKET, SO_LINGER, (char*)&lng, sizeof(lng)) < 0) {
    qDebug("Unable to set linger mode");
  }
  
  // Set the socket timeout
  struct timeval tv;
  tv.tv_sec = 10;
  tv.tv_usec = 0;
  
  if (setsockopt(m_data, SOL_SOCKET, SO_RCVTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    ::close(m_data);
    return false;
  }
  
  if (setsockopt(m_data, SOL_SOCKET, SO_SNDTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    ::close(m_data);
    return false;
  }

  kSocket.release();
  return true;
}

bool FtpSocket::ftpEPRTConnection()
{
  // TODO implement EPRT connections ;)
  //        EPRT |1|132.235.1.2|6275|
  //        EPRT |2|1080::8:800:200C:417A|5282|
  return false;
}

bool FtpSocket::ftpActiveConnection()
{
  union
  {
    struct sockaddr sa;
    struct sockaddr_in in;
  } sin;

  struct linger lng = { 0, 0 };
  ksocklen_t l;
  char buf[64];
  int on = 1;

  // Get control's ip
  l = sizeof(sin);
  if (KSocks::self()->getsockname(m_control, &sin.sa, &l) < 0)
    return false;

  // PORT is only for IPv4 connections
  if (sin.sa.sa_family != PF_INET)
    return false;

  m_data = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (m_data == 0)
    return false;

  if (setsockopt(m_data, SOL_SOCKET, SO_REUSEADDR, (char*)&on, sizeof(on)) == -1) {
    ::close(m_data);
    return false;
  }

  if (setsockopt(m_data, SOL_SOCKET, SO_LINGER, (char*)&lng, sizeof(lng)) == -1) {
    ::close(m_data);
    return false;
  }
  
  // Set the socket timeout
  struct timeval tv;
  tv.tv_sec = 10;
  tv.tv_usec = 0;
  
  if (setsockopt(m_data, SOL_SOCKET, SO_RCVTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    ::close(m_data);
    return false;
  }
  
  if (setsockopt(m_data, SOL_SOCKET, SO_SNDTIMEO, (struct timeval*) &tv, sizeof(tv)) == -1) {
    ::close(m_data);
    return false;
  }

  // If the user chose to force active port range, let select a port for him
  if (KFTPCore::Config::activeForcePort()) {
    srand(time(0L));
    
    int max = KFTPCore::Config::activeMaxPort();
    int min = KFTPCore::Config::activeMinPort();
    
    int startPort = min + rand() % (max - min + 1);
    int port = startPort;
    
    do {
      sin.in.sin_port = htons(port++);
      
      if (port > max)
        port = min;
        
      if (port == startPort) {
        ::close(m_data);
        return false;
      }
    } while (KSocks::self()->bind(m_data, (struct sockaddr*) &sin.in, sizeof(sin.in)) == -1);
  } else {
    sin.in.sin_port = 0;
  
    if (KSocks::self()->bind(m_data, (struct sockaddr*) &sin.in, sizeof(sin.in)) == -1) {
      ::close(m_data);
      m_data = 0;
      return false;
    }
  }

  if (KSocks::self()->listen(m_data, 1) < 0) {
    ::close(m_data);
    return false;
  }

  if (KSocks::self()->getsockname(m_data, &sin.sa, &l) < 0)
    return false;

  // If the user has chosen to set a different ip
  if (KFTPCore::Config::portForceIp()) {
    // Get the port we used when binding
    KSocketAddress *p_tmp = KSocketAddress::newAddress(&sin.sa, l);

    // Resolve IP first - using the port above
    KNetwork::KResolverResults p_list;

    p_list = KNetwork::KResolver::resolve(KFTPCore::Config::portIp(), p_tmp->serviceName());
    if (p_list.error() < 0) {
      // Well, we are unable to resolve the name, so we should use what we got
      // from control socket
    } else {
      // The name has been resolved and we have the address, so we should
      // use it
      sin.sa = *p_list[0].address().address();
      l = p_list[0].address().length();
    }

    delete p_tmp;
  }

  // Send the apropriate PORT command
  sprintf(buf,"PORT %d,%d,%d,%d,%d,%d",
          (unsigned char)sin.sa.sa_data[2],(unsigned char)sin.sa.sa_data[3],
          (unsigned char)sin.sa.sa_data[4],(unsigned char)sin.sa.sa_data[5],
          (unsigned char)sin.sa.sa_data[0],(unsigned char)sin.sa.sa_data[1]);

  if (!ftpSendCommand(buf) || m_responseBuf[0] != '2') {
    ::close(m_data);
    return false;
  }

  return true;
}

bool FtpSocket::ftpInitDataConnection()
{
  // Init a data connection
  bool result = false;

  if (getConfig("epsv_mode") == 1) {
    // We are already in EPSV mode, so the only thing we can init
    // is EPSV data connection.
    result = ftpEPSVConnection();
  } else {
    // Otherwise go one by one
    if (getConfig("can_epsv") == 1)
      result = ftpEPSVConnection();

    if (!result && getConfig("can_pasv") == 1)
      result = ftpPASVConnection();

    if (!result && getConfig("can_eprt") == 1)
      result = ftpEPRTConnection();

    if (result)
      setConfig("passive_mode", 1);

    if (!result) {
      result = ftpActiveConnection();

      if (result)
        setConfig("passive_mode", 0);
    }
  }

  return result;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////  DATA CONNECTION METHODS  ///////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

ssize_t FtpSocket::ftpDataRead(void *buf, long size)
{
  /* Use SSL if in TLS mode and PROTO is not cleartext */
  if (getConfig("using_ssl") == 1 && getConfig("prot_mode") != 'C') {
    return m_dataSSL->read(buf, size);
  } else {
    return KSocks::self()->read(m_dataConn, buf, size);
  }

  return 0;
}

ssize_t FtpSocket::ftpDataWrite(void *buf, long size)
{
  /* Use SSL if in TLS mode and PROTO is not cleartext */
  if (getConfig("using_ssl") == 1 && getConfig("prot_mode") != 'C') {
    return m_dataSSL->write(buf, size);
  } else {
    return KSocks::self()->write(m_dataConn, buf, size);
  }

  return 0;
}

int FtpSocket::ftpDataReadLine(char *buf, int max)
{
  if (max == 0)
    return 0;

#define BUF_LEN 512 * sizeof(char)

  char *read_buf = (char*) malloc(BUF_LEN);
  char *c = 0L;
  int read_bytes = 0;
  QCString s_buf;
  QString tmp = "";

  if (m_dataReadBuf.find('\n') == -1) {
    while (!c) {
      int b_read;
      memset(read_buf, 0, BUF_LEN);

      if ((b_read = ftpDataRead(read_buf, BUF_LEN-1)) == -1) {
        return -1;
      }

      if (b_read == 0)
        break;

      read_bytes += b_read;

      // Search for the \n (if it exists)
      c = strchr(read_buf, '\n');
      s_buf += read_buf;
    }

    tmp = s_buf;
  }

  free(read_buf);

  if (m_dataReadBuf.length() > 0)
    tmp.prepend(m_dataReadBuf);
  else if (read_bytes == 0)
    return 0;

  int n_pos = tmp.find('\n');
  if (n_pos == -1)
    n_pos = tmp.length() - 1;

  memset(buf, 0, max);
  strcpy(buf, tmp.mid(0, n_pos+1).ascii());
  read_bytes = strlen(buf);

  // Save the remaining bytes
  tmp.remove(0, n_pos+1);
  m_dataReadBuf = tmp;

  return read_bytes;
}

int FtpSocket::ftpPrepareTLS(int fd)
{
  if (getConfig("using_ssl") != 1)
    return fd;

  m_dataSSL = SSLNewData(fd);
  if (m_sslLastResult != 1) {
    ::close(fd);
    return -1;
  }

  return fd;
}

int FtpSocket::ftpInitDataSocket(bool noTLS)
{
  int data_sock;
  struct sockaddr addr;
  ksocklen_t l;
  fd_set mask;

  FD_ZERO(&mask);
  FD_SET(m_data, &mask);
  
  // If passive socket, return the currently connected socket
  if (getConfig("passive_mode") == 1)
    return noTLS ? m_data : ftpPrepareTLS(m_data);

  if (KSocks::self()->select(m_data + 1, &mask, NULL, NULL, 0L) == 0) {
    ::close(m_data);
    return -1;
  }

  l = sizeof(addr);
  if ((data_sock = KSocks::self()->accept(m_data, &addr, &l)) > 0)
    return noTLS ? data_sock : ftpPrepareTLS(data_sock);

  ::close(m_data);
  return -1;
}

bool FtpSocket::ftpDataCommand(const QString &command, const QString &path, char mode, bool noTLS,
                                    filesize_t offset)
{
  if (!m_isLoggedIn)
    return false;

  // If the protection is set to 'C'-lear then we shouldn't use TLS
  if (getConfig("prot_mode") == 'C')
    noTLS = true;

  // First, set the transfer mode
  QString cmd = "TYPE ";
  cmd += mode;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    return false;
  }

  // If in distributed FTPD mode, we must first send PRET <command> so the
  // server will know where to send us.
  if (getConfig("feat_pret") == 1) {
    QString p_cmd = "PRET ";
    p_cmd += command;
    if (!path.isEmpty()) {
      p_cmd += " ";
      p_cmd += path;
    }

    if (!ftpSendCommand(p_cmd) || m_responseBuf[0] != '2') {
      // PRET not supported or just permission/fs problems - have to check
      if (ftpIsRespCode("530") || ftpIsRespCode("550")) {
        // yep, the file could not be fetched, return false and don't even
        // try to init the transfer
        return false;
      }

      setConfig("feat_pret", 0);
    }
  }

  // Now try to init a data connection
  if (!ftpInitDataConnection()) {
    qDebug("Init failed.");
    return false;
  }

  // Check if we have an offset (for RETR/STOR)
  if (offset > 0) {
    char tmp[100];
    sprintf(tmp, "REST %ld", (long)offset);

    if (!ftpSendCommand(tmp) || m_responseBuf[0] != '3') {
      // Disable resuming
      setConfig("feat_resume", 0);
    }
  }

  // If path is set, add it after the command
  QString p_command = command;
  if (!path.isEmpty()) {
    p_command += " ";
    p_command += path;
  }

  if (!ftpSendCommand(p_command) || m_responseBuf[0] != '1') {
    return false;
  }

  if ((m_dataConn = ftpInitDataSocket(noTLS)) < 0) {
    ftpCloseDataCommand();
    return false;
  }

  return true;
}

bool FtpSocket::ftpCloseDataCommand()
{
  ftpCloseDataSockets();

  if (ftpReadResponse() != '2') {
    // No transfer complete msg ?!
    return false;
  }

  return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////  DIR LIST PARSING METHODS  //////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

FTPEntry FtpSocket::ftpStat(const KURL &url)
{
  KURL remoteUrl = getClientInfoUrl();
  remoteUrl.setPath(url.path());
  
  // Get info about a file
  FTPEntry entry;
  bool isDir = checkIsDir(remoteUrl);

  QString parentDir = remoteUrl.directory();

  if (isDir) {
    // It is a directory
    entry.type = 'd';
    entry.name = remoteUrl.fileName();
    entry.permissions = "drwxrwxrwx";

    return entry;
  }
  
  // Check the cache
  FTPCacheItem *item = FTPCache().listGetFromCache(remoteUrl.upURL());
  FTPDirList list;
  
  if (item) {
    list = item->getDirList();
  } else {
    m_stateInfo.enterSocketState(S_LIST);
    
    if (protoDirList(remoteUrl.upURL())) {
      FTPCache().listCache(remoteUrl.upURL(), m_lastDirList);
      list = m_lastDirList;
    }
    
    m_stateInfo.enterSocketState(S_IDLE);
  }
  
  for (FTPDirList::iterator i = list.begin(); i != list.end(); ++i) {
    if ((*i).name() == remoteUrl.fileName()) {
      entry = (*i).m_ftpEntry;
      break;
    }
  }
  
  return entry;
}

bool FtpSocket::protoDirList(const KURL &url)
{
  if (!m_isLoggedIn) {
    // We are not logged in
    return false;
  }

  QString path = url.path();

  if (getConfig("stat_listings") == 1) {
    // The server supports directory listings via the STAT command - that is it
    // sends the usual listings via the control channel.
    if (!ftpCwd(path)) {
      // Error while accessing the directory
      return false;
    }
  
    m_stateInfo.enterSocketState(S_LIST);
    m_lastDirList.clear();
    
    QString cmd = "STAT ";
    cmd += path;
    if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
      // The server has problems with the stat command after all, fallback
      setConfig("stat_listings", 0);
      m_stateInfo.enterSocketState(S_IDLE);
      
      return protoDirList(url);
    }
    
    // The output should now be ready, let's parse
    FTPEntry entry;
    FtpDirectoryParser d_parser;
    
    QStringList list = QStringList::split('\n', m_lastMultiResp);
    for (QStringList::iterator i = ++list.begin(); i != list.end(); ++i) {
      if (d_parser.parseLine((*i).mid(4), entry) && !entry.name.isEmpty()) {
        entry.name = m_remoteEncoding->decode(entry.name.ascii());
        entry.link = m_remoteEncoding->decode(entry.name.ascii());
      
        m_lastDirList.append(FTPDirectoryItem(entry));
      }
    }
    
    m_stateInfo.enterSocketState(S_IDLE);
  } else {
    if (!ftpDirChange(path))
      return false;
  
    m_stateInfo.enterSocketState(S_LIST);
    m_lastDirList.clear();
  
    FTPEntry entry;
    while (ftpReadList(entry)) {
      // Parse the whole dir listing
      if (!entry.name.isEmpty())
        m_lastDirList.append(FTPDirectoryItem(entry));
  
      if (m_stateInfo.abortInProgress())
        break;
    }
  
    (void) ftpDirClose();
    m_stateInfo.enterSocketState(S_IDLE);
  }

  sigLogUpdate(3, i18n("Directory listing complete."));
  return true;
}

bool FtpSocket::ftpDirChange(const QString &path, const QString &params)
{
  if (!m_isLoggedIn)
    return false;

  if (!ftpCwd(path)) {
    // Error while accessing the directory
    return false;
  }

  // TODO: use a config entry to specify a custom list command
  if (!ftpDataCommand("LIST -a", params, 'A')) {
    // List command & data connection failed
    return false;
  }

  return true;
}

bool FtpSocket::ftpDirClose()
{
  if (!ftpCloseDataCommand())
    return false;

 return true;
}

bool FtpSocket::ftpReadList(FTPEntry &entry)
{
  bool res = false;
  char buffer[1024];
  FtpDirectoryParser d_parser;

  while (ftpDataReadLine(buffer, sizeof(buffer)) != 0) {
    res = d_parser.parseLine(buffer, entry);

    if (res) {
      entry.name = m_remoteEncoding->decode(entry.name.ascii());
      entry.link = m_remoteEncoding->decode(entry.name.ascii());

      return res;
    }
  }

  return res;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////  FTP COMMAND IMPLEMENTATION  ////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int FtpSocket::ftpCwd(const QString &dir)
{
  if (!ftpSendCommand(QString("CWD %1").arg(dir)) || m_responseBuf[0] != '2') {
    if (ftpIsRespCode("550")) {
      // Dispatch error message and throw an exception so the processing will stop
      m_errorHandler->dispatchError(EC_UNABLE_TO_ENTER_DIR, ErrorData(dir));
      FTP_EXCEPTION;
    }

    return 0;
  }

  // Now send the PWD command so we know where we actually are
  if (!ftpSendCommand("PWD") || m_responseBuf[0] != '2')
    return 0;

  // 257 "/home/default/path"
  m_lastDir = m_remoteEncoding->decode(QCString(m_responseBuf));
  int first = m_lastDir.find('"') + 1;
  m_lastDir = m_lastDir.mid(first, m_lastDir.findRev('"') - first);

  return 1;
}

int FtpSocket::ftpCwdCheck(const QString &dir)
{
  return ftpSendCommand(QString("CWD %1").arg(dir)) && m_responseBuf[0] == '2' ? 1 : 0;
}

int FtpSocket::ftpSize(const QString &file, char mode)
{
  // Set the datatype
  QString cmd = "TYPE ";
  cmd += mode;
  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2')
    return -1;

  cmd = "SIZE ";
  cmd += file;
  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2')
    return -1;

  // Return the size
  m_lastSize = atol(m_responseBuf + 4);
  return 1;
}

int FtpSocket::ftpGet(const QString &source, const QString &destination)
{
  FILE *f;
  filesize_t offset = 0;
  char mode = 'I';
  QDir fs;
  
  m_stateInfo.enterSocketState(S_TRANSFER);

  /* Get the mode for the desired file (ascii/binary) */
  mode = KFTPCore::Config::self()->ftpMode(source);

  /* Check the sizes (and if the file exists) */
  filesize_t size_local = 0;
  if (ftpSize(source, mode) == -1) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_FILE_NOT_FOUND);
    FTP_EXCEPTION;
  }

  filesize_t size_remote = m_lastSize;

  /* Does the destination (local) file exist ? */
  if (fs.exists(destination)) {
    // If the error hasn't been handled, dispatch the signal and
    // abort processing.
    if (m_errorHandler->dispatchError(EC_FILE_EXISTS_DOWNLOAD, ErrorData(destination, F_STAT(source))))
      FTP_EXCEPTION;

    // If we are still here, that means that the signal has been
    // handled. Check the return argument from the handler.
    if (m_errorHandler->returnCode(EC_FILE_EXISTS_DOWNLOAD, ErrorData(destination)) != KIO::R_OVERWRITE) {
      setConfig("feat_resume", 1);

      // Check if we can resume
      f = fopen(destination.local8Bit(), "a");
      if (f != NULL) {
        if (fseek(f, 0, SEEK_END) != 0) {
          fclose(f);
          
          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_FD_ERR);
          FTP_EXCEPTION;
        }

        size_local = ftell(f);

        if (size_local < size_remote && size_local > 0) {
          offset = size_local;
        }
      } else {
        // Dispatch error message and throw an exception so the processing will stop
        m_errorHandler->dispatchError(EC_FD_ERR);
        FTP_EXCEPTION;
      }
    } else {
      setConfig("feat_resume", 0);

      // Overwrite the file
      f = fopen(destination.local8Bit(), "w");

      if (f == NULL) {
        // Dispatch error message and throw an exception so the processing will stop
        m_errorHandler->dispatchError(EC_FD_ERR);
        FTP_EXCEPTION;
      }
    }

    // Invalidate the error message
    m_errorHandler->errorDone(EC_FILE_EXISTS_DOWNLOAD, ErrorData(destination));
  } else {
    // Check if the file path exists, if not, create it
    QString dest_dir = fs.cleanDirPath(destination.mid(0, destination.findRev('/')));

    // Is the path available ?
    if (!fs.exists(dest_dir)) {
      // Create all dirs
      QString full_path;
      for(register int i = 1;i<=dest_dir.contains('/');i++) {
        full_path += "/" + dest_dir.section('/', i, i);
        if (!fs.exists(full_path)) {
          fs.mkdir(full_path);
        }
      }
    }

    // Create the file
    f = fopen(destination.local8Bit(), "w");

    if (f == NULL) {
      // Dispatch error message and throw an exception so the processing will stop
      m_errorHandler->dispatchError(EC_FD_ERR);
      FTP_EXCEPTION;
    }
  }

  if (size_remote == size_local && getConfig("feat_resume") == 1) {
    /* File is already downloaded - skip it if enabled */
    fclose(f);
    processedSize(size_local);

    m_stateInfo.enterSocketState(S_IDLE);
    return 1;
  }

  // Get the file's date-time via MDTM
  time_t p_modtime = 0;
  if (getConfig("feat_mdtm") == 1) {
    QString cmd = "MDTM ";
    cmd += source;

    if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
      setConfig("feat_mdtm", 0);
    } else {
      // 213 20031222105809
      // 213 YYYYMMDDhhmmss
      struct tm p_dt;
      QString tmp(m_responseBuf);

      tmp.remove(0, 4);
      p_dt.tm_year = tmp.left(4).toInt() - 1900;
      p_dt.tm_mon = tmp.mid(4, 2).toInt() - 1;
      p_dt.tm_mday = tmp.mid(6, 2).toInt();
      p_dt.tm_hour = tmp.mid(8, 2).toInt();
      p_dt.tm_min = tmp.mid(10, 2).toInt();
      p_dt.tm_sec = tmp.mid(12, 2).toInt();
      p_modtime = mktime(&p_dt);
    }
  }

  // Init data conn
  if (!ftpDataCommand("RETR", source, mode, false, offset)) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_DATA_CONN_ERR);
    FTP_EXCEPTION;
  }

  m_speedLimiter.transferStart();

  // Open the file and grab the data

/* Variable buffer config thingies - for download */
#define BUFMOD   512   /* Increase/Decrease buffer by BUFMOD bytes */
#define BUFDELTA 64    /* Fudge */
#define BUFMIN   4096  /* Minimum buffer size */
#define BUFMAX   32768 /* Maximum buffer size */

  char       *buffer;
  int        bufsize = BUFMIN;
  filesize_t bytes_received = offset;
  ssize_t b_read = 1;
  time_t stallTimeout = 0;

  setOffset(offset);
  emit sigResumedOffset(offset);

  buffer = (char *) malloc(sizeof(char) * bufsize);

  while (b_read) {
    int allowed = m_speedLimiter.allowedBytes();
    
    if (allowed > bufsize)
      allowed = bufsize;
    else if (allowed == 0)
      continue;
      
    b_read = ftpDataRead(buffer, allowed);
     
    if (b_read > 0) {
      fwrite(buffer, b_read, 1, f);
      bytes_received += b_read;
      
      processedSize(b_read);
    }
    
#ifdef VARIABLE_BUFFER
    if (b_read >= bufsize - BUFDELTA) { /* We've filled the whole buffer -> increase size */
      if (bufsize + BUFMOD <= BUFMAX) {
        bufsize += BUFMOD;
        buffer = (char *) realloc(buffer, bufsize);
      }
    } else if (b_read <= bufsize - BUFDELTA - 1) { /* We haven't filled the whole buffer -> decrease size */
      if (bufsize - BUFMOD >= BUFMIN) {
        bufsize -= BUFMOD;
        buffer = (char *) realloc(buffer, bufsize);
      }
    }
#endif

    if (b_read <= 0 && getSpeed() == 0) {
      if (stallTimeout == 0) {
        // We are stalled for the first time
        stallTimeout = time(0);
      } else {
        // We have been stalled for some time, let's get the duration
        if (time(0) - stallTimeout > KFTPCore::Config::dataTimeout()) {
          Socket::disconnect();

          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_TIMEOUT);
          FTP_EXCEPTION;
        }
      }
    } else {
      stallTimeout = 0;
    }

    if (m_stateInfo.abortInProgress())
      break;
  }
  fflush(f);
  fclose(f);

  free(buffer);

#undef BUFMOD
#undef BUFDELTA
#undef BUFMIN
#undef BUFMAX

  // Set the modification date we got at MDTM
  if (getConfig("feat_mdtm") == 1) {
    // Access time is actualy this moment ;) and modification time
    // should be set to what we got when doing MDTM

    utimbuf p_timbuf;
    p_timbuf.actime = time(0L);
    p_timbuf.modtime = p_modtime;
    utime(destination.latin1(), &p_timbuf);
  }

  if (!ftpCloseDataCommand()) {
    /* An error while closing ? */
  }

  if (bytes_received != size_remote) {
    /* Incomplete file download */
    qDebug("[WARNING] File download is incomplete");
  } else {
    ftpCheckCRC(source, destination);
  }

  emit sigLogUpdate(3, i18n("Transferred 1 byte.", "Transferred %n bytes.", bytes_received));

  m_stateInfo.enterSocketState(S_IDLE);
  
  return 1;
}

int FtpSocket::ftpPut(const QString &source, const QString &destination)
{
  int fd;
  filesize_t offset = 0;
  char mode = 'I';
  QDir fs;
  
  m_stateInfo.enterSocketState(S_TRANSFER);

  /* Get the mode for the desired file (ascii/binary) */
  mode = KFTPCore::Config::self()->ftpMode(source);

  /* Check the sizes (and if the file exists) */
  filesize_t size_local;
  filesize_t size_remote = 0;

  if (!fs.exists(source)) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_FILE_NOT_FOUND);
    FTP_EXCEPTION;
  } else {
    /* Get local filesize */
    fd = open(source.local8Bit(), O_RDONLY);

    if (fd < 0) {
      // Dispatch error message and throw an exception so the processing will stop
      m_errorHandler->dispatchError(EC_FD_ERR);
      FTP_EXCEPTION;
    } else {
      size_local = lseek(fd, 0, SEEK_END);
      lseek(fd, 0, SEEK_SET);
    }
  }

  /* Does the destination (remote) file exist ? */
  if (ftpSize(destination, mode) == 1) {
    // If the error hasn't been handled, dispatch the signal and
    // abort processing.
    if (m_errorHandler->dispatchError(EC_FILE_EXISTS_UPLOAD, ErrorData(destination, F_STAT(destination))))
      FTP_EXCEPTION;

    // If we are still here, that means that the signal has been
    // handled. Check the return argument from the handler.
    if (m_errorHandler->returnCode(EC_FILE_EXISTS_UPLOAD, ErrorData(destination)) != KIO::R_OVERWRITE) {
      setConfig("feat_resume", 1);
      size_remote = m_lastSize;

      if (size_remote < size_local) {
        offset = size_remote;
        lseek(fd, offset, SEEK_SET);
      }
    }

    // Invalidate the error message
    m_errorHandler->errorDone(EC_FILE_EXISTS_UPLOAD, ErrorData(destination));
  }

  if (size_remote == size_local && getConfig("feat_resume") == 1) {
    /* File is already on the server - skip it if enabled */
    close(fd);
    processedSize(size_remote);
    
    m_stateInfo.enterSocketState(S_IDLE);
    return 1;
  }

  // Does the destination directory exist ?
  if (size_remote == 0 && !ftpCwdCheck(KURL(destination).directory())) {
    // It doesn't - create it
    QString dest_dir = KURL(destination).directory();
    QString full_path;
    for(register int i = 1; i <= dest_dir.contains('/'); i++) {
      full_path += "/" + dest_dir.section('/', i, i);

      // Create the directory
      ftpSendCommand(QString("MKD %1").arg(full_path));
    }
  }

  // Init data conn
  if (!ftpDataCommand("STOR", destination, mode, false, offset)) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_DATA_CONN_ERR);
    FTP_EXCEPTION;
  }

  m_speedLimiter.transferStart();

/* Variable buffer config thingies - for upload */
#define BUFMOD   512   /* Increase/Decrease buffer by BUFMOD bytes */
#define BUFDELTA 64    /* Fudge */
#define BUFMIN   4096  /* Minimum buffer size */
#define BUFMAX   32768 /* Maximum buffer size */

  // Open the file and grab the data
  char       *buffer;
  int        bufsize = BUFMIN;
  filesize_t bytes_sent = offset;
  ssize_t b_read = 1;
  time_t stallTimeout = 0;

  setOffset(offset);
  emit sigResumedOffset(offset);

  buffer = (char *) malloc(sizeof(char) * bufsize);

  while (b_read) {
    int allowed = m_speedLimiter.allowedBytes();
    
    if (allowed > bufsize)
      allowed = bufsize;
    else if (allowed == 0)
      continue;
      
    b_read = read(fd, buffer, allowed);
    ssize_t b_sent = ftpDataWrite(buffer, b_read);
    
    if (b_sent > 0) {
      bytes_sent += b_sent;
      processedSize(b_sent);
    }

#ifdef VARIABLE_BUFFER
    if (b_sent >= bufsize - BUFDELTA) { /* We've filled the whole buffer -> increase size */
      if (bufsize + BUFMOD <= BUFMAX) {
        bufsize += BUFMOD;
        buffer = (char *) realloc(buffer, bufsize);
      }
    } else if (b_sent <= bufsize - BUFDELTA - 1) { /* We haven't filled the whole buffer -> decrease size */
      if (bufsize - BUFMOD >= BUFMIN) {
        bufsize -= BUFMOD;
        buffer = (char *) realloc(buffer, bufsize);
      }
    }
#endif

    if (b_sent <= 0) {
      if (stallTimeout == 0) {
        // We are stalled for the first time
        stallTimeout = time(0);
      } else {
        // We have been stalled for some time, let's get the duration
        if (time(0) - stallTimeout > KFTPCore::Config::dataTimeout()) {
          Socket::disconnect();

          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_TIMEOUT);
          FTP_EXCEPTION;
        }
      }
    } else {
      stallTimeout = 0;
    }

    if (m_stateInfo.abortInProgress())
      break;
  }
  close(fd);

  free(buffer);

#undef BUFMOD
#undef BUFDELTA
#undef BUFMIN
#undef BUFMAX

  if (!ftpCloseDataCommand()) {
    /* An error while closing ? */
  }

  if (bytes_sent != size_local) {
    /* Incomplete file upload */
    qDebug("[WARNING] File upload is incomplete");
  } else {
    ftpCheckCRC(destination, source);
  }

  emit sigLogUpdate(3, i18n("Transferred 1 byte.", "Transferred %n bytes.", bytes_sent));
  
  m_stateInfo.enterSocketState(S_IDLE);

  return 1;
}

int FtpSocket::ftpRename(const QString &source, const QString &destination)
{
  QString cmd = "RNFR ";
  cmd += source;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '3') {
    /* File does not exist */
    return -1;
  }

  cmd = "RNTO " + destination;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    /* Permission denied ? */
    return -1;
  }

  return 1;
}

int FtpSocket::ftpDelete(const QString &path)
{
  QString cmd = "DELE ";
  cmd += path;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    /* Unable to delete */
    return -1;
  }

  return 1;
}

int FtpSocket::ftpChmod(int mode, const QString &path)
{
  QString cmd;
  cmd.sprintf("SITE CHMOD %.3d %s", mode, path.ascii());

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    /* Unable to chmod */
    return -1;
  }

  return 1;
}

int FtpSocket::ftpMkdir(const QString &dir)
{
  QString cmd = "MKD ";
  cmd += dir;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    /* Unable to create directory */
    return -1;
  }

  return 1;
}

int FtpSocket::ftpRmdir(const QString &dir)
{
  QString cmd = "RMD ";
  cmd += dir;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    /* Unable to remove directory */
    return -1;
  }

  return 1;
}

void FtpSocket::ftpAbortFxp(FtpSocket *client)
{
  // Send the abort command
  ftpSendCommand("ABOR", false);
  client->ftpSendCommand("ABOR", false);

  qDebug("FXP starting reset wait phase.");
  
  // Read all that has been returned to us
  char resp;
  do {
    resp = ftpReadResponse();
  } while (resp != '2');
  
  qDebug("FXP reset #1 complete.");
  
  // Now read all that has been returned to the other host
  do {
    resp = client->ftpReadResponse();
  } while (resp != '2');
  
  qDebug("FXP abort complete.");
}

int FtpSocket::ftpFxpTransfer(const QString &source, const QString &destination, FtpSocket *client)
{
  int errCode = 0;
  int pos;
  QString cmd;
  QString ip_txt;

  if (m_socketAddress.sin_family != client->m_socketAddress.sin_family) {
    // Different families, will fail
    return -11;
  }

  // Do a FXP transfer - source is always this client
  char mode = KFTPCore::Config::self()->ftpMode(source);

  filesize_t offset = 0;
  filesize_t srcSize = 0;
  filesize_t dstSize = 0;

  // Does the source file exist ?
  if (ftpSize(source, mode) == 1) {
    srcSize = m_lastSize;
  } else {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_FILE_NOT_FOUND);
    FTP_EXCEPTION;
  }
  
  if (client->ftpSize(destination, mode) == 1) {
    // If the error hasn't been handled, dispatch the signal and
    // abort processing.
    if (m_errorHandler->dispatchError(EC_FILE_EXISTS_FXP, ErrorData(destination, F_STAT_FXP(destination), F_STAT(source))))
      FTP_EXCEPTION;
    
    // If we are still here, that means that the signal has been
    // handled. Check the return argument from the handler.
    if (m_errorHandler->returnCode(EC_FILE_EXISTS_FXP, ErrorData(destination)) != KIO::R_OVERWRITE) {
      setConfig("feat_resume", 1);
      dstSize = client->m_lastSize;

      if (dstSize < srcSize) {
        offset = dstSize;
      }
    }

    // Invalidate the error message
    m_errorHandler->errorDone(EC_FILE_EXISTS_FXP, ErrorData(destination));
  }

  // Does the destination directory exist ?
  if (dstSize == 0 && !client->ftpCwdCheck(KURL(destination).directory())) {
    // It doesn't - create it
    QString dest_dir = KURL(destination).directory();
    QString full_path;
    for(register int i = 1; i <= dest_dir.contains('/'); i++) {
      full_path += "/" + dest_dir.section('/', i, i);

      // Create the directory
      client->ftpSendCommand(QString("MKD %1").arg(full_path));
    }
  }
  
  /*
    WARNING!
    -------------------------------------------------------------------------------------
    
    From this point on (after the ssl stuff is initialised) any abortion must go trough
    the "abort" label! In any other case both sockets can become unusable!
  */
  bool sentSSCN = false;
  bool sendCPSV = false;
  bool changedTls = false;
  
  if (   (getConfig("using_ssl") == 1 || client->getConfig("using_ssl") == 1)
      && (getConfig("feat_sscn") == 0 || client->getConfig("feat_sscn") == 0)
      && (getConfig("feat_cpsv") == 0 || client->getConfig("feat_cpsv") == 0) ) {
    // We are in SSL mode and we don't have SSCN/CPSV support - just disable the SSL mode temporarily
    ftpChangeTLSDataProt('C');
    client->ftpChangeTLSDataProt('C');
    changedTls = true;
  }

  if (getConfig("using_ssl") == 1 && client->getConfig("using_ssl") == 1) {
    if (getConfig("feat_sscn") == 1 && client->getConfig("feat_sscn") == 1) {
      // Ok, both servers support SSCN (aka new extension to do FXP transfers over SSL), send the
      // proper command to enable SSCN
      if (!ftpSendCommand("SSCN ON") || m_responseBuf[0] != '2')
        return -1;

      if (!client->ftpSendCommand("SSCN ON") || client->m_responseBuf[0] != '2')
        return -1;

      sentSSCN = true;
    } else if (getConfig("feat_cpsv") == 1 && client->getConfig("feat_cpsv") == 1) {
      // Ok, server support CPSV, send it instead of PASV
      sendCPSV = true;
    }
  }

  if (dstSize == srcSize && getConfig("feat_resume") == 1) {
    errCode = 1;
    goto abort;
  }

  if (offset > 0) {
    QString tmp;
    tmp.sprintf("REST %lld", offset);

    if (!ftpSendCommand(tmp) || m_responseBuf[0] != '3') {
      errCode = -1;
      goto abort;
    }

    if (!client->ftpSendCommand(tmp) || client->m_responseBuf[0] != '3') {
      errCode = -1;
      goto abort;
    }
  }

  // Set TYPE for both clients
  cmd = "TYPE ";
  cmd += mode;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    errCode = -1;
    goto abort;
  }

  if (!client->ftpSendCommand(cmd) || client->m_responseBuf[0] != '2') {
    errCode = -1;
    goto abort;
  }
  
  // If in distributed FTPD mode, we must first send PRET <command> so the
  // server will know where to send us.
  if (getConfig("feat_pret") == 1) {
    QString p_cmd = "PRET RETR ";
    p_cmd += source;

    if (!ftpSendCommand(p_cmd) || m_responseBuf[0] != '2') {
      // PRET not supported or just permission/fs problems - have to check
      if (ftpIsRespCode("530") || ftpIsRespCode("550")) {
        // Dispatch error message and throw an exception so the processing will stop
        m_errorHandler->dispatchError(EC_FILE_NOT_FOUND);
        FTP_EXCEPTION;
      }

      setConfig("feat_pret", 0);
    }
  }

  // Init the PASV mode
  if (sendCPSV)
    cmd = "CPSV";
  else
    cmd = "PASV";
  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '2') {
    errCode = -2;
    goto abort;
  }

  ip_txt = m_responseBuf;
  pos = ip_txt.find('(')+1;
  ip_txt = ip_txt.mid(pos, ip_txt.find(')') - pos);

  // Init the PORT mode
  cmd = "PORT ";
  cmd += ip_txt;

  if (!client->ftpSendCommand(cmd) || client->m_responseBuf[0] != '2') {
    errCode = -2;
    goto abort;
  }

  // STOR on destination, RETR on source
  cmd = "STOR ";
  cmd += destination;

  if (!client->ftpSendCommand(cmd) || client->m_responseBuf[0] != '1') {
    errCode = -3;
    goto abort;
  }

  cmd = "RETR ";
  cmd += source;

  if (!ftpSendCommand(cmd) || m_responseBuf[0] != '1') {
    errCode = -3;
    goto abort;
  }
  
  // Set the offset
  setOffset(offset);
  emit sigResumedOffset(offset);
  
  // Disable the error handlers
  m_errorHandler->setHandlerEnabled(false);
  client->m_errorHandler->setHandlerEnabled(false);

  // Now wait, until the transfer is complete
  while (1) {
    // Check the server's response every 0.5 sec
    ::usleep(500000);

    // If there is anything to read - read it!
    try {
      if (ftpCanRead() || client->ftpCanRead()) {
        if (ftpReadResponse() != '\0') {
          // Read the last response from the other
          client->ftpReadResponse();
          break;
        } else if (client->ftpReadResponse() != '\0') {
          // Read the last response from this one
          ftpReadResponse();
          break;
        }
      }
    } catch (...) {
      // Ignore "timeouts"
    }

    // Check for abortion
    if (m_stateInfo.abortInProgress()) {
      ftpAbortFxp(client);
      break;
    }
  }
  
  // Reenable the error handlers
  m_errorHandler->setHandlerEnabled(true);
  client->m_errorHandler->setHandlerEnabled(true);

  if (m_stateInfo.abortInProgress()) {
    errCode = -4;
    goto abort;
  }

  // Finished ? Check the responses...
  if (m_responseBuf[0] != '2' || client->m_responseBuf[0] != '2') {
    errCode = -5;
    goto abort;
  }

  // No errors :)
  errCode = 1;

abort:
  if (errCode < 0) {
    // Abort transfer
    if (client->ftpCanRead())
      client->ftpReadResponse();

    if (ftpCanRead())
      ftpReadResponse();
  }
  
  if (changedTls) {
    ftpChangeTLSDataProt(getConfig("prot_mode"));
    client->ftpChangeTLSDataProt(client->getConfig("prot_mode"));
  }

  // Disable SSCN if it was activated
  if (sentSSCN) {
    ftpSendCommand("SSCN OFF");
    client->ftpSendCommand("SSCN OFF");
  }

  return errCode;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////  AUXILARY METHODS ///////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool FtpSocket::ftpCheckCRC(const QString &remote, const QString &local)
{
  // Currently not implemented
  /*if (getConfig("feat_xcrc") == 1 && ftpSendCommand(QString("XCRC %1").arg(remote)) && m_responseBuf[0] == '2') {
    // 250 361F12...
    long l_fileCRC = KCRCChecker::getFileCRC(local.ascii());
    QString fileCRC;
    fileCRC.sprintf("%08lX", l_fileCRC);

    QString serverCRC(m_responseBuf);
    serverCRC.remove(0, 4);
    serverCRC = serverCRC.stripWhiteSpace();

    if (fileCRC == serverCRC) {
      sigLogUpdate(3, i18n("CRC check matches - file integrity was verified."));
      return true;
    } else {
      sigLogUpdate(4, i18n("Warning: CRC check failed. Downloaded file may be corrupted."));
      return false;
    }
  }*/

  return true;
}

}
#include "ftpsocket.moc"
