/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Specialized list to display a list of photos
 * 
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */
#ifndef PHOTOLISTVIEW_H
#define PHOTOLISTVIEW_H

#include <kurl.h>
#include <qstring.h>
#include <qpixmap.h>
#include <qptrlist.h>
#include <kfileitem.h>
#include <klistview.h>
#include <qstringlist.h>
#include <qdragobject.h>


namespace KIO { class Job; class PreviewJob; }

class PhotoListViewItem;

/**
 * Provides all the functionality of a KListViewItem object with the addition
 * of an image preview mechanism and specialized signals.
 * @short KListView to suit our needs.
 * @author M. Asselstine <asselsm@gmail.com>
 */
class PhotoListView : public KListView
{
Q_OBJECT

public:
    PhotoListView(QWidget *parent, const char *name);  
    ~PhotoListView();  
      
    PhotoListViewItem* lastPhoto() const;
    PhotoListViewItem* firstSelectedPhoto() const;
    PhotoListViewItem* nextSelectedPhoto(PhotoListViewItem *current) const;
    PhotoListViewItem* getNextPhoto(PhotoListViewItem *current) const;
    
    void removeSelected();
    int numSelected() const;
    void getPreview(const KURL& url);
    void getPreviews(const KURL::List& url);
    const QPixmap& getBorderImage() { return m_border; }
    
protected:
    virtual QDragObject* dragObject();
    virtual void resizeEvent(QResizeEvent * e);
    virtual void viewportPaintEvent(QPaintEvent *e);
    virtual bool acceptDrag(QDropEvent *event) const;
      
signals:
    void hasValidSelection(bool);
    void hasSingleSelection(bool);
    
private slots:
    void slotSelectionUpdate();
    void showRMBMenu(QListViewItem*, const QPoint &pt, int);
    
    void slotResult(KIO::Job *);
    void slotFailed(const KFileItem*);
    void gotPreview(const KFileItem*, const QPixmap&);
    
private:
    QPixmap m_border;
    QPtrList<KIO::Job> m_previewJobs;
};


/**
 * Provides all the functionality of a KListViewItem with some specific additions
 * to allow for image preview and setting and retrieval of values in the context
 * of their use instead of simply via index.
 * @short KListViewItem to suit our needs.
 * @author M. Asselstine <asselsm@gmail.com>
 */
class PhotoListViewItem : public QObject, public KListViewItem 
{
Q_OBJECT

public:
    PhotoListViewItem(KListView *parent, const KURL &url);
    PhotoListViewItem(KListView *parent, const KURL &url, QListViewItem *after);
    ~PhotoListViewItem();
    
    void init(const KURL &url);
    
    virtual int width(const QFontMetrics &, const QListView *, int c) const;
    virtual void paintCell(QPainter *, const QColorGroup &cg, int column, int width, int alignment);
    
    void setDescription(const QString &str);
    void setTags(const QStringList &strlst);
    void setTitle(const QString &str);
    void setPublic(bool isPublic = TRUE);
    void setFamily(bool isPublic = TRUE);
    void setFriends(bool isPublic = TRUE);
    void setSize(const QString &size);
    
    QString title() const;
    QString description() const;
    QStringList tags() const;
    bool isPublic() const;
    bool isFamily() const;
    bool isFriends() const;
    QString size() const;
    const KURL& url() const;
    int rotation() const;
    
    QPixmap previewImage() const;
    
public slots:
    void rotatePhoto();
       
protected:
    void setHeight(int height);

private:
    KURL m_URL;
    bool m_public;
    bool m_family;
    bool m_friends;
    int m_rotation;
    QString m_desc;
    QString m_size;
    QString m_title;
    QStringList m_tags;
    
    QPixmap m_preview;    
    
private:
    QString accessString() const;
    
    friend class PhotoListView;
};

#endif
