// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/*
 * Parts of this file are
 * Copyright 2003 Waldo Bastian <bastian@kde.org>
 * Copyright 2005-2006 Wilfried Huss <Wilfried.Huss@gmx.at>
 *
 * These parts are free software; you can redistribute and/or modify
 * them under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.
 */

#include <config.h>

#include "ligaturepart.h"
#include "ligature.h"
#include "kvs_debug.h"
#include "toolindicator.h"

#include <k3urldrag.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kconfig.h>
#include <kedittoolbar.h>
#include <kglobal.h>
#include <kkeydialog.h>
#include <kicon.h>
#include <klibloader.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kstatusbar.h>
#include <kstandardaction.h>
#include <KTemporaryFile>
#include <ktoggleaction.h>
#include <ktogglefullscreenaction.h>
#include <ktoolbar.h>
#include <ktip.h>

#include <QApplication>
#include <QDragEnterEvent>
#include <QFileInfo>
#include <QKeyEvent>
#include <krecentfilesaction.h>
#include <cerrno>

#include "ligature.moc"

#define StatusBar_ID_PageNr 1
#define StatusBar_ID_PageSize 2
#define StatusBar_ID_Zoom 3


Ligature::Ligature(const QString& defaultMimeType)
  : KParts::MainWindow(),
    statusbar(0),
    view(0),
    recent(0),
    closeAction(0),
    reloadAction(0),
    fullScreenAction(0),
    isStatusBarShownInNormalMode(true),
    isToolBarShownInNormalMode(true),
    tmpFile(0)
{
  // create the viewer part

  // Try to load
  KLibFactory *factory = KLibLoader::self()->factory("ligaturepart");
  if (factory) {
    if (defaultMimeType == QString::null)
      view = static_cast<LigaturePart*>(factory->create(this, "LigaturePart"));
    else {
      QStringList args;
      args << defaultMimeType;
      view = static_cast<LigaturePart*>(factory->create(this, "LigaturePart", args));
    }
    if (!view) {
      kError() << "FATAL ERROR: Loading of the LigaturePart failed. Aborting..." << endl;
      ::exit(-1);
    }
  } else
    KMessageBox::detailedError(this,
                               i18n("<qt>A fatal error ocurred. An important library could not be found. Ligature will not work.</qt>"),
                               i18n("<qt><p>The Ligature application uses the <strong>ligaturepart</strong> library internally. "
                                    "This  library was not found on your computer.</p>"
                                    "<p>The problem is most likely a broken installation of the Ligature application. "
                                    "You could try to uninstall all instances of Ligature and then re-install the latest version.</p></qt>"),
                               i18n("Ligature - Critical Error"));

  if (view != 0) {
    QString version;
    int idx = view->metaObject()->indexOfClassInfo("Version");
    if (idx >= 0)
      version = view->metaObject()->classInfo(idx).value();

    QString expectedVersion("Version 1 for KDE 4");
    if (version != expectedVersion) {
      if (version.isEmpty())
        version = "Version 0, probaby from KDE 3.4 or older";
      KMessageBox::detailedError(this,
                                 i18n("<qt>A fatal error ocurred. An incompatible version of an important library was found. Ligature will not work.</qt>"),
                                 i18n("<qt><p>The Ligature application uses the <strong>ligaturepart</strong> library internally. "
                                      "A version of this library was found, but this version is <strong>not</strong> compatible with the current version "
                                      "of the Ligature application.</p>"
                                      "<p>Version found: <strong>%1</strong></p>"
                                      "<p>Version required: <strong>%2</strong></p>"
                                      "<p>The problem is most likely that two incompatible versions of Ligature are simultaneously installed on your computer. "
                                      "You could try to uninstall all instances of Ligature and then re-install the latest version.</p></qt>",
                                      version, expectedVersion),
                                 i18n("Ligature - Critical Error"));
      delete view;
      view = 0;
    }
  }


  if (view != 0)
    setCentralWidget(view->widget());

  // file menu
  if (view != 0) {
    KStandardAction::open(view, SLOT(slotFileOpen()), actionCollection());
    recent = KStandardAction::openRecent (this, SLOT(openUrl(const KUrl &)), actionCollection());
    reloadAction = actionCollection()->addAction("reload");
    reloadAction->setIcon(KIcon("reload"));
    reloadAction->setText(i18n("Reload"));
    connect(reloadAction, SIGNAL(triggered(bool)), view, SLOT(reload(void)));
    reloadAction->setShortcut(Qt::CTRL + Qt::Key_R);
    closeAction = KStandardAction::close(this, SLOT(slotFileClose()), actionCollection());
    connect(view, SIGNAL(fileOpened()), this, SLOT(addRecentFile()));
  }
  KStandardAction::quit (this, SLOT(slotQuit()), actionCollection());

  // view menu
  fullScreenAction = KStandardAction::fullScreen(this, SLOT(slotFullScreen()), this, this);
  actionCollection()->addAction( "fullscreen", fullScreenAction );

  createStandardStatusBarAction();
  setStandardToolBarMenuEnabled(true);

  KStandardAction::keyBindings(this, SLOT(slotConfigureKeys()), actionCollection());
  KStandardAction::configureToolbars(this, SLOT(slotEditToolbar()), actionCollection());
  actionCollection()->addAction(KStandardAction::TipofDay, "help_tipofday", this, SLOT(showTip()));

  if (view != 0) {
    // statusbar connects
    connect( view, SIGNAL( zoomChanged(const QString &) ), this,SLOT( slotChangeZoomText(const QString &) ) );
    connect( view, SIGNAL( pageChanged(const QString &) ), this,SLOT( slotChangePageText(const QString &) ) );
    connect( view, SIGNAL( sizeChanged(const QString &) ), this,SLOT( slotChangeSizeText(const QString &) ) );
    // Setup session management
    connect( this, SIGNAL( restoreDocument(const KUrl &, int) ), view, SLOT( restoreDocument(const KUrl &, int)));
    connect( this, SIGNAL( saveDocumentRestoreInfo(KConfig*) ), view, SLOT( saveDocumentRestoreInfo(KConfig*)));
  }

  setXMLFile( "ligature.rc" );
  createGUI(view);
  readSettings();
  checkActions();
  setAcceptDrops(true);

  // If ligature is started when another instance of ligature runs
  // in fullscreen mode, the menubar is switched off by default, which
  // is a nasty surprise for the user: there is no way to access the
  // menus. To avoid such complications, we switch the menubar on
  // explicitly.
  menuBar()->show();

  // Add statusbar-widgets for zoom, pagenr and format
  statusBar()->insertPermanentItem("", StatusBar_ID_PageNr, 0);
  statusBar()->insertPermanentFixedItem("XXXX%", StatusBar_ID_Zoom);
  statusBar()->changeItem("", StatusBar_ID_Zoom);

  ToolIndicator* toolIndicator = new ToolIndicator(statusBar());
  statusBar()->addPermanentWidget(toolIndicator, 0);

  statusBar()->insertPermanentItem("", StatusBar_ID_PageSize, 0);

  if (view != 0) {
    connect(view, SIGNAL(switchTool(int)), toolIndicator, SLOT(slotSwitchTool(int)));
    connect(view, SIGNAL(registerTool(int, QPixmap)), toolIndicator, SLOT(slotRegisterTool(int, QPixmap)));
  }
}


void Ligature::checkActions()
{
  if (view == 0)
    return;

  bool doc = !view->url().isEmpty();

  closeAction->setEnabled(doc);
  reloadAction->setEnabled(doc);
  fullScreenAction->setEnabled(doc);
}


Ligature::~Ligature()
{
  writeSettings();

  if (tmpFile) {
    tmpFile->setAutoRemove(true);
    delete tmpFile;
    tmpFile = 0;
  }

  delete view;
}


void Ligature::slotQuit()
{
  // If we are to quit the application while we operate in fullscreen
  // mode, we need to restore the visibility properties of the
  // statusbar, toolbar and the menus because these properties are
  // saved automatically ... and we don't want to start next time
  // without having menus.
  if (fullScreenAction->isChecked()) {
    kDebug(kvs::shell) << "Switching off fullscreen mode before quitting the application" << endl;
    showNormal();
    if (isStatusBarShownInNormalMode)
      statusBar()->show();
    if (isToolBarShownInNormalMode)
      toolBar()->show();
    menuBar()->show();
    if (view != 0)
      view->slotSetFullPage(false);
  }
  if ((view == 0) || (view->closeUrl())) {
    qApp->closeAllWindows();
    qApp->quit();
  }
}

bool Ligature::queryClose()
{
  if (view != 0)
    return view->closeUrl();
  else
    return true;
}


void Ligature::readSettings()
{
  resize(600, 300); // default size if the config file specifies no size
  setAutoSaveSettings( "General" ); // apply mainwindow settings (size, toolbars, etc.)

  KSharedConfig::Ptr config = KGlobal::config();
  config->setGroup("General");

  if (recent != 0) {
    recent->loadEntries(config.data(), "Recent Files");

    // Constant source of annoyance in KDVI < 1.0: the 'recent-files'
    // menu contains lots of files which don't exist (any longer). Thus,
    // we'll sort out the non-existent files here.
    QStringList items = recent->items();
    for ( QStringList::Iterator it = items.begin(); it != items.end(); ++it ) {
      KUrl url(*it);
      if (url.isLocalFile()) {
        QFileInfo info(url.path());
        if (!info.exists())
          recent->removeUrl(url);
      }
    }
  }
  config->setGroup("TipOfDay");
  if (config->readEntry("RunOnStart", QVariant(false)).toBool())
  {
    QTimer::singleShot(0, this, SLOT(showTip()));
  }
}


void Ligature::writeSettings()
{
  KSharedConfig::Ptr config = KGlobal::config();
  config->setGroup( "General" );
  if (recent != 0)
    recent->saveEntries(config.data(), "Recent Files");

  config->sync();
}


void Ligature::saveProperties(KConfig* config)
{
  // the 'config' object points to the session managed
  // config file.  anything you write here will be available
  // later when this app is restored
  emit saveDocumentRestoreInfo(config);
}


void Ligature::readProperties(KConfig* config)
{
  // the 'config' object points to the session managed
  // config file.  this function is automatically called whenever
  // the app is being restored.  read in here whatever you wrote
  // in 'saveProperties'
  if (view) {
    KUrl url (config->readPathEntry("URL"));
    if (url.isValid())
      emit restoreDocument(url, config->readEntry("Page", 1));
  }
}


void Ligature::addRecentFile()
{
  if (view == 0)
    return;

  // Get the URL of the opened file from the ligaturepart.
  KUrl actualURL = view->url();
  // To store the URL in the list of recent files, we remove the
  // reference part.
  actualURL.setRef(QString::null);
  recent->addUrl(actualURL);
  checkActions();
}

void Ligature::openUrl(const KUrl& url)
{
  if (view == 0)
    return;

  view->openUrl(url);
}


void Ligature::openStdin()
{
  if (view == 0)
    return;

  if (tmpFile) {
    tmpFile->setAutoRemove(true);
    delete tmpFile;
  }

  tmpFile = new KTemporaryFile;
  if (!tmpFile->open())
  {
    KMessageBox::error(this, i18n("Could not create temporary file: %1", strerror(errno)));
    return;
  }

  QByteArray buf(BUFSIZ, ' ');
  int read = 0;
  int written = 0;
  while ((read = fread(buf.data(), sizeof(char), buf.size(), stdin)) > 0)
  {
    written = tmpFile->write(buf.data(), read);
    if (read != written)
      break;
    qApp->processEvents();
  }

  if (read != 0)
  {
    KMessageBox::error(this, i18n("Could not open standard input stream: %1", strerror(errno)));
    return;
  }

  const QString tmpFileName = tmpFile->fileName();
  tmpFile->close();

  if (view->openUrl(KUrl::fromPathOrUrl(tmpFileName)))
    setCaption("stdin");
}


void Ligature::slotFullScreen()
{
  if (view == 0)
    return;

  if (fullScreenAction->isChecked()) {
    // In fullscreen mode, menu- tool- and statusbar are hidden. Save
    // the visibility flags of these objects here, so that they can
    // later be properly restored when we switch back to normal mode,
    // or before we leave the application in slotQuit()
    isStatusBarShownInNormalMode = !statusBar()->isHidden();
    statusBar()->hide();
    isToolBarShownInNormalMode = !toolBar()->isHidden();
    toolBar()->hide();
    menuBar()->hide();
    view->slotSetFullPage(true);

    // Go to fullscreen mode
    showFullScreen();

    KMessageBox::information(this, i18n("Use the Escape key to leave the fullscreen mode."), i18n("Entering Fullscreen Mode"), "leavingFullScreen");
  } else {
    showNormal();
    if (isStatusBarShownInNormalMode)
      statusBar()->show();
    if (isToolBarShownInNormalMode)
      toolBar()->show();
    menuBar()->show();
    view->slotSetFullPage(false);
  }
}


void Ligature::slotFileClose()
{
  if (view == 0)
    return;

  view->closeUrl();

  checkActions();
}

void Ligature::slotConfigureKeys()
{
#ifdef __GNUC__
#warning FIXME shortcut configuration
#endif
  /*
  KKeyDialog dlg( true, this );

  dlg.insert( actionCollection() );
  dlg.insert( view->actionCollection() );

  dlg.configure();
  */
}

void Ligature::slotEditToolbar()
{
  saveMainWindowSettings( KGlobal::config().data(), autoSaveGroup() );
  KEditToolbar dlg(factory());
  connect( &dlg, SIGNAL( newToolbarConfig() ), SLOT( slotNewToolbarConfig() ) );
  dlg.exec();
}


void Ligature::slotNewToolbarConfig()
{
  applyMainWindowSettings( KGlobal::config().data(), autoSaveGroup() );
}


void Ligature::showTip()
{
  KTipDialog::showTip(this, "ligature/tips", true);
}


void Ligature::dragEnterEvent(QDragEnterEvent *event)
{
  if (view == 0)
    return;

  if (K3URLDrag::canDecode(event))
  {
    KUrl::List urls;
    K3URLDrag::decode(event, urls);
    if (!urls.isEmpty())
    {
      KUrl url = urls.first();

      // Always try to open remote files
      if (!url.isLocalFile())
      {
        event->accept();
        return;
      }

      // For local files we only accept a drop, if we have a plugin for its
      // particular mimetype
      KMimeType::Ptr mimetype = KMimeType::findByUrl(url);
      kDebug(kvs::shell) << "[dragEnterEvent] Dragged URL is of type " << mimetype->comment() << endl;

      // Safety check
      if (view)
      {
        QStringList mimetypeList = view->supportedMimeTypes();
        kDebug(kvs::shell) << "[dragEnterEvent] Supported mime types: " << mimetypeList << endl;

        for (QStringList::Iterator it = mimetypeList.begin(); it != mimetypeList.end(); ++it)
        {
          if (mimetype->is(*it))
          {
            kDebug(kvs::shell) << "[dragEnterEvent] Found matching mimetype: " << *it << endl;
            event->accept();
            return;
          }
        }
        kDebug(kvs::shell) << "[dragEnterEvent] no matching mimetype found" << endl;
      }
    }
    event->ignore();
  }
}


void Ligature::dropEvent(QDropEvent *event)
{
  if (view == 0)
    return;

  KUrl::List urls;
  if (K3URLDrag::decode(event, urls) && !urls.isEmpty())
    view->openUrl(urls.first());
}


void Ligature::keyPressEvent(QKeyEvent *event)
{
  // The Escape Key is used to return to normal mode from fullscreen
  // mode
  if ((event->key() == Qt::Key_Escape) && (fullScreenAction->isChecked())) {
    showNormal();
    return;
  }
  // If we can't use the key event, pass it on
  event->ignore();
}


void Ligature::slotChangePageText(const QString &message)
{
  statusBar()->changeItem(" "+message+" ",StatusBar_ID_PageNr);
}


void Ligature::slotChangeSizeText(const QString &message)
{
  statusBar()->changeItem(" "+message+" ",StatusBar_ID_PageSize);
}


void Ligature::slotChangeZoomText(const QString &message)
{
  statusBar()->changeItem(" "+message+" ",StatusBar_ID_Zoom);
}
