// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: documentWidet
//
// Widget for displaying TeX DVI files.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004 Stefan Kebekus.
// Copyright (C) 2004-2006 Wilfried Huss <Wilfried.Huss@gmx.at>
//
// Distributed under the GPL.

#ifndef _documentwidget_h_
#define _documentwidget_h_

#include "ligature_export.h"

#include "dataView.h"
#include "hyperlink.h"
#include "selection.h"

#include <QPointer>
#include <QRegion>
#include <QTimer>
#include <QWidget>

class DocumentPageCache;
class PageView;
class QAction;
class QMouseEvent;
class QPaintEvent;
class RenderedDocumentPagePixmap;
class QMenu;

/* DocumentWidget */

class LIGATURECORE_EXPORT DocumentWidget : public QWidget, public DataView
{
  Q_OBJECT

public:
  DocumentWidget(PageView *sv, DocumentPageCache *cache);

  void setPageNumber(PageNumber);
  PageNumber getPageNumber() const {return pageNumber;}

  /** Returns the size of the widget without the page shadow. */
  QSize pageSize() const;
  /** Returns the bounding rectangle of the widget without the page shadow. */
  QRect pageRect() const;

  /** Draw a red vertical line at y-coordinate ycoord. The line is removed again
      after one second. This is used to make it easier to regain reading focus if
      the whole page is scrolled up or down. */
  void drawScrollGuide(int ycoord);

  /** Returns the size of the page, in pixels, using current.
      If the document does not specify a size (which happens, e.g.,
      for some DVI-files), the userPreferredSize is used.

      @warning This function does not return the current size of this
      widgets, but the size of the page in the document that is displayed
      by this widgets. The function is used to calculate the new size of
      the widget when the zoomfactor has been updated.

      The current size of the widget is returned by @ref pageSize().
      
      See also @ref DocumentPageCache::sizeOfPageInPixel() */
  QSize sizeOfPageInPixel();

  /** Resizes the widget, so that the size matches the size the page that is
      displayed by this widget in the current zoom level.

      @returns true if the size of the widget has been changed. */
  bool resizeWidget();
  
public slots:
  void          slotSwitchTool();

  void          select(const TextSelection&);
  void          selectAll();
  void          flash(int);

  /** Sets the size of the widget so that the page is of the given size.
      The widget gets slightly bigger because of the page shadow. */
  void setPageSize(const QSize&);
  void setPageSize(int width, int height);

signals:
  /** Passed through to the top-level kpart. */
  void setStatusBarText(const QString&);
  void localLink(const Hyperlink&);

  /** This signal is emitted when the selection needs to be cleared. */
  void clearSelection();

  /** Emitted on right click. */
  void showPopupMenu(const PageNumber& pageNumber, const QPoint& position);

protected slots:
  void slotShowPopupMenu(const PageNumber& pageNumber, const QPoint& position);

protected:
  virtual void paintEvent(QPaintEvent*);
  virtual void mousePressEvent(QMouseEvent*);
  virtual void mouseReleaseEvent(QMouseEvent*);
  virtual void mouseDoubleClickEvent(QMouseEvent*);


  /** This method is used by the DocumentWidget to find out of the
      mouse pointer hovers over a hyperlink, and to update the
      statusbar accordingly. Scrolling with the left mouse button
      pressed, and the text copy functions are also implemented here.
      Re-implementations of this method should do the following:

      0) Immediately return if pageNumber.isValid() == false,
         i.e. if no page number has been set

      1) Call the standard implementation using

          DocumentWidget::mouseMoveEvent(e);

      2) Ignore the QMouseEvent if a mouse button is pressed

      3) If no mouse button is pressed, analyze the mouse movement and
         take appropriate actions. To set statusbar text, do

         clearStatusBarTimer.stop();
         emit setStatusBarText( i18n("Whatever string") );

         To clear the statusbar, use the following code

         if (!clearStatusBarTimer.isActive())
           clearStatusBarTimer.start(200, true);

         This clears the statusbar after 200 msec and avoids awful
         flickering when the mouse is swiftly moved across various
         areas in the widget.

  */
  virtual void  mouseMoveEvent (QMouseEvent *);

protected:
  void updateSelection(const TextSelection& newTextSelection);

  /** Methods and counters used for the animation to mark the target of
      an hyperlink. */
  int           timerIdent;
  void          timerEvent( QTimerEvent *e );
  int           animationCounter;
  int           flashOffset;

  PageNumber    pageNumber;

  /* This timer is used to delay clearing of the statusbar. Clearing
     the statusbar is delayed to avoid awful flickering when the mouse
     moves over a block of text that contains source hyperlinks. The
     signal timeout() is connected to the method clearStatusBar() of
     *this. */
  QTimer        clearStatusBarTimer;

  /** Data structures used for marking text with the mouse.
      This points are stored in page coordinates. */
  QPoint       firstSelectedPoint;
  QRect        selectedRectangle;

  /** Pointer to the PageView that contains this
      widget. This pointer is used in the re-implementation of the
      paintEvent() method ---see the explanation there. */
  PageView *scrollView;
  QPointer<DocumentPageCache> documentCache;

  /** Currently selected Region. The region is stored in page coordinates
      therefore you have to transform it into screen coordinates before
      drawing. @see map() */
  QRegion selectedRegion;

  /** This is set to the index of the link over which the mouse pointer currently resides,
      and -1 if the no link is hovered.
      Is used when "Underline Links" is set to "Only on Hover". */
  int indexOfUnderlinedLink;

  /** Sets the cursor to an arrow if the move tool is selected, and to the text selection
      cursor if the selection tool is active. */
  virtual void setStandardCursor();

  /** Measure length and angle. If restrictAxis=true, the measurement is restricted
      to a vertical resp. horizontal line. The measurement results are displayed in
      the status bar. */
  void measureLength(bool restrictAxis);

  /** Creates a transformation matrix, which transforms from
      page coordinates to screen coordinates. */
  QMatrix transform() const;

  /** Creates a transformation matrix, which transforms from
      screen coordinates to page coordinates. */
  QMatrix inverseTransform() const;

  /** Map a point/rect/region from page coordinates to screen coordinates. */
  QPoint map(const QPoint&) const;
  QRect map(const QRect&) const;
  QRegion map(const QRegion&) const;

  /** Map a point/rect/region from screen coordinates to page coordinates. */
  QPoint inverseMap(const QPoint&) const;
  QRect inverseMap(const QRect&) const;
  QRegion inverseMap(const QRegion&) const;

  static QPointer<QMenu> contextMenu;

  static QPointer<QAction> addBookmarkAction;
  static QPointer<QAction> removeBookmarkAction;

private slots:
  /** This slot emits the signal setStatusBarText(QString::null) to
      clear the status bar. It is connected to the timeout slot of the
      clearStatusBarTimer. */
  void clearStatusBar();

  /** Hide the scroll guide. This slot is called one second after drawScrollGuide(). */
  void clearScrollGuide();

private:
  void performRectangleSelection(const QPoint& menuPos, const QRect& selectionRect);

  QRect linkFlashRect();

  QString printLength(int l) const;

  /** If this variable is positive draw a vertical line at this y-coordinate. */
  int scrollGuide;
  QTimer clearScrollGuideTimer;

  /** Color used by in the shadow drawing to check if the background color has been changed. */
  static QColor backgroundColorForCorners;

  /** The following tables store grey values for roundish shadow
      corners. They were shamelessly stolen from kdelibs/kdefx/kstyle.cpp. */
  static const int bottom_right_corner[16];
  static const int bottom_left_corner[16];
  static const int shadow_strip[4];

  /** If this is true the zoomlevel has changed and we need to update the
      selected region. */
  bool selectionNeedsUpdating;

  // Variables used by the measuring tool
  QPoint measuringOrigin;
  QPoint measuringPoint;
  QPoint oldMeasuringPoint;
  bool measuringInProgress;

  // Variable used by the rectangle selection tool
  QPoint rectSelectionStartPoint;
  QPoint rectSelectionEndPoint;
  QPoint oldRectSelectionEndPoint;
};


/** Stores data on the selected rectangle, to be used when the
 *  popup menu's Actions are triggered.
 */
class SelectedRectangle : public QObject
{
  Q_OBJECT
public:

  SelectedRectangle(const QRect& rectangle,
                    const QString & text,
                    RenderedDocumentPagePixmap& page_data);

signals:
  void setStatusBarText(const QString&);

public slots:
  void copyText() const;
  void copyImage();

private:
  const QRect& rectangle_;
  const QString& text_;
  RenderedDocumentPagePixmap& page_data_;
};

#endif
