// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/***************************************************************************
 *   Copyright (C) 2005-2006 by Wilfried Huss                              *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <config.h>

#include "bookmarkList.h"
#include "bookmark.h"
#include "kvs_debug.h"

#include <kiconloader.h>
#include <kglobal.h>
#include <klocale.h>

#include <Q3Header>
#include <QMenu>

BookmarkItem::BookmarkItem(BookmarkList* parent)
  : K3ListViewItem(parent), bookmarkList(parent)
{
}


void BookmarkItem::set(const PageNumber& _pageNumber, const QString& label)
{
  if (label == QString::null)
    setText(0, i18n("Page %1", _pageNumber));
  else
    setText(0, label);

  pageNumber = _pageNumber;

  // Allow renaming of bookmarks
  setRenameEnabled(0, true);
}


int BookmarkItem::compare(K3ListViewItem* i, int col, bool ascending) const
{
  Q_UNUSED(col);

  BookmarkItem* item = static_cast<BookmarkItem*>(i);
  if (ascending)
  {
    if (pageNumber > item->pageNumber)
      return 1;
    else if (pageNumber < item->pageNumber)
      return -1;
    else return 0;
  }
  else
  {
    if (pageNumber > item->pageNumber)
      return -1;
    else if (pageNumber < item->pageNumber)
      return 1;
    else return 0;
  }
}


BookmarkList::BookmarkList(QWidget* parent)
  : K3ListView(parent), DataView(), contextMenu(0)
{
  header()->hide();

  addColumn(i18n("Label"));

  setResizeMode(AllColumns);

  // Sort bookmarks in ascending order
  // The given column is irrelevant, because BookmarkItem
  // has a overloaded compare function.
  setSorting(0);

  setSelectionMode(Q3ListView::NoSelection);

  connect(this, SIGNAL(executed(Q3ListViewItem*)), this, SLOT(itemClicked(Q3ListViewItem*)));

  connect(this, SIGNAL(contextMenuRequested(Q3ListViewItem*, const QPoint&, int)),
          this, SLOT(slotContextMenu(Q3ListViewItem*, const QPoint&)));

  connect(this, SIGNAL(itemRenamed(Q3ListViewItem*, int, const QString&)),
          this, SLOT(slotItemRenamed(Q3ListViewItem*, int, const QString&)));

  KIconLoader* iconLoader = KIconLoader::global();
  currentBookmarkIcon = iconLoader->loadIcon("1rightarrow", K3Icon::Small, K3Icon::SizeSmall);
}


BookmarkList::~BookmarkList()
{
}


void BookmarkList::bookmarkAdded(const PageNumber& pageNumber, const QString& label)
{
  BookmarkItem* item = new BookmarkItem(this);
  item->set(pageNumber, label);
  if (pageNumber == currentPageNumber)
  {
    item->setPixmap(0, currentBookmarkIcon);
  }
}

void BookmarkList::bookmarkRenamed(const PageNumber& pageNumber, const QString& label)
{
  Q_UNUSED(pageNumber);
  Q_UNUSED(label);
}

void BookmarkList::bookmarkRemoved(const PageNumber& pageNumber)
{
  BookmarkItem* item = static_cast<BookmarkItem*>(firstChild());

  while (item)
  {
    if (item->getPageNumber() == pageNumber)
    {
      BookmarkItem* tempItem = item;
      item = static_cast<BookmarkItem*>(item->itemBelow());

      takeItem(tempItem);
      delete tempItem;
    }
    else
      item = static_cast<BookmarkItem*>(item->itemBelow());
  }
}


void BookmarkList::allBookmarksRemoved()
{
  clear();
}


void BookmarkList::itemClicked(Q3ListViewItem* _item)
{
  BookmarkItem* item = static_cast<BookmarkItem*>(_item);

  if (item)
    dataModel->setCurrentPageNumber(item->getPageNumber());
}


void BookmarkList::slotItemRenamed(Q3ListViewItem* _item, int col, const QString& label)
{
  Q_UNUSED(col);

  BookmarkItem* item = static_cast<BookmarkItem*>(_item);

  if (item)
    dataModel->renameBookmark(item->getPageNumber(), label);
}


void BookmarkList::setCurrentPage()
{
  currentPageNumber = dataModel->currentPageNumber();

  BookmarkItem* item = static_cast<BookmarkItem*>(firstChild());

  while (item)
  {
    if (item->getPageNumber() == currentPageNumber)
    {
      item->setPixmap(0, currentBookmarkIcon);
    }
    else
    {
      item->setPixmap(0, QPixmap());
    }
    item = static_cast<BookmarkItem*>(item->itemBelow());
  }
}


void BookmarkList::slotContextMenu(Q3ListViewItem* i, const QPoint& pos)
{
  BookmarkItem* item = static_cast<BookmarkItem*>(i);

  if (contextMenu == 0)
  {
    // Initialize Contextmenu
    contextMenu = new QMenu(this);

    addBookmarkAction        = contextMenu->addAction(SmallIcon("bookmark_add"), i18n("Add &Bookmark"));
    renameBookmarkAction     = contextMenu->addAction(i18n("&Rename Bookmark"));
    removeBookmarkAction     = contextMenu->addAction(SmallIcon("bookmark"), i18n("&Clear Bookmark"));
    removeAllBookmarksAction = contextMenu->addAction(i18n("Clear &All Bookmarks"));
  }

  // TODO: use new slot-based API, once it works reliable
  if (item)
  {
    renameBookmarkAction->setEnabled(true);
    removeBookmarkAction->setEnabled(true);
  }
  else
  {
    renameBookmarkAction->setEnabled(false);
    removeBookmarkAction->setEnabled(false);
  }

  if (childCount() > 0)
  {
    removeAllBookmarksAction->setEnabled(true);
  }
  else
  {
    removeAllBookmarksAction->setEnabled(false);
  }

  // Show Contextmenu
  QAction* selected = contextMenu->exec(pos);

  if (selected == addBookmarkAction)
  {
    dataModel->addBookmark(currentPageNumber, QString::null);
  }
  else if (selected == renameBookmarkAction && item)
  {
    item->startRename(0);
  }
  else if (selected == removeBookmarkAction && item)
  {
    dataModel->removeBookmark(item->getPageNumber());
  }
  else if (selected == removeAllBookmarksAction)
  {
    dataModel->removeAllBookmarks();
  }
}


void BookmarkList::setupObservers(DataModel* _dataModel)
{
  DataView::setupObservers(_dataModel);

  QList<PageNumber> keys = dataModel->bookmarks();
  for (unsigned int i = 0; i < keys.count(); i++)
  {
    bookmarkAdded(keys[i], dataModel->bookmarkLabel(keys[i]));
  }

  connect(dataModel, SIGNAL(currentPageNumberChanged()), this, SLOT(setCurrentPage()));

  connect(dataModel, SIGNAL(bookmarkAdded(const PageNumber&, const QString&)), this, SLOT(bookmarkAdded(const PageNumber&, const QString&)));
  connect(dataModel, SIGNAL(bookmarkRenamed(const PageNumber&, const QString&)), this, SLOT(bookmarkRenamed(const PageNumber&, const QString&)));
  connect(dataModel, SIGNAL(bookmarkRemoved(const PageNumber&)), this, SLOT(bookmarkRemoved(const PageNumber&)));
  connect(dataModel, SIGNAL(allBookmarksRemoved()), this, SLOT(allBookmarksRemoved()));
}

#include "bookmarkList.moc"
