/***************************************************************************
 *   Copyright (C) 2005 by Jeroen Wijnhout                                 *
 *   Jeroen.Wijnhout@kdemail.net                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/

#include <qdir.h>
#include <kdebug.h>

#include "sourceSplitter.h"

//#define DEBUG_SOURCESPLITTER


SourceFileSplitter::SourceFileSplitter(const QString& srclink, const QString& file)
{
  QString filepart = srclink, linepart;

  //if sourcefilename starts with a number
  //then there could be a mix up, i.e. src:123file.tex
  //line 123 and file.tex or line 12 and 3file.tex?
  bool possibleNumberMixUp = false;

#ifdef DEBUG_SOURCESPLITTER
  kDebug() << "SourceSplitter: srclink " << srclink << endl;
#endif

  //remove src: if necessary
  if (filepart.left(4) == "src:")
    filepart = srclink.mid(4);

  //split first
  unsigned int max = filepart.length();

  unsigned int i = 0;
  for (i = 0; i < max; ++i)
    if (!filepart[i].isDigit())
      break;

  linepart = filepart.left(i);
  filepart = filepart.mid(i);

  //check for number mix up
  if (filepart[0] != ' ' && (linepart.length() != 1))
    possibleNumberMixUp = true;

  //remove spaces
  filepart = filepart.trimmed();
  linepart = linepart.trimmed();

#ifdef DEBUG_SOURCESPLITTER
  kDebug() << "SourceSplitter: filepart " << filepart << " linepart " << linepart << endl;
#endif

  //test if the file exists
  m_fileInfo.setFile(QFileInfo(file).dir(true), filepart);
  bool fiExists = m_fileInfo.exists();

  //if it doesn't exist, but adding ".tex"
  if (!fiExists && QFileInfo(m_fileInfo.absoluteFilePath() + ".tex").exists())
    m_fileInfo.setFile(m_fileInfo.absoluteFilePath() + ".tex");

  //if that doesn't help either, perhaps the file started with a
  //number: move the numbers from the sourceline to the filename
  //one by one (also try to add .tex repeatedly)
  if (possibleNumberMixUp && !fiExists)
  {
    QFileInfo tempInfo(m_fileInfo);
    QString tempFileName = tempInfo.fileName();
    quint32 index, maxindex = linepart.length();
    bool found = false;
    for (index = 1; index < maxindex; ++index)
    {
      tempInfo.setFile(linepart.right(index) + tempFileName);
#ifdef DEBUG_SOURCESPLITTER
      kDebug() << "SourceSplitter: trying " << tempInfo.fileName() << endl;
#endif
      if (tempInfo.exists())
      {
        found = true;
        break;
      }
      tempInfo.setFile(linepart.right(index) + tempFileName + ".tex");
#ifdef DEBUG_SOURCESPLITTER
      kDebug() << "SourceSplitter: trying " << tempInfo.fileName() << endl;
#endif
      if (tempInfo.exists())
      {
        found = true;
        break;
      }
    }

    if (found)
    {
      m_fileInfo = tempInfo;
      linepart = linepart.left(maxindex - index);
    }
  }

  bool ok;
  m_line = linepart.toInt(&ok);
  if (!ok)
    m_line = 0;

#ifdef DEBUG_SOURCESPLITTER
  kDebug() << "SourceSplitter: result: file " << m_fileInfo.absoluteFilePath() << " line " << m_line << endl;
#endif
}
