/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "kblogger.h"

#include <QDropEvent>
#include <QDir>

#include <kglobal.h>
#include <klocale.h>
#include <kicon.h>
//#include <kprinter.h>
#include <kstatusbar.h>
#include <kedittoolbar.h>
//#include <ktempdir.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kstandarddirs.h>
//#include <kmessagebox.h>
#include <kapplication.h>
#include <ktoolbar.h>
#include <ktreewidgetsearchline.h>
#include <ksystemtrayicon.h>
#include <kwallet.h>

#include "postslist.h"
#include "itemsmanager.h"
#include "kbloggerconfig.h"
#include "configdialog.h"
#include "uploadmediadialog.h"

#include "config-kpa-kipi.h"
#ifdef HASKIPI
#include <libkipi/plugin.h>  // libKipi includes.
#include <libkipi/pluginloader.h>
#include "plugins/interface.h"
#endif

namespace KBlogger
{

Application::Application()
        : KXmlGuiWindow()
{
    kDebug();

    // HACK open the wallet on startup
    // we access the wallet directly from the profileconfigdialog and the backend
    // but this opening does not seem to work reliably, so we dummy open it here first
    KWallet::Wallet* mWallet = KWallet::Wallet::openWallet( "kdewallet", this->winId() );
    if ( mWallet ) {
        kDebug() << "Wallet successfully opened.";
    }
     // memory leak, but does not matter here, since it is destroyed with the app anyway
     // if we delete immediately we have to open the wallet twice
     //     delete mWallet; 

    mBackend = Backend::self(this);
    Q_ASSERT(mBackend);

    mMainWidget = MainWidget::self(actionCollection(), this); //NOTE MainWidget must be a KBlogger children!!
    Q_ASSERT(mMainWidget);

    //status bar connection.
    connect(mBackend, SIGNAL(statusBarMessage( const QString&)),
            statusBar(), SLOT(message( const QString&)));

    // accept dnd
    setAcceptDrops(true);

    //QDir::setSearchPaths("cache:", QStringList(KBloggerMedia::cachePath()) );
    //FIXME This line permit to QTextEdit to display media, exist another solution?
    //QDir::setCurrent ( KBloggerMedia::cachePath() );

    // tell the KXmlGuiWindow that this is indeed the main widget
    setCentralWidget(mMainWidget);

    // then, setup our actions
    setupActions();

    // add a status bar
    statusBar()->show();

    // a call to KXmlGuiWindow::setupGUI() populates the GUI
    // with actions, using KXMLGUI.
    // It also applies the saved mainwindow settings, if any, and ask the
    // mainwindow to automatically save settings if changed: window size,
    // toolbar position, icon size, etc.
    setupGUI();

#ifdef HASKIPI
    // Load KIPI Plugins.
    loadPlugins();
#endif

    show();
}

Application::~Application()
{
    kDebug();
}

void Application::setupActions()
{
    //QACTIONGROUP
    QActionGroup *postActionGroup = new QActionGroup(this);
    QActionGroup *mediaActionGroup = new QActionGroup(this);
    QActionGroup *kblogActionGroup = new QActionGroup(this);
    postActionGroup->setObjectName("postActionGroup");
    mediaActionGroup->setObjectName("mediaActionGroup");
    kblogActionGroup->setObjectName("kblogActionGroup");

    //FILE MENU
    (void) KStandardAction::quit(this, SLOT(quit()), actionCollection());
    (void) KStandardAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

    //POSTS MENU
    KLocalizedString ksNew = ki18nc( "a button to create a new post", "New" );
    KAction *createPostAction = new KAction(KIcon("document-new"), ksNew.toString(), this);
    actionCollection()->addAction( QLatin1String("create_post"), createPostAction );
    connect(createPostAction, SIGNAL(triggered(bool)), this, SLOT(createPost()));
    //postActionGroup->addAction(createPostAction);

    KAction *modifyAction = new KAction(KIcon("document-properties"), i18n("Modify"), this);
    actionCollection()->addAction( QLatin1String("modify_post"), modifyAction );
    connect(modifyAction, SIGNAL(triggered(bool)), mMainWidget, SLOT(modifyPost()));
    postActionGroup->addAction(modifyAction);

    KAction *trashAction = new KAction(KIcon("user-trash"), i18n("Move to trash / Remove from the server"), this);
    actionCollection()->addAction( QLatin1String("trash_post"), trashAction );
    connect(trashAction, SIGNAL(triggered(bool)), mMainWidget, SLOT(trashPost()));
    postActionGroup->addAction(trashAction);

    KAction *deleteAction = new KAction(KIcon("edit-delete"), i18n("Delete Locally"), this);
    actionCollection()->addAction( QLatin1String("delete_post"), deleteAction );
    connect(deleteAction, SIGNAL(triggered(bool)), mMainWidget, SLOT(deletePost()));
    postActionGroup->addAction(deleteAction);

    //MEDIA MENU
    KAction *createMediaAction = new KAction(KIcon("insert-image"), i18n("Create Generic Media"), this);
    actionCollection()->addAction( QLatin1String("create_generic_media"), createMediaAction );
    connect(createMediaAction, SIGNAL(triggered(bool)), this, SLOT(showUploadFileDialog()));

    KAction *openMediaWithExtAppAction = new KAction(KIcon("system-run"), i18n("Open with external application"), this);
    actionCollection()->addAction( QLatin1String("open_with_ext_app"), openMediaWithExtAppAction );
    connect(openMediaWithExtAppAction, SIGNAL(triggered(bool)), this, SLOT(openMediaWhitExtApp()));
    mediaActionGroup->addAction(openMediaWithExtAppAction);

    KAction *removeLocalMediaAction = new KAction(KIcon("user-trash"), i18n("Remove Local Media"), this);
    actionCollection()->addAction( QLatin1String("remove_local_media"), removeLocalMediaAction );
    connect(removeLocalMediaAction, SIGNAL(triggered(bool)), mMainWidget, SLOT( removeLocalMedia() ));
    mediaActionGroup->addAction(removeLocalMediaAction);

    //BLOGS ACTION MENU
    KAction *syncAction = new KAction(KIcon("go-up"), i18n("Sync Media and Entries"), this);
    actionCollection()->addAction( QLatin1String("sync_all"), syncAction );
    connect(syncAction, SIGNAL(triggered(bool)), mBackend, SLOT(sync()));
    kblogActionGroup->addAction(syncAction);

    KAction *updateSentPostListAction = new KAction(KIcon("mail-receive"), i18n("Update sent list"), this);
    actionCollection()->addAction( QLatin1String("update_sent_post_list"), updateSentPostListAction );
    connect(updateSentPostListAction, SIGNAL(triggered(bool)), mMainWidget, SLOT(updateAllSentPostList()));
    kblogActionGroup->addAction(updateSentPostListAction);

    KAction *uploadPostsAction = new KAction(KIcon("mail-send"), i18n("Upload Posts"), this);
    actionCollection()->addAction( QLatin1String("upload_posts"), uploadPostsAction );
    connect(uploadPostsAction, SIGNAL(triggered(bool)), mMainWidget, SLOT(uploadCurrentPosts()));
    kblogActionGroup->addAction(uploadPostsAction);

    KAction *uploadMediaAction = new KAction(KIcon("archive-extract"), i18n("Upload media"), this);
    actionCollection()->addAction( QLatin1String("upload_media"), uploadMediaAction );
    connect(uploadMediaAction, SIGNAL(triggered(bool)), mMainWidget, SLOT(uploadCurrentMedia()));
    kblogActionGroup->addAction(uploadMediaAction);

    //Search bar;
    KToolBar *searchBar = static_cast<KToolBar*>( addToolBar(tr("Search")) );
    searchBar->setObjectName("SearchBar");
    KLocalizedString ksSearch = ki18nc(  "the label for a search line edit", "Search:" );
    QLabel *searchLabel = new QLabel( ksSearch.toString(), searchBar);
    searchBar->addWidget( searchLabel );
    mSearchLine = new KTreeWidgetSearchLine( searchBar );
    searchBar->addWidget( mSearchLine );

    //Disable All groups;
    postActionGroup->setEnabled(false);
    mediaActionGroup->setEnabled(false);
    kblogActionGroup->setEnabled(false);
}

void Application::loadPlugins()
{
#ifdef HASKIPI
    kDebug();
    QStringList ignores;

    ignores.append( "HelloWorld" );
    ignores.append( "KameraKlient" );
    Plugins::Interface *pluginInterface;
    pluginInterface = new Plugins::Interface( this, "kipi interface" );
    mKipiPluginLoader = new KIPI::PluginLoader( ignores, pluginInterface );

    connect( mKipiPluginLoader, SIGNAL( replug() ),
             this, SLOT( slotKipiPluginPlug() ) );

    mKipiPluginLoader->loadPlugins();

#endif
}

void Application::slotKipiPluginPlug()
{
#ifdef HASKIPI
    kDebug();
    //TODO
    /*
        unplugActionList( QString::fromLatin1("file_actions_export") );
        unplugActionList( QString::fromLatin1("file_actions_import") );
        unplugActionList( QString::fromLatin1("image_actions") );
        unplugActionList( QString::fromLatin1("tool_actions") );
        unplugActionList( QString::fromLatin1("batch_actions") );
        unplugActionList( QString::fromLatin1("album_actions") );
     
        d->kipiImageActions.clear();
        d->kipiFileActionsExport.clear();
        d->kipiFileActionsImport.clear();
        d->kipiToolsActions.clear();
        d->kipiBatchActions.clear();
        d->kipiAlbumActions.clear();
     
        KIPI::PluginLoader::PluginList list = d->mKipiPluginLoader->pluginList();
     
        int cpt = 0;
     
        for( KIPI::PluginLoader::PluginList::Iterator it = list.begin() ; it != list.end() ; ++it )
        {
            KIPI::Plugin* plugin = (*it)->plugin();
     
            if ( !plugin || !(*it)->shouldLoad() )
                continue;
     
            ++cpt;
     
            //if(d->splashScreen)
              //  d->splashScreen->message(i18n("Loading: %1", (*it)->name()));
     
            plugin->setup( this );
     
            // Plugin category identification using KAction method based.
     
            QList<KAction*> actions = plugin->actions();
     
            for( QList<KAction*>::Iterator it2 = actions.begin(); it2 != actions.end(); ++it2 )
            {
                if ( plugin->category(*it2) == KIPI::IMAGESPLUGIN )
                {
                    d->kipiImageActions.append(*it2);
                }
                else if ( plugin->category(*it2) == KIPI::EXPORTPLUGIN )
                {
                    d->kipiFileActionsExport.append(*it2);
                }
                else if ( plugin->category(*it2) == KIPI::IMPORTPLUGIN )
                {
                    d->kipiFileActionsImport.append(*it2);
                }
                else if ( plugin->category(*it2) == KIPI::TOOLSPLUGIN )
                {
                    d->kipiToolsActions.append(*it2);
                }
                else if ( plugin->category(*it2) == KIPI::BATCHPLUGIN )
                {
                    d->kipiBatchActions.append(*it2);
                }
                else if ( plugin->category(*it2) == KIPI::COLLECTIONSPLUGIN )
                {
                    d->kipiAlbumActions.append(*it2);
                }
                else
                    DDebug() << "No menu found for a plugin!!!" << endl;
            }
        }
     
        // Create GUI menu in according with plugins.
     
        plugActionList( QString::fromLatin1("file_actions_export"), d->kipiFileActionsExport );
        plugActionList( QString::fromLatin1("file_actions_import"), d->kipiFileActionsImport );
        plugActionList( QString::fromLatin1("image_actions"),       d->kipiImageActions );
        plugActionList( QString::fromLatin1("tool_actions"),        d->kipiToolsActions );
        plugActionList( QString::fromLatin1("batch_actions"),       d->kipiBatchActions );
        plugActionList( QString::fromLatin1("album_actions"),       d->kipiAlbumActions );
    */
#endif
}

void Application::optionsPreferences()
{
    if ( configDialog::showDialog( "settings" ) )  {
        return;
    }
    configDialog *dialog = new configDialog( this, "settings", KBloggerConfig::self() );
    connect(dialog, SIGNAL( settingsChanged( const QString& ) ),
            mMainWidget, SLOT( reloadSettings() ) );
}

void Application::createPost()
{
    kDebug();
    mComposerDialog = new Composer( mMainWidget->currentBlogname() , this);
    mComposerDialog->exec();
    mComposerDialog = 0;
}

void Application::modifyPost(KBloggerPost* post)
{
    kDebug();

    if ( !post ) return;
    mComposerDialog = new Composer(post, this);
    mComposerDialog->exec();
    mComposerDialog = 0;
}

void Application::updateSearchLine(QTreeWidget *treeWidget)
{
    kDebug();
//        addTreeWidget (QTreeWidget *treeWidget)
    QTreeWidget *mTreeWidget = 0;
    mTreeWidget = mSearchLine->treeWidget();
    if (mTreeWidget)
        mSearchLine->removeTreeWidget(mTreeWidget);
    mSearchLine->addTreeWidget(treeWidget);
}

void Application::quit()
{
    kDebug();
    mMainWidget->saveStatus();
    QCoreApplication::quit();
}

void Application::showUploadFileDialog()
{
    kDebug();
    new UploadMediaDialog(QString(), mMainWidget->currentBlogname(), this);
}

void Application::openMediaWhitExtApp()
{
    kDebug();
    mMainWidget->openSelectedMedia();
}

} //namespace

#include "kblogger.moc"
