/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n.h>
#include "kz-file.h"

static void set_property    (GObject *object,
	        	     guint prop_id,
	        	     const GValue *value,
	        	     GParamSpec *pspec);
static void get_property    (GObject *object,
	        	     guint prop_id,
	        	     GValue *value,
	        	     GParamSpec *pspec);
static void dispose         (GObject *object);

static void      file_start      (KzIO *io);
static GIOStatus read_from_io    (KzIO *io,
				  GIOChannel *iochannel);
static GIOStatus write_to_io     (KzIO *io,
				  GIOChannel *iochannel);

typedef struct _KzFilePrivate KzFilePrivate;
struct _KzFilePrivate 
{
	gchar *filename;
};
#define KZ_FILE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_FILE, KzFilePrivate))

enum {
	PROP_0,
	PROP_FILENAME
};

#define BUFFER_SIZE 1024

G_DEFINE_TYPE(KzFile, kz_file, KZ_TYPE_IO)

static void
kz_file_class_init (KzFileClass *klass)
{
	GObjectClass *object_class;
	KzIOClass *io_class;

	object_class = (GObjectClass *) klass;
	io_class     = (KzIOClass *) klass;

	object_class->dispose      = dispose;
	object_class->set_property = set_property;
	object_class->get_property = get_property;
	
	io_class->read_from_io  = read_from_io;
	io_class->write_to_io   = write_to_io;
	io_class->io_start      = file_start;

	g_object_class_install_property(
		object_class,
		PROP_FILENAME,
		g_param_spec_string(
			"filename",
			_("Filename"),
			_("The Filename of the local file"),
			NULL,
			G_PARAM_READWRITE));
	g_type_class_add_private (object_class, sizeof(KzFilePrivate));
}


static void
kz_file_init (KzFile *file)
{
	KzFilePrivate *priv = KZ_FILE_GET_PRIVATE (file);

	priv->filename = NULL;
}


static void
dispose (GObject *object)
{
	KzFilePrivate *priv = KZ_FILE_GET_PRIVATE (object);

	if (priv->filename)
	{
		g_free(priv->filename);
	}
	priv->filename = NULL;

	if (G_OBJECT_CLASS (kz_file_parent_class)->dispose)
		G_OBJECT_CLASS (kz_file_parent_class)->dispose(object);
}


static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
	KzFilePrivate *priv = KZ_FILE_GET_PRIVATE (object);

	switch (prop_id)
	{
	case PROP_FILENAME:
		g_free(priv->filename);
		priv->filename = g_value_dup_string(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
	KzFilePrivate *priv = KZ_FILE_GET_PRIVATE (object);

	switch (prop_id)
	{
	case PROP_FILENAME:
		g_value_set_string(value, priv->filename);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}



KzFile *
kz_file_new (const gchar *uri)
{
	KzFile *file;

	file = g_object_new(KZ_TYPE_FILE,
			    "filename", uri,
			    NULL);
	
	return file;
}


static GIOStatus
kz_file_prepare_iochannel (KzIO *io)
{
	GIOStatus iostatus;
	gchar *filename;
	KzFilePrivate *priv = KZ_FILE_GET_PRIVATE (io);

	filename = priv->filename;

	if (kz_io_get_mode(io) == KZ_IO_READ)
	{
		if (g_file_test(filename, G_FILE_TEST_EXISTS) == FALSE)
		{
			KZ_IO_CLASS (kz_file_parent_class)->io_error(io);
			return G_IO_STATUS_ERROR;
		}
		io->iochannel = g_io_channel_new_file(filename, "r", NULL);
	}
	else if (kz_io_get_mode(io) == KZ_IO_WRITE)
		io->iochannel = g_io_channel_new_file(filename, "w", NULL);

	iostatus = g_io_channel_set_encoding(io->iochannel,
					     NULL,
					     NULL);
	return iostatus;
}


static GIOStatus
read_from_io(KzIO *io, GIOChannel *iochannel)
{
	GIOStatus iostatus;
	gsize bytes_read;
	gchar buffer[BUFFER_SIZE];

	/* Read the data into our buffer */
	iostatus = g_io_channel_read_chars(iochannel, buffer, 
					   sizeof(buffer),
					   &bytes_read,
					   NULL);

	if (iostatus == G_IO_STATUS_NORMAL)
	{	
		KZ_IO_CLASS (kz_file_parent_class)->io_to_buffer(io, bytes_read, buffer);
		/* Check for EOF */
		if (bytes_read == 0)
			iostatus = G_IO_STATUS_EOF;
	}
	return iostatus;
}


static GIOStatus
write_to_io(KzIO *io, GIOChannel *iochannel)
{
	KZ_IO_CLASS (kz_file_parent_class)->buffer_to_io(io, 0, NULL);
	return G_IO_STATUS_NORMAL;
}


static void
file_start (KzIO *io)
{
	GIOStatus iostatus;
	iostatus = kz_file_prepare_iochannel(io);
	
	if (iostatus != G_IO_STATUS_NORMAL)
	{
		KZ_IO_CLASS (kz_file_parent_class)->io_error(io);
		return;
	}

	/* call io_set_iochannel, start to loading */
	KZ_IO_CLASS (kz_file_parent_class)->io_set_channel(io);
}
