/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "kz-lirs.h"

static gboolean kz_lirs_is_supported      (KzBookmark  *bookmark,
					   const gchar *buf);
static gboolean kz_lirs_parse_from_string (KzBookmark  *bookmark, 
					   const gchar *buffer,
					   guint        length,
					   GError     **error);
static gchar  **kz_lirs_split             (const gchar *string,
					   gint max_tokens);


static KzBookmarkFileType lirs_file_type =
{
	priority_hint: 0,
	file_type:     "LIRS",
	is_supported:  kz_lirs_is_supported,
	init:          NULL,
	from_string:   kz_lirs_parse_from_string,
	to_string:     NULL,
};


KzBookmarkFileType *
kz_lirs_get_file_types (gint idx)
{
	if (idx == 0)
		return &lirs_file_type;
	else
		return NULL;
}


gboolean
kz_lirs_is_supported (KzBookmark  *bookmark, const gchar *buf)
{
	g_return_val_if_fail(buf, FALSE);

	if (strncmp(buf, "LIRS", 4) == 0)
		return TRUE;

	return FALSE;
}


static gchar  **
kz_lirs_split (const gchar *string, gint max_tokens)
{
	GSList *string_list = NULL, *slist;
	gchar **str_array= NULL, *s;
	GString *new_string = NULL;
	guint n = 0;

	new_string = g_string_sized_new(0);
	g_return_val_if_fail (string != NULL, NULL);

	s = (gchar*)string;

	while (n < max_tokens && *s != '\0')
	{
		if (*s == ',')
		{
			string_list = g_slist_prepend (string_list, 
						       g_strdup(new_string->str));
			g_string_free(new_string, TRUE);
			new_string = NULL;
			new_string = g_string_sized_new(0);
			n++;
		}
		else if (*s == '\\')
		{
			s++;
			g_string_append_c(new_string, *s);
		}
		else
		{
			g_string_append_c(new_string, *s);
		}
		s++;
	}
	
	if (*string)
	{
		n++;
		string_list = g_slist_prepend (string_list, g_strdup (string));
	}

	if (n < 8)
	{
		g_slist_free(string_list);
		return NULL;
	}
	
	str_array = g_new0 (gchar*, n + 1);

	str_array[n--] = NULL;
	for (slist = string_list; slist; slist = slist->next)
		str_array[n--] = slist->data;

	g_slist_free (string_list);

	return str_array;
}


gboolean
kz_lirs_parse_from_string (KzBookmark *bookmark,
			   const gchar *buffer, guint length,
			   GError **error)
{
	gchar *dest = NULL;
	gsize read_len, dest_len;
	gchar **lines    = NULL;
	gchar **elements = NULL;
	guint line_number = 0;
	gchar timestr[20];
	struct tm *t = NULL;

	g_return_val_if_fail(buffer, FALSE);

	/* convert to utf-8 encoding */
	dest = g_convert(buffer, length, "UTF-8", "EUC-JP",
			 &read_len, &dest_len, NULL);

	lines = g_strsplit(dest, "\n", -1);
	
	while (lines[line_number+1] != NULL)
	{
		KzBookmark *item;
		time_t date;

		elements = kz_lirs_split(lines[line_number], 9);
		if (!elements)
			break;

		item = kz_bookmark_new();
		/* kz_bookmark_set_editable(item, FALSE); */

		/* Last-Modified */
		date = strtol(elements[1], NULL, 10)
			+ strtol(elements[3], NULL, 10); /* adding time difference */
		kz_bookmark_set_last_modified(item, date);
		t = gmtime(&date);
		strftime(timestr, 20, "%Y/%m/%d %H:%M", t);
		/* URL */
		if (elements[5])
			kz_bookmark_set_link(item, elements[5]);
		/* Title */
		if (elements[6])
		{
			gchar *title = g_strdup_printf("%s %s",timestr, elements[6]);
			kz_bookmark_set_title(item, title);
			g_free(title);
		}
		/* Link of Antenna */
		if (elements[8])
			kz_bookmark_set_link(bookmark, elements[8]);

		g_strfreev(elements);

		kz_bookmark_append(KZ_BOOKMARK(bookmark), item);
		g_object_unref(G_OBJECT(item));

		line_number++;
	}

	g_strfreev(lines);

	/* sorting */
	kz_bookmark_sort(KZ_BOOKMARK(bookmark),
			 KZ_BOOKMARK_SORT_LAST_MODIFIED);

	return TRUE;
}
