/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <kstandarddirs.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <kdebug.h>

#include <katengine.h>

KatEngine::KatEngine()
{
    // Check if the kat database-file exists
    // Location is: "$KDEHOME/share/apps/kat/"
    KStandardDirs* sd = new KStandardDirs();
    QString fileName = sd->saveLocation( "data", "kat", true ) + "db.kat";
    delete sd;

    // if the database-file does not exist, create it
    QFile qf( fileName );
    if ( !qf.exists() )
        createMainDatabase();

    // open it
    openMainDatabase();

    // Fix tables to work with new version of kat
    checkTables();
    closeMainDatabase();

    // open it
    openMainDatabase();
}

KatEngine::~KatEngine()
{
    closeMainDatabase();
}

// A dumb method to fix the tables when migrating from previous version of kat
void KatEngine::checkTables()
{
    CppSQLite3Query q = m_db->execQuery( "select * from catalogs limit 1;" );

    bool needsFixing = false;
    try
    {
        if (!q.eof())
        {
            int version = q.getIntField("version");
        }
        q.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        q.finalize();
        kdDebug() << e.errorMessage() << endl;
        needsFixing = true;
    }

    if (needsFixing)
    {
        //kdDebug() << "Fixing old databases" << endl;
        openTransaction();
        try
        {
            // Create temp table needed for this catalog
            m_db->execDML( "create table indexerstore (catalogid integer, key varchar(5000), data integer);" );

            m_db->execDML (" alter table catalogs add column autoupdate integer; ");
            m_db->execDML (" alter table catalogs add column version integer; ");
            m_db->execDML (" alter table catalogs add column filesize integer; ");
            m_db->execDML (" alter table files add column lastupdatedate integer; ");

            m_db->execDML (" update catalogs set autoupdate=1; ");
            m_db->execDML (" update catalogs set version=1; ");
            m_db->execDML (" update catalogs set filesize=0; ");
            m_db->execDML (" update files set lastupdatedate=1150914141; ");
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << e.errorMessage() << endl;
        }
        commitTransaction();
        //kdDebug() << "Done fixing old databases" << endl;
    }
}

KatCatalog* KatEngine::readCatalog( int catalogId )
{
    KatCatalog* cat = 0L;

    QString DML = "select * from catalogs where catalogid = '" + QString::number( catalogId ) + "';";

    try
    {
        cat = new KatCatalog();
        CppSQLite3Query q = m_db->execQuery( DML );
        if ( !q.eof() )
        {
            cat->setDb( m_db );
            cat->setCatalogId( q.getIntField( "catalogid" ) );
            cat->setAutoUpdate( q.getIntField( "autoupdate" ) );
            cat->setName( q.getStringField( "name" ) );
            cat->setDescription( q.getStringField( "description" ) );
            cat->setPath( q.getStringField( "path" ) );
            cat->setNotes( q.getStringField( "notes" ) );
            cat->setAuthor( q.getStringField( "author" ) );
            cat->setVersion( q.getIntField( "version" ) );
            cat->setThumbnailSize( q.getIntField( "thumbnailsize" ) );
            cat->setUseExclusionList( q.getIntField( "useexclusionlist" ) );
            cat->setCreationDate( q.getIntField( "creationdate" ) );
            cat->setLastUpdateDate( q.getIntField( "lastupdatedate" ) );
            cat->setMetaData( q.getIntField( "metadata" ) );
            cat->setFiles( q.getIntField( "files" ) );
            cat->setFolders( q.getIntField( "folders" ) );
            cat->setFullTexts( q.getIntField( "fulltexts" ) );
            cat->setThumbnails( q.getIntField( "thumbnails" ) );
            cat->setWords( q.getIntField( "words" ) );
            cat->setFileSize( q.getIntField( "filesize" ) );
        }
        q.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
    }

    return cat;
}

int KatEngine::addCatalog( KatCatalog* cat )
{
    kdDebug()<<" KatEngine::addCatalog( KatCatalog* cat ): "<<cat<<endl;
    try
    {
        CppSQLite3Statement stmt = m_db->compileStatement( "insert into catalogs("
                                                                               "autoupdate, "
                                                                               "name, "
                                                                               "description, "
                                                                               "path, "
                                                                               "notes, "
                                                                               "author, "
                                                                               "version, "
                                                                               "thumbnailsize, "
                                                                               "useexclusionlist, "
                                                                               "creationdate, "
                                                                               "lastupdatedate, "
                                                                               "metadata, "
                                                                               "files, "
                                                                               "folders, "
                                                                               "fulltexts, "
                                                                               "thumbnails, "
                                                                               "words, "
                                                                               "filesize ) "
                                                           "values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?);" );
        stmt.bind(  1, cat->autoUpdate() );
        stmt.bind(  2, cat->name() );
        stmt.bind(  3, cat->description() );
        stmt.bind(  4, cat->path() );
        stmt.bind(  5, cat->notes() );
        stmt.bind(  6, cat->author() );
        stmt.bind(  7, cat->version() );
        stmt.bind(  8, cat->thumbnailSize() );
        stmt.bind(  9, cat->useExclusionList() );
        stmt.bind( 10, cat->creationDate() );
        stmt.bind( 11, cat->lastUpdateDate() );
        stmt.bind( 12, cat->metaData() );
        stmt.bind( 13, cat->files() );
        stmt.bind( 14, cat->folders() );
        stmt.bind( 15, cat->fullTexts() );
        stmt.bind( 16, cat->thumbnails() );
        stmt.bind( 17, cat->words() );
        stmt.bind( 18, ( long )cat->fileSize() );
        kdDebug()<<"KatEngine::addCatalog exec into database\n";
        stmt.execDML();
        stmt.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "KatEngine::addCatalog "<<e.errorMessage() << endl;
        return e.errorCode();
    }
    kdDebug()<<" m_db->lastRowId() ::::::::::"<<m_db->lastRowId()<<endl;
    cat->setCatalogId( m_db->lastRowId() );

    return 0;
}

int KatEngine::updateCatalog( KatCatalog* cat )
{
    try
    {
        CppSQLite3Statement stmt = m_db->compileStatement( "update catalogs set "
                                                                                "autoupdate = ?, "
                                                                                "name = ?, "
                                                                                "description = ?, "
                                                                                "path = ?, "
                                                                                "notes = ?, "
                                                                                "author = ?, "
                                                                                "version = ?, "
                                                                                "thumbnailsize = ?, "
                                                                                "useexclusionlist = ?, "
                                                                                "creationdate = ?, "
                                                                                "lastupdatedate = ?, "
                                                                                "metadata = ?, "
                                                                                "files = ?, "
                                                                                "folders = ?, "
                                                                                "fulltexts = ?, "
                                                                                "thumbnails = ?, "
                                                                                "words = ?, "
                                                                                "filesize = ? "
                                                            "where catalogid = '" + QString::number( cat->catalogId() ) + "';" );

        stmt.bind(  1, cat->autoUpdate() );
        stmt.bind(  2, cat->name() );
        stmt.bind(  3, cat->description() );
        stmt.bind(  4, cat->path() );
        stmt.bind(  5, cat->notes() );
        stmt.bind(  6, cat->author() );
        stmt.bind(  7, cat->version() );
        stmt.bind(  8, cat->thumbnailSize() );
        stmt.bind(  9, cat->useExclusionList() );
        stmt.bind( 10, cat->creationDate() );
        stmt.bind( 11, cat->lastUpdateDate() );
        stmt.bind( 12, cat->metaData() );
        stmt.bind( 13, cat->files() );
        stmt.bind( 14, cat->folders() );
        stmt.bind( 15, cat->fullTexts() );
        stmt.bind( 16, cat->thumbnails() );
        stmt.bind( 17, cat->words() );
        stmt.bind( 18, ( long )cat->fileSize() );

        stmt.execDML();
        stmt.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() <<"KatEngine::updateCatalog"<< e.errorMessage() << endl;
        return e.errorCode();
    }

    return 0;
}

KatCatalog* KatEngine::getCatalog( const QString &catalogName )
{
    KatCatalog* cat = 0L;

    QPtrList<KatCatalog> catalogs= readCatalogs();

    QPtrList<KatCatalog>::iterator end( catalogs.end() );
    for ( QPtrList<KatCatalog>::iterator it = catalogs.begin(); it != end; ++it )
    {
        cat = *it;
        if ( cat->name() == catalogName ) {
            break;
        }
    }
    return cat;
}

QPtrList<KatCatalog> KatEngine::readCatalogs()
{
    QPtrList<KatCatalog> catalogs;

    try
    {
        QString DML = "select * from catalogs;";
        CppSQLite3Query q = m_db->execQuery( DML );
        while ( !q.eof() )
        {
            KatCatalog* cat = new KatCatalog();

            cat->setDb( m_db );
            cat->setCatalogId( q.getIntField( "catalogid" ) );
            cat->setAutoUpdate( q.getIntField( "autoupdate" ) );
            cat->setName( q.getStringField( "name" ) );
            cat->setDescription( q.getStringField( "description" ) );
            cat->setPath( q.getStringField( "path" ) );
            cat->setNotes( q.getStringField( "notes" ) );
            cat->setAuthor( q.getStringField( "author" ) );
            cat->setVersion( q.getIntField( "version" ) );
            cat->setThumbnailSize( q.getIntField( "thumbnailsize" ) );
            cat->setUseExclusionList( q.getIntField( "useexclusionlist" ) );
            cat->setCreationDate( q.getIntField( "creationdate" ) );
            cat->setLastUpdateDate( q.getIntField( "lastupdatedate" ) );
            cat->setMetaData( q.getIntField( "metadata" ) );
            cat->setFiles( q.getIntField( "files" ) );
            cat->setFolders( q.getIntField( "folders" ) );
            cat->setFullTexts( q.getIntField( "fulltexts" ) );
            cat->setThumbnails( q.getIntField( "thumbnails" ) );
            cat->setWords( q.getIntField( "words" ) );
            cat->setFileSize( q.getIntField( "filesize" ) );

            catalogs.append( cat );

            q.nextRow();
        }
        q.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "readCatalogs() :: " << e.errorMessage() << endl;
    }

    return catalogs;
}

KatScanFolder* KatEngine::scanFolder( KatCatalog* cat )
{
    return KatScanFolder::scanFolderJob( cat, m_db );
}

KatInfoExtractor* KatEngine::extractInfo( KatCatalog* cat )
{
    return KatInfoExtractor::extractInfoJob( cat, m_db );
}

KatTempTable* KatEngine::tempTable( int catalogId )
{
    return KatTempTable::tempTable( m_db, catalogId );
}

// database functions

int KatEngine::createMainDatabase()
{
    // Location is: "$KDEHOME/share/apps/kat/"
    KStandardDirs* sd = new KStandardDirs();
    QString fileName = sd->saveLocation( "data", "kat", true) + "db.kat";
    delete sd;

    QFile::remove( fileName );
    QFile::remove( fileName + "-journal" );

    return createDatabase( fileName );
}

int KatEngine::createDatabase( const QString &fileName )
{
    CppSQLite3DB* db = 0L;

    try
    {
        db = new CppSQLite3DB();
        db->open( fileName );

        // create the tables and the indexes
        db->execDML( "create table catalogs("
                                            "catalogid integer primary key, "
                                            "autoupdate integer, "
                                            "name varchar(500), "
                                            "description varchar(500), "
                                            "path varchar(1000), "
                                            "notes varchar(2000), "
                                            "author varchar(100), "
                                            "version integer, "
                                            "thumbnailsize integer, "
                                            "useexclusionlist integer, "
                                            "creationdate varchar(12), "
                                            "lastupdatedate varchar(12), "
                                            "metadata integer, "
                                            "files integer, "
                                            "folders integer, "
                                            "fulltexts integer, "
                                            "thumbnails integer, "
                                            "words integer, "
                                            "filesize integer "
                                            ");" );

        db->execDML( "create table files("
                                            "fileid integer primary key, "
                                            "catalogid integer, "
                                            "fullname varchar(5000), "
                                            "filename varchar(300), "
                                            "parentid integer, "
                                            "filetype varchar(100), "
                                            "filesize integer, "
                                            "statuschangedate integer, "
                                            "modificationdate integer, "
                                            "lastaccessdate integer, "
                                            "lastupdatedate integer, "	// last updated by kat/daemon
                                            "username varchar(100), "
                                            "groupname varchar(100), "
                                            "permissions integer, "
                                            "mode integer, "
                                            "language varchar(100) "
                                            ");" );

        db->execDML( "create index files_parentid on files (parentid);" );
        db->execDML( "create index files_catalogid on files (catalogid);" );
        db->execDML( "create index files_filename on files (filename);" );
        db->execDML( "create unique index files_fullname on files (fullname);" );

        db->execDML( "create table words("
                                            "word varchar(100), "
                                            "wordid integer primary key"
                                            ");" );

        db->execDML( "create unique index words_word on words (word);" );
        db->execDML( "create unique index words_wordid on words (wordid);" );

        db->execDML( "create table wordfile("
                                            "wordid integer, "
                                            "fileid integer, "
                                            "occurrences integer "
                                            ");" );

        db->execDML( "create index wordfile_wordid on wordfile (wordid);" );
        db->execDML( "create index wordfile_fileid on wordfile (fileid);" );


        db->execDML( "create table metadata("
                                            "fileid integer, "
                                            "field varchar(100), "
                                            "type varchar(100), "
                                            "value varchar(5000) "
                                            ");" );

        db->execDML( "create index metadata_fileid on metadata (fileid);" );
        db->execDML( "create index metadata_field on metadata (field);" );
        db->execDML( "create index metadata_type on metadata (type);" );
        db->execDML( "create index metadata_value on metadata (value);" );


        db->execDML( "create table thumbnails("
                                            "fileid integer primary key, "
                                            "thumbnaildata blob, "
                                            "thumbnaildatalength integer "
                                            ");" );

        db->execDML( "create table fulltexts("
                                            "fileid integer primary key, "
                                            "fulltextdata blob, "
                                            "fulltextdatalength integer "
                                            ");" );

        // Create temp table needed for this catalog
        db->execDML( "create table indexerstore (catalogid integer, key varchar(5000), data integer);" );

        db->close();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() <<"KatEngine::createDatabase "<< e.errorMessage() << endl;
        delete db;
        return e.errorCode();
    }

    delete db;
    return 0;
}

int KatEngine::openTransaction()
{
    try
    {
        m_db->execDML( "begin transaction;" );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "KatEngine::openTransaction :"<< e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

int KatEngine::commitTransaction()
{
    try
    {
        m_db->execDML( "commit transaction;" );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "KatEngine::commitTransaction "<<e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

int KatEngine::rollbackTransaction()
{
    try
    {
        m_db->execDML( "rollback transaction;" );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() <<"KatEngine::rollbackTransaction "<< e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

CppSQLite3DB* KatEngine::openDatabase( const QString &fileName )
{
    CppSQLite3DB* db = 0L;

    try
    {
        db = new CppSQLite3DB();
        db->open( fileName );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "KatEngine::openDatabase :"<<fileName << e.errorMessage() << endl;
    }

    return db;
}

int KatEngine::openMainDatabase()
{
    // Location is: "$KDEHOME/share/apps/kat/"
    KStandardDirs* sd = new KStandardDirs();
    QString fileName = sd->saveLocation( "data", "kat", true) + "db.kat";
    delete sd;

    m_db = openDatabase( fileName );

    if ( m_db == 0L )
        return -1;
    else
        return 0;
}

int KatEngine::closeDatabase( CppSQLite3DB* db )
{
    try
    {
        db->close();
        delete db;
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() <<"KatEngine::closeDatabase :"<< e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

int KatEngine::closeMainDatabase()
{
    return closeDatabase( m_db );
}

QMap<int,QString> KatEngine::getFileChildren( int catalogId, int parentId )
{
    QMap<int, QString> children;
    QString DML;

    try
    {
        DML = "select fileid, fullname from files "
              "where catalogid = '" + QString::number( catalogId ) + "' and parentid = '" + QString::number( parentId ) + "';";
        CppSQLite3Query q = m_db->execQuery( DML );

        while ( !q.eof() )
        {
            int fileId = q.getIntField( "fileid" );
            QString fullName = q.getStringField( "fullname" );

            children.insert( fileId, fullName );

            q.nextRow();
        }
        q.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "getFileChildren " << e.errorMessage() << endl;
        return children;
    }

    return children;
}

KatInformation KatEngine::readFileInformation(  const QString &fullName, KatInformation::Type type )
{
    int fileId = 0;
    KatInformation info;
    CppSQLite3Query q;

    try
    {
        QString DML = "select fileid from files where fullname = '" + fullName + "';";
        q = m_db->execQuery( DML );

        if ( !q.eof() )
            fileId = q.getIntField( "fileid" );
        q.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << k_funcinfo << e.errorMessage() << endl;
        q.finalize();
    }

    if ( fileId )
        return readFileInformation( fileId, type );
    else
        return info;
}

KatInformation KatEngine::readFileInformation( int fileId, KatInformation::Type type )
{
    KatInformation info;
    QString DML;

    // Read the basic information present in the files table
    if ( type & KatInformation::FileInfo )
    {
        try
        {
            DML = "select * from files where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query q = m_db->execQuery( DML );

            if ( !q.eof() )
            {
                info.fileId = q.getIntField( "fileid" );
                info.catalogId = q.getIntField( "catalogid" );
                info.fullName = q.getStringField( "fullname" );
                info.fileName = q.getStringField( "filename" );
                info.parentId = q.getIntField( "parentid" );
                info.fileType = q.getStringField( "filetype" );
                info.fileSize = q.getIntField( "filesize" );
                info.statusChangeDate = q.getIntField( "statuschangedate" );
                info.modificationDate = q.getIntField( "modificationdate" );
                info.lastAccessDate = q.getIntField( "lastaccessdate" );
                info.lastUpdateDate = q.getIntField( "lastupdatedate" );
                // TODO: Add the new fields (username, group, permissions, mode)
                info.language = q.getStringField( "language" );
            }
            q.finalize();
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << "readFileInformation (fileinfo) " << e.errorMessage() << endl;
            return info;
        }
    }

    // Read the thumbnail present in the thumbnails table
    if ( type & KatInformation::Thumbnail )
    {
        try
        {
            DML = "select * from thumbnails where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query q = m_db->execQuery( DML );

            if ( !q.eof() )
            {
                info.thumbnailDataLength = q.getIntField( "thumbnaildatalength" );
                info.hasThumbnail = true;

                QBuffer buffer( info.thumbnailData );
                buffer.open( IO_WriteOnly );
                int len = info.thumbnailDataLength;
                buffer.writeBlock( (char*)q.getBlobField( "thumbnaildata" , len ), (long)len );
                buffer.close();
            }
            q.finalize();
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << "readFileInformation (thumbnail) " << e.errorMessage() << endl;
            return info;
        }
    }

    // Read the metadata information present in the metadata table
    if ( type & KatInformation::MetaData )
    {
        QString metaData = "";
        try
        {
            DML = "select * from metadata where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query q = m_db->execQuery( DML );

            while ( !q.eof() )
            {
                info.hasMetaData = true;

                QString field = q.getStringField( "field" );
                QString type = q.getStringField( "type" );
                QString value = q.getStringField( "value" );

                metaData = metaData + field + "|" + type + "|" + value + "|";

                q.nextRow();
            }
            q.finalize();

            info.metaData = metaData;
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << "readFileInformation (metadata) " << e.errorMessage() << endl;
            return info;
        }
    }

    // Check if the file has a fulltext record
    if ( type & KatInformation::FullText )
    {
        try
        {
            DML = "select fileid from fulltexts where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query q = m_db->execQuery( DML );

            if ( !q.eof() )
                info.hasFullText = true;

            q.finalize();
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << "readFileInformation (fulltext) " << e.errorMessage() << endl;
            return info;
        }
    }

    return info;
}

blobInfo KatEngine::readFullTextRecord( int fileId )
{
    QString DML;
    blobInfo bi;
    bi.blobLen = 0;

    // Read the fulltext present in the fulltext table
    try
    {
        DML = "select * from fulltexts where fileid = '" + QString::number( fileId ) + "';";
        CppSQLite3Query q = m_db->execQuery( DML );

        if ( !q.eof() )
        {
            bi.blobLen = q.getIntField( "fulltextdatalength" );

            QBuffer buffer( bi.blobData );
            buffer.open( IO_WriteOnly );
            int len = bi.blobLen;
            buffer.writeBlock( (char*)q.getBlobField( "fulltextdata" , len ), (long)len );
            buffer.close();
        }
        q.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() <<"KatEngine::readFullTextRecord "<< e.errorMessage() << endl;
        return bi;
    }

    return bi;
}

QValueList<int> KatEngine::doSearch( const QString &what, bool filenames, bool descriptions, bool fulltexts )
{
    QValueList<int> results;
    QString DML = "";

    // TODO: Find a regular expression to eliminate all possible problems with words in SQL
    QString swhat( what );
    swhat = swhat.replace( QRegExp( "'" ), "''" );

    try
    {
        if ( fulltexts )
        {
            DML = DML + "select distinct fileid from wordfile join words on wordfile.wordid = words.wordid "
                        "where word like '%" + swhat + "%'";

            if ( filenames || descriptions )
                DML = DML + " union ";
        }

        if ( filenames )
        {
            DML = DML + "select fileid from files where filename like '%" + swhat + "%'";

            if ( descriptions )
                DML = DML + " union ";
        }

        if ( descriptions )
            DML = DML + "select fileid from metadata where value like '%" + swhat + "%'";

        DML = DML + ";";

        CppSQLite3Query q = m_db->execQuery( DML );

        while ( !q.eof() )
        {
            int fileId = q.getIntField( "fileid" );
            results.push_back( fileId );

            q.nextRow();
        }
        q.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
    }

    return results;
}

QMap<QString, int> KatEngine::readFileWords( int fileId )
{
    QString DML;
    QMap<QString, int> words;

    if ( fileId == -1 )
    {
        // read all words from the catalog
    }
    else
    {
        // read the words found in the specified file
        try
        {
            openTransaction();

            DML = "select wordfile.occurrences, words.word from wordfile "
                        "join words on wordfile.wordid = words.wordid "
                        "where wordfile.fileid = '" + QString::number( fileId ) + "';";

            CppSQLite3Query q = m_db->execQuery( DML );
            while ( !q.eof() )
            {
                words[ q.getStringField( "words.word" ) ] = q.getIntField( "wordfile.occurrences" );
                q.nextRow();
            }
            q.finalize();

            commitTransaction();

        }
        catch ( CppSQLite3Exception& e )
        {
            rollbackTransaction();
            kdDebug() << e.errorMessage() << endl;
        }
    }

    return words;
}

bool KatEngine::folderHasChildren( int catalogId, int fileId )
{
    int childrenNumber = 0;

    try
    {
        childrenNumber = m_db->execScalar( "select count(*) from files  "
                                           "where parentid = '" + QString::number( fileId ) + "' "
                                           "and catalogid = '" + QString::number( catalogId ) + "' "
                                           "and filetype = 'inode/directory';");
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "folderHasChildren " << e.errorMessage() << endl;
        return false;
    }

    if ( childrenNumber != 0 )
        return true;
    else
        return false;
}

int KatEngine::deleteCatalog( KatCatalog* cat )
{
    QString DML;

    kdDebug() << "KatEngine::deleteCatalog " << cat->name() << endl;

    // get the catalog id
    int catalogId = cat->catalogId();

    try {
        // open transaction
        openTransaction();

        // remove the catalog from CATALOGS table
        m_db->execDML( "delete from catalogs where catalogid = " + QString::number( catalogId ) + ";" );

        // get the files belonging to the catalog
        DML = "select fileid from files where catalogid = '" + QString::number( catalogId ) + "';";
        CppSQLite3Query qf = m_db->execQuery( DML );
        while ( !qf.eof() )
        {
            int fileId = qf.getIntField( "fileid" );

            // delete the WORDFILE records belonging to the file
            m_db->execDML( "delete from wordfile where fileid = '" + QString::number( fileId ) + "';" );

            // delete the metadata belonging to the file
            m_db->execDML( "delete from metadata where fileid = '" + QString::number( fileId ) + "';" );

            // delete the fulltexts belonging to the file
            m_db->execDML( "delete from fulltexts where fileid = '" + QString::number( fileId ) + "';" );

            // delete the thumbnails belonging to the file
            m_db->execDML( "delete from thumbnails where fileid = '" + QString::number( fileId ) + "';" );

            qf.nextRow();
        }
        qf.finalize();

        // remove the files of the catalog from the FILES table
        m_db->execDML( "delete from files where catalogid = '" + QString::number( catalogId ) + "';" );

        // delete the WORDS which do not have a record in WORDFILE
        m_db->execDML( "delete from words where wordid not in (select distinct wordid from wordfile);" );

        commitTransaction();
    }
    catch ( CppSQLite3Exception& e )
    {
        rollbackTransaction();
        kdDebug() << "deleteCatalog " << e.errorMessage() << endl;
        return -1;
    }

    return 0;
}

int KatEngine::deleteAllCatalogs()
{
    QString DML;

    kdDebug() << "KatEngine::deleteAllCatalog " << endl;

    try {
        // open transaction
        openTransaction();

        // clear CATALOGS table
        m_db->execDML( "delete from catalogs;" );

        // clear FILES table
        m_db->execDML( "delete from files;" );

        // clear WORDFILE table
        m_db->execDML( "delete from wordfile;" );

        // clear WORDS table
        m_db->execDML( "delete from words;" );

        // delete the metadata
        m_db->execDML( "delete from metadata;" );

        // delete the fulltexts
        m_db->execDML( "delete from fulltexts;" );

        // delete the thumbnails
        m_db->execDML( "delete from thumbnails;" );

        commitTransaction();
    }
    catch ( CppSQLite3Exception& e )
    {
        rollbackTransaction();
        kdDebug() << "deleteAllCatalogs " << e.errorMessage() << endl;
        return -1;
    }

    return 0;

}

int KatEngine::importCatalog( QString& fileName )
{
    int result;
    int catalogId;
    QString DML;
    CppSQLite3Statement stmt;

    kdDebug() << "KatEngine::importCatalog " << fileName << endl;

    // open the source database file
    CppSQLite3DB* s_db = openDatabase( fileName );

    try {
        openTransaction();

        // import the catalog
        DML = "select * from catalogs;";
        CppSQLite3Query q = s_db->execQuery( DML );
        if ( !q.eof() )
        {
            int autoupdate = q.getIntField( "autoupdate" );
            int version = q.getIntField( "version" );
            if ( !autoupdate )
                autoupdate = 1;
            if ( !version )
                version = 1;

            stmt = m_db->compileStatement( "insert into catalogs("
                                                                "autoupdate, "
                                                                "name, "
                                                                "description, "
                                                                "path, "
                                                                "notes, "
                                                                "author, "
                                                                "version, "
                                                                "thumbnailsize, "
                                                                "useexclusionlist, "
                                                                "creationdate, "
                                                                "lastupdatedate, "
                                                                "metadata, "
                                                                "files, "
                                                                "folders, "
                                                                "fulltexts, "
                                                                "thumbnails, "
                                                                "words, "
                                                                "filesize ) "
                                            "values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?);" );

            stmt.bind(  1, q.getIntField( "autoupdate" ) );
            stmt.bind(  2, q.getStringField( "name" ) );
            stmt.bind(  3, q.getStringField( "description" ) );
            stmt.bind(  4, q.getStringField( "path" ) );
            stmt.bind(  5, q.getStringField( "notes" ) );
            stmt.bind(  6, q.getStringField( "author" ) );
            stmt.bind(  7, q.getIntField( "version" ) );
            stmt.bind(  8, q.getIntField( "thumbnailsize" ) );
            stmt.bind(  9, q.getIntField( "useexclusionlist" ) );
            stmt.bind( 10, q.getIntField( "creationdate" ) );
            stmt.bind( 11, q.getIntField( "lastupdatedate" ) );
            stmt.bind( 12, q.getIntField( "metadata" ) );
            stmt.bind( 13, q.getIntField( "files" ) );
            stmt.bind( 14, q.getIntField( "folders" ) );
            stmt.bind( 15, q.getIntField( "fulltexts" ) );
            stmt.bind( 16, q.getIntField( "thumbnails" ) );
            stmt.bind( 17, q.getIntField( "words" ) );
            stmt.bind( 18, q.getIntField( "filesize" ) );

            stmt.execDML();
            stmt.finalize();

            catalogId = m_db->lastRowId();
        }
        q.finalize();

        // import the words (which do not exist already in the main database)
        DML = "select * from words;";
        CppSQLite3Query qw = s_db->execQuery( DML );
        while ( !qw.eof() )
        {
            QString word = qw.getStringField( "word" );
            int sourceWordId = qw.getIntField( "wordid" );

            // check if the word already exists in the main database
            if ( getWordId( word ) == -1 )
            {
                // the word does not exists yet, so insert it
                stmt = m_db->compileStatement( "insert into words(word) values (?);" );
                stmt.bind(  1, qw.getStringField( "word" ) );

                stmt.execDML();
                stmt.finalize();
            }

            qw.nextRow();
        }
        qw.finalize();


        // import the files
        DML = "select * from files;";
        CppSQLite3Query qf = s_db->execQuery( DML );
        while ( !qf.eof() )
        {
            int sourceFileId = qf.getIntField( "fileid" );
            int destFileId;

            stmt = m_db->compileStatement( "insert into files("
                                                              "catalogid, "
                                                              "fullname, "
                                                              "filename, "
                                                              "parentid, "
                                                              "filetype, "
                                                              "filesize, "
                                                              "statuschangedate, "
                                                              "modificationdate, "
                                                              "lastaccessdate, "
                                                              "lastupdatedate, "
                                                              // "username, "
                                                              // "groupname, "
                                                              // "permissions, "
                                                              // "mode, "
                                                              "language "
                                           ") values (?,?,?,?,?,?,?,?,?,?,?);" );

            stmt.bind(  1, catalogId );
            stmt.bind(  2, qf.getStringField( "fullname" ) );
            stmt.bind(  3, qf.getStringField( "filename" ) );
            stmt.bind(  4, qf.getIntField( "parentid" ) );
            stmt.bind(  5, qf.getStringField( "filetype" ) );
            stmt.bind(  6, qf.getIntField( "filesize" ) );
            stmt.bind(  7, qf.getIntField( "statuschangedate" ) );
            stmt.bind(  8, qf.getIntField( "modificationdate" ) );
            stmt.bind(  9, qf.getIntField( "lastaccessdate" ) );
            stmt.bind( 10, qf.getIntField( "lastupdatedate" ) );
            // TODO: Add the new fields (username, groupname, permissions, mode)
            stmt.bind( 11, qf.getStringField( "language" ) );

            stmt.execDML();
            stmt.finalize();

            destFileId = m_db->lastRowId();

            // import the thumbnail
            DML = "select * from thumbnails where fileid = '" + QString::number( sourceFileId ) + "';";
            CppSQLite3Query qt = s_db->execQuery( DML );
            if ( !qt.eof() )
            {
                int thumbnailDataLength = qt.getIntField( "thumbnaildatalength" );

                stmt = m_db->compileStatement( "insert into thumbnails(fileid, thumbnaildata, thumbnaildatalength) values(?,?,?);" );
                stmt.bind(  1, destFileId );
                stmt.bind(  2, qt.getBlobField( "thumbnaildata", thumbnailDataLength ), thumbnailDataLength );
                stmt.bind(  3, thumbnailDataLength );

                stmt.execDML();
                stmt.finalize();
            }

            // import the metadata
            DML = "select * from metadata where fileid = '" + QString::number( sourceFileId ) + "';";
            CppSQLite3Query qm = s_db->execQuery( DML );
            if ( !qm.eof() )
            {
                stmt = m_db->compileStatement( "insert into metadata(fileid, field, type, value) values(?,?,?,?);" );
                stmt.bind(  1, destFileId );
                stmt.bind(  2, qm.getStringField( "field" ) );
                stmt.bind(  3, qm.getStringField( "type" ) );
                stmt.bind(  4, qm.getStringField( "value" ) );

                stmt.execDML();
                stmt.finalize();
            }

            // import the fulltexts
            DML = "select * from fulltexts where fileid = '" + QString::number( sourceFileId ) + "';";
            CppSQLite3Query qft = s_db->execQuery( DML );
            if ( !qft.eof() )
            {
                int fullTextDataLength = qft.getIntField( "fulltextdatalength" );

                stmt = m_db->compileStatement( "insert into fulltexts(fileid, fulltextdata, fulltextdatalength) values(?,?,?);" );
                stmt.bind(  1, destFileId );
                stmt.bind(  2, qft.getBlobField( "fulltextdata", fullTextDataLength ), fullTextDataLength );
                stmt.bind(  3, fullTextDataLength );

                stmt.execDML();
                stmt.finalize();
            }

            // import the wordfile record for current file
            DML = "select * from wordfile, words on wordfile.wordid = words.wordid where fileid = '" + QString::number( sourceFileId ) + "';";
            CppSQLite3Query qw = s_db->execQuery( DML );
            while ( !qw.eof() )
            {
                QString word = qw.getStringField( "word" );
                int sourceWordId = qw.getIntField( "wordid" );
                int destWordId;

                // get the wordid of the word from the main database
                destWordId = getWordId( word );

                stmt = m_db->compileStatement( "insert into wordfile(fileid, wordid, occurrences) values (?,?,?);" );
                stmt.bind(  1, destFileId );
                stmt.bind(  2, destWordId );
                stmt.bind(  3, qw.getIntField( "occurrences" ) );

                stmt.execDML();
                stmt.finalize();

                qw.nextRow();
            }
            qw.finalize();


            qf.nextRow();
        }
        qf.finalize();


        commitTransaction();
        closeDatabase( s_db );
    }
    catch ( CppSQLite3Exception& e )
    {
        rollbackTransaction();
        closeDatabase( s_db );

        kdDebug() << "importCatalog " << e.errorMessage() << endl;
        return -1;
    }

    return 0;
}

int KatEngine::exportCatalog( KatCatalog* cat, QString& fileName )
{
    int result;
    QString DML;
    CppSQLite3Statement stmt;

    kdDebug() << "KatEngine::exportCatalog " << cat->name() << " in " << fileName << endl;

    // create the destination database file
    if ( ( result = createDatabase( fileName ) )!= 0 )
        return result;
    kdDebug()<<" after create temp database \n";
    // open the destination database file
    CppSQLite3DB* d_db = openDatabase( fileName );
    kdDebug()<<" Export Database : opendatabase d_db :"<<d_db<<endl;
    try {
        // open transaction
        d_db->execDML( "begin transaction;" );

        // export the catalog
        stmt = d_db->compileStatement( "insert into catalogs("
                                                             "catalogid, "
                                                             "autoupdate, "
                                                             "name, "
                                                             "description, "
                                                             "path, "
                                                             "notes, "
                                                             "author, "
							     "version, "
                                                             "thumbnailsize, "
                                                             "useexclusionlist, "
                                                             "creationdate, "
                                                             "lastupdatedate, "
                                                             "metadata, "
                                                             "files, "
                                                             "folders, "
                                                             "fulltexts, "
                                                             "thumbnails, "
                                                             "words, "
                                                             "filesize "
                                        "values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?);" );
        stmt.bind(  1, cat->catalogId() );
        stmt.bind(  2, cat->autoUpdate() );
        stmt.bind(  3, cat->name() );
        stmt.bind(  4, cat->description() );
        stmt.bind(  5, cat->path() );
        stmt.bind(  6, cat->notes() );
        stmt.bind(  7, cat->author() );
        stmt.bind(  8, cat->version() );
        stmt.bind(  9, cat->thumbnailSize() );
        stmt.bind( 10, cat->useExclusionList() );
        stmt.bind( 11, cat->creationDate() );
        stmt.bind( 12, cat->lastUpdateDate() );
        stmt.bind( 13, cat->metaData() );
        stmt.bind( 14, cat->files() );
        stmt.bind( 15, cat->folders() );
        stmt.bind( 16, cat->fullTexts() );
        stmt.bind( 17, cat->thumbnails() );
        stmt.bind( 18, cat->words() );
        stmt.bind( 19, ( long )cat->fileSize() );

        stmt.execDML();
        stmt.finalize();

        // export the files
        DML = "select * from files where catalogid = '" + QString::number( cat->catalogId() ) + "';";
        CppSQLite3Query q = m_db->execQuery( DML );
        while ( !q.eof() )
        {
            int fileId = q.getIntField( "fileid" );

            stmt = d_db->compileStatement( "insert into files("
                                                              "fileid, "
                                                              "catalogid, "
                                                              "fullname, "
                                                              "filename, "
                                                              "parentid, "
                                                              "filetype, "
                                                              "filesize, "
                                                              "statuschangedate, "
                                                              "modificationdate, "
                                                              "lastaccessdate, "
                                                              "lastupdatedate, "
                                                              // "username, "
                                                              // "groupname, "
                                                              // "permissions, "
                                                              // "mode, "
                                                              "language "
                                           ") values (?,?,?,?,?,?,?,?,?,?,?);" );

            stmt.bind(  1, q.getIntField( "fileid" ) );
            stmt.bind(  2, q.getIntField( "catalogid" ) );
            stmt.bind(  3, q.getStringField( "fullname" ) );
            stmt.bind(  4, q.getStringField( "filename" ) );
            stmt.bind(  5, q.getIntField( "parentid" ) );
            stmt.bind(  6, q.getStringField( "filetype" ) );
            stmt.bind(  7, q.getIntField( "filesize" ) );
            stmt.bind(  8, q.getIntField( "statuschangedate" ) );
            stmt.bind(  9, q.getIntField( "modificationdate" ) );
            stmt.bind( 10, q.getIntField( "lastaccessdate" ) );
            stmt.bind( 11, q.getIntField( "lastupdatedate" ) );
            // TODO: Add the new fields (username, groupname, permissions, mode)
            stmt.bind( 12, q.getStringField( "language" ) );

            stmt.execDML();
            stmt.finalize();

            // export the thumbnail
            DML = "select * from thumbnails where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query qt = m_db->execQuery( DML );
            if ( !qt.eof() )
            {
                int thumbnailDataLength = qt.getIntField( "thumbnaildatalength" );

                stmt = d_db->compileStatement( "insert into thumbnails(fileid, thumbnaildata, thumbnaildatalength) values(?,?,?);" );
                stmt.bind(  1, qt.getIntField( "fileid" ) );
                stmt.bind(  2, qt.getBlobField( "thumbnaildata", thumbnailDataLength ), thumbnailDataLength );
                stmt.bind(  3, thumbnailDataLength );

                stmt.execDML();
                stmt.finalize();
            }

            // export the metadata
            DML = "select * from metadata where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query qm = m_db->execQuery( DML );
            while ( !qm.eof() )
            {
                stmt = d_db->compileStatement( "insert into metadata(fileid, field, type, value) values(?,?,?,?);" );
                stmt.bind(  1, qm.getIntField( "fileid" ) );
                stmt.bind(  2, qm.getStringField( "field" ) );
                stmt.bind(  3, qm.getStringField( "type" ) );
                stmt.bind(  4, qm.getStringField( "value" ) );

                stmt.execDML();
                stmt.finalize();

                qm.nextRow();
            }

            // export the fulltext
            DML = "select * from fulltexts where fileid = '" + QString::number( fileId ) + "';";
            CppSQLite3Query qft = m_db->execQuery( DML );
            if ( !qft.eof() )
            {
                int fullTextDataLength = qft.getIntField( "fulltextdatalength" );

                stmt = d_db->compileStatement( "insert into fulltexts(fileid, fulltextdata, fulltextdatalength) values(?,?,?);" );
                stmt.bind(  1, qft.getIntField( "fileid" ) );
                stmt.bind(  2, qft.getBlobField( "fulltextdata", fullTextDataLength ), fullTextDataLength );
                stmt.bind(  3, fullTextDataLength );

                stmt.execDML();
                stmt.finalize();
            }

            q.nextRow();
        }
        q.finalize();

        // export the wordfile records
        DML = "select wordfile.* from wordfile, files on wordfile.fileid = files.fileid "
                    "where files.catalogid = '" + QString::number( cat->catalogId() ) + "';";
        CppSQLite3Query qwf = m_db->execQuery( DML );
        while ( !qwf.eof() )
        {
            stmt = d_db->compileStatement( "insert into wordfile(fileid, wordid, occurrences) values (?,?,?);" );
            stmt.bind(  1, qwf.getIntField( "fileid" ) );
            stmt.bind(  2, qwf.getIntField( "wordid" ) );
            stmt.bind(  3, qwf.getIntField( "occurrences" ) );

            stmt.execDML();
            stmt.finalize();

            qwf.nextRow();
        }
        qwf.finalize();

        // export the words records
        DML = "select distinct words.* from wordfile, files, words on wordfile.fileid = files.fileid and wordfile.wordid = words.wordid "
                    "where files.catalogid = '" + QString::number( cat->catalogId() ) + "';";
        CppSQLite3Query qw = m_db->execQuery( DML );
        while ( !qw.eof() )
        {
            stmt = d_db->compileStatement( "insert into words(word, wordid) values (?,?);" );
            stmt.bind(  1, qw.getStringField( "word" ) );
            stmt.bind(  2, qw.getIntField( "wordid" ) );

            stmt.execDML();
            stmt.finalize();

            qw.nextRow();
        }
        qw.finalize();


        d_db->execDML( "commit transaction;" );
        closeDatabase( d_db );
    }
    catch ( CppSQLite3Exception& e )
    {
        d_db->execDML( "rollback transaction;" );
        closeDatabase( d_db );

        kdDebug() << "exportCatalog catching exception : " << e.errorMessage() << endl;
        return -1;
    }

    return 0;
}

int KatEngine::getWordId( QString& word )
{
    QString DML;
    int wordId;

    // TODO: Find a regular expression to eliminate all possible problems with words in SQL
    QString sword( word );
    sword = sword.replace( QRegExp( "'" ), "''" );

    try
    {
        DML = "select wordid from words where word = '" + sword + "';";
        CppSQLite3Query q = m_db->execQuery( DML );
        if ( q.eof() )
        {
            // word absent
            wordId = -1;
        }
        else
        {
            // word already present
            wordId = q.getIntField( "wordid" );
        }
        q.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << "Exception : int KatEngine::getWordId( QString& word ) :"<<e.errorMessage() << endl;
    }

    return wordId;
}

// NOTE - THIS WILL ONLY WORK FOR CATALOGS USED BY DAEMON -- NEED TO FIX THIS
// TEMPORARY UNTIL A MORE PERMANENT SOL. IS THOUGHT
void KatEngine::pruneCatalog( KatCatalog* cat )
{
    QString DML = "select fileid,fullname from files where catalogid=" + QString::number ( cat->catalogId() ) +  " limit 200 ";
    int iter = 0;
    QString fullName;
    int fileId;
    KIO::filesize_t fileSize = 0;
    QString fileType;
    QValueList<int> fileList;

    openTransaction();

    while ( true ) {
        fileList.clear();
        while (fileList.count() < 200 ) {
            try {
                CppSQLite3Query q = m_db->execQuery( DML + "offset " + QString::number( iter * 200 ) + ";" );
                iter ++;

                if ( q.eof() ) {
                    q.finalize();
                    break;
                }

                while ( !q.eof() ) {
                    fullName = q.getStringField( "fullname" );
                    QFileInfo fi( fullName );
                    if ( !fi.exists() )
                        fileList.append( q.getIntField( "fileid" ) );
                    q.nextRow();
                }
                q.finalize();
            } catch ( CppSQLite3Exception& e ) {
                kdDebug() << e.errorMessage() << endl;
                break;
            }
        }

        if (fileList.empty())
            break;

        QValueList<int>::Iterator it;
        QValueList<int>::Iterator end( fileList.end() );
        for (it = fileList.begin(); it != end; ++it)
        {
            fileId = *it;

            try
            {
                CppSQLite3Query q = m_db->execQuery ("select filetype,filesize from files where fileid='" + QString::number(fileId) + "';" );

                if (!q.eof()) {
                    fileType = q.getStringField( "filetype" );
                    fileSize = q.getIntField( "filesize" );
                }
                q.finalize();
            }
            catch ( CppSQLite3Exception& e )
            {
                kdDebug() << e.errorMessage() << " : " << *it << endl;
            }

            // Delete the current file from the database
            try
            {
                m_db->execDML( "delete from files where fileid = " + QString::number(fileId) + ";" );

                if (fileType == "inode/directory")
                    cat->setFolders (cat->folders() - 1);
                else
                {
                    cat->setFiles (cat->files() - 1);
                    cat->setFileSize (cat->fileSize() - fileSize);
                }

                if (m_db->execDML( "delete from fulltexts where fileid = " + QString::number(fileId) + ";" ))
                    cat->setFullTexts(cat->fullTexts() - 1);

                if (m_db->execDML( "delete from thumbnails where fileid = " + QString::number(fileId) + ";" ))
                    cat->setThumbnails(cat->thumbnails() - 1);

                if (m_db->execDML( "delete from metadata where fileid = " + QString::number(fileId) + ";" ))
                    cat->setMetaData(cat->metaData() - 1);

                //FIXME -- the corresponding words must be deleted from the 'words' table
                //NOTE - It might be better not to delete from the words file since they may crop up again
                //      when new files are added to the database
                if (m_db->execDML( "delete from wordfile where fileid = " + QString::number(fileId) + ";" ))
                    cat->setWords( cat->words() - 1 );

            }
            catch ( CppSQLite3Exception& e )
            {
                kdDebug() << e.errorMessage() << " : " << fileId << endl;
            }
        }
    }

    // delete the WORDS which do not have a record in WORDFILE
    m_db->execDML( "delete from words where wordid not in (select distinct wordid from wordfile);" );

    commitTransaction();

    updateCatalog ( cat );
}

void KatEngine::deleteFiles( KatCatalog* cat, QStringList files )
{
    int fileId;
    KIO::filesize_t fileSize = 0;
    QString fileType;

    openTransaction();

    while (!files.empty())
    {
        fileId = 0;

        // Get the first file to act upon
        QString file = files[0];
        files.pop_front();
        try
        {
            CppSQLite3Query q = m_db->execQuery ("select filetype,filesize,fileid from files where fullname='" + file + "';" );

            if (!q.eof()) {
                fileId = q.getIntField( "fileid" );
                fileType = q.getStringField( "filetype" );
                fileSize = q.getIntField( "filesize" );
            }
            q.finalize();
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << e.errorMessage() << " : " << file << endl;
        }

        if (!fileId)
            continue;

        try
        {
            // Now find all the files/folders whose parentid = fileid
            CppSQLite3Query q = m_db->execQuery ("select fullname from files where parentid='" + QString::number(fileId) + "';" );

            while (!q.eof())
            {
                files.append(q.getStringField("fullname"));
                q.nextRow();
            }
            q.finalize();
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << e.errorMessage() << endl;
        }

        // Delete the current file from the database
        try
        {
            m_db->execDML( "delete from files where fileid = " + QString::number(fileId) + ";" );

            if (fileType == "inode/directory")
                cat->setFolders (cat->folders() - 1);
            else
            {
                cat->setFiles (cat->files() - 1);
                cat->setFileSize (cat->fileSize() - fileSize);
            }

            if (m_db->execDML( "delete from fulltexts where fileid = " + QString::number(fileId) + ";" ))
                cat->setFullTexts(cat->fullTexts() - 1);

            if (m_db->execDML( "delete from thumbnails where fileid = " + QString::number(fileId) + ";" ))
                cat->setThumbnails(cat->thumbnails() - 1);

            if (m_db->execDML( "delete from metadata where fileid = " + QString::number(fileId) + ";" ))
                cat->setMetaData(cat->metaData() - 1);

                //FIXME -- the corresponding words must be deleted from the 'words' table
                //NOTE - It might be better not to delete from the words file since they may crop up again
                //      when new files are added to the database
            if (m_db->execDML( "delete from wordfile where fileid = " + QString::number(fileId) + ";" ))
                cat->setWords( cat->words() - 1 );
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << e.errorMessage() << " : " << fileId << endl;
        }
    }

    // delete the WORDS which do not have a record in WORDFILE
    m_db->execDML( "delete from words where wordid not in (select distinct wordid from wordfile);" );

    commitTransaction();

    updateCatalog ( cat );
}

