/***************************************************************************
                          karchiveurpart.cpp  -  karchiveur component extension
                             -------------------
    begin                : Tue Aug 15 2000
    copyright            : (C) 2001 by Eric Coquelle
    email                : coquelle@caramail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include "karchiveurpart.h"
#include "cgpg.h"

#include <klocale.h>
#include <kinstance.h>
#include <kaction.h>
#include <kaboutdata.h>
#include <kparts/partmanager.h>
#include <kdebug.h>
#include <kfileitem.h>


static const char *description = I18N_NOOP("KArchiver for KDE KPart\nAn easy tool to manage all your compressed files!\nReleased under GPL license");

extern "C"
{
    /**
     * This function is the 'main' function of this part.  It takes
     * the form 'void *init_lib<library name>()'.  It always returns a
     * new factory object
     */
    void *init_libkarchiver()
    {
        return new KarchiverFactory;
    }
}

KInstance *KarchiverFactory::s_instance = 0L;

KarchiverFactory::KarchiverFactory()
{
    KGlobal::locale()->insertCatalogue( QString::fromLatin1("karchiver") );
}

KarchiverFactory::~KarchiverFactory()
{
    if (s_instance)
    {
        delete s_instance->aboutData();
        delete s_instance;
    }

    s_instance = 0;
}

QObject *KarchiverFactory::createObject(QObject *parent, const char *name, const char*,
                               const QStringList& )
{
    QObject *obj = new KarchiverPart((QWidget*)parent, name);
    emit objectCreated(obj);
    return obj;
}

/**
 * We need one static instance of the factory for our C 'main'
 * function
 */
 KInstance *KarchiverFactory::instance()
{
    if ( !s_instance )
        s_instance = new KInstance( aboutData() );
    return s_instance;
}

KAboutData *KarchiverFactory::aboutData()
{
  KAboutData *about = new KAboutData("karchiver", "KArchiver", VERSION,
				     description, KAboutData::License_GPL,
				     "(c) 2000-2006, Eric Coquelle");

  return about;
}



/* Class KarchiverPart */
KarchiverPart::KarchiverPart(QWidget *parent, const char *name)
    : KParts::ReadOnlyPart(parent, name)
{
  QString directory;
  setInstance(KarchiverFactory::instance());

  archivecontent = new  KarchiverPartListView (parent, "main_widget");
  archivecontent->setFocusPolicy(QWidget::ClickFocus);
  setWidget(archivecontent);
  
  connect(archivecontent, SIGNAL(activatedisplayfiles( bool )), this, SLOT(slotEnableView( bool )) );
  
  archiveExtract = new KAction(i18n("&Extract"), "extract", 0, this, SLOT(slotExtractArchive()), actionCollection(), "extract");
  archiveView = new KAction(i18n("&View"), "exec", 0, this, SLOT(slotDisplayFiles()), actionCollection(), "view");

  m_extension = new KarchiverBrowserExtension(this);
  setXMLFile("karchiver_partui.rc");

  directory.sprintf("karchivertmp.%d/", getpid());
  tmpdir = locateLocal( "tmp", directory );

  if (tmpdir.isNull())
    kdWarning() <<i18n("Failed to create temporary "
            "attachment directory %1")
            .arg(directory) << endl;

  directory.sprintf("karchivertmp.%d/extract/", getpid());
  tmpdir = locateLocal( "tmp", directory );

  if (tmpdir.isNull())
    kdWarning() <<i18n("Failed to create temporary "
            "attachment directory %1")
            .arg(directory) << endl;
  
  m_statusbar=new KArchiverStatusBarExtension(this);
  connect(this, SIGNAL(displayStatusBarText(QString, QColor)), m_statusbar, SLOT(slotDisplayMessages(QString, QColor)));
  setProgressInfoEnabled( false );
  
  connect(archivecontent,SIGNAL(clicked ( QListViewItem * )), this, SLOT(slotMouseClick ( QListViewItem * )));
  connect(m_statusbar->cbdirectories, SIGNAL(activated(int)), this, SLOT(slotComboDirectoriesActivated(int)));

  
  viewbydirectories=true;
  CArchive::setWidgetListView(archivecontent);
  CArchive::setWidgetComboDirectories(m_statusbar->cbdirectories);
  CArchive::setDisplayIcons(true);
  CArchive::setKindOfDate(LOCALISED_DATE);
  CArchive::setWidgetProgressBar(m_statusbar->progress);
  CArchive::setOverwriteFilesWhenExtracting(false);
  CArchive::setIconSize(16);
  CArchive::setReadArchiveWithStream(false);
  
  archiveobj=NULL;
  archiveoperation=NULL;


  enableMenus(false);
}

KarchiverPart::~KarchiverPart()
{
  QString directory;
  KProcess process;
  directory.sprintf("karchivertmp.%d/", getpid());
  tmpdir = locateLocal( "tmp", directory );

  process.clearArguments();
  process << "rm" << "-Rf" <<tmpdir;
  process.start(KProcess::Block);
  if(archiveobj!=NULL)
    delete archiveobj;
  delete m_statusbar;
}

/** User wants to go to a subfolder */
void KarchiverPart::slotComboDirectoriesActivated(int index){
  QString path;
  int i;

  if(!archiveobj->supportDisplayByDirs())
    return;

  for(i=0;i<=index;i++)
  {
    path+=m_statusbar->cbdirectories->text(i);
    if(!path.endsWith("/"))
      path+="/";
  }
  path=path.left(path.length()-1);
  archiveobj->displayArchiveContent(m_statusbar->cbdirectories->text(index),path);
}

void KarchiverPart::slotMouseClick ( QListViewItem * li)
{
  bool controlkeypressed=false;
  if( li!=0 )
  {
    if(archiveobj->supportDisplayByDirs() && viewbydirectories && archiveobj->isDirectory(li->text(0)) && (!controlkeypressed))
      archiveobj->displayArchiveContent(li->text(0), li->text(6));//Display content of selected directory
  }
}

bool KarchiverPart::openFile()
{
    return openArchive(m_file);
}

void KarchiverPart::slotEnableView(bool enable)
{
  archiveView->setEnabled( enable );
}

/**enable/disable the toolbar entries */
void KarchiverPart::enableMenus(bool b)
{
    archiveExtract->setEnabled(b);
    archiveView->setEnabled(false); //Activated only if at least one file is selected, cf. slotEnableView
}

/** Prepare to display archive content */
bool KarchiverPart::openArchive(const QString & archivename )
{
kdDebug()<<QString("KarchiverPartListView::openArchive %1\n").arg(archivename);

  QString errormessage;
  //led->setColor(QColor("red"));
  kdDebug()<<QString("KarchiverPartListView::displayArchiveConten of %1").arg(archivename)<<endl;
  if(archiveoperation!=NULL)
    delete archiveoperation;
  archiveoperation = new CArchiveOperationDisplay( archivename, false, viewbydirectories, archiveobj,  m_statusbar->progress);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  
  errormessage=archiveoperation->canDisplayArchive();
  if(!errormessage.isEmpty())
  {
    kdDebug()<<QString("KarchiverPart::openArchive - Error:%1").arg(errormessage)<<endl;
    emit displayStatusBarText(errormessage, QColor("orange"));
    return false;
  }

  emit displayStatusBarText(i18n("Opening archive..."), QColor("red"));
  enableMenus(false);
  archiveoperation->displayArchiveContent();
  return true;
}

/** This slot will extract the archive content */
void KarchiverPart::slotExtractArchive()
{
  enableMenus(false);
  emit displayStatusBarText(i18n("Extract files..."),QColor("red"));
  archiveoperation=new CArchiveOperationExtract(archiveobj, m_statusbar->progress);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  
  archiveoperation->extractCurrentArchive();
}

/** This slot will extract selected file and view it with its associated software */
void KarchiverPart::slotDisplayFiles()
{
  emit displayStatusBarText(i18n("View all selected files..."),QColor("red"));
  enableMenus(false);
  archiveoperation=new CArchiveOperationViewFiles(archiveobj, m_statusbar->progress, tmpdir);
  archiveoperation->displaySelectedFiles();
}

/** Messages from CArchiveOperation and childs */
void KarchiverPart::slotMessageArchiveOperation(int message, QString messagetext)
{
  kdDebug()<<QString("KarchiveurApp::slotMessageArchiveOperation %1 %2").arg(message).arg(messagetext)<<endl;
  switch(message)
  {
    case DISPLAY_ACHIEVED:
    {
      archiveobj=archiveoperation->getArchiveObject();//Fetch back the archive object, as a new one has been created in CArchiveOperation...
      if(archiveobj->supportDisplayByDirs())
      {
        m_statusbar->cbdirectories->show();
      }
      else
        m_statusbar->cbdirectories->hide();
      emit displayStatusBarText(messagetext, QColor("green"));
      break;
    }
    case EXTRACT_ACHIEVED:
    {
      emit displayStatusBarText(messagetext, QColor("green"));
      break;
    }
    case VIEW_FILES_ACHIEVED:
    {
      emit displayStatusBarText(messagetext, QColor("green"));
      break;
    }
    default:
    case CANNOT_PERFORM_OPERATION:
    {
      emit displayStatusBarText(messagetext, QColor("orange"));
      break;
    }
  }
  delete archiveoperation;
  m_statusbar->progress->reset();
  archiveoperation=NULL;
  if(archiveobj!=NULL)
    enableMenus(true);
  else
    enableMenus(false);
}



/* ClassKarchiverPartListView*/
KarchiverPartListView::KarchiverPartListView( QWidget *parent, const char *name ) : KListView(parent, name)
{
  createFileListView();
  
  setIconText("Karchiver Kparts");
}

KarchiverPartListView::~KarchiverPartListView()
{
}

void KarchiverPartListView::prepareDisplayOfArchive(QString& archivename)
{
  setCaption(archivename);
  clear();
  for(int i=0;i<7;i++)
    setColumnWidth(i,50);
}

/** The selection changed. Check if at least one item is selected, to
  * enable/disable the view action */
void KarchiverPartListView::slotSelectionChanged()
{
  nbselitems=0;
  QListViewItem *it = firstChild();
  while (it)
	{
	  if (it->isSelected())
	      nbselitems++;
	  it = it->itemBelow();
	}
	emit( activatedisplayfiles((nbselitems==0) ? false : true));
}

void KarchiverPartListView::createFileListView()
{
	resize(200,150);
	addColumn(i18n("Name"));
	addColumn(i18n("Size"));
	addColumn(i18n("Time"));
	addColumn(i18n("Date"));
	addColumn(i18n("Owner"));
	addColumn(i18n("Permissions"));
	addColumn(i18n("Path"));
	setMultiSelection(TRUE);
	setColumnWidth(6,100);
	setColumnAlignment(1,Qt::AlignRight);
	setColumnAlignment(2,Qt::AlignRight);
	setColumnAlignment(3,Qt::AlignRight);
	setColumnAlignment(4,Qt::AlignRight);
	setColumnAlignment(5,Qt::AlignRight);
	setShowSortIndicator(true);
        setSorting(6);
	show();

	connect(this, SIGNAL(selectionChanged()), this, SLOT(slotSelectionChanged()));
}

void KarchiverPartListView::resizeEvent ( QResizeEvent * newSize) {
	resize( newSize->size() );
}



/** ClassKarchiverBrowserExtension*/
KarchiverBrowserExtension::KarchiverBrowserExtension(KarchiverPart *parent)
    : KParts::BrowserExtension(parent, "KarchiverBrowserExtension")
{
}

KarchiverBrowserExtension::~KarchiverBrowserExtension()
{
}



KArchiverStatusBarExtension::KArchiverStatusBarExtension( KParts::ReadOnlyPart * parent ) : KParts::StatusBarExtension( parent )
{
  timer=new QTimer();
  connect(timer, SIGNAL(timeout ()), this, SLOT(slotDisplayDefaultStatusBarMessage()));
    
  //Additionnal init to the statusbar
  statustext = new QLabel(statusBar());
  statustext->setAlignment( Qt::AlignLeft );
  statustext->setFrameStyle( QFrame::NoFrame );
  statustext->setFrameStyle( QFrame::Panel | QFrame::Raised );
  statustext->setText(i18n(IDS_STATUS_DEFAULT));
  addStatusBarItem( statustext, 1, true );
  
  cbdirectories=new KComboBox(statusBar(),"directory_combo");
  cbdirectories->hide();
  addStatusBarItem( cbdirectories, 1, true );

  progress = new QProgressBar(statusBar(),"Progress_Bar");
  progress->setFixedHeight ( statustext->fontMetrics().height() -4 );
  addStatusBarItem( progress, 3, true );

  led=new CLed(statusBar(),"statusbar_led");
  led->setLook(KLed::Sunken);
  led->setFixedHeight ( statustext->fontMetrics().height()-4 );
  led->setFixedWidth ( statustext->fontMetrics().height()-4 );
  addStatusBarItem( led, 1, true );
}

KArchiverStatusBarExtension::~KArchiverStatusBarExtension()
{
}

void KArchiverStatusBarExtension::slotDisplayMessages(QString statusmessage, QColor ledcolor)
{
  if(ledcolor==QColor("orange"))
    timer->start(3000, true);
  statustext->setText(statusmessage);
  led->setColor(ledcolor);
}

void KArchiverStatusBarExtension::slotDisplayDefaultStatusBarMessage()
{
  statustext->setText(i18n(IDS_STATUS_DEFAULT));
}

#include "karchiveurpart.moc"
