/*
    collectionbuilder.h - Build a collection of ROMs

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#ifndef KAMEFUCOLLECTIONBUILDER_H
#define KAMEFUCOLLECTIONBUILDER_H

#include <qobject.h>
#include <kurl.h> // Forward declaration doesn't work for KURL::List
#include <kdemacros.h>
#include "rommetainformation.h" // Forward declaration of container typedef is a pain.
#include <kfileitem.h> // idem as above.
class KFileItem;

namespace Kamefu 
{

/**
Collection Builder is the service that list a list of directories defined in a configuration file. After retriving the metadata, it adds the info to the database.

During listing of all the files, if a file is a zip file, the collection builder uncompress the zip file to see if the zip file contains a ROM image. Using the temp file (unzipped), the Collection Builder will be able to retrive metadata from the file.

For each file, Collection Builder should use a class obtained from RomMetaDataFactory. If no specific class is required, the RomMetaDataFactory will return the DefaultRomMetaData that read the metadata from KFileMetaInfo. This will be useful for exemple for the MAME system to use MAME database to retrive meta information.

When it has retrived the meta information, it push the meta information to database using Collection Access service.

@author Michaël Larouche
*/
class KDE_EXPORT CollectionBuilder : public QObject
{
	Q_OBJECT
public:
	CollectionBuilder(QObject *parent = 0, const char *name = 0);
	
	~CollectionBuilder();

	/**
	 * Set the directories to scan when building 
	 * Note that it erase the old list.
	 */
	void setDirectoryList(const KURL::List &directoryList);

signals:
	/**
	 * A single ROM file was listed.
	 */ 
	void singleRomListed(const Kamefu::RomMetaInformation &romMetaInfo);
	/**
	 * Listing of all directories is finished.
	 */
	void collectionBuildFinished();
	
	/**
	 * A directory begin to be listed, tell the numbers of file.
	 */
	void beginListDirectory(int numberItems);
	/**
	 * A signel item was listed.
	 */
	void itemListed();

public slots:
	/**
	 * Start listing the directories to build the collectionbuilder
	 * @param recursive Recurse into sub-directies.
	 */
	void startBuild(bool recursive = true);

	/**
	 * Cancel the build of the database.
	 */
	void cancelBuild();

private slots:
	/**
	 * Called when KDirLister found new files.
	 * Now it's just do the recursive magic.
	 */
	void slotNewFiles(const KFileItemList& list);
	/**
	 * @brief Insert items into database.
	 * Insert the file list into the database. It expect filtered items.
	 * Called when the listing of KDirLister is finished.
	 */
	void slotInsertItems(const KFileItemList &list);
	/**
	 * Called when the build is cancelled.
	 */
	void slotBuildCanceled();
	/**
	 * Called when KDirLister has finished all its job.
	 */
	void slotListingComplete();
	/**
	 * Called when KDirLister has finished to list a directory.
	 */
	void slotDirectoryFinished();
private:
	class Private;
	Private *d;
};

}

#endif
