/*
    kfile_gba.cpp - GBA(Game Boy Advance) Rom image KFile Plugin

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

     *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "kfile_gba.h"

#include <kgenericfactory.h>
#include <kdebug.h>
#include <kmdcodec.h>

#include <qfile.h>
#include <qdatastream.h>

#include <nintendomakerlist.h>

typedef KGenericFactory<KGbaFilePlugin> kgbafileFactory;

K_EXPORT_COMPONENT_FACTORY(kfile_gba, kgbafileFactory( "kfile_gba" ))

KGbaFilePlugin::KGbaFilePlugin(QObject *parent, const char *name, const QStringList& args)
	: KFilePlugin(parent, name, args)
{
	//add the mimetype here:
	KFileMimeTypeInfo *info = addMimeTypeInfo( "application/x-rom-gba" );

	KFileMimeTypeInfo::GroupInfo *groupGeneral = 0L;
	groupGeneral = addGroupInfo(info, "romGeneralInfo", i18n("General"));
	
	// General Group
	addItemInfo(groupGeneral, "internalName", i18n("Internal Name"), QVariant::String);
	addItemInfo(groupGeneral, "country", i18n("Country"), QVariant::String);
	addItemInfo(groupGeneral, "editor", i18n("Developer"), QVariant::String);
	addItemInfo(groupGeneral, "md5Hash", i18n("MD5 Hash"), QVariant::String);
	
	// GBA specific group
	KFileMimeTypeInfo::GroupInfo *groupGBA = 0L;
	groupGBA = addGroupInfo(info, "gbaInfo", i18n("GBA Info"));
	addItemInfo(groupGBA, "gbaGameCode", i18n("Game Code"), QVariant::String);
}

bool KGbaFilePlugin::readInfo( KFileMetaInfo& info, uint /*what*/)
{
	KFileMetaInfoGroup groupGeneral = appendGroup(info, "romGeneralInfo");
	KFileMetaInfoGroup groupGBA = appendGroup(info, "gbaInfo");

	QByteArray internalName(12), gameCode(4), makerCode(2);
	QString md5Hash, countryName(i18n("Unknown")), editorName(i18n("Unknown"));

	QFile gba_file(info.path());

	if(gba_file.open(IO_ReadOnly))
	{
		QDataStream stream(&gba_file);
		stream.setByteOrder(QDataStream::LittleEndian);
		
		// Skip the JUMP ARM command and nintendo logo
		gba_file.at(0xA0);

		// Read the game title (internalName)
		stream.readRawBytes(internalName.data(), internalName.size());
		// Read the game code
		stream.readRawBytes(gameCode.data(), gameCode.size());
		// Read the maker code
		stream.readRawBytes(makerCode.data(), makerCode.size());

		gba_file.reset(); // Make sure that the md5 hash read the entire ROM.

		// Generate the MD5 Hash
		KMD5 context(0L);
		context.update(gba_file);
		md5Hash = context.hexDigest();
	
		// Get the information
		// Get the editor name from the maker code
		kdDebug() << k_funcinfo << "Read makerCode: " << QString(makerCode) << endl;

		int result=0;
		bool ok;
		QString sMaker(makerCode);
		result = sMaker.toInt(&ok, 16);

		kdDebug() << k_funcinfo << "Convert editorIndex: " << QString::number(result, 16) << endl;

		result = (result >> 4) * 36 + (result & 0x0f);
		if(result > 0 || result <= nintendoMakerListLength)
			editorName = nintendoMakerList[result];

		kdDebug() << k_funcinfo << "Editor name: " << editorName << endl;

		// Try to guess the country from the gamecode
		if(gameCode[3] == 'J')
		{
			countryName = i18n("Japan");
		}
		else if(gameCode[3] == 'P')
		{
			countryName = i18n("Europe");
		}
		else if(gameCode[3] == 'E')
		{
			countryName = i18n("America");
		}

		gba_file.close();
	}

	// General group
	appendItem(groupGeneral, "internalName", QString(internalName));
	appendItem(groupGeneral, "country", countryName);
	appendItem(groupGeneral, "editor", editorName);
	appendItem(groupGeneral, "md5Hash", md5Hash);
	
	// GBA group
	appendItem(groupGBA, "gbaGameCode", QString(gameCode));

	return true;
}

#include "kfile_gba.moc"
