/*
    kcheckdirectoryselector.cpp - Show a treeview of all directories with a checkbox.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    Based on code in amarok/directorylist.[h/cpp] by
    (C) 2003 Scott Wheeler <wheeler@kde.org>
    (C) 2004 Max Howell <max.howell@methylblue.com>
    (C) 2004 Mark Kretschmann <markey@web.de>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "kcheckdirectoryselector.h"

#include <qlabel.h>
#include <qtooltip.h>
#include <qheader.h>

#include <klocale.h>
#include <kurl.h>
#include <kdirlister.h>
#include <kdebug.h>

namespace Kamefu {

namespace UI {

// Class KCheckDirectorySelector
class KCheckDirectorySelector::Private
{
public:
	Private() : recursive(false)
	{}
	QStringList dirs;
	bool recursive;
};

KCheckDirectorySelector::KCheckDirectorySelector(QWidget *parent, const char *name, bool recursive)
 : QListView(parent, name), d(new Private())
{
	kdDebug() << k_funcinfo << endl;
	d->recursive = recursive;

	addColumn( QString::null );
	setRootIsDecorated( true );
	setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );
	setResizeMode( QListView::LastColumn );

	header()->hide();
	
	new KCheckDirectorySelectorItem(this);
}

KCheckDirectorySelector::~KCheckDirectorySelector()
{
	delete d;
}

QStringList& KCheckDirectorySelector::dirs()
{
	return d->dirs;	
}

void KCheckDirectorySelector::setDirs(const QStringList &dirs)
{
	d->dirs = dirs;
}

void KCheckDirectorySelector::addDirectory(const QString &directory)
{
	d->dirs.append(directory);
}

void KCheckDirectorySelector::removeDirectory(const QStringList::Iterator &directory)
{
	d->dirs.erase(directory);
}

bool KCheckDirectorySelector::recursive() const
{
	return d->recursive;
}

void KCheckDirectorySelector::setRecursive(bool recursive)
{
	d->recursive = recursive;
}

// Class KCheckDirectorySelectorItem
class KCheckDirectorySelectorItem::Private
{
public:
	Private(KCheckDirectorySelector *theParent) 
		: lister(true), 
		  url("file:/"), 
		  listed(false),
		  parent(0L)
	{
		parent = theParent;
		lister.setDirOnlyMode(true);
	}

	KDirLister lister;
	KURL       url;
	bool       listed;
	KCheckDirectorySelector *parent;
};

KCheckDirectorySelectorItem::KCheckDirectorySelectorItem(KCheckDirectorySelector *parent)
    : QCheckListItem(parent, "/", QCheckListItem::CheckBox),
	  d(new Private(parent))
{
	connect( &d->lister, SIGNAL(newItems( const KFileItemList& )), SLOT(newItems( const KFileItemList& )) );
	setOpen( true );
	setVisible( true );
}


KCheckDirectorySelectorItem::KCheckDirectorySelectorItem( QListViewItem *parent, const KURL &url )
    : QCheckListItem( parent, url.fileName(), QCheckListItem::CheckBox  )
    , d( new Private( static_cast<KCheckDirectorySelector*>(parent->listView()) ) )
{
	d->url = url;
	setExpandable( true );
	connect( &d->lister, SIGNAL(newItems(const KFileItemList& )), SLOT(newItems(const KFileItemList& )) );
	connect( &d->lister, SIGNAL(completed()), SLOT(completed()) );
	connect( &d->lister, SIGNAL(canceled()), SLOT(completed()) );
}

KCheckDirectorySelectorItem::~KCheckDirectorySelectorItem()
{
	delete d;
}

QString KCheckDirectorySelectorItem::fullPath() const
{
	QString path;
	
	for( const QListViewItem *item = this; item != listView()->firstChild(); item = item->parent() )
	{
		path.prepend( item->text(0) );
		path.prepend('/');
	}
	
	kdDebug() << k_funcinfo << "Full path: " << path << endl;
	return path;
}

void KCheckDirectorySelectorItem::setOpen(bool b)
{
	if ( !d->listed )
	{
		d->lister.openURL(d->url, true);
		d->listed = true;
		kdDebug() << k_funcinfo << "Listing directory: " << d->url << endl;
	}
	
	QListViewItem::setOpen(b);
}

void KCheckDirectorySelectorItem::stateChange(bool b)
{
	if( d->parent->recursive() )
		for( QListViewItem *item = firstChild(); item; item = item->nextSibling() )
			static_cast<QCheckListItem*>(item)->QCheckListItem::setOn( b );
	
	// Update folder list
	QStringList::Iterator it = d->parent->dirs().find( d->url.path() );
	if ( isOn() ) 
	{
		kdDebug() << k_funcinfo << "Adding the directory \"" << d->url.path() << "\" to selected directories." << endl;
		if ( it == d->parent->dirs().end() )
			d->parent->addDirectory(d->url.path());
	
	}
	else
	{
		kdDebug() << k_funcinfo << "Removing the directory \"" << d->url.path() << "\" from selected directories." << endl;
		d->parent->removeDirectory(it);
	}
	
	// Redraw parent items
	listView()->triggerUpdate();
}

void KCheckDirectorySelectorItem::activate()
{
	kdDebug() << k_funcinfo << endl;
	if( !isDisabled() )
		QCheckListItem::activate();
}

bool KCheckDirectorySelectorItem::isDisabled() const
{
	return d->parent->recursive() && parent() && parent()->isOn();
}

void KCheckDirectorySelectorItem::newItems(const KFileItemList &list) //SLOT
{
	for( KFileItemListIterator it( list ); *it; ++it )
	{
		KCheckDirectorySelectorItem *item = new KCheckDirectorySelectorItem( this, (*it)->url() );
		item->setOn( d->parent->recursive() && isOn() || d->parent->dirs().contains( item->fullPath() ) );
		item->setPixmap( 0, (*it)->pixmap( KIcon::SizeSmall ) );
	}
}

void KCheckDirectorySelectorItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
	bool dirty = false;
	
	// Figure out if a child folder is activated
	for ( uint i = 0; i < d->parent->dirs().count(); i++ )
	{
		if ( d->parent->dirs()[i].startsWith( d->url.path() ) )
		{
			dirty = true;
		}
	}
	
	// Use a different color if this folder has an activated child folder
	QColorGroup _cg = cg;
	if ( dirty ) _cg.setColor( QColorGroup::Text, listView()->colorGroup().link() );
	
	QCheckListItem::paintCell( p, isDisabled() ? listView()->palette().disabled() : _cg, column, width, align );
}

}

}

#include "kcheckdirectoryselector.moc"
