/***************************************************************************
                          xineconfig.cpp - config dialog for xine parameters
                             -------------------
    begin                : Son Mai 4 2003
    revision             : $Revision: 1.20 $
    last modified        : $Date: 2004/04/24 08:58:22 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by J. Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <kdebug.h>
#include <kseparator.h>

#include <qlabel.h>
#include <qscrollview.h>

#include "xineconfig.h"
#include "xineconfig.moc"


XineConfigEntry::XineConfigEntry(QWidget* parent, QGridLayout* grid, int row, xine_cfg_entry_t* entry)
{

  key = QString(entry->key);
  valueChanged = false;
  numValue = entry->num_value;
  stringValue = entry->str_value;

  

  switch (entry->type)
  {
    case XINE_CONFIG_TYPE_UNKNOWN: break;
    case XINE_CONFIG_TYPE_STRING:
    {
      stringEdit = new KLineEdit(entry->str_value, parent);
      grid->addWidget(stringEdit, row ,0);
      connect(stringEdit, SIGNAL(textChanged(const QString&)), this, SLOT(slotStringChanged(const QString&)));
      break;
    }
    case XINE_CONFIG_TYPE_ENUM:
    {
      enumEdit = new KComboBox(parent);
      int i = 0;
      while (entry->enum_values[i])
      {
        enumEdit->insertItem(entry->enum_values[i]);
        i++;
      }
      enumEdit->setCurrentItem(entry->num_value);
      grid->addWidget(enumEdit, row, 0);
      connect(enumEdit, SIGNAL(activated(int)), this, SLOT(slotNumChanged(int)));
      break;
    }
    case XINE_CONFIG_TYPE_NUM:
    {
      numEdit = new QSpinBox(-999999, 999999, 1, parent);
      numEdit->setValue(entry->num_value);
      grid->addWidget(numEdit, row, 0);
      connect(numEdit, SIGNAL(valueChanged(int)), this, SLOT(slotNumChanged(int)));
      break;
    }
    case XINE_CONFIG_TYPE_RANGE:
    {
      numEdit = new QSpinBox(parent);
      numEdit->setValue(entry->num_value);
      numEdit->setRange(entry->range_min, entry->range_max);
      grid->addWidget(numEdit, row, 0);
      connect(numEdit, SIGNAL(valueChanged(int)), this, SLOT(slotNumChanged(int)));
      break;
    }       
    case XINE_CONFIG_TYPE_BOOL:
    {
      boolEdit = new QCheckBox(parent);
      boolEdit->setChecked(entry->num_value);
      grid->addWidget(boolEdit, row, 0);
      connect(boolEdit, SIGNAL(toggled(bool)), this, SLOT(slotBoolChanged(bool)));
      break;
    }        
  }

  QString keyName(entry->key);
  keyName.remove( 0, keyName.find(".") + 1 );

  QLabel* description = new QLabel(keyName + "\n" + QString::fromLocal8Bit(entry->description), parent);
  description->setAlignment( QLabel::WordBreak | QLabel::AlignVCenter );
  grid->addWidget(description, row, 1);

  KSeparator* separator = new KSeparator(KSeparator::Horizontal, parent);
  grid->addMultiCellWidget(separator, row+1, row+1, 0, 1);
}


XineConfigEntry::~XineConfigEntry() {}


void XineConfigEntry::slotNumChanged(int val)
{
   numValue = val;
   valueChanged = true;
}


void XineConfigEntry::slotBoolChanged(bool val) 
{
   numValue = val;
   valueChanged = true;
}


void XineConfigEntry::slotStringChanged(const QString& val)
{
   stringValue = val;
   valueChanged = true;
}


bool XineConfigEntry::ValueChanged() const
{
  return valueChanged;
}

  
const QString& XineConfigEntry::GetKey() const
{
  return key;
}

  
int XineConfigEntry::GetNumValue() const
{
  return numValue;
}

  
const QString& XineConfigEntry::GetStringValue() const
{
  return stringValue;
}  




/************** XINE CONFIG CLASS **************************/

XineConfig::XineConfig(const xine_t* const xine) :
       KDialogBase(KDialogBase::Tabbed, i18n("xine Engine Parameters"), KDialogBase::Ok|KDialogBase::Apply|KDialogBase::Cancel, KDialogBase::Cancel)

{
  setInitialSize(QSize(650,500), true);

  xineEngine = (xine_t*)xine;
 
  xine_cfg_entry_t* ent;
  int row = 0;

  QString currentPage("");
  
  QString pageName;
  
  QScrollView* sv = NULL;
  QWidget* parent = NULL;
  QGridLayout* grid = NULL;


/*********** read in xine config entries ***********/    
  ent = new xine_cfg_entry_t; 
  xine_config_get_first_entry(xineEngine, ent);

  do
  {
   if (ent->exp_level <= 20)
    {
    
      pageName = QString(ent->key);
      pageName = pageName.left(pageName.find("."));

     // if (pageName == "decoder") continue;
      
      if (pageName == currentPage)
      {
        row += 2;  
      }
      else
      {
        parent = addPage(pageName);
        parent->setFixedSize(620,400);       
        sv = new QScrollView(parent);
        sv->setResizePolicy(QScrollView::AutoOneFit);
        sv->setFixedSize(620,400);
        parent = new QWidget(sv->viewport());
        sv->addChild(parent);
       
        grid = new QGridLayout(parent, 20 ,2);
        grid->setColStretch(1,8);
        grid->setSpacing(10);
        grid->setMargin(10);
       
        currentPage = pageName;
        row = 0;   
      }
      

    entrys.append(new XineConfigEntry(parent, grid, row, ent));
    delete ent;
    ent = new xine_cfg_entry_t;   
   }   
                         
  } while(xine_config_get_next_entry(xineEngine, ent));

 delete ent;
/********************/ 

 connect(this, SIGNAL(okClicked()), SLOT(slotOkPressed()));
 connect(this, SIGNAL(applyClicked()), SLOT(slotApplyPressed()));
}


XineConfig::~XineConfig()

{
 entrys.setAutoDelete(true); 
 entrys.clear();
 kdDebug() << "xine config dialog destructed" << endl;
}


void XineConfig::slotOkPressed()
{
  slotApplyPressed();
  QDialog::close();
}


/*************** apply changed entries *****************/

void XineConfig::slotApplyPressed()
{
  xine_cfg_entry_t *entry;

  
  for(uint i = 0; i < entrys.count(); i++) /* check all entries */
  {
    if (entrys.at(i)->ValueChanged())  /* changed? */
    {
      entry = new xine_cfg_entry_t;
      if (xine_config_lookup_entry(xineEngine, entrys.at(i)->GetKey(), entry))
      {
        kdDebug(555) << "Apply: " << entrys.at(i)->GetKey() << "\n";
     
        entry->num_value = entrys.at(i)->GetNumValue();
        
        if (entrys.at(i)->GetStringValue()) 
          entry->str_value = (char*) (const char*)entrys.at(i)->GetStringValue().latin1(); 

        xine_config_update_entry(xineEngine, entry);            
        delete entry;

        entrys.at(i)->SetValueUnchanged();   
      }    
    }
  }    
}







