/***************************************************************************
                           videowindow.cpp 
                           -------------------
    begin                : Fre Apr 18 2003
    revision             : $Revision: 1.68 $
    last modified        : $Date: 2004/04/25 08:50:24 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by J. Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


 
#include <klocale.h>
#include <kurl.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kapplication.h>

#include <qdir.h>
#include <qcursor.h>
#include <qimage.h>

#include <xine/xineutils.h>

#include <X11/keysym.h>
#ifdef HAVE_XTEST
#include <X11/extensions/XTest.h>
#endif

#include <string.h>
#include <stdlib.h>

#include "convert.h"
#include "videowindow.h"
#include "videowindow.moc"


#define TIMER_EVENT_PLAYBACK_FINISHED 100
#define TIMER_EVENT_NEW_CHANNELS      101
#define TIMER_EVENT_NEW_TITLE         102
#define TIMER_EVENT_NEW_INFO          103
#define TIMER_EVENT_NEW_PROGRESS_INFO 104
#define TIMER_EVENT_CHANGE_CURSOR     105
#define TIMER_EVENT_GRAB_KEYBOARD     106
#define TIMER_EVENT_NEW_MRL_REFERENCE 107
#define TIMER_EVENT_NEW_XINE_MESSAGE  108
#define TIMER_EVENT_RESTART_PLAYBACK  200
#define TIMER_EVENT_RESIZE_PARENT     300


VideoWindow::VideoWindow(QWidget *parent, const char *name, const QString& prefAudio, const QString& prefVideo, bool verbose, bool startManual)
             : QWidget(parent,name), QThread()
{
  xineEngine = NULL; xineStream = NULL; audioDriver = NULL; videoDriver = NULL;
  eventQueue = NULL; xinePost = NULL;  xineRunning = false;
  xineDisplay = NULL; currentZoom = 100;
  startXineManual = startManual; filtersEnabled = true; visualPluginName = QString::null;
  deinterlaceFilter = NULL; deinterlacerDialog = NULL; deinterlaceEnabled = false;
  devicePath = QString::null;
  xineVerbose = verbose;
  autoresizeEnabled = false; videoFrameWidth = 0; videoFrameHeight = 0;
  
  preferedAudio = prefAudio;
  preferedVideo = prefVideo;

  setPaletteBackgroundColor(QColor(0,0,0));

  playMode = NORMAL_PLAY;

  filterList.setAutoDelete( true ); /*** delete post plugin on removing from list ***/

  connect(&posTimer, SIGNAL(timeout()), this, SLOT(slotGetPosition()));
  connect(&lengthInfoTimer, SIGNAL(timeout()), this, SLOT(slotEmitLengthInfo()));
  connect(&screensaverTimer, SIGNAL(timeout()), this, SLOT(slotFakeKeyEvent()));
  connect(&mouseHideTimer, SIGNAL(timeout()), this, SLOT(slotHideMouse()));

  setMouseTracking( true );
  setUpdatesEnabled( false ); //prevent qt overwrites videowin black
}


VideoWindow::~VideoWindow()
{
  /* "careful" shutdown, maybe xine initialization was not successful */

  xineRunning = false;

  /* stop all timers */
  posTimer.stop();
  screensaverTimer.stop();
  mouseHideTimer.stop();

  if (xineStream)
  {
    xine_close(xineStream);
  }

  if (running())
  {

    XEvent ev;
    ev.type = Expose;
    ev.xexpose.display = xineDisplay;
    ev.xexpose.window = winId();
    ev.xexpose.x = x();
    ev.xexpose.y = y();
    ev.xexpose.width = width();
    ev.xexpose.height = height();
    ev.xexpose.count = 0;

    XSendEvent( x11Display(), winId(), False, ExposureMask, &ev );  /* send a fake expose event */
    XFlush( x11Display() );

    if( !wait(1000) )  /* wait a second for thread exiting */
    {
       kdWarning(555) << "XEvent thread don't exit. Terminating it..." << endl;
       terminate();
    }
  }
       
  kdDebug(555) << "Shut down xine engine\n";

  if (xineDisplay)
     XLockDisplay( xineDisplay );

  if (xinePost)
  {
    kdDebug(555) << "Dispose visual plugin: " << visualPluginName << endl;
    postAudioSource = xine_get_audio_source (xineStream);
    xine_post_wire_audio_port (postAudioSource, audioDriver);
    xine_post_dispose (xineEngine, xinePost);
  }  
  if (eventQueue)
     xine_event_dispose_queue(eventQueue);
  if (xineStream)   
     xine_dispose(xineStream);
  if (audioDriver)   
     xine_close_audio_driver(xineEngine, audioDriver);
  if (videoDriver)   
     xine_close_video_driver(xineEngine, videoDriver);
  if (xineEngine)
     xine_exit(xineEngine);
  xineEngine = NULL;

  if (xineDisplay)
    XUnlockDisplay( xineDisplay ); 

  if (xineDisplay)
    XCloseDisplay(xineDisplay);  /* close xine display */
  xineDisplay = NULL;

  kdDebug(555) << "xine closed\n";
}


void VideoWindow::SaveXineConfig()
{
  if (!devicePath.isNull())
  {
    kdDebug(555) << "Set CD/VCD/DVD path back" << endl;
    xine_cfg_entry_t config;

    xine_config_lookup_entry (xineEngine, "input.cdda_device", &config);
    if ( devicePath == config.str_value )
    {
      config.str_value = (char*)cachedCDPath.latin1();
      xine_config_update_entry (xineEngine, &config);
    }  

    xine_config_lookup_entry (xineEngine, "input.vcd_device", &config);
    if ( devicePath == config.str_value )
    {
      config.str_value = (char*)cachedVCDPath.latin1();
      xine_config_update_entry (xineEngine, &config);
    }  

    xine_config_lookup_entry (xineEngine, "input.dvd_device", &config);
    if ( devicePath == config.str_value )
    {
      config.str_value = (char*)cachedDVDPath.latin1();
      xine_config_update_entry (xineEngine, &config);
    }  
  }  
    
  kdDebug(555) << "Save " << configFile << "\n";
  xine_config_save(xineEngine, configFile);
}  


/***************************************************
 *         CALLBACKS
 ***************************************************/

void VideoWindow::DestSizeCallback(void* p, int /*video_width*/, int /*video_height*/, double /*video_aspect*/,
                       int* dest_width, int* dest_height, double* dest_aspect)

{
  if (p == NULL) return;
  VideoWindow* vw = (VideoWindow*) p;

  *dest_width = vw->width();
  *dest_height = vw->height();
  *dest_aspect = vw->displayRatio;
}


void VideoWindow::FrameOutputCallback(void* p, int video_width, int video_height, double video_aspect,
                          int* dest_x, int* dest_y, int* dest_width, int* dest_height,
                          double* dest_aspect, int* win_x, int* win_y)

{
  if (p == NULL) return;
  VideoWindow* vw = (VideoWindow*) p;

  *dest_x = 0;
  *dest_y = 0 ;
  *dest_width = vw->width();
  *dest_height = vw->height();
  *win_x = vw->globX;
  *win_y = vw->globY;
  *dest_aspect = vw->displayRatio;

 /* correct size with video aspect */
  if (video_aspect >= vw->displayRatio)
    video_width  = (int) ( (double) (video_width * video_aspect / vw->displayRatio + 0.5) );
   else
    video_height = (int) ( (double) (video_height * vw->displayRatio / video_aspect) + 0.5);

 /*** set the size of main window (KMP) to fit the size of video stream ***/
  if (vw->autoresizeEnabled)
  {
    if ( (video_width != vw->videoFrameWidth) || (video_height != vw->videoFrameHeight) )
    {
      if ( (vw->parentWidget()) && (!vw->parentWidget()->isFullScreen()) && (vw->posTimer.isActive()) && (video_width > 0) && (video_height > 0) )
      {
        vw->videoFrameWidth = video_width;
        vw->videoFrameHeight = video_height;

        vw->newParentSize = vw->parentWidget()->size() - QSize((vw->width() - video_width), vw->height() - video_height);
        kdDebug(555) << "Resize video window to: " << vw->newParentSize.width() << "x" << vw->newParentSize.height() << "..." << endl;
        kdDebug(555) << "...with video frame: " << video_width << "x" << video_height << ", video aspect ratio: " << video_aspect << endl;

        /* we should not do a resize() inside a xine thread,
           but post an event to the main thread */
        KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_RESIZE_PARENT ) );
      }
    }
  }  
}


/*
 * XINE EVENT THREAD
 * only the QT event thread should do GUI operations,
 * we use QApplication::postEvent() and a reimplementation of QObject::timerEvent() to
 * make sure all critical jobs are done within the QT main thread context
 *
 * for more information see http://doc.trolltech.com/3.1/threads.html
 */

void VideoWindow::XineEventListener(void *p, const xine_event_t* xineEvent)
{

  if (p == NULL) return;
  VideoWindow* vw = (VideoWindow*) p;

 // KApplication::kApplication()->lock(); /* lock QT event thread */
  
  switch (xineEvent->type)
  {
    case XINE_EVENT_UI_PLAYBACK_FINISHED:
       {
         kdDebug(555) << "xine event: playback finished" << endl;
         KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_PLAYBACK_FINISHED ) );
         break; 
       }
    case XINE_EVENT_UI_CHANNELS_CHANGED:  /* new channel informations */
      {
        kdDebug(555) << "xine event: channels changed" << endl;
        int i,j,channels;
        char* lang = new char[128];     /* more than enough */
        QString num;

        vw->audioCh.clear();
        vw->subCh.clear();

        /*** get audio channels  ***/
        channels = QMAX(6, xine_get_stream_info(vw->xineStream, XINE_STREAM_INFO_MAX_AUDIO_CHANNEL));
        for(i = 0; i < channels; i++)
        {
          if (xine_get_audio_lang(vw->xineStream, i, lang))
            vw->audioCh << lang;
           else
            vw->audioCh << i18n("Ch ") + num.setNum(i+1);
        }     
      
        /*** get subtitle channels ***/
    
        channels = QMAX(6, xine_get_stream_info(vw->xineStream, XINE_STREAM_INFO_MAX_SPU_CHANNEL));
        for(j = 0; j < channels; j++)
        {
          if (xine_get_spu_lang(vw->xineStream, j, lang))
            vw->subCh << lang;
           else
            vw->subCh << i18n("Ch ") + num.setNum(j+1);
        }    

        delete lang;  /* or not? */

        vw->currentAudio = xine_get_param(vw->xineStream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL);
        vw->currentSub = xine_get_param(vw->xineStream, XINE_PARAM_SPU_CHANNEL);
 
        if (vw->currentAudio > (i-1))
        {
          vw->slotSetAudioChannel(0);
          vw->currentAudio = -1;
        } 
          
        if (vw->currentSub > (j-1))
        {
          vw->slotSetSubtitleChannel(0);
          vw->currentSub = -1;
         }

         KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_CHANNELS ) );
         break;
      }
    case XINE_EVENT_UI_SET_TITLE:  /* set new title */
       {
        kdDebug(555) << "xine event: ui set title" << endl;
        xine_ui_data_t* xd = (xine_ui_data_t*)xineEvent->data;
        vw->currentTitle = QString::fromLocal8Bit( (char*)xd->str );
        vw->trackInfoRow = vw->currentTitle + " ** " + vw->GetStreamInfo() + vw->extraInfo;

        KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_TITLE ) );
        KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_INFO ) );
        break;
       }
     case XINE_EVENT_PROGRESS:
       {
        kdDebug(555) << "xine event: progress info" << endl;
        xine_progress_data_t* pd = (xine_progress_data_t*)xineEvent->data;

        QString perc;
        perc.setNum(pd->percent);
        perc.append(" %");
        
        vw->currentProgress = QString::fromLocal8Bit( pd->description );
        vw->currentProgress.append("  ");
        vw->currentProgress.append(perc);
        
        if (pd->percent != 100)
          KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_PROGRESS_INFO ) );
         else
          KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_INFO ) );

        break;
        }
     case XINE_EVENT_DROPPED_FRAMES:
      {
        kdDebug(555) << "xine event: dropped frames" << endl;
        xine_dropped_frames_t* dropped = (xine_dropped_frames_t*)xineEvent->data;
        
        kdWarning(555) << "skipped frames: " << dropped->skipped_frames/10 << "% - discarded frames: " << dropped->discarded_frames/10 << "%" << endl;

        break;
       }  
     case XINE_EVENT_SPU_BUTTON:
       {
         kdDebug(555) << "xine event: spu button" << endl;
         xine_spu_button_t* button = (xine_spu_button_t*)xineEvent->data;

         if (button->direction == 1) /* enter a button */
         {          
           kdDebug(555) << "DVD Menu: Mouse entered button" << endl;
           vw->DVDButtonEntered = true;
         }  
         else
         {
           kdDebug(555) << "DVD Menu: Mouse leaved button" << endl;
           vw->DVDButtonEntered = false;
         }  

         KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_CHANGE_CURSOR ) );
         break;
       }  
     case XINE_EVENT_UI_NUM_BUTTONS:
       {
         kdDebug(555) << "xine event: ui num buttons" << endl;
       /*  xine_ui_data_t *buttons = (xine_ui_data_t *)xineEvent->data;
         if(buttons->num_buttons >= 1)
           vw->DVDMenuEntered = true; 
          else
           vw->DVDMenuEntered = false;
    
         kdDebug(555) << "DVD Menu: Num buttons=" << buttons->num_buttons << endl;
         KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_GRAB_KEYBOARD ) ); */
         break;
       }
     case XINE_EVENT_MRL_REFERENCE:
       {
         kdDebug(555) << "xine event: mrl reference" << endl;
         xine_mrl_reference_data_t* mrldata = (xine_mrl_reference_data_t*)xineEvent->data;
         vw->newMrlReference = mrldata->mrl;

         KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_MRL_REFERENCE ) );
         break;
       }
     case XINE_EVENT_UI_MESSAGE:
       {
         kdDebug(555) << "xine event: xine message" << endl;
         
         xine_ui_message_data_t *data = (xine_ui_message_data_t *)xineEvent->data;
         QString message;

         /* some code take from the xine-ui - Copyright (C) 2000-2003 the xine project */
         switch(data->type)
         {
           case XINE_MSG_NO_ERROR:
             {               
              /* copy strings, and replace '\0' separators by '\n' */
               char* s = data->messages;
               char* d = new char[2000];
     
               while(s && (*s != '\0') && ((*s + 1) != '\0'))
               {
                  switch(*s)
                  {
                    case '\0':
                      {
                         *d = '\n';
                         break;
                      }
                    default:
                      {
                         *d = *s;
                          break;
                      }
                  }   
                  s++;
                  d++;
               }
               *++d = '\0';

               message = d;
               delete d;
               break;
             }
           case XINE_MSG_GENERAL_WARNING:
            {
              message = i18n("General Warning: \n");

              if(data->explanation)
                message = message + ((char *) data + data->explanation) + " " + ((char *) data + data->parameters);
              else
                message = message + i18n("No Informations available.");

              break;
            }
          case XINE_MSG_SECURITY:
            {
              message = i18n("Security Warning: \n");

              if(data->explanation)
                message = message + ((char *) data + data->explanation) + " " + ((char *) data + data->parameters);

              break;
            }  
          case XINE_MSG_UNKNOWN_HOST:
            {
              message = i18n("The host you're trying to connect is unknown.\nCheck the validity of the specified hostname. ");
                if(data->explanation)
                  message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
          case XINE_MSG_UNKNOWN_DEVICE:
            {
              message = i18n("The device name you specified seems invalid. ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
          case XINE_MSG_NETWORK_UNREACHABLE:
            {
              message = i18n("The network looks unreachable.\nCheck your network setup and the server name. ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
           case XINE_MSG_AUDIO_OUT_UNAVAILABLE:
            {
              message = i18n("Audio output unavailable. Device is busy. ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
          case XINE_MSG_CONNECTION_REFUSED:
            {
              message = i18n("The connection was refused.\nCheck the host name. ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
          case XINE_MSG_FILE_NOT_FOUND:
            {
              message = i18n("The specified file or url was not found. Please check it. ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
#ifdef XINE_MSG_PERMISSION_ERROR    /* since xine-lib 1.0rc4 */
          case XINE_MSG_PERMISSION_ERROR:
            {
              message = i18n("Permission to this source was denied. ");
             // if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
#endif
          case XINE_MSG_READ_ERROR:
            {
              message = i18n("The source can't be read.\nMaybe you don't have enough rights for this, or source doesn't contain data (e.g: no disc in drive). ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }  
          case XINE_MSG_LIBRARY_LOAD_ERROR:
            {
              message = i18n("A problem occur while loading a library or a decoder: ");
              if(data->explanation)
                message = message + ((char *) data + data->parameters);
              break;
            }  
          case XINE_MSG_ENCRYPTED_SOURCE:
            {
              message = i18n("The source seems encrypted, and can't be read. ");
              if (vw->currentMRL.contains("dvd:/"))
                message = message + i18n("\nYour DVD is probably crypted. According to your country laws, you can or can't use libdvdcss to be able to read this disc. ");
              if(data->explanation)
                message = message + "(" + ((char *) data + data->parameters) + ")";
              break;
            }
          default:
            {
              message = i18n("Unknown error: \n");
              if(data->explanation)
                message = message + ((char *) data + data->explanation) + " " + ((char *) data + data->parameters);
              break;
            }
          }

         vw->xineMessage = message;
         KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_XINE_MESSAGE ) );
         break;
       }
     default:
       {
        // kdDebug(555) << "xine event: unhandled type " << xineEvent->type << endl;
         break;
       }    
   }

 //  KApplication::kApplication()->unlock();
}


void VideoWindow::timerEvent( QTimerEvent* tevent )
{
  switch ( tevent->timerId() )
  {
    case TIMER_EVENT_PLAYBACK_FINISHED:
    {
      emit signalPlaybackFinished();
      break;
    }
    case TIMER_EVENT_NEW_CHANNELS:
    {
      emit signalNewChannels( audioCh, subCh, currentAudio, currentSub);
      break;
    }
    case TIMER_EVENT_NEW_TITLE:
    {
       emit signalNewTitle( currentTitle );
       break;
    }
    case TIMER_EVENT_NEW_INFO:
    {
       emit signalNewInfo( trackInfoRow );
       break;
    }
    case TIMER_EVENT_NEW_PROGRESS_INFO:
    {
       emit signalNewInfo( currentProgress );
       break;
    }
    case TIMER_EVENT_CHANGE_CURSOR:
    {
      if (DVDButtonEntered)
        setCursor(QCursor(Qt::PointingHandCursor));
       else
        setCursor(QCursor(Qt::ArrowCursor));
      break;
    }      
    case TIMER_EVENT_GRAB_KEYBOARD:
    {
      if (DVDMenuEntered)
        grabKeyboard();
       else
        releaseKeyboard();
      break;  
    }      
    case TIMER_EVENT_NEW_MRL_REFERENCE:
    {
       emit signalNewMrlReference( newMrlReference );
       break;
    }
    case TIMER_EVENT_NEW_XINE_MESSAGE:
    {
       emit signalNewXineMessage();
       break;
    }
    case TIMER_EVENT_RESTART_PLAYBACK:
    {
      PlayMRL( currentMRL, currentTitle, false );
      slotChangePosition( savedPos );
      break;
    }
    case TIMER_EVENT_RESIZE_PARENT:
    {
      parentWidget()->resize( newParentSize );
      break;
    }    
    default: break;
  }
}      


/******************* new video driver *********************/

void VideoWindow::VideoDriverChangedCallback(void* p, xine_cfg_entry_t* entry)
{
  if (p == NULL) return;
  VideoWindow* vw = (VideoWindow*) p;
  xine_video_port_t* oldVideoDriver = vw->videoDriver;
  xine_video_port_t* noneVideoDriver;

  int pos, time, length;
  
  kdDebug(555) << "New video driver: " << entry->enum_values[entry->num_value] << endl;
  
  noneVideoDriver = xine_open_video_driver(vw->xineEngine, "none",
                    XINE_VISUAL_TYPE_NONE, NULL);
  if (!noneVideoDriver)
  {
     kdError(555) << "Can't init Video Driver 'none', operation aborted." << endl;
     return;
  }

  bool playing = false;
  if (xine_get_status(vw->xineStream) == XINE_STATUS_PLAY)
  {
    playing = true;
    vw->savedPos = 0;
    if ( xine_get_pos_length(vw->xineStream, &pos, &time, &length) )
     vw->savedPos = pos;
  }   

  xine_close(vw->xineStream);

  /* wire filters to "none" driver so the old one can be safely disposed */
  vw->videoDriver = noneVideoDriver;
  vw->UnwireFilters();
  vw->WireFilters();
  
  if (vw->xinePost)
  {
     kdDebug(555) << "Dispose visual plugin: " << vw->visualPluginName << endl;
     vw->postAudioSource = xine_get_audio_source (vw->xineStream);
     xine_post_wire_audio_port (vw->postAudioSource, vw->audioDriver);
     xine_post_dispose (vw->xineEngine, vw->xinePost);
     vw->xinePost = NULL;
  }

  xine_event_dispose_queue(vw->eventQueue);
  xine_dispose(vw->xineStream);
  
  xine_close_video_driver(vw->xineEngine, oldVideoDriver);

  vw->videoDriver = xine_open_video_driver(vw->xineEngine,
                           entry->enum_values[entry->num_value], XINE_VISUAL_TYPE_X11,
                           (void *) &(vw->visual));

  if (!vw->videoDriver)
  {
     QApplication::beep();
     vw->trackInfoRow = i18n("Error: Can't init new Video Driver %1 - using %2!").arg(entry->enum_values[entry->num_value]).arg(vw->videoDriverName);
     KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_INFO ) );
     playing = false;
     vw->videoDriver = xine_open_video_driver(vw->xineEngine,
                           vw->videoDriverName, XINE_VISUAL_TYPE_X11,
                           (void *) &(vw->visual));
  }
  else
  {
     vw->videoDriverName = entry->enum_values[entry->num_value];
     vw->trackInfoRow = i18n("Using Video Driver: ") + vw->videoDriverName;
     KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_INFO ) );
  }

  vw->xineStream = xine_stream_new(vw->xineEngine, vw->audioDriver, vw->videoDriver);
  vw->eventQueue = xine_event_new_queue (vw->xineStream);
  xine_event_create_listener_thread(vw->eventQueue, &VideoWindow::XineEventListener, p);

  /* rewire filters to the new driver */
  vw->UnwireFilters();
  vw->WireFilters();

  /* "none" can now be disposed too */
  xine_close_video_driver(vw->xineEngine, noneVideoDriver);
  
  if (playing)
    KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_RESTART_PLAYBACK ) );
}


/*********************** new audio driver *************************/

void VideoWindow::AudioDriverChangedCallback(void* p, xine_cfg_entry_t* entry)
{
  if (p == NULL) return;
  VideoWindow* vw = (VideoWindow*) p;

  int pos, time, length;

  kdDebug(555) << "New audio driver: " << entry->enum_values[entry->num_value] << endl;

  vw->UnwireFilters();

  bool playing = false;
  if (xine_get_status(vw->xineStream) == XINE_STATUS_PLAY)
  {
    playing = true;
    vw->savedPos = 0;
    if ( xine_get_pos_length(vw->xineStream, &pos, &time, &length) )
     vw->savedPos = pos;
  }   

  xine_close(vw->xineStream);

  if (vw->xinePost)
  {
     kdDebug(555) << "Dispose visual plugin: " << vw->visualPluginName << endl;
     vw->postAudioSource = xine_get_audio_source (vw->xineStream);
     xine_post_wire_audio_port (vw->postAudioSource, vw->audioDriver);
     xine_post_dispose (vw->xineEngine, vw->xinePost);
     vw->xinePost = NULL;
  }

  xine_event_dispose_queue(vw->eventQueue);
  xine_dispose(vw->xineStream);
  xine_close_audio_driver(vw->xineEngine, vw->audioDriver);
  vw->audioDriver = NULL;

  vw->audioDriver = xine_open_audio_driver(vw->xineEngine, entry->enum_values[entry->num_value], NULL);

  if (!vw->audioDriver)
  {
     QApplication::beep();
     vw->trackInfoRow = i18n("Error: Can't init new Audio Driver %1 - using %2!").arg(entry->enum_values[entry->num_value]).arg(vw->audioDriverName);
     KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_INFO ) );
     playing = false;
     vw->audioDriver = xine_open_audio_driver(vw->xineEngine, vw->audioDriverName, NULL);
  }
  else
  {
     vw->audioDriverName = entry->enum_values[entry->num_value];
     vw->trackInfoRow = i18n("Using Audio Driver: ") + vw->audioDriverName;
     KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_NEW_INFO ) );
  }

  vw->xineStream = xine_stream_new(vw->xineEngine, vw->audioDriver, vw->videoDriver);
  vw->eventQueue = xine_event_new_queue (vw->xineStream);
  xine_event_create_listener_thread(vw->eventQueue, &VideoWindow::XineEventListener, p);

  vw->WireFilters();

  if (playing)
    KApplication::kApplication()->postEvent( vw, new QTimerEvent( TIMER_EVENT_RESTART_PLAYBACK ) );
}
    

/************ change path to logo file *************/
    
void VideoWindow::LogoPathChangedCallback(void* p, xine_cfg_entry_t* entry)
{
  if (p==NULL) return;
  VideoWindow* vw = (VideoWindow*) p;

  vw->logoPath = entry->str_value;
}  


/**********************************************
 *      EVENT LOOP
 *********************************************/

void VideoWindow::run()
{
  kdDebug(555) << "Start event loop...\n";

  XEvent event;

  while( xineRunning )
  {
     XNextEvent ( xineDisplay, &event );
     XLockDisplay( xineDisplay );
     
     if (event.type == Expose)
     {
       if (event.xexpose.count == 0)
       {
         xine_gui_send_vo_data (xineStream, XINE_GUI_SEND_EXPOSE_EVENT, &event);
       }
     }
        
     XUnlockDisplay( xineDisplay );
  }

  kdDebug(555) << "Exiting event loop...\n";
}    
  


/**********************************************************
 *           INIT XINE ENGINE
 *********************************************************/

void VideoWindow::polish()
{
  if (!startXineManual)  /* start xine engine automatically? */
  {
    if (!InitXine()) emit signalQuit(); /* no success */
  }  
}    

 
bool VideoWindow::InitXine()
{
  if (xineRunning) return true;
 
  bool cmdlAudioEnum = false, cmdlVideoEnum = false;

  GlobalPosChanged(); /* get global pos of the window */
   
/**** INIT XINE DISPLAY  ****/

/*  
 *   xfree 4.4/x.org 6.7: init threads only if we are in the kpart, see main.cpp for more info
 */

#ifdef FamilyInternet6  /* in X.h since xfree version 4.3.99.10 */
  if (startXineManual)
  {
#endif
    kdDebug(555) << "XInitThreads()" << endl;
    XInitThreads();
#ifdef FamilyInternet6
  }
#endif  

  xineDisplay = XOpenDisplay( getenv("DISPLAY") );

  if (!xineDisplay)
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("Failed to connect to X-Server!"));
    return false;
  }

  xineScreen = DefaultScreen( xineDisplay );
  xineWindow = winId();

  XLockDisplay( xineDisplay );      

  XSelectInput( xineDisplay, xineWindow, ExposureMask );


/* determine display aspect ratio  */
  double resHor = DisplayWidth( xineDisplay, xineScreen )*1000 / DisplayWidthMM( xineDisplay, xineScreen );
  double resVer = DisplayHeight( xineDisplay, xineScreen )*1000 / DisplayHeightMM( xineDisplay, xineScreen );

  displayRatio = resVer / resHor;
  //if (QABS( displayRatio - 1.0 ) < 0.01) displayRatio   = 1.0;

  kdDebug(555) << "Display aspect ratio (v/h): " << displayRatio << endl;

  XUnlockDisplay( xineDisplay );

/**** INIT XINE ENGINE ****/

  kdDebug(555) << "Using xine version " << xine_get_version_string() << "\n";

  xineEngine = xine_new();
  if (!xineEngine)
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("Can't init xine Engine!"));
    return false;
  }

  if (xineVerbose)
   xine_engine_set_param( xineEngine, XINE_ENGINE_PARAM_VERBOSITY, 99 );

/* load configuration */

  configFile = QDir::homeDirPath();
  configFile.append("/.kaffeine/config");

  if (!QFile::exists(configFile))
    kdWarning(555) << "No config file found, will create one...\n";
  else
    xine_config_load (xineEngine, configFile);


   kdDebug(555) << "Post-init xine engine\n";
   xine_init(xineEngine);

/** set xine parameters **/

  const char* const* drivers = NULL;
  char** audioChoices = NULL;
  int i = 0;

  drivers = xine_list_audio_output_plugins (xineEngine);
  audioChoices = new char*[15];
  for(i = 0; i < 15; i++) audioChoices[i] = new char[10];
  audioChoices[0] = const_cast<char*>("auto");
  i = 0;
  while(drivers[i])
  {
    audioChoices[i+1] = (char*)drivers[i];
    if (preferedAudio == drivers[i]) cmdlAudioEnum = true;
    i++;
  }  
  audioChoices[i+1] = NULL;
  if (preferedAudio == "auto") cmdlAudioEnum = true;
  
  char* audioInfo = new char[200];
  strcpy(audioInfo, i18n("Audiodriver to use (default: auto)").local8Bit() );
  i = xine_config_register_enum(xineEngine, "gui.audiodriver", 0,
                   audioChoices, audioInfo, NULL, 10, &VideoWindow::AudioDriverChangedCallback, this);

  if (cmdlAudioEnum)
    audioDriverName = preferedAudio;
   else 
    audioDriverName = audioChoices[i];

  kdDebug(555) << "Use audio driver " << audioDriverName << "\n";

  i = 0;
  char** videoChoices = NULL;
  drivers = xine_list_video_output_plugins (xineEngine);
  videoChoices = new char*[15];
  for(i = 0; i < 15; i++) videoChoices[i] = new char[10];
  i = 0;
  videoChoices[0] = const_cast<char*>("auto");
  while(drivers[i])
  {
    videoChoices[i+1] = (char*)drivers[i];
    if (preferedVideo == drivers[i]) cmdlVideoEnum = true;
    i++;
  }
  videoChoices[i+1] = NULL;
  if (preferedVideo == "auto") cmdlVideoEnum = true;

  char* videoInfo = new char[200];
  strcpy( videoInfo, i18n("Videodriver to use (default: auto)").local8Bit() );
  i = xine_config_register_enum(xineEngine, "gui.videodriver", 0,
                   videoChoices, videoInfo, NULL, 10, &VideoWindow::VideoDriverChangedCallback, this);

  if (cmdlVideoEnum)
    videoDriverName = preferedVideo;
   else 
    videoDriverName = videoChoices[i];

  kdDebug(555) << "Use video driver " << videoDriverName << "\n";

  

/* init video driver */
  visual.display          = xineDisplay;
  visual.screen           = xineScreen;
  visual.d                = xineWindow;
  visual.dest_size_cb     = &VideoWindow::DestSizeCallback;
  visual.frame_output_cb  = &VideoWindow::FrameOutputCallback;
  visual.user_data        = (void*)this;

  kdDebug(555) << "Init video driver\n";

  videoDriver = xine_open_video_driver(xineEngine,
               videoDriverName,  XINE_VISUAL_TYPE_X11,
               (void *) &(visual));

  if (!videoDriver && videoDriverName != "auto")
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("Can't init Video Driver '%1' - trying 'auto'...").arg(videoDriverName));
    videoDriverName = "auto";
    videoDriver = xine_open_video_driver(xineEngine,
                  videoDriverName,  XINE_VISUAL_TYPE_X11,
                  (void *) &(visual));
  }

  if (!videoDriver)
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("All Video Drivers failed to initialize!"));
    return false;
  }

/* init audio driver */  
  kdDebug(555) << "Init audio driver\n";
  
  audioDriver = xine_open_audio_driver (xineEngine, audioDriverName, NULL);

  if (!audioDriver && audioDriverName != "auto")
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("Can't init Audio Driver '%1' - trying 'auto'...").arg(audioDriverName));
    audioDriverName = "auto";
    audioDriver = xine_open_audio_driver (xineEngine, audioDriverName, NULL);
  }  

  if (!audioDriver)
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("All Audio Drivers failed to initialize!"));
    return false;
  }

//kdDebug(555) << "Open stream\n";
  
  xineStream  = xine_stream_new(xineEngine, audioDriver, videoDriver);
  if (!xineStream)
  {
    QApplication::beep();
    KMessageBox::error(0, i18n("Can't create a new xine Stream!"));
    return false;
  }
  
/** event handling **/

  eventQueue = xine_event_new_queue (xineStream);
  xine_event_create_listener_thread(eventQueue, &VideoWindow::XineEventListener, (void*)this);

  
/* set path to logo file */

  QStringList dirs = KGlobal::dirs()->findDirs("data","kaffeine");
  logoPath = dirs[0];  // global kde path
  logoPath.append("logo.avi");

 // kdDebug(555) << "Logopath: " << logoPath << "\n";

  char* logoInfo = new char[200];
  strcpy( logoInfo, i18n("Absolute path to the logo URL (png file for example)").local8Bit() );
  logoPath = xine_config_register_string(xineEngine, "gui.logo_path", logoPath,
                     logoInfo, NULL, 10,
                     &VideoWindow::LogoPathChangedCallback, this);

/** get a keycode for a faked key event to hold down screensaver **/

#ifdef HAVE_XTEST
int a,b,c,d;
  haveXTest = XTestQueryExtension(x11Display(), &a, &b, &c, &d);
  if (haveXTest)
    xTestKeycode = XKeysymToKeycode(x11Display(), XK_Shift_L);
  
#endif

       
  kdDebug(555) << "xine init successful\n";
  xineRunning = true;

 /** start event thread **/
  start();

  parentWidget()->polish(); /* make sure gui will build correctly */
  
  return true;
}



/************************************************
 *              PLAY MRL
 ************************************************/


bool VideoWindow::PlayMRL(const QString& mrl, const QString& title, bool returnInfo)
{
  lengthInfoTimer.stop();
  
 // if (xine_get_status(xineStream) == XINE_STATUS_PLAY)
 //   xine_stop(xineStream);

  emit signalNewInfo(i18n("Opening..."));
  setCursor(QCursor(Qt::WaitCursor));

  currentMRL = mrl;

 /* check for external subtitle file or save url */
  extraInfo = QString::null;
  QString ref;
  
  for (int i = 1; i <= mrl.contains('#'); i++)
  {
    ref = mrl.section('#', i, i);
    if (ref.section(':', 0, 0) == "subtitle")
      extraInfo = extraInfo + " ** " + i18n("Subtitles: ") + ref.section(':', 1);
    if (ref.section(':', 0, 0) == "save")
      extraInfo = extraInfo + " ** " + i18n("Save as: ") + ref.section(':', 1);
  }
 
  kdDebug(555) << "Playing: " << currentMRL.local8Bit() << endl;

  if (!xine_open(xineStream, currentMRL.local8Bit() )) /** pass mrl local 8Bit encoded **/
  {
    SendXineError( returnInfo );
    setCursor(QCursor(Qt::ArrowCursor));
    return false;
  }  

/**** use visualization ? ****/

  if ( (xine_get_stream_info (xineStream, XINE_STREAM_INFO_HAS_AUDIO)) &&
       (!xine_get_stream_info (xineStream, XINE_STREAM_INFO_HAS_VIDEO)) )
  {

    if (visualPluginName && (!xinePost))
    {
      kdDebug(555) << "Init visual plugin: " << visualPluginName << endl;
      xinePost = xine_post_init (xineEngine, visualPluginName, 0,
          &audioDriver,
          &videoDriver);

      postAudioSource = xine_get_audio_source (xineStream);
      postInput = (xine_post_in_t*)xine_post_input (xinePost, const_cast<char*>("audio in"));
      xine_post_wire (postAudioSource, postInput);
    }
  }
  else
  {
    if (xinePost)
    {
      kdDebug(555) << "Dispose visual plugin: " << visualPluginName << endl;
      postAudioSource = xine_get_audio_source (xineStream);
      xine_post_wire_audio_port (postAudioSource, audioDriver);
      xine_post_dispose (xineEngine, xinePost);
      xinePost = NULL;
    }
  }
  
/*** play ***/

  if (!xine_play(xineStream, 0,0))
  {
    SendXineError( returnInfo );
    setCursor(QCursor(Qt::ArrowCursor));
    return false;
  }

/* do the stream have chapters ? */
  if (xine_get_stream_info(xineStream, XINE_STREAM_INFO_HAS_CHAPTERS))
    emit signalHasChapters(true);
   else
    emit signalHasChapters(false);


/** information requirement **/

  currentTitle = title;

  if ( (title.contains('.')) || (title.contains(":/")) ) /* tags was still read? */
  {
    QString metaTitle = QString::fromLocal8Bit( xine_get_meta_info(xineStream, XINE_META_INFO_TITLE) );
    if ( (metaTitle) && (!metaTitle.isEmpty()) )  //usable meta info
    {
      QString metaArtist = QString::fromLocal8Bit(xine_get_meta_info(xineStream, XINE_META_INFO_ARTIST));
      QString metaAlbum = QString::fromLocal8Bit(xine_get_meta_info(xineStream, XINE_META_INFO_ALBUM));
      QString metaTrack = ""; /* not implemented in xine-lib */
      QString metaInfo = metaString;
      metaInfo.replace( "artist", metaArtist );
      metaInfo.replace( "title",  metaTitle );
      metaInfo.replace( "album",  metaAlbum );
      metaInfo.replace( "track",  metaTrack );;
      currentTitle = metaInfo;
      if ((metaInfo != title) && (returnInfo))
      {
        emit signalMetaInfo(metaInfo);
      }
    }
    if (returnInfo)
    {
      QString length = GetLengthInfo();
      if (!length.isNull())
        emit signalLengthInfo( length );
       else
        lengthInfoTimer.start( 1000 ); /* wait for available track length info */
    }      
  }

  trackInfoRow = currentTitle;

  QString streamInfo = GetStreamInfo();
  if (returnInfo)
  {
     emit signalStreamInfo(streamInfo);
  }

  streamInfo = streamInfo + extraInfo;
                                                                                                        
  trackInfoRow.append( streamInfo.prepend(" ** ") );
  emit signalNewInfo(trackInfoRow);
   
  slotSetAudioChannel(0); //refresh channel info

  posTimer.start(500);

  setCursor(QCursor(Qt::ArrowCursor));
                
  emit signalShowOSD(currentTitle);

  return true;
}


/****** error processing ****/

void VideoWindow::SendXineError( bool returnInfo )
{
  QString error;
  int errCode = xine_get_error(xineStream);

  switch (errCode)
  {
    case XINE_ERROR_NO_INPUT_PLUGIN:    
    case XINE_ERROR_NO_DEMUX_PLUGIN:
    {
      error = i18n("No plugin found to handle this resource");
      break;
    }
    case XINE_ERROR_DEMUX_FAILED:
    {
      error = i18n("Resource seems to be broken");
      break;
    }
    case XINE_ERROR_MALFORMED_MRL:
    {
      error = i18n("Requested resource does not exist");
      break;
    }
    case XINE_ERROR_INPUT_FAILED:
    {
      error = i18n("Resource can not be opened");
      break;
    }
    default:
    {
      error = i18n("Generic error");
      break;
    }  
  }
    
  emit signalNewInfo( i18n("Error: ") + error );

  if (returnInfo)
  {
      emit signalStreamInfo( error );
      emit signalLengthInfo( i18n("*Error*") );
  }  
}


/****************** postprocessing filter management ****************/

void VideoWindow::GetPostprocessFilterNames( QStringList& filters )
{
  const char* const* plugins = xine_list_post_plugins_typed(xineEngine, XINE_POST_TYPE_VIDEO_FILTER);

  for (int i = 0; plugins[i]; i++)
  {
    filters << plugins[i];
  }  
}


void VideoWindow::slotCreateFilter( const QString& name, QWidget* parent )
{
  UnwireFilters();

  PostFilter* filter = new PostFilter( name, xineEngine, audioDriver, videoDriver, parent );
  connect( filter, SIGNAL( signalDeleteMe( PostFilter* ) ), this, SLOT( slotDeleteFilter( PostFilter* ) ));
  filterList.append( filter );

  WireFilters();
}  
  

void VideoWindow::slotRemoveAllFilters()
{
  UnwireFilters();
  if (deinterlaceFilter)
  {
    delete deinterlaceFilter;
    deinterlaceFilter = NULL;
  }  
  while (filterList.count())
    filterList.removeLast();
  WireFilters();
}  
  
  
void VideoWindow::slotDeleteFilter( PostFilter* filter )
{
  UnwireFilters();
  filterList.remove( filter );
  WireFilters();
}


void VideoWindow::UnwireFilters()
{
  xine_post_wire_video_port(  xine_get_video_source( xineStream), videoDriver );
 /* Waiting a small chunk of time helps to avoid crashing */
  // xine_usec_sleep(500000);
}


void VideoWindow::WireFilters()
{
  QPtrList<PostFilter> activeList;

  if (filterList.count() && filtersEnabled)
    activeList = filterList;
  
  if (deinterlaceFilter && deinterlaceEnabled )
    activeList.insert (0, deinterlaceFilter);

  if (activeList.count())
  {
    xine_post_wire_video_port( activeList.at( activeList.count()-1 )->GetOutput(), videoDriver );

    for (uint i = activeList.count()-1; i >0; i--)
    {
      xine_post_wire( activeList.at( i-1 )->GetOutput(), activeList.at( i )->GetInput() );
    }

    xine_post_wire( xine_get_video_source( xineStream ), activeList.at( 0 )->GetInput() );
  }
}


void VideoWindow::slotEnableFilters(bool enable)
{
  filtersEnabled = enable;

  UnwireFilters();
  WireFilters();
}     
    

/****  visual plugin **********/


void VideoWindow::GetVisualPlugins(QStringList& visuals) const
{
  const char* const* plugins = xine_list_post_plugins_typed(xineEngine, XINE_POST_TYPE_AUDIO_VISUALIZATION);

  for (int i = 0; plugins[i]; i++)
  {
    visuals << plugins[i];
  }
}   


/**************** change visualization plugin *****************/

void VideoWindow::SetVisualPlugin(const QString& visual)
{
  if ( visualPluginName == visual ) return;
  
  kdDebug(555) << "New visualization plugin: " << visual << endl;

  if (visual == "none")
    visualPluginName = QString::null;
   else
    visualPluginName = visual;

  if (xinePost)
  {
    xine_post_out_t *pp;

    pp = xine_get_audio_source (xineStream);
    xine_post_wire_audio_port (pp, audioDriver);
    xine_post_dispose (xineEngine, xinePost);
    xinePost = NULL;
  }



  if ( (xine_get_status( xineStream ) == XINE_STATUS_PLAY)
       && (!xine_get_stream_info(xineStream, XINE_STREAM_INFO_HAS_VIDEO)) && (visualPluginName) )
  {

    xinePost = xine_post_init (xineEngine, visualPluginName, 0, &audioDriver, &videoDriver);
    postAudioSource = xine_get_audio_source(xineStream);
    postInput = (xine_post_in_t*)xine_post_input(xinePost, const_cast<char*>("audio in"));
    xine_post_wire( postAudioSource, postInput );
  }
}


/*****/


void VideoWindow::GetAutoplayPlugins(QStringList& autoPlayList) const
{
  char** pluginIds = NULL;
  int i = 0;

  pluginIds = (char**)xine_get_autoplay_input_plugin_ids(xineEngine);

  while(pluginIds[i])
  {
    autoPlayList << pluginIds[i];
    
    autoPlayList << xine_get_input_plugin_description(xineEngine, pluginIds[i]);    
    i++;
  }
}

 
bool VideoWindow::GetAutoplayPluginMrl(const QString& plugin, QStringList& list)
{
  char** mrls = NULL;
  int num;
  int i = 0;

  mrls = xine_get_autoplay_mrls(xineEngine, plugin, &num);

  if (mrls)
  {
    while (mrls[i])
      {
        list << mrls[i];
        i++;
      }  

    return true;
   } 
   else 
   {
     QString error(i18n("Error: No "));
     error.append(plugin);
     error.append(i18n(", or wrong path to device."));
     signalNewInfo(error);
     return false;
   }
}


void VideoWindow::slotSetVolume(int vol)
{
  xine_set_param(xineStream, XINE_PARAM_AUDIO_VOLUME, -vol);
}


void VideoWindow::mouseMoveEvent(QMouseEvent* mev)
{
   if (!xineRunning) return;

 //  kdDebug(555) << "mouse move event" << endl;
   
   if (cursor().shape() == Qt::BlankCursor)
   {
     setCursor(QCursor(Qt::ArrowCursor));
   }  

    x11_rectangle_t   rect;
    xine_event_t      event;
    xine_input_data_t input;

    rect.x = mev->x();
    rect.y = mev->y();
    rect.w = 0;
    rect.h = 0;

    xine_gui_send_vo_data (xineStream,
               XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                (void*)&rect);

    event.type        = XINE_EVENT_INPUT_MOUSE_MOVE;
    event.data        = &input;
    event.data_length = sizeof(input);
    input.button      = 0; 
    input.x           = rect.x;
    input.y           = rect.y;
    xine_event_send (xineStream, &event);
}


void VideoWindow::mousePressEvent(QMouseEvent* mev)
{
   if (!xineRunning) return;
   int cur = cursor().shape();

 //  kdDebug(555) << "mouse press event" << endl;

   if (mev->button() == Qt::MidButton)
   {
     emit signalToggleFullscreen();
     mev->accept();
     return;
   }  

   if (mev->button() == Qt::RightButton)
   {
     if ( (cur == Qt::ArrowCursor) || (cur == Qt::BlankCursor) )
     {
       emit signalShowContextMenu(mev->globalPos());
       mev->accept();
       return;
     }  
   }  

   if (mev->button() == Qt::LeftButton)
   {
     if ( (cur == Qt::ArrowCursor) || (cur == Qt::BlankCursor) )
     {
       emit signalShowFullscreenPanel(mev->globalPos());
       mev->accept();
       return;
     }  

     x11_rectangle_t   rect;
     xine_event_t      event;
     xine_input_data_t input;

     rect.x = mev->x();
     rect.y = mev->y();
     rect.w = 0;
     rect.h = 0;

     xine_gui_send_vo_data (xineStream,
                XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                   (void*)&rect);

     event.type        = XINE_EVENT_INPUT_MOUSE_BUTTON;
     event.data        = &input;
     event.data_length = sizeof(input);
     input.button      = 1;
     input.x           = rect.x;
     input.y           = rect.y;
     xine_event_send (xineStream, &event);
     mev->accept();
   }
}


void VideoWindow::PlayNextChapter() const
{

  xine_event_t xev;

  xev.type = XINE_EVENT_INPUT_NEXT;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}  


void VideoWindow::PlayPreviousChapter() const
{

  xine_event_t xev;

  xev.type = XINE_EVENT_INPUT_PREVIOUS;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}  


void VideoWindow::slotStopPlayback()
{
  posTimer.stop();
  if (logoPath.isNull())
  {
    if (xine_get_status(xineStream) == XINE_STATUS_PLAY)
      xine_stop(xineStream);
  }
  else
  {
    PlayLOGO();
  }  
}


void VideoWindow::PlayLOGO() 
{
  if(!xine_open(xineStream, logoPath))
  {
    kdWarning(555) << "Invalid logo file\n";
    if (xine_get_status(xineStream) == XINE_STATUS_PLAY)
      xine_stop(xineStream);
    logoPath = QString::null;
    return;
  }  

  xine_play(xineStream, 0,0);
}


void VideoWindow::SetDevice(const QString& device)
{
  kdDebug(555) << "Set CD/VCD/DVD device to " << device << "\n";
  devicePath = device;

  xine_cfg_entry_t config;

  xine_config_lookup_entry (xineEngine, "input.cdda_device", &config);
  cachedCDPath = config.str_value;
  config.str_value = (char*)device.latin1();
  xine_config_update_entry (xineEngine, &config);

  xine_config_lookup_entry (xineEngine, "input.vcd_device", &config);
  cachedVCDPath = config.str_value;
  config.str_value = (char*)device.latin1();
  xine_config_update_entry (xineEngine, &config);

  xine_config_lookup_entry (xineEngine, "input.dvd_device", &config);
  cachedDVDPath = config.str_value;
  config.str_value = (char*)device.latin1();
  xine_config_update_entry (xineEngine, &config);
}


void VideoWindow::SetStreamSaveDir(const QString& dir)
{
  xine_cfg_entry_t config;

  if (!xine_config_lookup_entry (xineEngine, "misc.save_dir", &config)) return; /* older xine-lib */

  kdDebug(555) << "Set misc.save_dir to: " << dir << endl;
  config.str_value = (char*)dir.latin1();
  xine_config_update_entry (xineEngine, &config);
}  


const QString VideoWindow::GetStreamSaveDir()
{
  xine_cfg_entry_t config;

  if (!xine_config_lookup_entry (xineEngine, "misc.save_dir", &config)) return QString::null; /* older xine-lib */

  return QString( config.str_value );
}


void VideoWindow::SetBroadcasterPort(const uint port)
{
  kdDebug(555) << "Set broadcaster port to " << port << endl;
  xine_set_param(xineStream, XINE_PARAM_BROADCASTER_PORT, port);
}  


void VideoWindow::slotSpeedPause()
{
  xine_set_param(xineStream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
  posTimer.stop();
  signalNewInfo("Pause.");
}


void VideoWindow::slotSpeedNormal()
{
  xine_set_param(xineStream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
  posTimer.start(500);
  signalNewInfo(trackInfoRow);
}


QString VideoWindow::GetSupportedExtensions() const
{  
  return xine_get_file_extensions( xineEngine );
}


void VideoWindow::slotSetAudioChannel(int ch)
{
  xine_set_param(xineStream, XINE_PARAM_AUDIO_CHANNEL_LOGICAL, ch-1);
}  


void VideoWindow::slotSetSubtitleChannel(int ch)
{
  xine_set_param(xineStream, XINE_PARAM_SPU_CHANNEL, ch-1);
}


void VideoWindow::slotSetFileSubtitles(QString newMRL)
{
  int pos;
  int time;
  int length;
  
  QString oldMRL = currentMRL;
  
  if ( !xine_get_pos_length(xineStream, &pos, &time, &length) )
  {
    kdDebug(555) << "No valid stream position information" << endl;
    return;
  }    
  
  if (xine_get_status(xineStream) == XINE_STATUS_PLAY)
    xine_stop(xineStream);
    
  posTimer.stop();
  
  if(!PlayMRL(newMRL, currentTitle, true))
     PlayMRL(oldMRL, currentTitle, true);
     
  slotChangePosition(pos);
}


void VideoWindow::slotTogglePlayMode()
{
  switch (playMode)
 {
   case NORMAL_PLAY:
   {
     playMode = REPEAT_PLAY;
     break;
   }
   case REPEAT_PLAY:
   {
     playMode = PERCENT_PLAY;
     break;
   }
   case PERCENT_PLAY:
   {
     playMode = NORMAL_PLAY;
     break;
   }
 }
}


void VideoWindow::slotGetPosition()
{
  if (!xineRunning) return;

  int pos;
  int time;
  int length;
  
  double perc;

  QString t;
  
  if ( !xine_get_pos_length(xineStream, &pos, &time, &length) )
  {
    kdDebug(555) << "No valid stream position information" << endl;
    return;
  }  
  
  switch (playMode)
  { 
    case PERCENT_PLAY:
    {
      perc = pos/655,35;
      time = (int) (perc);
      if ( (time < 0) || (time > 100) )
      {
        emit signalNewPosition( pos , "  --%  " );
        return;
      }  
      t = t.setNum(time);
      t = QString(" %1%2 ").arg(t).arg("%");
      emit signalNewPosition( pos, t );
      break;
    }  
    case REPEAT_PLAY:
    {
      time = length - time;      
    }
    case NORMAL_PLAY:
    {
      if (time < 0)
      {
        emit signalNewPosition( pos , "-:--:--" );
        return;
      } 
      emit signalNewPosition( pos, msToTimeString(time) );
      break;
    }
  }
}


void VideoWindow::slotChangePosition(int pos)
{
  if (!xineRunning) return;

  int pause = !xine_get_param(xineStream, XINE_PARAM_SPEED);
  if ( (xine_get_status(xineStream) == XINE_STATUS_PLAY) && (xine_get_stream_info(xineStream, XINE_STREAM_INFO_SEEKABLE)) )
  {
    posTimer.stop();
    xine_play(xineStream, pos, 0);
    posTimer.start(500,false);
  }
  
  if (pause)
    slotSpeedPause();
}


void VideoWindow::slotEject()
{
  xine_eject(xineStream);
}

void VideoWindow::slotEnableAutoresize(bool enable)
{
  autoresizeEnabled = enable;
  if (!autoresizeEnabled)
  {
    videoFrameHeight = 0;
    videoFrameWidth = 0;
  }  
}


/***************************************
 *    tvtime deinterlacer plugin       *
 ***************************************/

void VideoWindow::CreateDeinterlacePlugin( const QString& config )
{
  deinterlacerDialog = new DeinterlacerConfigDialog( 0, "deinterlacefilterconfigdialog" );

  deinterlaceFilter = new PostFilter( config.section(':',0,0), xineEngine, audioDriver, videoDriver, deinterlacerDialog->GetMainWidget() );
  if( !deinterlaceFilter->GetInput() || !deinterlaceFilter->GetOutput() )
  {
    delete deinterlaceFilter;
    deinterlaceFilter = NULL;
  }

  slotSetDeinterlaceConfig( config );
}
  

const QString VideoWindow::GetDeinterlaceConfig() const
{
  if (deinterlaceFilter)
    return deinterlaceFilter->GetConfig();

  return DEFAULT_TVTIME_CONFIG;
}  


void VideoWindow::slotSetDeinterlaceConfig( const QString& config )
{
   if (deinterlaceFilter)
     deinterlaceFilter->SetConfig( config );
}  


void VideoWindow::slotToggleDeinterlace()
{
  if (deinterlaceFilter)
  {
    deinterlaceEnabled = !deinterlaceEnabled;
    kdDebug(555) << "Deinterlace enabled: " << deinterlaceEnabled << endl;
    UnwireFilters();
    WireFilters();
  }
  else
  {
    /* fallback - this method is deprecated */
    if (xine_get_param(xineStream, XINE_PARAM_VO_DEINTERLACE))
      xine_set_param(xineStream, XINE_PARAM_VO_DEINTERLACE, false);
     else
      xine_set_param(xineStream, XINE_PARAM_VO_DEINTERLACE, true);
  }
}

/**************************/


void VideoWindow::slotAspectRatioAuto()
{
  xine_set_param(xineStream, XINE_PARAM_VO_ASPECT_RATIO, XINE_VO_ASPECT_AUTO);
}  


void VideoWindow::slotAspectRatio4_3()
{
  xine_set_param(xineStream, XINE_PARAM_VO_ASPECT_RATIO, XINE_VO_ASPECT_4_3);
}


void VideoWindow::slotAspectRatio16_9()
{
  xine_set_param(xineStream, XINE_PARAM_VO_ASPECT_RATIO, XINE_VO_ASPECT_ANAMORPHIC);
}


void VideoWindow::slotAspectRatioSquare()
{
  xine_set_param(xineStream, XINE_PARAM_VO_ASPECT_RATIO, XINE_VO_ASPECT_SQUARE);
}


void VideoWindow::slotZoomOut()
{
  if ((currentZoom - 5) >= 100)
  {
    currentZoom -= 5;
    xine_set_param(xineStream, XINE_PARAM_VO_ZOOM_X, currentZoom);
    xine_set_param(xineStream, XINE_PARAM_VO_ZOOM_Y, currentZoom);
  }
}


void VideoWindow::slotZoomIn()
{
  if ((currentZoom + 5) <= XINE_VO_ZOOM_MAX)
  {
    currentZoom += 5;
    xine_set_param(xineStream, XINE_PARAM_VO_ZOOM_X, currentZoom);
    xine_set_param(xineStream, XINE_PARAM_VO_ZOOM_Y, currentZoom);
  }
}


void VideoWindow::slotZoomOff()
{
  xine_set_param(xineStream, XINE_PARAM_VO_ZOOM_X, 100);
  xine_set_param(xineStream, XINE_PARAM_VO_ZOOM_Y, 100);
  currentZoom = 100;
}


QString VideoWindow::GetStreamInfo()
{
  QString streamInfo;

  streamInfo = streamInfo + "(" + xine_get_meta_info(xineStream, XINE_META_INFO_INPUT_PLUGIN) + ") ";

  if (xine_get_stream_info(xineStream, XINE_STREAM_INFO_HAS_VIDEO))
  {
     streamInfo.append(xine_get_meta_info(xineStream, XINE_META_INFO_VIDEOCODEC));
     streamInfo.append(" (");
     streamInfo.append( QString::number(xine_get_stream_info(xineStream, XINE_STREAM_INFO_VIDEO_WIDTH)) );
     streamInfo.append("x");
     streamInfo.append( QString::number(xine_get_stream_info(xineStream, XINE_STREAM_INFO_VIDEO_HEIGHT)) );
     streamInfo.append(") ");
  }

  if (xine_get_stream_info(xineStream, XINE_STREAM_INFO_HAS_AUDIO))
  {
     streamInfo.append(xine_get_meta_info(xineStream, XINE_META_INFO_AUDIOCODEC));
     streamInfo.append(" (");
     streamInfo.append( QString::number(xine_get_stream_info(xineStream, XINE_STREAM_INFO_AUDIO_BITRATE)/1000) );
     streamInfo.append("kb)");
  }

  return streamInfo;
}


QString VideoWindow::GetLengthInfo()
{
  int pos, time, length;

  bool ok = xine_get_pos_length(xineStream, &pos, &time, &length);

  if ( (ok) && (length > 0) )
  {
    return msToTimeString(length);
  }
  
  return QString::null;
}  


void VideoWindow::slotEmitLengthInfo()
{
  QString length = GetLengthInfo();
  if (!length.isNull())
  {
    lengthInfoTimer.stop();
    emit signalLengthInfo( length );
  }  
}


void VideoWindow::slotSetScreensaverTimeout( int ssTimeout )
{
  screensaverTimeout = ssTimeout;

#ifdef HAVE_XTEST
  if (screensaverTimeout > 0)
    screensaverTimer.start(screensaverTimeout* 60000); //min
   else
    screensaverTimer.stop();
#endif    
}


int VideoWindow::GetScreensaverTimeout() const
{
  return screensaverTimeout;
}


void VideoWindow::GlobalPosChanged()
{
  QPoint g = mapToGlobal(QPoint(0,0));
  globX = g.x();
  globY = g.y();
//  kdDebug(555) << "VideoWindow: x: " << globX << " y: " << globY << endl;
}    


void VideoWindow::slotFakeKeyEvent()
{
#ifdef HAVE_XTEST

// kdDebug(555) << "Fake key event...\n";

 if (haveXTest)
 {
   XTestFakeKeyEvent(x11Display(), xTestKeycode, true, CurrentTime);
   XTestFakeKeyEvent(x11Display(), xTestKeycode, false, CurrentTime);
   XSync(x11Display(), false);
 }  

#endif 

}


const xine_t* const VideoWindow::GetXineEngine()const
{
  return xineEngine;
}

  
/************ video settings ****************/

void VideoWindow::GetVideoSettings(int& hue, int& sat, int& contrast, int& bright,
                                   int& audioAmp, int& avOffset, int& spuOffset) const
{
  hue = xine_get_param(xineStream, XINE_PARAM_VO_HUE);
  sat = xine_get_param(xineStream, XINE_PARAM_VO_SATURATION);
  contrast = xine_get_param(xineStream, XINE_PARAM_VO_CONTRAST);
  bright = xine_get_param(xineStream, XINE_PARAM_VO_BRIGHTNESS);

  audioAmp = xine_get_param(xineStream, XINE_PARAM_AUDIO_AMP_LEVEL);
  avOffset = xine_get_param(xineStream, XINE_PARAM_AV_OFFSET);
  spuOffset = xine_get_param(xineStream, XINE_PARAM_SPU_OFFSET);
}


void VideoWindow::slotSetHue(int hue)
{
  xine_set_param(xineStream, XINE_PARAM_VO_HUE, hue);
}
  

void VideoWindow::slotSetSaturation(int sat)
{
  xine_set_param(xineStream, XINE_PARAM_VO_SATURATION, sat);
}
  

void VideoWindow::slotSetContrast(int contrast)
{
  xine_set_param(xineStream, XINE_PARAM_VO_CONTRAST, contrast);
}
  

void VideoWindow::slotSetBrightness(int bright)
{
  xine_set_param(xineStream, XINE_PARAM_VO_BRIGHTNESS, bright);
}
  

void VideoWindow::slotSetAudioAmp(int amp)
{
  xine_set_param(xineStream, XINE_PARAM_AUDIO_AMP_LEVEL, amp);
}
  

void VideoWindow::slotSetAVOffset(int av)
{
  xine_set_param(xineStream, XINE_PARAM_AV_OFFSET, av);
}
  

void VideoWindow::slotSetSpuOffset(int spu)
{
  xine_set_param(xineStream, XINE_PARAM_SPU_OFFSET, spu);
}



/**************** equalizer *****************/

void VideoWindow::slotSetEq30(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_30HZ, -val);
}


void VideoWindow::slotSetEq60(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_60HZ, -val);
}


void VideoWindow::slotSetEq125(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_125HZ, -val);
}


void VideoWindow::slotSetEq250(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_250HZ, -val);
}


void VideoWindow::slotSetEq500(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_500HZ, -val);
}


void VideoWindow::slotSetEq1k(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_1000HZ, -val);
}


void VideoWindow::slotSetEq2k(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_2000HZ, -val);
}


void VideoWindow::slotSetEq4k(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_4000HZ, -val);
}


void VideoWindow::slotSetEq8k(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_8000HZ, -val);
}


void VideoWindow::slotSetEq16k(int val)
{
  xine_set_param(xineStream, XINE_PARAM_EQ_16000HZ, -val);
}

/*************** dvd menus ******************/

void VideoWindow::slotMenu1()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU1;
  xev.data = NULL;
  xev.data_length = 0;
  
  xine_event_send(xineStream, &xev);
}


void VideoWindow::slotMenu2()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU2;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
  
}


void VideoWindow::slotMenu3()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU3;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}


void VideoWindow::slotMenu4()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU4;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}


void VideoWindow::slotMenu5()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU5;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}


void VideoWindow::slotMenu6()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU6;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}


void VideoWindow::slotMenu7()
{
  xine_event_t xev;
  xev.type = XINE_EVENT_INPUT_MENU7;
  xev.data = NULL;
  xev.data_length = 0;

  xine_event_send(xineStream, &xev);
}


void VideoWindow::slotDVDMenuLeft()
{
  xine_event_t xev;
  xev.data = NULL;
  xev.data_length = 0;
  xev.type = XINE_EVENT_INPUT_LEFT;

  xine_event_send(xineStream, &xev);
}

  
void VideoWindow::slotDVDMenuRight()
{
  xine_event_t xev;
  xev.data = NULL;
  xev.data_length = 0;
  xev.type = XINE_EVENT_INPUT_RIGHT;

  xine_event_send(xineStream, &xev);
}

    
void VideoWindow::slotDVDMenuUp()
{
  xine_event_t xev;
  xev.data = NULL;
  xev.data_length = 0;
  xev.type = XINE_EVENT_INPUT_UP;

  xine_event_send(xineStream, &xev);
}

  
void VideoWindow::slotDVDMenuDown()
{
  xine_event_t xev;
  xev.data = NULL;
  xev.data_length = 0;
  xev.type = XINE_EVENT_INPUT_DOWN;

  xine_event_send(xineStream, &xev);
}

  
void VideoWindow::slotDVDMenuSelect()
{
  xine_event_t xev;
  xev.data = NULL;
  xev.data_length = 0;
  xev.type = XINE_EVENT_INPUT_SELECT;

  xine_event_send(xineStream, &xev);
}
  
/*
void VideoWindow::keyPressEvent(QKeyEvent *e)
{  
  if (keyboardGrabber() == this)
  {
    xine_event_t xev;
    xev.data = NULL;
    xev.data_length = 0;

    switch(e->key())
    {
      case Qt::Key_Up:
        xev.type = XINE_EVENT_INPUT_UP;
        break;
      case Qt::Key_Down:
        xev.type = XINE_EVENT_INPUT_DOWN;
        break;
      case Qt::Key_Left:
        xev.type = XINE_EVENT_INPUT_LEFT;
        break;
      case Qt::Key_Right:
        xev.type = XINE_EVENT_INPUT_RIGHT;
        break;
      case Qt::Key_Return:
        xev.type = XINE_EVENT_INPUT_SELECT;
        break;
      default:
        e->ignore();
        return;
    }
    xine_event_send(xineStream, &xev);
      
    e->accept();
    return;
  }
  e->ignore();
}
*/


/******** mouse hideing at fullscreen ****/


void VideoWindow::StartMouseHideTimer()
{
  mouseHideTimer.start(5000);
}
  

void VideoWindow::StopMouseHideTimer()
{
  mouseHideTimer.stop();
}
  

void VideoWindow::slotHideMouse()
{
  if (cursor().shape() == Qt::ArrowCursor)
  {
    setCursor(QCursor(Qt::BlankCursor));
  }  
}   
  

/************************************************************
 *   Take a Screenshot                                      *
 ************************************************************/


void VideoWindow::GetScreenshot(uchar*& rgb32BitData, int& videoWidth, int& videoHeight, double& scaleFactor) const
{

  uint8_t   *yuv = NULL, *y = NULL, *u = NULL, *v =NULL;
  
  int        width, height, ratio, format;
  double     desired_ratio, image_ratio;

  if (!xine_get_current_frame (xineStream, &width, &height, &ratio, &format, NULL))
    return;

  yuv = new uint8_t[((width+8) * (height+1) * 2)];
  if (yuv == NULL)
    {
      kdError(555) << "Not enough memory to make screenshot!" << endl;
      return;
    }  

  xine_get_current_frame (xineStream, &width, &height, &ratio, &format, yuv);

  videoWidth = width;
  videoHeight = height;

/*
 * convert to yv12 if necessary
 */

  switch (format) {
  case XINE_IMGFMT_YUY2:
    {
      uint8_t *yuy2 = yuv;

      yuv = new uint8_t[(width * height * 2)];
      if (yuv == NULL)
      {
        kdError(555) << "Not enough memory to make screenshot!" << endl;
        return;
      }  
      y = yuv;
      u = yuv + width * height;
      v = yuv + width * height * 5 / 4;

      yuy2Toyv12 (y, u, v, yuy2, width, height);

      delete [] yuy2;
    }
    break;
  case XINE_IMGFMT_YV12:
    y = yuv;
    u = yuv + width * height;
    v = yuv + width * height * 5 / 4;

    break;
  default:
    {
      kdWarning(555) << "Screenshot: Format " << (char *) &format << "not supported!\n";
      delete [] yuv;
      return;
    }  
  }

  /*
   * convert to rgb
   */

  rgb32BitData = yv12ToRgb (y, u, v, width, height);


  image_ratio = (double) width / (double) height;


  switch (ratio) {
  case XINE_VO_ASPECT_ANAMORPHIC:  /* anamorphic  */
//  case XINE_VO_ASPECT_PAN_SCAN:  /* depreciated */
    desired_ratio = 16.0 /9.0;
    break;
  case XINE_VO_ASPECT_DVB:         /* 2.11:1 */
    desired_ratio = 2.11/1.0;
    break;
  case XINE_VO_ASPECT_SQUARE:      /* square pels */
//  case XINE_VO_ASPECT_DONT_TOUCH:  /* depreciated */
    desired_ratio = image_ratio;
    break; 
  default:
    kdWarning(555) << "Screenshot: Unknown aspect ratio: " << ratio << " - using 4:3" << endl;
  case XINE_VO_ASPECT_4_3:         /* 4:3   */
    desired_ratio = 4.0 / 3.0;
    break;
  }

  scaleFactor = desired_ratio / image_ratio;

  delete [] yuv;
}














