/***************************************************************************
                          postfilter.cpp - wrapper for xine's postprocessing filters
                             -------------------
    begin                : Son Dez 7 2003
    revision             : $Revision: 1.9 $
    last modified        : $Date: 2004/04/26 11:04:57 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by Juergen  Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kseparator.h>
#include <klocale.h>
#include <kdebug.h>

#include <qlayout.h>
#include <qlabel.h>

#include <stdio.h>
 
#include "postfilter.h"
#include "postfilter.moc"



PostFilter::PostFilter(const QString& name, xine_t* engine, xine_audio_port_t* audioDriver,
              xine_video_port_t* videoDriver, QWidget *parent) : QObject(parent), data(NULL), groupBox(NULL)
{
  filterName = name;
  kdDebug(555) << "Create Postprocessing Filter: " << filterName << endl;

  xineEngine = engine;

  xinePost = xine_post_init( xineEngine, filterName, 0, &audioDriver, &videoDriver );
  if( xinePost )
  {
    xine_post_in_t* inputAPI = NULL;

    groupBox = new QGroupBox( name, parent );
    groupBox->setSizePolicy ( QSizePolicy (QSizePolicy::Minimum, QSizePolicy::Fixed) );
    QGridLayout* grid = new QGridLayout( groupBox, 2, 2 );
    grid->setMargin( 20 );
    grid->setSpacing( 5 );
    int row = 0;

    if ((inputAPI = (xine_post_in_t*)xine_post_input( xinePost, const_cast<char*>("parameters")) ))
    {
      xinePostAPI = (xine_post_api_t*)inputAPI->data;
      xinePostDescr = xinePostAPI->get_param_descr();
      xinePostParameter = xinePostDescr->parameter;

      data = new char[ xinePostDescr->struct_size ];
      xinePostAPI->get_parameters( xinePost, data );

      QLabel* descr;

      while (xinePostParameter->type != POST_PARAM_TYPE_LAST)
      {
        kdDebug(555) << "Parameter: " << xinePostParameter->name << endl;
        if (xinePostParameter->readonly) continue;

        switch (xinePostParameter->type)
        {
          case POST_PARAM_TYPE_INT:
          {
            if (xinePostParameter->enum_values)
            {
              PostFilterParameterCombo* parameter = new
                PostFilterParameterCombo( xinePostParameter->name, xinePostParameter->offset,
                                          *(int*)(data+xinePostParameter->offset),
                                          xinePostParameter->enum_values, groupBox );
              connect( parameter, SIGNAL( signalIntValue(int, int)), this, SLOT( slotApplyIntValue(int, int) ));
              parameterList.append( parameter );

              grid->addWidget( parameter->getWidget(), row, 0 );
            }
            else
            {
              PostFilterParameterInt* parameter = new
                PostFilterParameterInt( xinePostParameter->name, xinePostParameter->offset,
                                        *(int*)(data+xinePostParameter->offset),
                                        (int)xinePostParameter->range_min, (int)xinePostParameter->range_max,
                                        groupBox );

              connect( parameter, SIGNAL( signalIntValue(int, int) ), this, SLOT( slotApplyIntValue(int, int) ));
              parameterList.append( parameter );

              grid->addWidget( parameter->getWidget(), row, 0 );
            }
            break;
          }
          case POST_PARAM_TYPE_DOUBLE:
          {
            PostFilterParameterDouble* parameter = new
              PostFilterParameterDouble( xinePostParameter->name, xinePostParameter->offset,
                                         *(double*)(data+xinePostParameter->offset),
                                         (double)xinePostParameter->range_min, (double)xinePostParameter->range_max,
                                         groupBox );

            connect( parameter, SIGNAL( signalDoubleValue(int, double) ), this, SLOT( slotApplyDoubleValue(int, double) ));
            parameterList.append( parameter );

            grid->addWidget( parameter->getWidget(), row, 0 );
            break;
          }
          case POST_PARAM_TYPE_CHAR:
          {
            PostFilterParameterChar* parameter = new
              PostFilterParameterChar( xinePostParameter->name, xinePostParameter->offset,
                                       (char*)(data+xinePostParameter->offset), xinePostParameter->size,
                                       groupBox );

            connect( parameter, SIGNAL( signalCharValue(int, const QString&) ), this, SLOT( slotApplyCharValue(int, const QString& ) ));
            parameterList.append( parameter );

            grid->addWidget( parameter->getWidget(), row, 0 );
            break;
          }
          case POST_PARAM_TYPE_STRING:
          case POST_PARAM_TYPE_STRINGLIST: break; /* not implemented */
          case POST_PARAM_TYPE_BOOL:
          {
            PostFilterParameterBool* parameter = new
              PostFilterParameterBool( xinePostParameter->name, xinePostParameter->offset,
                                       (bool) *(int*)(data+xinePostParameter->offset), groupBox );

            connect( parameter, SIGNAL( signalIntValue(int, int) ), this, SLOT( slotApplyIntValue(int, int) ));
            parameterList.append( parameter );

            grid->addWidget( parameter->getWidget(), row, 0 );
            break;
          }
          default: break;
        }

        descr = new QLabel( QString::fromLocal8Bit( xinePostParameter->description ), groupBox );
        descr->setAlignment( QLabel::WordBreak | QLabel::AlignVCenter );
        grid->addWidget( descr, row, 1 );
        row++;
        xinePostParameter++;
      }
    }
    KSeparator* sep = new KSeparator( KSeparator::Horizontal, groupBox );
    grid->addMultiCellWidget( sep, row, row, 0, 1 );
    row++;
    KPushButton* deleteButton = new KPushButton( i18n("Delete Filter"), groupBox );
    deleteButton->setSizePolicy ( QSizePolicy (QSizePolicy::Minimum, QSizePolicy::Fixed) );
    connect( deleteButton, SIGNAL( clicked() ), this, SLOT( slotDeletePressed() ));
    grid->addWidget( deleteButton, row, 0 );

    if( inputAPI )
    {
      KPushButton* helpButton = new KPushButton( i18n("Help"), groupBox );
      helpButton->setSizePolicy ( QSizePolicy (QSizePolicy::Minimum, QSizePolicy::Fixed) );
      connect( helpButton, SIGNAL( clicked() ), this, SLOT( slotHelpPressed() ));
      grid->addWidget( helpButton, row, 1 );
    }
 
    if (parent)
      groupBox->show();
  }
}



PostFilter::~PostFilter()
{
  kdDebug(555) << "Delete Postprocessing Filter: " << filterName << endl;
  if(xinePost)
  {
    delete groupBox;
    delete data;
    xine_post_dispose(xineEngine, xinePost);
  }
}


void PostFilter::slotApplyIntValue(int offset, int val)
{
  kdDebug(555) << "PostFilter: " << filterName << " Apply integer value " << val << " on offset " << offset << endl;
  *(int*)(data+offset) = val;
  xinePostAPI->set_parameters( xinePost, data );
}


void PostFilter::slotApplyDoubleValue(int offset, double val)
{
  kdDebug(555) << "PostFilter: " << filterName << " Apply double value " << val << " on offset " << offset << endl;
  *(double*)(data+offset) = val;
  xinePostAPI->set_parameters( xinePost, data );
}


void PostFilter::slotApplyCharValue(int offset, const QString& val)
{
  kdDebug(555) << "PostFilter: " << filterName << " Apply char value '" << val << "' on offset " << offset << endl;
  sprintf((char *)(data+offset), "%s", val.latin1());
  xinePostAPI->set_parameters( xinePost, data );
}


xine_post_in_t* PostFilter::GetInput() const
{
  kdDebug(555) << "PostFilter: Get input 'video'" << endl;
  if(xinePost)
    return (xine_post_in_t*)xine_post_input( xinePost, const_cast<char*>("video") );
  else
    return NULL;
}
  

xine_post_out_t* PostFilter::GetOutput() const
{
  kdDebug(555) << "PostFilter: Get output" << endl;

  if(xinePost)
  {
    const char *const *outs = xine_post_list_outputs( xinePost );
    return (xine_post_out_t*)xine_post_output( xinePost, (char *) *outs);
  }
  else
    return NULL;
}


void PostFilter::slotHelpPressed()
{
  kdDebug(555) << "PostFilter: Help pressed" << endl;

  PostFilterHelp* filterHelp = new PostFilterHelp( NULL, filterName.latin1(), xinePostAPI->get_help());
  filterHelp->exec();

  delete filterHelp;
}


QString PostFilter::GetConfig()
{
 /*
  *  returns a string like "filtername:parameter=value,parameter=value,..."
  */

  QString configString;
  QTextOStream configStream(&configString);

  configStream << filterName << ":";
  for (uint i = 0; i < parameterList.count(); i++)
  {
    configStream << parameterList.at( i )->name() << "=" << parameterList.at( i )->getValue();
    if( i != parameterList.count()-1 )
      configStream << ",";
  }

  kdDebug(555) << "PostFilter: GetConfig " << configString << endl;

  return configString;
}

void PostFilter::SetConfig( const QString &configString )
{
 /*
  *  expects a string like filtername:parameter=value,parameter=value,...
  *  or filtername:parameter="value",parameter="value",...
  */

  kdDebug(555) << "PostFilter: SetConfig " << configString << endl;

  QString configStr;
  if (configString.section(':',0,0) == filterName)
  {
    configStr = configString.section(':',1,1);
  }
  else
  {
    kdWarning(555) << "Config string doesn't match filter name " << filterName << endl;
    kdDebug(555) << "Don't apply new configuration" << endl;
    return;
  }  

  for( int i = 0; i < configStr.contains(',') + 1; i++ )
  {
    QString parameterConfig = configStr.section(',', i, i);
    QString parameterName = parameterConfig.section('=', 0, 0);
    QString parameterValue = parameterConfig.section('=', 1, 1);
    parameterValue = parameterValue.remove('"');

    for (uint j = 0; j < parameterList.count(); j++)
    {
      if( parameterName == parameterList.at( j )->name() )
        parameterList.at( j )->setValue( parameterValue );
    }
  }
}


PostFilterParameterInt::PostFilterParameterInt( const QString& _name, int _offset, int value, int min, int max, QWidget* parent )
  : PostFilterParameter (_name, _offset, parent )
{
  numInput = new KIntNumInput( value, parent );
  numInput->setRange( min, max, 1, false);
  connect( numInput, SIGNAL( valueChanged( int ) ), this, SLOT( slotIntValue( int ) ));
}


PostFilterParameterDouble::PostFilterParameterDouble( const QString& _name, int _offset, double value, double min, double max, QWidget* parent )
  : PostFilterParameter (_name, _offset, parent )
{
  numInput = new KDoubleNumInput( parent );
  numInput->setValue( value );
  numInput->setRange( min, max, 0.01, false);
  connect( numInput, SIGNAL( valueChanged( double ) ), this, SLOT( slotDoubleValue( double ) ));
}


PostFilterParameterChar::PostFilterParameterChar( const QString& _name, int _offset, char *value, int size, QWidget* parent )
  : PostFilterParameter (_name, _offset, parent )
{
  charInput = new KLineEdit( value, parent );
  charInput->setMaxLength(size);
  connect( charInput, SIGNAL( returnPressed( const QString& ) ), this, SLOT( slotCharValue( const QString& ) ));
}


PostFilterParameterCombo::PostFilterParameterCombo( const QString& _name, int _offset, int value, char **enums, QWidget* parent )
  : PostFilterParameter (_name, _offset, parent )
{
  comboBox = new KComboBox( parent );
  for (int i = 0; enums[i]; i++)
  {
    comboBox->insertItem(enums[i]);
  }
  comboBox->setCurrentItem( value );
  connect( comboBox, SIGNAL( activated( int )), this, SLOT( slotIntValue( int ) ));
}


PostFilterParameterBool::PostFilterParameterBool( const QString& _name, int _offset, bool value, QWidget* parent )
  : PostFilterParameter (_name, _offset, parent )
{
  checkBox = new QCheckBox( parent );
  checkBox->setChecked( value );
  connect( checkBox, SIGNAL( toggled( bool ) ), this, SLOT( slotBoolValue( bool ) ));
}


PostFilterHelp::PostFilterHelp(QWidget *parent, const char *name, const char *text)
 : KDialogBase( parent, name, true, QString(name) + " - " + i18n("Help"), KDialogBase::Close )
{
  setInitialSize( QSize(500,500) );

  QWidget* mainWidget = makeMainWidget();
  QGridLayout* grid = new QGridLayout( mainWidget, 1, 1 );
  grid->setSpacing( 5 );

  QString help = QString::fromUtf8( text );
  textEdit = new QTextEdit( text, QString::null, mainWidget, name );
  textEdit->setReadOnly(true);
  grid->addWidget( textEdit, 0, 0 );
}


PostFilterHelp::~PostFilterHelp()
{
  delete textEdit;
}

