// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "button.h"
#include "entry.h"
#include "user_property.h"
#include "widget_manip.h"

#include <k3dsdk/aqsis_properties.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/inode.h>
#include <k3dsdk/ipersistent_container.h>
#include <k3dsdk/iproperty_collection.h>
#include <k3dsdk/renderman_properties.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/tokens.h>
#include <k3dsdk/user_properties.h>

#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/combobox.h>
#include <gtkmm/entry.h>
#include <gtkmm/label.h>
#include <gtkmm/stock.h>
#include <gtkmm/table.h>

#include <boost/format.hpp>

namespace libk3dngui
{

///////////////////////////////////////////////////////////////////////////////
// add_user_property

add_user_property::add_user_property(k3d::inode& Object, k3d::icommand_node& Parent) :
	base(Object, Parent, "add_user_property"),
	m_type(new Gtk::ComboBox()),
	m_types(Gtk::ListStore::create(m_type_columns)),
	m_name(init_value(std::string(""))),
	m_label(init_value(std::string(""))),
	m_description(init_value(std::string("")))
{
	set_title(_("Add User Property:"));
	set_role("add_user_property");
	set_position(Gtk::WIN_POS_CENTER);

	add_type("bool", _("Boolean"));
	add_type("color", _("Color"));
	add_type("double", _("Double"));
	add_type("string", _("String"));
	add_type("vector3", _("Vector3"));
	add_type("matrix4", _("Matrix4"));
	add_type("aqsis_displacement_layer", _("Aqsis Displacement Layer"));
	add_type("aqsis_surface_layer", _("Aqsis Surface Layer"));

	m_type->set_model(m_types);
	m_type->set_active(0);
	m_type->pack_start(m_type_columns.label);
	tooltips().set_tip(*m_type, _("Choose property type"));

	Gtk::Table* const table = Gtk::manage(new Gtk::Table(4, 2, false));

	table->attach(*Gtk::manage(new Gtk::Label(_("Type:"))), 0, 1, 0, 1, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(m_type), 1, 2, 0, 1);

	table->attach(*Gtk::manage(new Gtk::Label(_("Name:"))), 0, 1, 1, 2, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "name", entry::proxy(m_name))), 1, 2, 1, 2);

	table->attach(*Gtk::manage(new Gtk::Label(_("Label:"))), 0, 1, 2, 3, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "label", entry::proxy(m_label))), 1, 2, 2, 3);

	table->attach(*Gtk::manage(new Gtk::Label(_("Description:"))), 0, 1, 3, 4, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "description", entry::proxy(m_description))), 1, 2, 3, 4);

	Gtk::HButtonBox* const box = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
	box->pack_start(*Gtk::manage(new button::control(*this, "cancel", _("Cancel"), Gtk::Stock::CANCEL)
		<< connect_button(sigc::mem_fun(*this, &add_user_property::close))));
	box->pack_start(*Gtk::manage(new button::control(*this, "add", _("Add"), Gtk::Stock::ADD)
		<< connect_button(sigc::mem_fun(*this, &add_user_property::on_add))));

	Gtk::VBox* const vbox = Gtk::manage(new Gtk::VBox(false, 10));
	vbox->set_border_width(10);
	vbox->pack_start(*Gtk::manage(new Gtk::Label(k3d::string_cast(boost::format(_("Add a user property to %1%:")) % node().name()))), Gtk::PACK_SHRINK);
	vbox->pack_start(*table, Gtk::PACK_SHRINK);
	vbox->pack_start(*box, Gtk::PACK_SHRINK);

	add(*vbox);

	show_all();
}

void add_user_property::add_type(const std::string& Type, const std::string& Label)
{
	Gtk::TreeModel::Row row = *m_types->append();
	row[m_type_columns.type] = Type;
	row[m_type_columns.label] = Label;
}

void add_user_property::on_add()
{
	k3d::iproperty_collection* const property_collection = dynamic_cast<k3d::iproperty_collection*>(&node());
	return_if_fail(property_collection);

	k3d::ipersistent_container* const persistent_container = dynamic_cast<k3d::ipersistent_container*>(&node());
	return_if_fail(persistent_container);

	const std::string type = m_type->get_active()->get_value(m_type_columns.type);
	const std::string name = m_name.value();
	const std::string label = m_label.value();
	const std::string description = m_description.value();

	if(type == "bool")
		k3d::user::create_property<k3d::user::bool_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), false);
	else if(type == "color")
		k3d::user::create_property<k3d::user::color_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::color(1, 1, 1));
	else if(type == "double")
		k3d::user::create_property<k3d::user::double_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), 0.0);
	else if(type == "string")
		k3d::user::create_property<k3d::user::string_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), std::string());
	else if(type == "vector3")
		k3d::user::create_property<k3d::user::vector3_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::vector3(0, 0, 0));
	else if(type == "matrix4")
		k3d::user::create_property<k3d::user::matrix4_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::identity3D());
	else if(type == "aqsis_displacement_layer")
		k3d::user::create_property<k3d::aqsis::displacement_layer_connection_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), static_cast<k3d::inode*>(0));
	else if(type == "aqsis_surface_layer")
		k3d::user::create_property<k3d::aqsis::surface_layer_connection_property>(name, label, description, node().document(), *property_collection, *persistent_container, &node(), static_cast<k3d::inode*>(0));
	else		
		k3d::log() << error << "unknown user property type: [" << type << "]" << std::endl;

	close();
}

///////////////////////////////////////////////////////////////////////////////
// add_renderman_property

add_renderman_property::add_renderman_property(k3d::inode& Object, k3d::icommand_node& Parent) :
	base(Object, Parent, "add_renderman_property"),
	m_parameter_type(new Gtk::ComboBox()),
	m_type(new Gtk::ComboBox()),
	m_parameter_types(Gtk::ListStore::create(m_parameter_type_columns)),
	m_types(Gtk::ListStore::create(m_type_columns)),
	m_list(init_value(std::string(""))),
	m_name(init_value(std::string(""))),
	m_label(init_value(std::string(""))),
	m_description(init_value(std::string("")))
{
	set_title(_("Add RenderMan Property:"));
	set_role("add_renderman_property");
	set_position(Gtk::WIN_POS_CENTER);

	add_parameter_type("attribute", _("Attribute"));
	add_parameter_type("option", _("Option"));
	
	add_type("integer", _("Integer"));
	add_type("real", _("Real"));
	add_type("string", _("String"));
	add_type("point", _("Point"));
	add_type("vector", _("Vector"));
	add_type("normal", _("Normal"));
	add_type("color", _("Color"));
	add_type("hpoint", _("HPoint"));
//	add_type("matrix", _("Matrix"));
		
	m_parameter_type->set_model(m_parameter_types);
	m_parameter_type->set_active(0);
	m_parameter_type->pack_start(m_parameter_type_columns.label);
	tooltips().set_tip(*m_parameter_type, _("Choose parameter type"));
	
	m_type->set_model(m_types);
	m_type->set_active(0);
	m_type->pack_start(m_type_columns.label);
	tooltips().set_tip(*m_type, _("Choose property type"));

	Gtk::Table* const table = Gtk::manage(new Gtk::Table(6, 2, false));

	table->attach(*Gtk::manage(new Gtk::Label(_("Parameter Type:"))), 0, 1, 0, 1, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(m_parameter_type), 1, 2, 0, 1);
	
	table->attach(*Gtk::manage(new Gtk::Label(_("Type:"))), 0, 1, 1, 2, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(m_type), 1, 2, 1, 2);

	table->attach(*Gtk::manage(new Gtk::Label(_("List:"))), 0, 1, 2, 3, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "list", entry::proxy(m_list))), 1, 2, 2, 3);
	
	table->attach(*Gtk::manage(new Gtk::Label(_("Name:"))), 0, 1, 3, 4, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "name", entry::proxy(m_name))), 1, 2, 3, 4);

	table->attach(*Gtk::manage(new Gtk::Label(_("Label:"))), 0, 1, 4, 5, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "label", entry::proxy(m_label))), 1, 2, 4, 5);

	table->attach(*Gtk::manage(new Gtk::Label(_("Description:"))), 0, 1, 5, 6, Gtk::SHRINK, Gtk::SHRINK);
	table->attach(*Gtk::manage(new entry::control(*this, "description", entry::proxy(m_description))), 1, 2, 5, 6);

	Gtk::HButtonBox* const box = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
	box->pack_start(*Gtk::manage(new button::control(*this, "cancel", _("Cancel"), Gtk::Stock::CANCEL)
		<< connect_button(sigc::mem_fun(*this, &add_renderman_property::close))));
	box->pack_start(*Gtk::manage(new button::control(*this, "add", _("Add"), Gtk::Stock::ADD)
		<< connect_button(sigc::mem_fun(*this, &add_renderman_property::on_add))));

	Gtk::VBox* const vbox = Gtk::manage(new Gtk::VBox(false, 10));
	vbox->set_border_width(10);
	vbox->pack_start(*Gtk::manage(new Gtk::Label(k3d::string_cast(boost::format(_("Add a RenderMan property to %1%:")) % node().name()))), Gtk::PACK_SHRINK);
	vbox->pack_start(*table, Gtk::PACK_SHRINK);
	vbox->pack_start(*box, Gtk::PACK_SHRINK);

	add(*vbox);

	show_all();
}

void add_renderman_property::add_parameter_type(const std::string& Type, const std::string& Label)
{
	Gtk::TreeModel::Row row = *m_parameter_types->append();
	row[m_parameter_type_columns.type] = Type;
	row[m_parameter_type_columns.label] = Label;
}

void add_renderman_property::add_type(const std::string& Type, const std::string& Label)
{
	Gtk::TreeModel::Row row = *m_types->append();
	row[m_type_columns.type] = Type;
	row[m_type_columns.label] = Label;
}

void add_renderman_property::on_add()
{
	k3d::iproperty_collection* const property_collection = dynamic_cast<k3d::iproperty_collection*>(&node());
	return_if_fail(property_collection);

	k3d::ipersistent_container* const persistent_container = dynamic_cast<k3d::ipersistent_container*>(&node());
	return_if_fail(persistent_container);

	const std::string ptype = m_parameter_type->get_active()->get_value(m_parameter_type_columns.type);
	const std::string type = m_type->get_active()->get_value(m_type_columns.type);
	const std::string list = m_list.value();
	const std::string name = m_name.value();
	const std::string label = m_label.value();
	const std::string description = m_description.value();

	k3d::irenderman_property::parameter_type_t parameter_type;
	if(ptype == "attribute")
		parameter_type = k3d::irenderman_property::ATTRIBUTE;
	else if(ptype == "option")
		parameter_type = k3d::irenderman_property::OPTION;
	else
	{
		k3d::log() << "unknown RenderMan parameter list type" << std::endl;
		return;
	}
	
	if(type == "integer")
		k3d::ri::create_property<k3d::ri::integer_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), 0);
	else if(type == "real")
		k3d::ri::create_property<k3d::ri::real_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), 0.0);
	else if(type == "string")
		k3d::ri::create_property<k3d::ri::string_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::ri::string(""));
	else if(type == "point")
		k3d::ri::create_property<k3d::ri::point_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::vector3(0, 0, 0));
	else if(type == "vector")
		k3d::ri::create_property<k3d::ri::vector_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::vector3(0, 0, 0));
	else if(type == "normal")
		k3d::ri::create_property<k3d::ri::normal_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::vector3(0, 0, 0));
	else if(type == "color")
		k3d::ri::create_property<k3d::ri::color_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::color(0, 0, 0));
	else if(type == "hpoint")
		k3d::ri::create_property<k3d::ri::hpoint_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, &node(), k3d::vector4(0, 0, 0, 0));
/*
	else if(type == "matrix")
		k3d::ri::create_property<k3d::ri::matrix_property>(parameter_type, list, name, label, description, node().document(), *property_collection, *persistent_container, node(), k3d::identity3D());
*/
	else		
		k3d::log() << error << "unknown RenderMan property type: [" << type << "]" << std::endl;

	close();
}

} // namespace libk3dngui

