// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your argument) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the public interface for the K-3D Next Generation User Interface (NGUI) plugin
		\author Tim Shead (tshead@k-3d.com)
*/

#include "application_state.h"
#include "button.h"
#include "document_state.h"
#include "file_selection.h"
#include "main_document_window.h"
#include "messages.h"
#include "module.h"
#include "open_uri.h"
#include "splash_box.h"
#include "tutorial_menu.h"
#include "tutorial_message.h"
#include "tutorial_recorder.h"
#include "user_interface.h"
#include "utility.h"
#include "widget_manip.h"
#include "window.h"

#include <k3dsdk/application.h>
#include <k3dsdk/command_node.h>
#include <k3dsdk/command_tree.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/data.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icommand_tree.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/iuser_interface.h>
#include <k3dsdk/iuser_interface_plugin.h>
#include <k3dsdk/log.h>
#include <k3dsdk/module.h>
#include <k3dsdk/result.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/system.h>
#include <k3dsdk/version.h>

#include <glibmm/main.h>

#include <gtkmm/accelkey.h>
#include <gtkmm/accelmap.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/label.h>
#include <gtkmm/main.h>
#include <gtkmm/settings.h>
#include <gtkmm/stock.h>
#include <gtkmm/window.h>

#include <gtk/gtkgl.h>

#include <boost/date_time/gregorian/gregorian.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

#include <k3dsdk/fstream.h>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/format.hpp>

#include <iostream>

namespace libk3dngui
{

namespace detail
{

void handle_error(const std::string& Message, bool& Quit, bool& Error)
{
	k3d::log() << error << Message << std::endl;

	Quit = true;
	Error = true;
}

const boost::filesystem::path hotkey_path()
{
	return k3d::system::get_home_directory() / boost::filesystem::path(".k3d/hotkeys", boost::filesystem::native);
}

void setup_default_hotkeys()
{
	Gtk::AccelMap::add_entry("<k3d-document>/actions/create/PolyCube", Gtk::AccelKey("F5").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/create/PolyCylinder", Gtk::AccelKey("F6").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/create/PolyGrid", Gtk::AccelKey("F8").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/create/PolySphere", Gtk::AccelKey("F7").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/delete", Gtk::AccelKey("Delete").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/duplicate", Gtk::AccelKey("d").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/instantiate", Gtk::AccelKey("d").get_key(), Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/redo", Gtk::AccelKey("z").get_key(), Gdk::CONTROL_MASK | Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/tools/move_tool", Gtk::AccelKey("w").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/tools/parent_tool", Gtk::AccelKey("p").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/tools/rotate_tool", Gtk::AccelKey("e").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/tools/scale_tool", Gtk::AccelKey("r").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/tools/select_tool", Gtk::AccelKey("q").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/tools/unparent", Gtk::AccelKey("p").get_key(), Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/edit/undo", Gtk::AccelKey("z").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/file/close", Gtk::AccelKey("w").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/file/new", Gtk::AccelKey("n").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/file/open", Gtk::AccelKey("o").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/file/quit", Gtk::AccelKey("q").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/file/save", Gtk::AccelKey("s").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/file/save_as", Gtk::AccelKey("s").get_key(), Gdk::CONTROL_MASK | Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/help/manual", Gtk::AccelKey("F1").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/layout/fullscreen", Gtk::AccelKey("F11").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/Delete", Gtk::AccelKey("BackSpace").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/Dissolve", Gtk::AccelKey("BackSpace").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/ExtrudeFaces", Gtk::AccelKey("m").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/FlipOrientation", Gtk::AccelKey("i").get_key(), Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/GrowSelection", Gtk::AccelKey("equal").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/SelectEdgeLoops", Gtk::AccelKey("semicolon").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/SelectEdgeRings", Gtk::AccelKey("l").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/SubdivideEdges", Gtk::AccelKey("k").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/modify/SubdivideFaces", Gtk::AccelKey("k").get_key(), Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/render/render_frame", Gtk::AccelKey("F12").get_key(), Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/render/render_preview", Gtk::AccelKey("F12").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/render/render_region", Gtk::AccelKey("F12").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_all", Gtk::AccelKey("a").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_child", Gtk::AccelKey("bracketleft").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_faces", Gtk::AccelKey("4").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_invert", Gtk::AccelKey("i").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_lines", Gtk::AccelKey("3").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_nodes", Gtk::AccelKey("1").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_none", Gtk::AccelKey("space").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_parent", Gtk::AccelKey("p").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_points", Gtk::AccelKey("2").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/select/select_sibling", Gtk::AccelKey("bracketright").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/aim_selection", Gtk::AccelKey("a").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/assign_hotkeys", Gtk::AccelKey("Insert").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/frame_selection", Gtk::AccelKey("f").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/hide_selection", Gtk::AccelKey("h").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/hide_unselected", Gtk::AccelKey("h").get_key(), Gdk::SHIFT_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/set_camera", Gtk::AccelKey("c").get_key(), Gdk::ModifierType(0));
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/show_all", Gtk::AccelKey("h").get_key(), Gdk::CONTROL_MASK);
	Gtk::AccelMap::add_entry("<k3d-document>/actions/view/show_selection", Gtk::AccelKey("h").get_key(), Gdk::CONTROL_MASK | Gdk::SHIFT_MASK);
}

const std::string black_box_recorder_started = "Black Box Recorder Started:";
const std::string black_box_recorder_stopped = "Black Box Recorder Stopped:";

/////////////////////////////////////////////////////////////////////////////
// assign_hotkeys_popup

class assign_hotkeys_popup :
	public window
{
	typedef window base;
public:
	assign_hotkeys_popup() :
		base("assign_hotkeys", 0)
	{
		set_title(_("Assign Hotkeys:"));
		set_role("assign_hotkeys");
		set_keep_above(true);

		Gtk::HButtonBox* const box2 = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
		box2->pack_start(*Gtk::manage(
			new button::control(*this, "close", Gtk::Stock::CLOSE)
			<< connect_button(sigc::mem_fun(*this, &assign_hotkeys_popup::safe_close))
			), Gtk::PACK_SHRINK);

		Gtk::VBox* const box1 = new Gtk::VBox(false, 10);
		box1->set_border_width(10);
		box1->pack_start(*Gtk::manage(new Gtk::Label() << line_wrap() << center_justify() << set_markup(_("<big><b>Assign Hotkeys Mode</b></big>"))));
		box1->pack_start(*Gtk::manage(new Gtk::Label(_("To assign hotkeys, hover the mouse over a menu item, and hit the desired hotkey combination.  Use \"delete\" to remove hotkeys.  Close this window to turn-off Assign Hotkeys Mode.")) << line_wrap() << center_justify()));
		box1->pack_start(*Gtk::manage(box2));

		add(*Gtk::manage(box1));

		show_all();

		application_state().assign_hotkeys().changed_signal().connect(sigc::mem_fun(*this, &assign_hotkeys_popup::on_assign_hotkeys));
	}

	void on_assign_hotkeys()
	{
		if(!application_state().assign_hotkeys().value())
			close();
	}

	bool on_safe_to_close()
	{
		application_state().assign_hotkeys().set_value(false);
		return false;
	}
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// user_interface_implementation

class user_interface_implementation :
	public k3d::command_node::implementation,
	public k3d::iuser_interface,
	public k3d::iuser_interface_plugin,
	public k3d::ideletable,
	public user_interface,
	public sigc::trackable
{
	typedef k3d::command_node::implementation base;

public:
	user_interface_implementation() :
		base("ui", 0),
		m_show_tutorials(false),
		m_record_tutorials(false),
		m_bbr(init_value(true)),
		m_bbr_script_engine(0)
	{
		/// Redirect glib-based logging to our own standard logging mechanism
		g_log_set_handler("", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("Gdk", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("GdkPixbuf", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("Glib", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("Glib-GObject", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("Gtk", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("atkmm", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("gdkmm", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("glibmm", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("gtkmm", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);
		g_log_set_handler("pangomm", static_cast<GLogLevelFlags>(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION), log_handler, 0);

		application_state().assign_hotkeys().changed_signal().connect(sigc::mem_fun(*this, &user_interface_implementation::on_assign_hotkeys));
		m_bbr.changed_signal().connect(sigc::mem_fun(*this, &user_interface_implementation::on_black_box_recorder));
	}

	~user_interface_implementation()
	{
		const boost::filesystem::path hotkey_path = detail::hotkey_path();
		boost::filesystem::create_directories(hotkey_path.branch_path());
		k3d::log() << info << "Saving hotkeys to " << hotkey_path.native_file_string() << std::endl;
		Gtk::AccelMap::save(hotkey_path.native_file_string());

		stop_black_box_recorder();
	}

	static void log_handler(const gchar* LogDomain, GLogLevelFlags LogLevel, const gchar* Message, gpointer UserData)
	{
		std::string message(Message ? Message : "");
		std::replace(message.begin(), message.end(), '\n', ' ');
		std::replace(message.begin(), message.end(), '\r', ' ');

		switch(LogLevel)
		{
			case G_LOG_LEVEL_ERROR:
				k3d::log() << critical << LogDomain << ": " << message << std::endl;
				break;
			case G_LOG_LEVEL_CRITICAL:
				k3d::log() << error << LogDomain << ": " << message << std::endl;
				break;
			case G_LOG_LEVEL_WARNING:
				k3d::log() << warning << LogDomain << ": " << message << std::endl;
				break;
			case G_LOG_LEVEL_MESSAGE:
			case G_LOG_LEVEL_INFO:
				k3d::log() << info << LogDomain << ": " << message << std::endl;
				break;
			case G_LOG_LEVEL_DEBUG:
				k3d::log() << debug << LogDomain << ": " << message << std::endl;
				break;
			default:
				k3d::log() << LogDomain << ": " << message << std::endl;
				break;
		}
	}

	const arguments_t parse_command_line(const arguments_t& Arguments, const boost::filesystem::path& SharePath, bool& Quit, bool& Error)
	{
		// Store the share path where we keep our resources ...
		m_share_path = SharePath;

		// Keep track of whether to display a splash screen or not ...
		bool show_splash = true;

		// We want GTK to parse an optional RC file from the user's home directory ...
		const boost::filesystem::path rc_file = k3d::system::get_home_directory() / boost::filesystem::path(".k3d/gtkrc", boost::filesystem::native);
		boost::filesystem::create_directories(rc_file.branch_path());
		if(!boost::filesystem::exists(rc_file))
		{
			k3d::filesystem::ofstream stream(rc_file);
			stream << "# You can add your own K-3D-specific GTK styles here.\n\n";

			stream << "# Un-comment this to modify icon sizes.\n";
			stream << "# gtk-icon-sizes = \"mini-commander-icon=24,24:print-manager=64,64:panel-button=32,32:gtk-dnd=48,48:gtk-menu=32,32:panel-menu=48,48:gtk-large-toolbar=48,48:gtk-small-toolbar=32,32:gtk-button=32,32:gtk-dialog=64,64\"\n\n";

			stream << "# Add styles that will apply to all K-3D widgets here.\n";
			stream << "style \"k3d-style\"\n";
			stream << "{\n\n";

			stream << "}\n\n";

			stream << "class \"*\" style \"k3d-style\"\n";
		}
		if(boost::filesystem::exists(rc_file))
		{
			Gtk::RC::add_default_file(rc_file.native_file_string());
		}
		else
		{
			k3d::log() << warning << "Could not locate or create " << rc_file.native_file_string() << " resource file" << std::endl;
		}

		const Glib::StringArrayHandle default_files = Gtk::RC::get_default_files();
		for(Glib::StringArrayHandle::const_iterator file = default_files.begin(); file != default_files.end(); ++file)
			k3d::log() << info << "Loading GTK resources from " << *file << std::endl;

		// gtkmm expects to parse and modify argc / argv, so create some temporary storage for it to fiddle with ...
		std::vector<char*> argv_buffer;
		argv_buffer.push_back("k3d");
		for(arguments_t::const_iterator argument = Arguments.begin(); argument != Arguments.end(); ++argument)
			argv_buffer.push_back(const_cast<char*>(argument->c_str()));
		int argc = argv_buffer.size();
		char** argv = &argv_buffer[0];

		m_main.reset(new Gtk::Main(argc, argv));

		// Give gtkglext a chance at the startup arguments ...
		if(!gtk_gl_init_check(&argc, &argv))
		{
			detail::handle_error("Could not initialize gtkglext", Quit, Error);
			return arguments_t();
		}

		// We return any "unused" arguments ...
		arguments_t unused;

		// For each command-line argument ...
		for(arguments_t::const_iterator argument = Arguments.begin(); argument != Arguments.end(); ++argument)
		{
			if((*argument) == "--batch")
			{
				application_state().batch_mode() = true;
			}
			else if((*argument) == "--no-splash")
			{
				show_splash = false;
			}
			else if((*argument) == "--tutorials")
			{
				++argument;
				if(argument == Arguments.end())
				{
					detail::handle_error("You must supply a directory path with the --tutorials argument!", Quit, Error);
					return arguments_t();
				}

				m_tutorials_path = boost::filesystem::path(*argument, boost::filesystem::native);
			}
			else if((*argument) == "--show-tutorials")
			{
				m_show_tutorials = true;
			}
			else if((*argument) == "--record-tutorials")
			{
				m_record_tutorials = true;
			}
			else
			{
				unused.push_back(*argument);
			}
		}

		Gtk::Window::set_default_icon(load_pixbuf(m_share_path / "ngui/pixmap", "default_icon.png"));

		if(show_splash)
			m_splash_box.reset(new splash_box(m_share_path / "ngui/pixmap"));

		return unused;
	}

	void startup_message_handler(const std::string& Message)
	{
		if(m_splash_box.get())
			m_splash_box->on_startup_message(Message);
	}

	void display_user_interface()
	{
		detail::setup_default_hotkeys();

		const boost::filesystem::path hotkey_path = detail::hotkey_path();
		k3d::log() << info << "Loading hotkeys from " << hotkey_path.native_file_string() << std::endl;
		Gtk::AccelMap::load(hotkey_path.native_file_string());

		k3d::application().close_signal().connect(sigc::mem_fun(*this, &user_interface_implementation::on_application_close));

		m_splash_box.reset();

		k3d::idocument* const document = k3d::application().create_document();
		return_if_fail(document);
		
		populate_new_document(*document);

		document_state* const state = new document_state(*document);
		new main_document_window(*state, *this);

		if(m_show_tutorials)
			create_tutorial_menu();

		if(m_record_tutorials)
			create_tutorial_recorder();

		k3d::command_tree().command_signal().connect(sigc::mem_fun(*this, &user_interface_implementation::on_command));
		if(m_bbr.value())
			start_black_box_recorder();
	}

	void start_event_loop()
	{
		m_main->run();
	}

	void stop_event_loop()
	{
		m_main->quit();
	}

	void on_application_close()
	{
	}

	bool batch_mode()
	{
		return application_state().batch_mode();
	}

	void browser_navigate(const std::string& URL)
	{
		return_if_fail(libk3dngui::open_uri(URL));
	}

	void message(const std::string& Message, const std::string& Title)
	{
		libk3dngui::message(Message, Title);
	}

	void error_message(const std::string& Message, const std::string& Title)
	{
		libk3dngui::error_message(Message, Title);
	}

	unsigned int query_message(const std::string& Message, const std::string& Title, const unsigned int DefaultButton, const std::vector<std::string> Buttons)
	{
		return libk3dngui::query_message(Message, Title, DefaultButton, Buttons);
	}

	bool tutorial_message(const std::string& Message)
	{
		k3d::command_tree().command_signal().emit(*this, k3d::icommand_node::INTERACTIVE, "tutorial_message", Message);
		return libk3dngui::tutorial_message(Message);
	}

	void tutorial_mouse_message(const std::string& Message, const mouse_action_t Action, const k3d::key_modifiers Modifiers)
	{
		libk3dngui::tutorial_mouse_message(Message, Action, Modifiers);
	}

	bool get_file_path(const k3d::ipath_property::mode_t Mode, const std::string& Type, const std::string& Prompt, const boost::filesystem::path& OldPath, boost::filesystem::path& Result)
	{
		return libk3dngui::get_file_path(Mode, Type, Prompt, OldPath, Result);
	}

	bool show(iunknown& Object)
	{
		k3d::log() << error << k3d_file_reference << ": not implemented!" << std::endl;
		return false;
	}

	sigc::connection get_timer(const double FrameRate, sigc::slot0<void> Slot)
	{
		return_val_if_fail(FrameRate != 0.0, sigc::connection());

		const unsigned int interval = static_cast<unsigned int>(1000.0 / FrameRate);
		return Glib::signal_timeout().connect(sigc::bind_return(Slot, true), interval);
	}

	bool execute_command(const std::string& Command, const std::string& Arguments)
	{
		if(Command == "tutorial_message")
			return tutorial_message(Arguments);

		return base::execute_command(Command, Arguments);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<user_interface_implementation>,
			k3d::interface_list<k3d::iuser_interface_plugin> > factory(
			get_class_id(),
			"NextGenerationUI",
			"Next Generation User Interface (NGUI)",
			"");

		return factory;
	}

	k3d_data(bool, no_name, change_signal, no_undo, local_storage, no_constraint, no_property, no_serialization)& black_box_recorder()
	{
		return m_bbr;
	}

private:
	void on_black_box_recorder()
	{
		if(m_bbr.value())
		{
			start_black_box_recorder();
			return;
		}

		if(!application_state().batch_mode())
		{
			std::vector<std::string> buttons;
			buttons.push_back(_("OK"));
			buttons.push_back(_("Cancel"));

			if(1 != query_message(_("Disable Black Box Recorder?  The Black Box Recorder is a powerful tool for reporting bugs to the K-3D development team."), _("Disable Black Box Recorder?"), 1, buttons))
			{
				m_bbr.set_value(true);
				return;
			}
		}

		stop_black_box_recorder();
	}

	void start_black_box_recorder()
	{
		if(m_bbr_log_stream.is_open())
			return;

		if(!m_bbr_script_engine)
			m_bbr_script_engine = dynamic_cast<k3d::iscript_engine*>(k3d::create_plugin(k3d::classes::K3DScriptEngine()));
		return_if_fail(m_bbr_script_engine);

		const boost::filesystem::path bbr_log_path = k3d::system::get_home_directory() / boost::filesystem::path(".k3d/blackbox.log", boost::filesystem::native);
		if(boost::filesystem::exists(bbr_log_path))
		{
			std::stringstream buffer;
			k3d::filesystem::ifstream stream(bbr_log_path);
			buffer << stream.rdbuf();
			stream.close();

			if(buffer.str().find(detail::black_box_recorder_started) != std::string::npos &&
				buffer.str().rfind(detail::black_box_recorder_stopped) == std::string::npos)
			{
				unsigned long i = 1;
				boost::filesystem::path storage = bbr_log_path;
				while(boost::filesystem::exists(storage))
					storage = k3d::system::get_home_directory() / boost::filesystem::path(".k3d/blackbox.log." + k3d::string_cast(i++), boost::filesystem::native);

				try
				{
					boost::filesystem::rename(bbr_log_path, storage);
				}
				catch(std::exception& e)
				{
					k3d::log() << error << "Error archiving blackbox log: " << bbr_log_path.native_file_string() << ": " << e.what() << std::endl;
				}

				if(!application_state().batch_mode())
				{
					std::string command_line;
					command_line += "k3d-bug-buddy";
					command_line += " \"" + storage.native_file_string() + "\"";
					if(!k3d::system::run_process(command_line))
					{
						// We fall-back on our old nag message if bug-buddy can't be run for any reason ...
						boost::format warning_message(_("It appears that a previous K-3D session may have terminated unexpectedly.  A logfile has been saved as\n\n%1%\n\nPlease send this file to the K-3D development team for analysis."));
						warning_message % storage.native_file_string();
						message(warning_message.str(), _("Black Box Recorder:"));
					}
				}
			}
		}

		m_bbr_log_stream.open(bbr_log_path);
		return_if_fail(m_bbr_log_stream.good());

		std::ostringstream buffer;
		buffer << detail::black_box_recorder_started << " " << boost::posix_time::second_clock::universal_time() << " UTC\n";
		buffer << "Package: " << K3D_PACKAGE << "\n";
		buffer << "Version: " << K3D_VERSION << "\n";
		buffer << "Platform: " << K3D_HOST << "\n";
		buffer << "Compiler: " << __VERSION__ << "\n";
		buffer << "Build Time: " << __DATE__ << " " << __TIME__ << " local\n";

		m_bbr_script_engine->bless_script(m_bbr_log_stream);
		m_bbr_script_engine->append_comment(m_bbr_log_stream, buffer.str());
		m_bbr_log_stream.flush();
	}

	void stop_black_box_recorder()
	{
		if(!m_bbr_log_stream.is_open())
			return;

		std::ostringstream buffer;
		buffer << detail::black_box_recorder_stopped << " " << boost::posix_time::second_clock::universal_time() << " UTC\n";

		m_bbr_script_engine->append_comment(m_bbr_log_stream, buffer.str());
		m_bbr_log_stream.flush();
		m_bbr_log_stream.close();
	}

	void on_command(k3d::icommand_node& CommandNode, const k3d::icommand_node::type Type, const std::string& Name, const std::string& Arguments)
	{
		// We only want user interface commands ... !
		if(Type != k3d::icommand_node::INTERACTIVE)
			return;

		// And only if recording is enabled ...
		if(!m_bbr.value())
			return;

		// Sanity checks ...
		return_if_fail(m_bbr_script_engine);
		return_if_fail(m_bbr_log_stream.is_open());

		m_bbr_script_engine->append_command(m_bbr_log_stream, CommandNode, Name, Arguments);
		m_bbr_log_stream.flush();
	}

	void on_assign_hotkeys()
	{
		if(application_state().assign_hotkeys().value())
		{
			new detail::assign_hotkeys_popup();
			Gtk::Settings::get_default()->property_gtk_can_change_accels() = true;
		}
		else
		{
			Gtk::Settings::get_default()->property_gtk_can_change_accels() = false;
		}
	}

	/// Set to true iff we should display the tutorial menu at startup
	bool m_show_tutorials;
	/// Set to true iff we should begin recording a tutorial immediately at startup
	bool m_record_tutorials;
	/// Stores the share path where resources are stored
	boost::filesystem::path m_share_path;
	/// Stores the path where tutorials should be displayed
	boost::filesystem::path m_tutorials_path;
	/// Stores the main loop
	std::auto_ptr<Gtk::Main> m_main;
	/// Stores the (optional) splash screen
	std::auto_ptr<splash_box> m_splash_box;
	/// Set to true while the black box recorder is enabled
	k3d_data(bool, no_name, change_signal, no_undo, local_storage, no_constraint, no_property, no_serialization) m_bbr;
	/// Stream for recording black box data
	k3d::filesystem::ofstream m_bbr_log_stream;
	/// Script engine for recording black box data
	k3d::iscript_engine* m_bbr_script_engine;
};

/////////////////////////////////////////////////////////////////////////////
// user_interface_factory

k3d::iplugin_factory& user_interface_factory()
{
	return user_interface_implementation::get_factory();
}

} // namespace libk3dngui

