// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Provides a UI for recording interactive tutorials
		\author Tim Shead (tshead@k-3d.com)
*/

#include "application_state.h"
#include "button.h"
#include "check_menu_item.h"
#include "file_selection.h"
#include "icons.h"
#include "image_menu_item.h"
#include "image_toggle_button.h"
#include "menubar.h"
#include "menu_item.h"
#include "messages.h"
#include "scripting.h"
#include "tutorial_recorder.h"
#include "toolbar.h"
#include "utility.h"
#include "widget_manip.h"
#include "window.h"

#include <k3dsdk/application.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/command_tree.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/data.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/iuser_interface.h>
#include <k3dsdk/options.h>
#include <k3dsdk/scripting.h>
#include <k3dsdk/string_cast.h>

#include <boost/filesystem/path.hpp>

#include <gtkmm/box.h>
#include <gtkmm/paned.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/textview.h>

#include <sstream>

namespace libk3dngui
{

class tutorial_recorder :
	public window
{
	typedef window base;

public:
	tutorial_recorder() :
		base("tutorial_recorder", 0),
		m_script_engine(dynamic_cast<k3d::iscript_engine*>(k3d::create_plugin(k3d::classes::K3DScriptEngine()))),
		m_unsaved_changes(false),
		m_recording(init_name("recording") + init_label(_("Recording")) + init_description(_("Tells whether the tutorial recorder records user actions")) + init_value(true)),
		m_running(false)
	{
		assert_warning(m_script_engine);

		k3d::command_tree().command_signal().connect(sigc::mem_fun(*this, &tutorial_recorder::on_command));
		m_recording.changed_signal().connect(sigc::mem_fun(*this, &tutorial_recorder::on_edit_recording));

		menubar::control* const menubar = new menubar::control(*this, "menus");

		Gtk::Menu* const menu_file = new Gtk::Menu();
		menu_file->items().push_back(*Gtk::manage(connect(new image_menu_item::control(*menubar, "file_new", Gtk::Stock::NEW), sigc::mem_fun(*this, &tutorial_recorder::on_file_new))));
		menu_file->items().push_back(*Gtk::manage(connect(new image_menu_item::control(*menubar, "file_open", Gtk::Stock::OPEN), sigc::mem_fun(*this, &tutorial_recorder::on_file_open))));
		menu_file->items().push_back(Gtk::Menu_Helpers::SeparatorElem());
		menu_file->items().push_back(*Gtk::manage(connect(new image_menu_item::control(*menubar, "file_save", Gtk::Stock::SAVE), sigc::mem_fun(*this, &tutorial_recorder::on_file_save))));
		menu_file->items().push_back(*Gtk::manage(connect(new image_menu_item::control(*menubar, "file_save_as", Gtk::Stock::SAVE_AS), sigc::mem_fun(*this, &tutorial_recorder::on_file_save_as))));
		menu_file->items().push_back(*Gtk::manage(connect(new image_menu_item::control(*menubar, "file_revert", Gtk::Stock::REVERT_TO_SAVED), sigc::mem_fun(*this, &tutorial_recorder::on_file_revert))));
		menu_file->items().push_back(Gtk::Menu_Helpers::SeparatorElem());
		menu_file->items().push_back(*Gtk::manage(connect(new image_menu_item::control(*menubar, "file_close", Gtk::Stock::CLOSE), sigc::mem_fun(*this, &tutorial_recorder::safe_close))));

		Gtk::Menu* const menu_edit = new Gtk::Menu();
		menu_edit->items().push_back(*Gtk::manage(new check_menu_item::control(*menubar, "edit_recording", check_menu_item::proxy(m_recording), _("Recording"))));
		menu_edit->items().push_back(*Gtk::manage(
			new menu_item::control(*menubar, "edit_record_message", _("Record Message")) <<
			connect_menu_item(sigc::mem_fun(*this, &tutorial_recorder::on_edit_record_message))));

		menu_edit->items().push_back(*Gtk::manage(
			new menu_item::control(*menubar, "edit_stop_recording", _("Stop Recording")) <<
			connect_menu_item(sigc::mem_fun(*this, &tutorial_recorder::on_edit_stop_recording))));

		menu_edit->items().push_back(Gtk::Menu_Helpers::SeparatorElem());

		menu_edit->items().push_back(*Gtk::manage(
			new menu_item::control(*menubar, "edit_play", _("Play")) <<
			connect_menu_item(sigc::mem_fun(*this, &tutorial_recorder::on_edit_play))));

		menu_edit->items().push_back(*Gtk::manage(
			new menu_item::control(*menubar, "edit_play_from_cursor", _("Play From Cursor")) <<
			connect_menu_item(sigc::mem_fun(*this, &tutorial_recorder::on_edit_play_from_cursor))));

		menubar->items().push_back(Gtk::Menu_Helpers::MenuElem(_("_File"), *manage(menu_file)));
		menubar->items().push_back(Gtk::Menu_Helpers::MenuElem(_("_Edit"), *manage(menu_edit)));

		toolbar::control* const toolbar = new toolbar::control(*this, "toolbar");
		toolbar->row(0).pack_start(*Gtk::manage(
			new image_toggle_button::control(*toolbar, "record",
				image_toggle_button::proxy(m_recording),
				load_icon("record", Gtk::ICON_SIZE_BUTTON),
				load_icon("record", Gtk::ICON_SIZE_BUTTON)) <<
			make_toolbar_button()), Gtk::PACK_SHRINK);

		toolbar->row(0).pack_start(*Gtk::manage(
			new button::control(*toolbar, "stop_recording",
				*Gtk::manage(new Gtk::Image(load_icon("stop", Gtk::ICON_SIZE_BUTTON)))) <<
			connect_button(sigc::mem_fun(*this, &tutorial_recorder::on_edit_stop_recording)) <<
			make_toolbar_button()), Gtk::PACK_SHRINK);

		toolbar->row(0).pack_start(*Gtk::manage(
			new button::control(*toolbar, "play",
				*Gtk::manage(new Gtk::Image(load_icon("play", Gtk::ICON_SIZE_BUTTON)))) <<
			connect_button(sigc::mem_fun(*this, &tutorial_recorder::on_edit_play)) <<
			make_toolbar_button()), Gtk::PACK_SHRINK);

		toolbar->row(0).pack_start(*Gtk::manage(
			new button::control(*toolbar, "play_from_cursor",
				*Gtk::manage(new Gtk::Image(load_icon("play_from_cursor", Gtk::ICON_SIZE_BUTTON)))) <<
			connect_button(sigc::mem_fun(*this, &tutorial_recorder::on_edit_play_from_cursor)) <<
			make_toolbar_button()), Gtk::PACK_SHRINK);

		toolbar->row(0).pack_start(*Gtk::manage(
			new button::control(*toolbar, "record_message", _("Record Message")) <<
			connect_button(sigc::mem_fun(*this, &tutorial_recorder::on_edit_record_message)) <<
			make_toolbar_button()), Gtk::PACK_SHRINK);

		Gtk::HBox* const hbox1 = new Gtk::HBox(false);
		hbox1->pack_start(*Gtk::manage(menubar), Gtk::PACK_SHRINK);
		hbox1->pack_start(*Gtk::manage(toolbar), Gtk::PACK_SHRINK);

		m_script.set_wrap_mode(Gtk::WRAP_CHAR);

		Gtk::ScrolledWindow* const scrolled_window = new Gtk::ScrolledWindow();
		scrolled_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		scrolled_window->add(m_script);

		Gtk::VPaned* const vpaned = new Gtk::VPaned();
		vpaned->pack1(m_message);
		vpaned->pack2(*Gtk::manage(scrolled_window));

		Gtk::VBox* const vbox1 = new Gtk::VBox(false);
		vbox1->pack_start(*Gtk::manage(hbox1), Gtk::PACK_SHRINK);
		vbox1->pack_start(*Gtk::manage(vpaned), Gtk::PACK_EXPAND_WIDGET);

		add(*Gtk::manage(vbox1));
		set_role("tutorial_recorder");
		resize(600, 300);

		file_new();
		show_all();

		m_script.get_buffer()->signal_changed().connect(sigc::mem_fun(*this, &tutorial_recorder::on_script_changed));
	}

private:
	~tutorial_recorder()
	{
		delete dynamic_cast<k3d::ideletable*>(m_script_engine);
	}

	void on_file_new()
	{
		if(!safe_to_close())
			return;

		file_new();
	}

	void on_file_open()
	{
		if(!safe_to_close())
			return;

		boost::filesystem::path filepath;
		if(!get_file_path(k3d::ipath_property::READ, k3d::options::path::tutorials(), _("Open File:"), m_path, filepath))
			return;

		file_open(filepath);
	}

	void on_file_save()
	{
		file_save();
	}

	void on_file_save_as()
	{
		file_save_as();
	}

	void on_file_revert()
	{
		if(!safe_to_close())
			return;

		if(m_path.empty())
			file_new();
		else
			file_open(m_path);
	}

	void on_edit_recording()
	{
		update_title();
	}

	void on_edit_record_message()
	{
		k3d::application().user_interface().tutorial_message(m_message.get_buffer()->get_text());
	}

	void on_edit_stop_recording()
	{
		m_recording.set_value(false);
	}

	void on_edit_play()
	{
		on_edit_stop_recording();

		m_running = true;
		update_title();

		std::istringstream script(m_script.get_buffer()->get_text());
		k3d::iscript_engine::context_t context;
		execute_script(script, get_title(), context);

		m_running = false;
		update_title();
	}

	void on_edit_play_from_cursor()
	{
		on_edit_stop_recording();
		m_running = true;
		update_title();

		// We have to get the entire script first, so we can identify the correct scripting engine ...
		Glib::RefPtr<Gtk::TextBuffer> buffer = m_script.get_buffer();
		std::istringstream script(std::string(buffer->get_text()));
		k3d::iplugin_factory* const language = k3d::recognize_script_language(script);
		if(language)
			{
				std::istringstream script(std::string(buffer->get_text(buffer->get_iter_at_mark(buffer->get_insert()), buffer->end())));
				k3d::iscript_engine::context_t context;
				libk3dngui::execute_script(script, get_title(), context, language->class_id());
			}
		else
			{
				error_message(
					_("Could not determine scripting language.  K-3D supports multiple scripting languages, but the language for this script was\n"
					"not recognized. Most K-3D script engines use some type of \"magic token\" at the beginning of a script to recognize it,\n"
					"e.g. \"#k3dscript\" in the first 12 characters of a script for K-3D's built-in K3DScript engine.  If you are writing a K-3D script,\n"
					"check the documentation for the scripting language you're writing in to see how to make it recognizable."),
					k3d::string_cast(boost::format(_("Play %1%:")) % get_title()));
			}

		m_running = false;
		update_title();
	}

	void on_script_changed()
	{
		m_unsaved_changes = true;
		update_title();
	}

	void file_new()
	{
		return_if_fail(m_script_engine);

		m_message.get_buffer()->set_text(_("-- Enter a message to be recorded --"));

		std::stringstream script;
		m_script_engine->bless_script(script);
		m_script.get_buffer()->set_text(script.str());

		m_path = boost::filesystem::path();
		m_unsaved_changes = false;
		update_title();
	}

	void file_open(const boost::filesystem::path& Path)
	{
		k3d::filesystem::ifstream stream(Path);

		std::stringstream script;
	        stream.get(*script.rdbuf(), '\0');
		m_script.get_buffer()->set_text(script.str());

		m_path = Path;
		m_unsaved_changes = false;
		update_title();
	}

	bool file_save()
	{
		if(m_path.empty())
			return file_save_as();

		k3d::filesystem::ofstream stream(m_path);
		stream << m_script.get_buffer()->get_text();

		m_unsaved_changes = false;
		update_title();
		return true;
	}

	bool file_save_as()
	{
		if(!get_file_path(k3d::ipath_property::WRITE, k3d::options::path::tutorials(), _("Save Tutorial As:"), m_path, m_path))
			return false;

		k3d::filesystem::ofstream stream(m_path);
		stream << m_script.get_buffer()->get_text();

		m_unsaved_changes = false;
		update_title();
		return true;
	}

	void on_command(k3d::icommand_node& CommandNode, const k3d::icommand_node::type Type, const std::string& Command, const std::string& Arguments)
	{
		// Sanity checks ...
		return_if_fail(m_script_engine);
		return_if_fail(Command.size());

		// If we aren't recording ...
		if(!m_recording.value())
			return;

		// Skip everything but UI events ...
		if(Type != k3d::icommand_node::INTERACTIVE)
			return;

		// Don't record our own events ...
		if(k3d::command_node::is_descendant(this, &CommandNode))
			return;

		std::stringstream buffer;
		m_script_engine->append_command(buffer, CommandNode, Command, Arguments);

		m_script.get_buffer()->insert(m_script.get_buffer()->end(), buffer.str());
		Gtk::TextBuffer::iterator end = m_script.get_buffer()->end();
		m_script.scroll_to(end, 0.0);

		m_unsaved_changes = true;
	}

	bool on_safe_to_close()
	{
		// If we're in batch mode, go for it!
		if(application_state().batch_mode())
			return true;

		// If we don't have any unsaved changes, go for it!
		if(!m_unsaved_changes)
			return true;

		// Prompt the user to save changes ...
		std::vector<std::string> buttons;
		buttons.push_back(_("Save Changes"));
		buttons.push_back(_("Discard Changes"));
		buttons.push_back(_("Cancel"));
		std::string message = k3d::string_cast(boost::format(_("Save %1% before proceeding? Unsaved changes will be lost (No Undo)")) % get_script_title());

		const int choice = query_message(message, get_script_title(), 1, buttons);
		switch(choice)
			{
				case 0:
					return false;
				case 1:
					return file_save();
				case 2:
					return true;
				case 3:
					return false;
			}

		return false;
	}

	std::string get_script_title()
	{
		return m_path.empty() ? _("Untitled Tutorial") : m_path.leaf();
	}

	void update_title()
	{
		std::string title = get_script_title();
		if(m_unsaved_changes)
			title += _(" [changed]");
		if(m_recording.value())
			title += _(" [recording]");
		if(m_running)
			title += _(" [running]");

		set_title(title);
	}

	/// Script engine for the script being recorded
	k3d::iscript_engine* const m_script_engine;
	/// Stores the file path (could be empty)
	boost::filesystem::path m_path;
	/// Set to true iff there are unsaved changes
	bool m_unsaved_changes;
	/// Set to true iff recording is in progress
	k3d_data(bool, immutable_name, change_signal, no_undo, local_storage, no_constraint, no_property, no_serialization) m_recording;
	/// Set to true iff tutorial playback is in progress
	bool m_running;
	/// Stores an interactive message
	Gtk::TextView m_message;
	/// Stores the script being recorded
	Gtk::TextView m_script;
};

/*
void tutorial_recorder::on_record()
{
	m_recording = ToggleButton(control_record).GetState();

	if(m_recording)
		{
			Widget(control_stop).SetSensitive(true);
			Widget(control_play).SetSensitive(false);
			Widget(control_play_from_cursor).SetSensitive(false);
			Widget(control_recordmessage).SetSensitive(true);
		}
	else
		{
			Widget(control_stop).SetSensitive(false);
			Widget(control_play).SetSensitive(true);
			Widget(control_play_from_cursor).SetSensitive(true);
			Widget(control_recordmessage).SetSensitive(false);
		}

	update_titlebar();
}
*/

void create_tutorial_recorder()
{
	new tutorial_recorder();
}

} // namespace libk3dngui


