#ifndef NGUI_TOOL_SELECTION_H
#define NGUI_TOOL_SELECTION_H

// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include "selection.h"
#include "utility.h"
#include "viewport.h"

#include <k3dsdk/geometry.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/iselectable.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/property.h>
#include <k3dsdk/transform.h>

namespace libk3dngui
{

/////////////////////////////////////////////////////////////////////////////
// tool_selection

class tool_selection
{
public:
	tool_selection(k3d::idocument& Document, document_state& DocumentState) :
		m_document(Document),
		m_document_state(DocumentState),
		m_current_motion(MOTION_NONE),
		m_box_selection(0, 0, 0, 0)
	{
	}

	k3d::idocument& document()
	{
		return m_document;
	}

protected:
	/// Enumerates motions
	typedef enum
	{
		MOTION_NONE,
		MOTION_CLICK_DRAG,	// LMB click + drag
		MOTION_DRAG,		// LMB down + drag
		MOTION_BOX_SELECT,	// LMB down + drag draws selection window
		MOTION_PAINT_SELECT
	} motion_t;

	/// Enumerates selection types
	typedef enum
	{
		SELECTION_ADD,
		SELECTION_SUBTRACT,
		SELECTED_OBJECT,
		DESELECTED_OBJECT,
		NOTHING
	} mouse_down_content_t;

	/// Stores the owning document
	k3d::idocument& m_document;
	document_state& m_document_state;

	/// Saves current selection motion
	motion_t m_current_motion;
	/// Saves mouse down action
	mouse_down_content_t m_mouse_down_content;
	/// Stores the current selection window
	k3d::rectangle m_box_selection;
	/// Saves clicked selection
	k3d::selection::record m_mouse_down_selection;
	/// Save action name for tutorials
	std::string m_tutorial_action;

	void redraw_all()
	{
		k3d::gl::redraw_all(m_document, k3d::gl::irender_engine::ASYNCHRONOUS);
	}

	void pick_selectables(k3d::selection::records& PickedSelectables, viewport::control& Viewport, const k3d::vector2& Coordinates)
	{
		// Find what's under the mouse pointer
		m_mouse_down_selection = Viewport.pick_object(Coordinates, PickedSelectables);

		// Nullify selection in component mode if the parent node isn't selected
		/** \todo Don't select components from deselected objects */
		if(k3d::inode* node = k3d::selection::get_node(m_mouse_down_selection))
			{
				switch(m_document_state.selection_mode().value())
					{
						case SELECT_POINTS:
						case SELECT_LINES:
						case SELECT_FACES:
							if(!m_document_state.is_selected(node))
								m_mouse_down_selection = k3d::selection::record::empty_record();;
						break;
						case SELECT_NODES:
						break;
						default:
							assert_not_reached();
						break;
					}
			}
	}

	const k3d::rectangle normalize(const k3d::rectangle& Rectangle)
	{
		return k3d::rectangle(
			std::min(Rectangle.left, Rectangle.right),
			std::max(Rectangle.left, Rectangle.right),
			std::min(Rectangle.top, Rectangle.bottom),
			std::max(Rectangle.top, Rectangle.bottom));
	}

	/// Stores a GC for drawing selections
	Glib::RefPtr<Gdk::GC> m_selection_gc;

	/// Returns a GC for drawing selections
	Glib::RefPtr<Gdk::GC> selection_gc(viewport::control& Viewport)
	{
		if(!m_selection_gc)
		{
			Gdk::Color selection_color = convert(k3d::color(0.2, 1.0, 1.0));
			Viewport.get_default_colormap()->alloc_color(selection_color);
			m_selection_gc = Gdk::GC::create(Viewport.get_window());
			m_selection_gc->set_foreground(selection_color);
			m_selection_gc->set_function(Gdk::XOR);
			m_selection_gc->set_line_attributes(1, Gdk::LINE_ON_OFF_DASH, Gdk::CAP_BUTT, Gdk::JOIN_MITER);
		}

		return m_selection_gc;
	}

	/// Draws selection window
	void draw_rubber_band(viewport::control& Viewport)
	{
		Glib::RefPtr<Gdk::Drawable> Drawable = Viewport.get_window();
		Glib::RefPtr<Gdk::GC> GC = selection_gc(Viewport);

		const k3d::rectangle rectangle = normalize(m_box_selection);
		Drawable->draw_rectangle(GC, false, static_cast<int>(rectangle.left), static_cast<int>(rectangle.top), static_cast<int>(rectangle.width()), static_cast<int>(rectangle.height()));
	}

	void on_box_select_motion(viewport::control& Viewport, const k3d::vector2& Coordinates)
	{
		draw_rubber_band(Viewport);
		m_box_selection.right = Coordinates[0];
		m_box_selection.bottom = Coordinates[1];
		draw_rubber_band(Viewport);
	}

	void on_box_select_objects(viewport::control& Viewport, const k3d::vector2& Coordinates, const k3d::rectangle& SelectionRegion)
	{
		const k3d::selection::records selection = Viewport.get_selectable_objects(normalize(SelectionRegion));

		switch(m_mouse_down_content)
		{
			//case MOTION_RUBBER_BAND_SELECT:
			case SELECTION_ADD:
				{
					m_document_state.select(selection);
				}
				break;

			//case MOTION_RUBBER_BAND_DESELECT:
			case SELECTION_SUBTRACT:
				{
					m_document_state.deselect(selection);
				}
				break;
			case SELECTED_OBJECT:
			case DESELECTED_OBJECT:
			case NOTHING:
				{
					m_document_state.deselect_all();
					m_document_state.select(selection);
				}
				break;
			default:
				assert_not_reached();
		}
	}
};

} // namespace libk3dngui

#endif // !NGUI_TOOL_SELECTION_H

